extern crate alloc;

use alloc::vec::Vec;
use alloc::boxed::Box;
use alloc::format;
use core::cell::UnsafeCell;

use ross_eeprom::DeviceInfo;
use ross_logger::{log_debug, log_error};

use crate::module::config_module::filter::Filter;
use crate::module::config_module::extractor::Extractor;
use crate::module::config_module::producer::Producer;
use crate::helper::cell_helper::get_from_cell;
use crate::helper::type_helper::{UnsafeLogger, UnsafeConfigModule, UnsafeCanProtocol};

pub mod filter;
pub mod extractor;
pub mod producer;

pub struct ConfigModule {
}

impl<'a> ConfigModule {
    pub fn new() -> UnsafeConfigModule {
        UnsafeCell::new(ConfigModule {})
    }

    pub fn init(_module: &'a UnsafeConfigModule, logger: &'a UnsafeLogger) {
        log_debug!(logger, "Config module initialized.");
    }

    pub fn add_event_processor(&mut self, filters: Vec<Box<dyn Filter>>, extractor: Box<dyn Extractor>, producer: Box<dyn Producer>, protocol: &'a UnsafeCanProtocol<'a>, logger: &'a UnsafeLogger, device_info: &'a DeviceInfo) {
        get_from_cell(protocol).add_packet_handler(Box::new(move |packet, can| {
            let mut filters_passed = true;

            for filter in filters.iter() {
                if !filter.filter(packet) {
                    filters_passed = false;
                }
            }

            if !filters_passed {
                return;
            }

            let value = extractor.extract(packet);
            let packet = producer.produce(&value, device_info);

            if let Err(err) = get_from_cell(protocol).send_packet(&packet) {
                log_error!(
                    logger,
                    "Failed to send packet ({:?}).",
                    err
                );
            } else {
                log_debug!(logger, "Sent packet ({:?}).", packet);
            }
        }), true).unwrap();
    }

    pub fn tick(&mut self) {}
}
