extern crate alloc;

use core::convert::Infallible;
use alloc::boxed::Box;
use alloc::format;
use alloc::collections::BTreeMap;
use debouncr::{debounce_16, Edge, Debouncer, Repeat16};
use core::cell::UnsafeCell;
use ross_logger::{log_debug, log_error};
use embedded_hal::digital::v2::InputPin;

use ross_eeprom::DeviceInfo;
use ross_protocol::convert_packet::ConvertPacket;
use ross_protocol::event::internal_event::{InternalButtonPressedEvent, InternalButtonReleasedEvent};

use crate::helper::cell_helper::get_from_cell;
use crate::helper::type_helper::{UnsafeLogger, UnsafeButtonModule, UnsafeCanProtocol};

pub struct ButtonModule {
    buttons: BTreeMap<u8, (Box<dyn InputPin<Error = Infallible>>, Debouncer<u16, Repeat16>)>,
}

impl ButtonModule {
    pub fn new() -> UnsafeButtonModule {
        UnsafeCell::new(ButtonModule {
            buttons: BTreeMap::new(),
        })
    }

    pub fn init<'a>(_module: &'a UnsafeButtonModule, logger: &'a UnsafeLogger) {
        log_debug!(logger, "Button module initialized.");
    }

    pub fn add_button(&mut self, index: u8, input: Box<dyn InputPin<Error = Infallible>>, logger: &UnsafeLogger) {
        let debouncer = debounce_16(input.is_high().unwrap());

        log_debug!(logger, "Button ({}) has been added to the button module.", index);

        self.buttons.insert(index, (input, debouncer));
    }

    pub fn tick(&mut self, logger: &UnsafeLogger, protocol: &UnsafeCanProtocol, device_info: &DeviceInfo) {
        for button in self.buttons.iter_mut() {
            let transition = button.1.1.update(button.1.0.is_high().unwrap());
            match transition {
                Some(Edge::Rising) => {
                    log_debug!(logger, "Button ({}) has been pressed.", button.0);

                    let internal_button_pressed_event = InternalButtonPressedEvent {
                        device_address: device_info.device_address,
                        index: *button.0,
                    };

                    if let Err(err) = get_from_cell(protocol).send_packet(&internal_button_pressed_event.to_packet()) {
                        log_error!(logger, "Failed to send `internal_button_pressed_event` ({:?}).", err);
                    } else {
                        log_debug!(logger, "Sent `internal_button_pressed_event` ({:?}).", internal_button_pressed_event);
                    }
                },
                Some(Edge::Falling) => {
                    log_debug!(logger, "Button ({}) has been released.", button.0);

                    let internal_button_released_event = InternalButtonReleasedEvent {
                        device_address: device_info.device_address,
                        index: *button.0,
                    };

                    if let Err(err) = get_from_cell(protocol).send_packet(&internal_button_released_event.to_packet()) {
                        log_error!(logger, "Failed to send `internal_button_released_event` ({:?}).", err);
                    } else {
                        log_debug!(logger, "Sent `internal_button_released_event` ({:?}).", internal_button_released_event);
                    }
                },
                None => {},
            };
        }
    }
}
