#![no_std]
#![no_main]
#![feature(alloc_error_handler)]

extern crate alloc;

use panic_itm as _;

use alloc::format;
use alloc_cortex_m::CortexMHeap;
use bxcan::filter::Mask32;
use bxcan::Can as BxCan;
use core::alloc::Layout;
use core::cell::UnsafeCell;
use cortex_m_rt::entry;
use eeprom24x::{Eeprom24x, SlaveAddr};
use nb::block;
use stm32f1xx_hal_bxcan::can::Can as HalCan;
use stm32f1xx_hal_bxcan::i2c::{BlockingI2c, Mode};
use stm32f1xx_hal_bxcan::pac::{CorePeripherals, Peripherals};
use stm32f1xx_hal_bxcan::prelude::*;
use stm32f1xx_hal_bxcan::serial::{Config as SerialConfig, Serial};

use ross_eeprom::Eeprom;
use ross_logger::{log_debug, log_warning, LogLevel, Logger};
use ross_protocol::interface::can::Can;
use ross_protocol::interface::usart::Usart;
use ross_protocol::protocol::Protocol;
use ross_firmware::config::*;
use ross_firmware::helper::can_helper::calc_can_btr;
use ross_firmware::helper::cell_helper::get_from_cell;
use ross_firmware::helper::heap_helper::allocate_heap;
use ross_firmware::module::programmer_module::ProgrammerModule;

#[global_allocator]
static ALLOCATOR: CortexMHeap = CortexMHeap::empty();

#[entry]
fn main() -> ! {
    let dp = Peripherals::take().unwrap();
    let cp = CorePeripherals::take().unwrap();

    let mut flash = dp.FLASH.constrain();
    let mut rcc = dp.RCC.constrain();

    let clocks = rcc
        .cfgr
        .use_hse(8.mhz())
        .sysclk(72.mhz())
        .hclk(72.mhz())
        .pclk1(36.mhz())
        .pclk2(72.mhz())
        .freeze(&mut flash.acr);

    let config = Config {
        heap_config: Some(HeapConfig { size: 4096 }),
        eeprom_config: Some(EepromConfig { bitrate: 400_000 }),
        can_config: Some(CanConfig {
            bitrate: 50_000,
            tseg1: 13,
            tseg2: 2,
            sjw: 1,
        }),
        usart_config: Some(UsartConfig { baudrate: 115_200 }),
    };

    allocate_heap(&config.unwrap_heap_config(), &ALLOCATOR);

    let logger = UnsafeCell::new(Logger::new(LogLevel::Debug, cp.ITM));

    log_debug!(logger, "Firmware initialized.");

    let mut gpioa = dp.GPIOA.split(&mut rcc.apb2);

    let mut afio = dp.AFIO.constrain(&mut rcc.apb2);
    let mut gpiob = dp.GPIOB.split(&mut rcc.apb2);

    let mut eeprom = {
        let i2c1 = {
            let scl = gpiob.pb6.into_alternate_open_drain(&mut gpiob.crl);
            let sda = gpiob.pb7.into_alternate_open_drain(&mut gpiob.crl);
            // TODO: put better values in thge last 4 arguments
            BlockingI2c::i2c1(
                dp.I2C1,
                (scl, sda),
                &mut afio.mapr,
                Mode::standard(config.unwrap_eeprom_config().bitrate.hz()),
                clocks,
                &mut rcc.apb1,
                10,
                10,
                10,
                10,
            )
        };

        let eeprom = Eeprom24x::new_24x02(i2c1, SlaveAddr::Alternative(false, false, false));

        Eeprom::new(eeprom, 0)
    };

    let device_info = eeprom.read_device_info().unwrap();

    log_debug!(
        logger,
        "Loaded device information from EEPROM ({:?}).",
        device_info
    );

    let can_protocol = {
        let mut can1 = {
            let can = HalCan::new(dp.CAN1, &mut rcc.apb1, dp.USB);

            let rx = gpioa.pa11.into_floating_input(&mut gpioa.crh);
            let tx = gpioa.pa12.into_alternate_push_pull(&mut gpioa.crh);
            can.assign_pins((tx, rx), &mut afio.mapr);

            BxCan::new(can)
        };

        can1.configure(|c| {
            c.set_bit_timing(calc_can_btr(&config.unwrap_can_config(), clocks.pclk1().0));
            c.set_loopback(false);
            c.set_silent(false);
        });

        let mut filters = can1.modify_filters();
        filters.enable_bank(0, Mask32::accept_all());
        drop(filters);

        block!(can1.enable()).unwrap();

        let can = Can::new(can1);
        UnsafeCell::new(Protocol::new(device_info.device_address, can))
    };

    let usart_protocol = {
        let pin_tx = gpioa.pa9.into_alternate_push_pull(&mut gpioa.crh);
        let pin_rx = gpioa.pa10;

        let serial = Serial::usart1(
            dp.USART1,
            (pin_tx, pin_rx),
            &mut afio.mapr,
            SerialConfig::default().baudrate(config.unwrap_usart_config().baudrate.bps()),
            clocks,
            &mut rcc.apb2,
        );

        let usart = Usart::new(serial);
        UnsafeCell::new(Protocol::new(device_info.device_address, usart))
    };

    let programmer_module = ProgrammerModule::new();
    ProgrammerModule::init(&programmer_module, &can_protocol, &usart_protocol, &logger, &device_info);

    loop {
        if let Err(err) = get_from_cell(&can_protocol).tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        if let Err(err) = get_from_cell(&usart_protocol).tick() {
            log_warning!(logger, "Unexpected error occurred ({:?}).", err);
        }

        get_from_cell(&programmer_module).tick();
    }
}

#[alloc_error_handler]
fn oom(_: Layout) -> ! {
    loop {}
}
