#![no_std]
#![no_main]
#![feature(alloc_error_handler)]

use panic_itm as _;

use alloc_cortex_m::CortexMHeap;
use bxcan::filter::Mask32;
use bxcan::*;
use core::alloc::Layout;
use cortex_m::iprint;
use cortex_m_rt::entry;
use eeprom24x::{Eeprom24x, SlaveAddr};
use embedded_hal::serial::{Read, Write};
use nb::block;
use stm32f1xx_hal_bxcan::can::Can;
use stm32f1xx_hal_bxcan::i2c::{BlockingI2c, Mode};
use stm32f1xx_hal_bxcan::pac::{CorePeripherals, Peripherals, ITM};
use stm32f1xx_hal_bxcan::prelude::*;
use stm32f1xx_hal_bxcan::serial::{Config as SerialConfig, Serial};
use stm32f1xx_hal_bxcan::time::MonoTimer;

use ross_eeprom::{RossDeviceInfo, RossEeprom};
use ross_protocol::ross_convert_packet::*;
use ross_protocol::ross_event::ross_configurator_event::*;
use ross_protocol::ross_event::ross_programmer_event::RossProgrammerHelloEvent;
use ross_protocol::ross_interface::ross_can::{RossCan, RossCanError};
use ross_protocol::ross_interface::ross_usart::{RossUsart, RossUsartError};
use ross_protocol::ross_packet::RossPacket;

const EEPROM_BITRATE: u32 = 400_000;

const CAN_BITRATE: u32 = 50_000;
const CAN_TSEG1: u32 = 13;
const CAN_TSEG2: u32 = 2;
const CAN_SJW: u32 = 1;
const CAN_TIMEOUT_MS: u32 = 500;

const SERIAL_BAUDRATE: u32 = 115_200;

const DEBUG: bool = true;
const HEAP_SIZE: usize = 4096;

#[global_allocator]
static ALLOCATOR: CortexMHeap = CortexMHeap::empty();

static mut ITM_PERIPHERAL: Option<ITM> = None;

macro_rules! debug {
    ($fmt:expr) => {
        if DEBUG {
            iprint!(&mut unsafe { ITM_PERIPHERAL.as_mut().unwrap() }.stim[0], concat!($fmt, "\r\n"));
        }
    };
    ($fmt:expr, $($arg:tt)*) => {
        if DEBUG {
            iprint!(&mut unsafe { ITM_PERIPHERAL.as_mut().unwrap() }.stim[0], concat!($fmt, "\r\n"), $($arg)*);
        }
    };
}

#[entry]
fn main() -> ! {
    let dp = Peripherals::take().unwrap();
    let cp = CorePeripherals::take().unwrap();

    let mut flash = dp.FLASH.constrain();
    let mut rcc = dp.RCC.constrain();

    let clocks = rcc
        .cfgr
        .use_hse(8.mhz())
        .sysclk(72.mhz())
        .hclk(72.mhz())
        .pclk1(36.mhz())
        .pclk2(72.mhz())
        .freeze(&mut flash.acr);

    unsafe {
        ITM_PERIPHERAL = Some(cp.ITM);
    }

    debug!("Firmware initialized.");

    let mut gpioa = dp.GPIOA.split(&mut rcc.apb2);

    let mut afio = dp.AFIO.constrain(&mut rcc.apb2);
    let mut gpiob = dp.GPIOB.split(&mut rcc.apb2);
    let timer = MonoTimer::new(cp.DWT, cp.DCB, clocks);

    let mut eeprom = {
        let i2c1 = {
            let scl = gpiob.pb6.into_alternate_open_drain(&mut gpiob.crl);
            let sda = gpiob.pb7.into_alternate_open_drain(&mut gpiob.crl);
            // TODO: put better values in thge last 4 arguments
            BlockingI2c::i2c1(
                dp.I2C1,
                (scl, sda),
                &mut afio.mapr,
                Mode::standard(EEPROM_BITRATE.hz()),
                clocks,
                &mut rcc.apb1,
                10,
                10,
                10,
                10,
            )
        };

        let eeprom = Eeprom24x::new_24x02(i2c1, SlaveAddr::Alternative(false, false, false));

        RossEeprom::new(eeprom, 0)
    };

    let device_info = eeprom.read_device_info().unwrap();

    let mut can = {
        let mut can1 = {
            let can = Can::new(dp.CAN1, &mut rcc.apb1, dp.USB);

            let rx = gpioa.pa11.into_floating_input(&mut gpioa.crh);
            let tx = gpioa.pa12.into_alternate_push_pull(&mut gpioa.crh);
            can.assign_pins((tx, rx), &mut afio.mapr);

            bxcan::Can::new(can)
        };

        can1.configure(|c| {
            c.set_bit_timing(calc_can_btr(clocks.pclk1().0));
            c.set_loopback(false);
            c.set_silent(false);
        });

        let mut filters = can1.modify_filters();
        filters.enable_bank(0, Mask32::accept_all());
        drop(filters);

        block!(can1.enable()).unwrap();

        RossCan::new(can1)
    };

    let mut usart = {
        let pin_tx = gpioa.pa9.into_alternate_push_pull(&mut gpioa.crh);
        let pin_rx = gpioa.pa10;

        let serial = Serial::usart1(
            dp.USART1,
            (pin_tx, pin_rx),
            &mut afio.mapr,
            SerialConfig::default().baudrate(SERIAL_BAUDRATE.bps()),
            clocks,
            &mut rcc.apb2,
        );

        RossUsart::new(serial)
    };

    allocate_heap();

    loop {
        let usart_packet = get_usart_packet(&device_info, &mut usart);

        let can_packet = transmit_can_packet(&usart_packet, &mut can, &timer);

        if let Ok(packet) = can_packet {
            transmit_usart_packet(&packet, &mut usart);
        }
    }
}

fn get_usart_packet<S: Read<u8> + Write<u8>>(
    device_info: &RossDeviceInfo,
    usart: &mut RossUsart<S>,
) -> RossPacket {
    loop {
        match usart.try_get_packet() {
            Ok(packet) => match RossConfiguratorHelloEvent::try_from_packet(&packet) {
                Ok(_) => {
                    let programmer_hello_event = RossProgrammerHelloEvent {
                        programmer_address: device_info.device_address,
                        firmware_version: device_info.firmware_version,
                    };

                    transmit_usart_packet(&programmer_hello_event.to_packet(), usart)
                }
                Err(_) => {
                    return packet;
                }
            },
            Err(err) => {
                if let RossUsartError::NoPacketReceived = err {
                    continue;
                }

                debug!("Failed to get packet ({:?}).", err);
            }
        };
    }
}

fn transmit_can_packet<I: Instance>(
    packet: &RossPacket,
    can: &mut RossCan<I>,
    timer: &MonoTimer,
) -> Result<RossPacket, ()> {
    loop {
        if let Err(err) = can.try_send_packet(packet) {
            debug!("Failed to send can packet ({:?}).", err);
        } else {
            debug!("Sent can packet ({:?}).", packet);
        }

        let start_time = timer.now();

        loop {
            if start_time.elapsed() / (timer.frequency().0 / 1000) >= CAN_TIMEOUT_MS {
                debug!("No acknowledgement received before timeout.");
                break;
            }

            match can.try_get_packet() {
                Ok(packet) => return Ok(packet),
                Err(err) => {
                    if let RossCanError::NoPacketReceived = err {
                        continue;
                    }

                    debug!("Failed to get packet ({:?}).", err);
                    continue;
                }
            };
        }

        return Err(());
    }
}

fn transmit_usart_packet<S: Read<u8> + Write<u8>>(packet: &RossPacket, usart: &mut RossUsart<S>) {
    if let Err(err) = usart.try_send_packet(packet) {
        debug!("Failed to send usart packet ({:?}).", err);
    } else {
        debug!("Sent usart packet ({:?}).", packet);
    }
}

const fn calc_can_btr(clock_rate: u32) -> u32 {
    let brp = clock_rate / CAN_BITRATE / (CAN_TSEG1 + CAN_TSEG2);

    (brp - 1) | ((CAN_TSEG1 - 1) << 16) | ((CAN_TSEG2 - 1) << 20) | ((CAN_SJW - 1) << 24)
}

fn allocate_heap() {
    let start = cortex_m_rt::heap_start() as usize;
    unsafe { ALLOCATOR.init(start, HEAP_SIZE) }
}

#[alloc_error_handler]
fn oom(_: Layout) -> ! {
    loop {}
}
