extern crate alloc;

use alloc::boxed::Box;
use core::cell::RefCell;
use eeprom24x::addr_size::TwoBytes;
use eeprom24x::page_size::B32;
use stm32f1xx_hal_bxcan::delay::Delay;
use stm32f1xx_hal_bxcan::prelude::*;

use ross_eeprom::{Eeprom, EepromError};
use ross_protocol::event::general::DataEvent;
use ross_protocol::event::programmer::ProgrammerStartConfigUpgradeEvent;

use crate::helper::buffer_helper::BufferHelper;

const BUFFER_SIZE: usize = 1024;

#[derive(Debug)]
pub enum ConfigUpgraderError {
    UpgradeAlreadyStarted,
    UpgradeNotStarted,
    TooMuchData,
    EepromError(EepromError),
}

pub struct ConfigUpgrader<'a, I2C>
where
    I2C: _embedded_hal_blocking_i2c_WriteRead<Error = nb::Error<stm32f1xx_hal_bxcan::i2c::Error>>
        + _embedded_hal_blocking_i2c_Write<Error = nb::Error<stm32f1xx_hal_bxcan::i2c::Error>>,
{
    eeprom: &'a RefCell<Eeprom<I2C, B32, TwoBytes>>,
    buffer_helper: RefCell<BufferHelper<BUFFER_SIZE>>,
    waiting_for_data: bool,
    config_size: usize,
    eeprom_offset: RefCell<usize>,
}

impl<'a, I2C> ConfigUpgrader<'a, I2C>
where
    I2C: _embedded_hal_blocking_i2c_WriteRead<Error = nb::Error<stm32f1xx_hal_bxcan::i2c::Error>>
        + _embedded_hal_blocking_i2c_Write<Error = nb::Error<stm32f1xx_hal_bxcan::i2c::Error>>,
{
    pub fn new(eeprom: &'a RefCell<Eeprom<I2C, B32, TwoBytes>>) -> Self {
        Self {
            eeprom,
            buffer_helper: RefCell::new(BufferHelper::new()),
            waiting_for_data: false,
            config_size: 0,
            eeprom_offset: RefCell::new(0),
        }
    }

    pub fn start_upgrade(
        &mut self,
        event: &ProgrammerStartConfigUpgradeEvent,
    ) -> Result<(), ConfigUpgraderError> {
        if self.waiting_for_data {
            return Err(ConfigUpgraderError::UpgradeAlreadyStarted);
        }

        self.waiting_for_data = true;
        self.config_size = event.config_size as usize;
        *self.eeprom_offset.borrow_mut() = 0;

        Ok(())
    }

    pub fn handle_data_event(
        &mut self,
        event: &DataEvent,
        delay: &mut Delay,
    ) -> Result<bool, ConfigUpgraderError> {
        if !self.waiting_for_data {
            return Err(ConfigUpgraderError::UpgradeNotStarted);
        }

        let new_eeprom_offset = *self.eeprom_offset.borrow()
            + self.buffer_helper.borrow().get_buffer_offset()
            + event.data.len();

        if new_eeprom_offset > self.config_size {
            self.finish_upgrade()?;
            return Err(ConfigUpgraderError::TooMuchData);
        }

        let last_data_event = new_eeprom_offset == self.config_size;

        let mut eeprom_error = None;

        self.buffer_helper.borrow_mut().add_data(
            &event.data,
            last_data_event,
            Box::new(|buffer| {
                if let Err(err) = self.eeprom.borrow_mut().write_config_data(
                    &buffer.to_vec(),
                    *self.eeprom_offset.borrow() as u32,
                    delay,
                ) {
                    eeprom_error = Some(err);
                    return;
                }

                *self.eeprom_offset.borrow_mut() += buffer.len();
            }),
        );

        if let Some(err) = eeprom_error {
            self.finish_upgrade()?;
            return Err(ConfigUpgraderError::EepromError(err));
        }

        if last_data_event {
            self.finish_upgrade()?;
        }

        Ok(last_data_event)
    }

    fn finish_upgrade(&mut self) -> Result<(), ConfigUpgraderError> {
        if !self.waiting_for_data {
            return Err(ConfigUpgraderError::UpgradeNotStarted);
        }

        self.waiting_for_data = false;
        self.config_size = 0;
        *self.eeprom_offset.borrow_mut() = 0;

        Ok(())
    }
}
