extern crate alloc;

use alloc::boxed::Box;
use core::cell::RefCell;
use stm32f1xx_hal_bxcan::flash::{Error as FlashError, FlashWriter};

use ross_protocol::event::general_event::DataEvent;
use ross_protocol::event::programmer_event::ProgrammerStartUploadEvent;

use crate::helper::buffer_helper::BufferHelper;

const FLASH_OFFSET: usize = 0x0001_0000;
const FLASH_SECTOR_SIZE: usize = 1024;

#[derive(Debug)]
pub enum FirmwareUpgraderError {
    UpgradeAlreadyStarted,
    UpgradeNotStarted,
    TooMuchData,
    EraseError(FlashError),
    WriteError(FlashError),
}

pub struct FirmwareUpgrader<'a> {
    flash_writer: RefCell<FlashWriter<'a>>,
    buffer_helper: RefCell<BufferHelper<FLASH_SECTOR_SIZE>>,
    waiting_for_data: bool,
    firmware_size: usize,
    flash_offset: RefCell<usize>,
}

impl<'a> FirmwareUpgrader<'a> {
    pub fn new(flash_writer: FlashWriter<'a>) -> Self {
        Self {
            flash_writer: RefCell::new(flash_writer),
            buffer_helper: RefCell::new(BufferHelper::new()),
            waiting_for_data: false,
            firmware_size: 0,
            flash_offset: RefCell::new(0),
        }
    }

    pub fn start_upgrade(
        &mut self,
        event: &ProgrammerStartUploadEvent,
    ) -> Result<(), FirmwareUpgraderError> {
        if self.waiting_for_data {
            return Err(FirmwareUpgraderError::UpgradeAlreadyStarted);
        }

        self.waiting_for_data = true;
        self.firmware_size = event.firmware_size as usize;
        *self.flash_offset.borrow_mut() = FLASH_OFFSET;

        Ok(())
    }

    pub fn handle_data_event(&mut self, event: &DataEvent) -> Result<bool, FirmwareUpgraderError> {
        if !self.waiting_for_data {
            return Err(FirmwareUpgraderError::UpgradeNotStarted);
        }

        if *self.flash_offset.borrow() + event.data.len() > FLASH_OFFSET + self.firmware_size {
            return Err(FirmwareUpgraderError::TooMuchData);
        }

        let last_data_event = *self.flash_offset.borrow()
            + self.buffer_helper.borrow().get_buffer_offset()
            + event.data.len()
            == FLASH_OFFSET + self.firmware_size;

        let mut erase_error = None;
        let mut write_error = None;

        self.buffer_helper.borrow_mut().add_data(
            &event.data,
            last_data_event,
            Box::new(|buffer| {
                if let Err(err) = self
                    .flash_writer
                    .borrow_mut()
                    .erase(*self.flash_offset.borrow() as u32, buffer.len())
                {
                    erase_error = Some(err);
                    return;
                }

                if let Err(err) = self
                    .flash_writer
                    .borrow_mut()
                    .write(*self.flash_offset.borrow() as u32, buffer)
                {
                    write_error = Some(err);
                    return;
                }

                *self.flash_offset.borrow_mut() += buffer.len();
            }),
        );

        if let Some(err) = erase_error {
            self.finish_upgrade()?;
            return Err(FirmwareUpgraderError::EraseError(err));
        }

        if let Some(err) = write_error {
            self.finish_upgrade()?;
            return Err(FirmwareUpgraderError::WriteError(err));
        }

        if last_data_event {
            self.finish_upgrade()?;
        }

        Ok(last_data_event)
    }

    fn finish_upgrade(&mut self) -> Result<(), FirmwareUpgraderError> {
        if !self.waiting_for_data {
            return Err(FirmwareUpgraderError::UpgradeNotStarted);
        }

        self.waiting_for_data = false;
        self.firmware_size = 0;
        *self.flash_offset.borrow_mut() = 0;

        Ok(())
    }
}
