//! This module defines the instance snapshot subsystem of Rojo.
//!
//! Snapshots define a way to define the instance tree of a project as a pure
//! function of the filesystem by providing a lightweight instance 'snapshot'
//! type, a method to generate minimal patches, and a method that applies those
//! patches.
//!
//! Changes in Rojo go through a pipeline of transformations once they hit the
//! snapshot subsystem.
//!
//! 1. Instance snapshots are generated by Rojo's snapshot middleware,
//!    representing a set of instances and their metadata. These will
//!    usually contain data that hasn't actually changed, and how coarsely
//!    the snapshots happen is defined outside this part of the code.
//!
//!    See `src/snapshot_middleware` for implementation.
//!
//! 2. Input snapshots are turned into `PatchSet` objects by Rojo's diffing
//!    algorithm via `compute_patch_set`. This operation doesn't mutate the
//!    instance tree, so work at this point can be thrown away.
//!
//!    `PatchSet` prescribe what changes need to be applied to the current
//!    tree to get the next tree. It isn't useful for describing those
//!    changes after the fact, since Rojo needs to assign IDs to created
//!    instances, which happens during patch application, not generation!
//!
//!    See `src/snapshot/patch_compute.rs` for implementation.
//!
//! 3. Patch sets are applied to the tree with `apply_patch_set`, which
//!    mutates the relevant instances. `apply_patch_set` returns a new
//!    object, `AppliedPatchSet`. Applied patch sets describe the transform
//!    that was applied, and are suitable for cases where another tree needs
//!    to be synchronized with Rojo's, like the Rojo Studio plugin.
//!
//!    See `src/snapshot/patch_apply.rs` for implementation.
//!
//! The aim with this approach is to reduce the number of bugs that arise from
//! attempting to manually update instances in response to filesystem updates.
//! Instead of surgically identifying what needs to change, we can do rough
//! "damage-painting", running our relatively fast snapshot function over
//! anything that could have changed and running it through a diffing function
//! to minimize the set of real changes.
//!
//! Building out a snapshot reconciler is mostly overkill for scripts, since
//! their relationships are mostly simple and well-defined. It becomes very
//! important, however, when dealing with large opaque model files and
//! user-defined plugins.

#![allow(dead_code)]

mod instance_snapshot;
mod metadata;
mod patch;
mod patch_apply;
mod patch_compute;
mod tree;

pub use instance_snapshot::InstanceSnapshot;
pub use metadata::*;
pub use patch::*;
pub use patch_apply::apply_patch_set;
pub use patch_compute::compute_patch_set;
pub use tree::*;

#[cfg(test)]
mod tests;
