use std::fs;
use std::path::PathBuf;
use std::time::Duration;

use libxm::*;
use rodio::Source;

pub const BUFFER_SIZE: usize = 2048;

pub struct XMSource {
    xm_context: XMContext,
    buffer: [f32; BUFFER_SIZE],
    buffer_index: usize,
    sample_rate: u32,
}

impl XMSource {
    pub fn new(xm_context: XMContext, sample_rate: u32) -> Self {
        XMSource {
            xm_context,
            buffer: [0.0; BUFFER_SIZE],
            buffer_index: 0,
            sample_rate
        }
    }

    pub fn from_bytes(bytes: &[u8], sample_rate: u32) -> Self {
        let xm = libxm::XMContext::new(bytes, sample_rate)
            .expect("failed to parse xm");

        XMSource::new(xm, sample_rate)
    }

    pub fn from_file(path: PathBuf, sample_rate: u32) -> Self {
        let file = fs::read(path).expect("couldn't read file");
        let xm = libxm::XMContext::new(&file, sample_rate)
            .expect("failed to parse xm");

        XMSource::new(xm, sample_rate)
    }
}

impl Source for XMSource {
    fn current_frame_len(&self) -> Option<usize> {
        Some(BUFFER_SIZE - self.buffer_index)
    }

    fn channels(&self) -> u16 { 2 }

    fn sample_rate(&self) -> u32 { self.sample_rate }

    fn total_duration(&self) -> Option<Duration> { None }
}

impl Iterator for XMSource {
    type Item = f32;

    fn next(&mut self) -> Option<Self::Item> {
        self.buffer_index += 1;

        if self.buffer_index >= BUFFER_SIZE {
            self.xm_context.generate_samples(&mut self.buffer);
            self.buffer_index = 0;
        }

        Some(self.buffer[self.buffer_index])
    }
}
