extern crate roaring;
mod iter;
use roaring::RoaringTreemap;

use iter::outside_in;
use proptest::arbitrary::any;
use proptest::collection::btree_set;
use proptest::proptest;
use std::iter::FromIterator;

#[test]
fn array() {
    let original = (0..2000).collect::<RoaringTreemap>();
    let clone = RoaringTreemap::from_iter(&original);
    let clone2 = RoaringTreemap::from_iter(original.clone());

    assert_eq!(clone, original);
    assert_eq!(clone2, original);
}

#[test]
fn bitmap() {
    let original = (0..6000).collect::<RoaringTreemap>();
    let clone = RoaringTreemap::from_iter(&original);
    let clone2 = RoaringTreemap::from_iter(original.clone());

    assert_eq!(clone, original);
    assert_eq!(clone2, original);
}

#[test]
fn arrays() {
    let original = ((0..2000).chain(1_000_000..1_002_000).chain(2_000_000..2_001_000))
        .collect::<RoaringTreemap>();
    let clone = RoaringTreemap::from_iter(&original);
    let clone2 = RoaringTreemap::from_iter(original.clone());

    assert_eq!(clone, original);
    assert_eq!(clone2, original);
}

#[test]
fn bitmaps() {
    let original = ((0..6000).chain(1_000_000..1_012_000).chain(2_000_000..2_010_000))
        .collect::<RoaringTreemap>();
    let clone = RoaringTreemap::from_iter(&original);
    let clone2 = RoaringTreemap::from_iter(original.clone());

    assert_eq!(clone, original);
    assert_eq!(clone2, original);
}

#[test]
fn bitmaps_iterator() {
    let original = ((0..6000).chain(1_000_000..1_012_000).chain(2_000_000..2_010_000))
        .collect::<RoaringTreemap>();
    let clone = RoaringTreemap::from_bitmaps(original.bitmaps().map(|(p, b)| (p, b.clone())));
    let clone2 = original.bitmaps().map(|(p, b)| (p, b.clone())).collect::<RoaringTreemap>();

    assert_eq!(clone, original);
    assert_eq!(clone2, original);
}

proptest! {
    #[test]
    fn iter(values in btree_set(any::<u64>(), ..=10_000)) {
        let bitmap = RoaringTreemap::from_sorted_iter(values.iter().cloned()).unwrap();

        assert!(values.into_iter().eq(bitmap.into_iter()));
    }
}

#[test]
fn rev() {
    let values = (1..3)
        .chain(1_000_000..1_012_003)
        .chain(2_000_001..2_000_003)
        .chain(2_000_000_000_001..2_000_000_000_003);
    let bitmap = RoaringTreemap::from_iter(values.clone());

    assert!(values.into_iter().rev().eq(bitmap.iter().rev()));
}

proptest! {
    #[test]
    fn rev_iter(values in btree_set(any::<u64>(), ..=10_000)) {
        let bitmap = RoaringTreemap::from_sorted_iter(values.iter().cloned()).unwrap();

        assert!(values.into_iter().rev().eq(bitmap.iter().rev()));
    }
}

#[test]
fn interleaved() {
    let values = (1..3)
        .chain(1_000_000..1_012_003)
        .chain(2_000_001..2_000_003)
        .chain(2_000_000_000_001..2_000_000_000_003);
    let bitmap = RoaringTreemap::from_iter(values.clone());

    assert!(outside_in(values).eq(outside_in(bitmap)));
}

proptest! {
    #[test]
    fn interleaved_iter(values in btree_set(any::<u64>(), 50_000..=100_000)) {
        let bitmap = RoaringTreemap::from_sorted_iter(values.iter().cloned()).unwrap();

        assert!(outside_in(values).eq(outside_in(bitmap)));
    }
}
