extern crate rlu;

const ANSWER: f64 = 1.0;

fn main() {
    // Compute the matrix and right-hand-side vector that define
    // the linear system, Ax = b.
    let (n, rowind, colst, nz) = lhr01();

    // Set exact solution; then compute right-hand-side vector.
    let x0 = vec![ANSWER; n];

    let opts = rlu::Options {
        expand_ratio: 2.0,

        ..rlu::Options::default()
    };

    let mut b = mat_vec(n, &rowind, &colst, &nz, &x0);
    let mut rhs: Vec<&mut [f64]> = vec![&mut b];

    let lu = rlu::factor(n, &rowind, &colst, &nz, None, &opts).unwrap();

    rlu::solve(&lu, &mut rhs, false).unwrap();

    println!("resid = {}", residual(&b));
}

fn mat_vec(n: usize, rowind: &[usize], colst: &[usize], nz: &[f64], x: &[f64]) -> Vec<f64> {
    let mut y = vec![0.0; n];
    for j in 0..n {
        let start = colst[j];
        let end = colst[j + 1];

        for ii in start..end {
            let i = rowind[ii];
            y[i] += nz[ii] * x[j];
        }
    }
    y
}

fn residual(x: &[f64]) -> f64 {
    let mut norm = f64::NEG_INFINITY;
    for i in 0..x.len() {
        //let abs = (x[i] - b[i]).abs();
        let abs = (x[i] - ANSWER).abs();
        if abs > norm {
            norm = abs;
        }
    }
    norm
}

fn lhr01() -> (usize, Vec<usize>, Vec<usize>, Vec<f64>) {
    let n: usize = 1477;

    let rowind = vec![
        0, 62, 83, 84, 1, 63, 83, 84, 2, 64, 83, 84, 3, 65, 83, 84, 4, 66, 83, 84, 5, 67, 83, 84,
        6, 68, 83, 84, 7, 69, 83, 84, 8, 70, 83, 84, 9, 71, 83, 84, 10, 72, 83, 84, 11, 73, 83, 84,
        12, 74, 83, 84, 13, 75, 83, 84, 14, 76, 83, 84, 15, 77, 83, 84, 16, 78, 83, 84, 17, 79, 83,
        84, 18, 80, 83, 84, 19, 83, 84, 61, 83, 20, 84, 21, 84, 87, 84, 88, 62, 81, 82, 90, 63, 81,
        82, 91, 64, 81, 82, 92, 65, 81, 82, 93, 66, 81, 82, 94, 67, 81, 82, 95, 68, 81, 82, 96, 69,
        81, 82, 97, 70, 81, 82, 98, 71, 81, 82, 99, 72, 81, 82, 100, 73, 81, 82, 101, 74, 81, 82,
        102, 75, 81, 82, 103, 76, 81, 82, 104, 77, 81, 82, 105, 78, 81, 82, 106, 79, 81, 82, 107,
        80, 81, 82, 108, 81, 82, 61, 81, 89, 82, 82, 87, 111, 82, 88, 112, 90, 109, 110, 114, 91,
        109, 110, 115, 92, 109, 110, 116, 93, 109, 110, 117, 94, 109, 110, 118, 95, 109, 110, 119,
        96, 109, 110, 120, 97, 109, 110, 121, 98, 109, 110, 122, 99, 109, 110, 123, 100, 109, 110,
        124, 101, 109, 110, 125, 102, 109, 110, 126, 103, 109, 110, 127, 104, 109, 110, 128, 105,
        109, 110, 129, 106, 109, 110, 130, 107, 109, 110, 131, 108, 109, 110, 132, 109, 110, 89,
        109, 113, 22, 110, 110, 111, 135, 110, 112, 162, 114, 133, 134, 164, 115, 133, 134, 165,
        116, 133, 134, 166, 117, 133, 134, 167, 118, 133, 134, 168, 119, 133, 134, 169, 120, 133,
        134, 170, 121, 133, 134, 171, 122, 133, 134, 172, 123, 133, 134, 173, 124, 133, 134, 174,
        125, 133, 134, 175, 126, 133, 134, 176, 127, 133, 134, 177, 128, 133, 134, 178, 129, 133,
        134, 179, 130, 133, 134, 180, 131, 133, 134, 181, 132, 133, 134, 182, 133, 134, 113, 133,
        163, 134, 161, 184, 134, 135, 185, 134, 162, 186, 164, 183, 188, 165, 183, 189, 166, 183,
        190, 167, 183, 191, 168, 183, 192, 169, 183, 193, 170, 183, 194, 171, 183, 195, 172, 183,
        196, 173, 183, 197, 174, 183, 198, 175, 183, 199, 176, 183, 200, 177, 183, 201, 178, 183,
        202, 179, 183, 203, 180, 183, 204, 181, 183, 205, 182, 183, 206, 183, 163, 183, 187, 184,
        23, 185, 186, 211, 137, 156, 157, 188, 212, 213, 214, 216, 218, 220, 222, 224, 226, 228,
        230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 138, 156, 157, 189, 212, 214, 215,
        216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250,
        139, 156, 157, 190, 212, 214, 216, 217, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236,
        238, 240, 242, 244, 246, 248, 250, 140, 156, 157, 191, 212, 214, 216, 218, 219, 220, 222,
        224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 141, 156, 157, 192,
        212, 214, 216, 218, 220, 221, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244,
        246, 248, 250, 142, 156, 157, 193, 212, 214, 216, 218, 220, 222, 223, 224, 226, 228, 230,
        232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 143, 156, 157, 194, 212, 214, 216, 218,
        220, 222, 224, 225, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 144,
        156, 157, 195, 212, 214, 216, 218, 220, 222, 224, 226, 227, 228, 230, 232, 234, 236, 238,
        240, 242, 244, 246, 248, 250, 145, 156, 157, 196, 212, 214, 216, 218, 220, 222, 224, 226,
        228, 229, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 146, 156, 157, 197, 212,
        214, 216, 218, 220, 222, 224, 226, 228, 230, 231, 232, 234, 236, 238, 240, 242, 244, 246,
        248, 250, 147, 156, 157, 198, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 233,
        234, 236, 238, 240, 242, 244, 246, 248, 250, 148, 156, 157, 199, 212, 214, 216, 218, 220,
        222, 224, 226, 228, 230, 232, 234, 235, 236, 238, 240, 242, 244, 246, 248, 250, 149, 156,
        157, 200, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 237, 238, 240,
        242, 244, 246, 248, 250, 150, 156, 157, 201, 212, 214, 216, 218, 220, 222, 224, 226, 228,
        230, 232, 234, 236, 238, 239, 240, 242, 244, 246, 248, 250, 151, 156, 157, 202, 212, 214,
        216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 241, 242, 244, 246, 248,
        250, 152, 156, 157, 203, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236,
        238, 240, 242, 243, 244, 246, 248, 250, 153, 156, 157, 204, 212, 214, 216, 218, 220, 222,
        224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 245, 246, 248, 250, 154, 156, 157,
        205, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244,
        246, 247, 248, 250, 155, 156, 157, 206, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230,
        232, 234, 236, 238, 240, 242, 244, 246, 248, 249, 250, 156, 157, 212, 214, 216, 218, 220,
        222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 251, 136, 156,
        187, 213, 215, 217, 219, 221, 223, 225, 227, 229, 231, 233, 235, 237, 239, 241, 243, 245,
        247, 249, 251, 24, 157, 209, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234,
        236, 238, 240, 242, 244, 246, 248, 250, 25, 157, 160, 210, 212, 214, 216, 218, 220, 222,
        224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 157, 162, 211, 188,
        207, 208, 212, 213, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240,
        242, 244, 246, 248, 250, 253, 189, 207, 208, 212, 214, 215, 216, 218, 220, 222, 224, 226,
        228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 254, 190, 207, 208, 212, 214,
        216, 217, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248,
        250, 255, 191, 207, 208, 212, 214, 216, 218, 219, 220, 222, 224, 226, 228, 230, 232, 234,
        236, 238, 240, 242, 244, 246, 248, 250, 256, 192, 207, 208, 212, 214, 216, 218, 220, 221,
        222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 257, 193, 207,
        208, 212, 214, 216, 218, 220, 222, 223, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242,
        244, 246, 248, 250, 258, 194, 207, 208, 212, 214, 216, 218, 220, 222, 224, 225, 226, 228,
        230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 259, 195, 207, 208, 212, 214, 216,
        218, 220, 222, 224, 226, 227, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250,
        260, 196, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 229, 230, 232, 234, 236,
        238, 240, 242, 244, 246, 248, 250, 261, 197, 207, 208, 212, 214, 216, 218, 220, 222, 224,
        226, 228, 230, 231, 232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 262, 198, 207, 208,
        212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 233, 234, 236, 238, 240, 242, 244,
        246, 248, 250, 263, 199, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232,
        234, 235, 236, 238, 240, 242, 244, 246, 248, 250, 264, 200, 207, 208, 212, 214, 216, 218,
        220, 222, 224, 226, 228, 230, 232, 234, 236, 237, 238, 240, 242, 244, 246, 248, 250, 265,
        201, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 239,
        240, 242, 244, 246, 248, 250, 266, 202, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226,
        228, 230, 232, 234, 236, 238, 240, 241, 242, 244, 246, 248, 250, 267, 203, 207, 208, 212,
        214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 243, 244, 246,
        248, 250, 268, 204, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234,
        236, 238, 240, 242, 244, 245, 246, 248, 250, 269, 205, 207, 208, 212, 214, 216, 218, 220,
        222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242, 244, 246, 247, 248, 250, 270, 206,
        207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230, 232, 234, 236, 238, 240, 242,
        244, 246, 248, 249, 250, 271, 207, 208, 212, 214, 216, 218, 220, 222, 224, 226, 228, 230,
        232, 234, 236, 238, 240, 242, 244, 246, 248, 250, 251, 187, 207, 213, 215, 217, 219, 221,
        223, 225, 227, 229, 231, 233, 235, 237, 239, 241, 243, 245, 247, 249, 251, 252, 208, 209,
        208, 210, 274, 208, 211, 301, 253, 272, 273, 303, 254, 272, 273, 304, 255, 272, 273, 305,
        256, 272, 273, 306, 257, 272, 273, 307, 258, 272, 273, 308, 259, 272, 273, 309, 260, 272,
        273, 310, 261, 272, 273, 311, 262, 272, 273, 312, 263, 272, 273, 313, 264, 272, 273, 314,
        265, 272, 273, 315, 266, 272, 273, 316, 267, 272, 273, 317, 268, 272, 273, 318, 269, 272,
        273, 319, 270, 272, 273, 320, 271, 272, 273, 321, 272, 273, 252, 272, 302, 273, 300, 323,
        273, 274, 324, 273, 301, 325, 303, 322, 327, 304, 322, 328, 305, 322, 329, 306, 322, 330,
        307, 322, 331, 308, 322, 332, 309, 322, 333, 310, 322, 334, 311, 322, 335, 312, 322, 336,
        313, 322, 337, 314, 322, 338, 315, 322, 339, 316, 322, 340, 317, 322, 341, 318, 322, 342,
        319, 322, 343, 320, 322, 344, 321, 322, 345, 322, 302, 322, 326, 323, 26, 324, 325, 350,
        276, 295, 296, 327, 351, 352, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375,
        377, 379, 381, 383, 385, 387, 389, 277, 295, 296, 328, 351, 353, 354, 355, 357, 359, 361,
        363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 278, 295, 296, 329,
        351, 353, 355, 356, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383,
        385, 387, 389, 279, 295, 296, 330, 351, 353, 355, 357, 358, 359, 361, 363, 365, 367, 369,
        371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 280, 295, 296, 331, 351, 353, 355, 357,
        359, 360, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 281,
        295, 296, 332, 351, 353, 355, 357, 359, 361, 362, 363, 365, 367, 369, 371, 373, 375, 377,
        379, 381, 383, 385, 387, 389, 282, 295, 296, 333, 351, 353, 355, 357, 359, 361, 363, 364,
        365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 283, 295, 296, 334, 351,
        353, 355, 357, 359, 361, 363, 365, 366, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385,
        387, 389, 284, 295, 296, 335, 351, 353, 355, 357, 359, 361, 363, 365, 367, 368, 369, 371,
        373, 375, 377, 379, 381, 383, 385, 387, 389, 285, 295, 296, 336, 351, 353, 355, 357, 359,
        361, 363, 365, 367, 369, 370, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 286, 295,
        296, 337, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 372, 373, 375, 377, 379,
        381, 383, 385, 387, 389, 287, 295, 296, 338, 351, 353, 355, 357, 359, 361, 363, 365, 367,
        369, 371, 373, 374, 375, 377, 379, 381, 383, 385, 387, 389, 288, 295, 296, 339, 351, 353,
        355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 376, 377, 379, 381, 383, 385, 387,
        389, 289, 295, 296, 340, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375,
        377, 378, 379, 381, 383, 385, 387, 389, 290, 295, 296, 341, 351, 353, 355, 357, 359, 361,
        363, 365, 367, 369, 371, 373, 375, 377, 379, 380, 381, 383, 385, 387, 389, 291, 295, 296,
        342, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 382,
        383, 385, 387, 389, 292, 295, 296, 343, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369,
        371, 373, 375, 377, 379, 381, 383, 384, 385, 387, 389, 293, 295, 296, 344, 351, 353, 355,
        357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 386, 387, 389,
        294, 295, 296, 345, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377,
        379, 381, 383, 385, 387, 388, 389, 295, 296, 351, 353, 355, 357, 359, 361, 363, 365, 367,
        369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 390, 275, 295, 326, 352, 354, 356,
        358, 360, 362, 364, 366, 368, 370, 372, 374, 376, 378, 380, 382, 384, 386, 388, 390, 27,
        296, 348, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381,
        383, 385, 387, 389, 28, 296, 299, 349, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369,
        371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 296, 301, 350, 327, 346, 347, 351, 352,
        353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387,
        389, 392, 328, 346, 347, 351, 353, 354, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373,
        375, 377, 379, 381, 383, 385, 387, 389, 393, 329, 346, 347, 351, 353, 355, 356, 357, 359,
        361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 394, 330, 346,
        347, 351, 353, 355, 357, 358, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381,
        383, 385, 387, 389, 395, 331, 346, 347, 351, 353, 355, 357, 359, 360, 361, 363, 365, 367,
        369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 396, 332, 346, 347, 351, 353, 355,
        357, 359, 361, 362, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389,
        397, 333, 346, 347, 351, 353, 355, 357, 359, 361, 363, 364, 365, 367, 369, 371, 373, 375,
        377, 379, 381, 383, 385, 387, 389, 398, 334, 346, 347, 351, 353, 355, 357, 359, 361, 363,
        365, 366, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 399, 335, 346, 347,
        351, 353, 355, 357, 359, 361, 363, 365, 367, 368, 369, 371, 373, 375, 377, 379, 381, 383,
        385, 387, 389, 400, 336, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 370,
        371, 373, 375, 377, 379, 381, 383, 385, 387, 389, 401, 337, 346, 347, 351, 353, 355, 357,
        359, 361, 363, 365, 367, 369, 371, 372, 373, 375, 377, 379, 381, 383, 385, 387, 389, 402,
        338, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 374, 375, 377,
        379, 381, 383, 385, 387, 389, 403, 339, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365,
        367, 369, 371, 373, 375, 376, 377, 379, 381, 383, 385, 387, 389, 404, 340, 346, 347, 351,
        353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 378, 379, 381, 383, 385,
        387, 389, 405, 341, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373,
        375, 377, 379, 380, 381, 383, 385, 387, 389, 406, 342, 346, 347, 351, 353, 355, 357, 359,
        361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 382, 383, 385, 387, 389, 407, 343,
        346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381,
        383, 384, 385, 387, 389, 408, 344, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367,
        369, 371, 373, 375, 377, 379, 381, 383, 385, 386, 387, 389, 409, 345, 346, 347, 351, 353,
        355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377, 379, 381, 383, 385, 387, 388,
        389, 410, 346, 347, 351, 353, 355, 357, 359, 361, 363, 365, 367, 369, 371, 373, 375, 377,
        379, 381, 383, 385, 387, 389, 390, 326, 346, 352, 354, 356, 358, 360, 362, 364, 366, 368,
        370, 372, 374, 376, 378, 380, 382, 384, 386, 388, 390, 391, 347, 348, 412, 347, 349, 413,
        347, 350, 414, 392, 411, 416, 393, 411, 417, 394, 411, 418, 395, 411, 419, 396, 411, 420,
        397, 411, 421, 398, 411, 422, 399, 411, 423, 400, 411, 424, 401, 411, 425, 402, 411, 426,
        403, 411, 427, 404, 411, 428, 405, 411, 429, 406, 411, 430, 407, 411, 431, 408, 411, 432,
        409, 411, 433, 410, 411, 434, 411, 391, 411, 415, 412, 29, 413, 414, 441, 137, 158, 159,
        1408, 138, 158, 159, 1409, 139, 158, 159, 1410, 140, 158, 159, 1411, 141, 158, 159, 1412,
        142, 158, 159, 1413, 143, 158, 159, 1414, 144, 158, 159, 1415, 145, 158, 159, 1416, 146,
        158, 159, 1417, 147, 158, 159, 1418, 148, 158, 159, 1419, 149, 158, 159, 1420, 150, 158,
        159, 1421, 151, 158, 159, 1422, 152, 158, 159, 1423, 153, 158, 159, 1424, 154, 158, 159,
        1425, 155, 158, 159, 1426, 158, 159, 136, 158, 1407, 159, 161, 159, 160, 1429, 159, 162,
        1430, 276, 297, 298, 1408, 277, 297, 298, 1409, 278, 297, 298, 1410, 279, 297, 298, 1411,
        280, 297, 298, 1412, 281, 297, 298, 1413, 282, 297, 298, 1414, 283, 297, 298, 1415, 284,
        297, 298, 1416, 285, 297, 298, 1417, 286, 297, 298, 1418, 287, 297, 298, 1419, 288, 297,
        298, 1420, 289, 297, 298, 1421, 290, 297, 298, 1422, 291, 297, 298, 1423, 292, 297, 298,
        1424, 293, 297, 298, 1425, 294, 297, 298, 1426, 297, 298, 275, 297, 1407, 298, 300, 298,
        299, 1429, 298, 301, 1430, 1408, 1427, 1428, 1453, 1409, 1427, 1428, 1454, 1410, 1427,
        1428, 1455, 1411, 1427, 1428, 1456, 1412, 1427, 1428, 1457, 1413, 1427, 1428, 1458, 1414,
        1427, 1428, 1459, 1415, 1427, 1428, 1460, 1416, 1427, 1428, 1461, 1417, 1427, 1428, 1462,
        1418, 1427, 1428, 1463, 1419, 1427, 1428, 1464, 1420, 1427, 1428, 1465, 1421, 1427, 1428,
        1466, 1422, 1427, 1428, 1467, 1423, 1427, 1428, 1468, 1424, 1427, 1428, 1469, 1425, 1427,
        1428, 1470, 1426, 1427, 1428, 1471, 1427, 1428, 1407, 1427, 1452, 1428, 1428, 1429, 1474,
        1428, 1430, 1476, 1453, 1472, 1473, 1454, 1472, 1473, 1455, 1472, 1473, 1456, 1472, 1473,
        1457, 1472, 1473, 1458, 1472, 1473, 1459, 1472, 1473, 1460, 1472, 1473, 1461, 1472, 1473,
        1462, 1472, 1473, 1463, 1472, 1473, 1464, 1472, 1473, 1465, 1472, 1473, 1466, 1472, 1473,
        1467, 1472, 1473, 1468, 1472, 1473, 1469, 1472, 1473, 1470, 1472, 1473, 1471, 1472, 1473,
        1472, 1473, 1452, 1472, 1473, 1475, 1473, 1474, 1473, 1476, 1270, 1289, 1290, 1294, 1295,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324,
        1326, 1328, 1330, 1332, 1335, 1356, 1271, 1289, 1290, 1294, 1296, 1297, 1298, 1300, 1302,
        1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1336, 1357, 1272, 1289, 1290, 1294, 1296, 1298, 1299, 1300, 1302, 1304, 1306, 1308, 1310,
        1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1337, 1358, 1273, 1289,
        1290, 1294, 1296, 1298, 1300, 1301, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318,
        1320, 1322, 1324, 1326, 1328, 1330, 1332, 1338, 1359, 1274, 1289, 1290, 1294, 1296, 1298,
        1300, 1302, 1303, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326,
        1328, 1330, 1332, 1339, 1360, 1275, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1305,
        1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1340,
        1361, 1276, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1307, 1308, 1310, 1312,
        1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1341, 1362, 1277, 1289, 1290,
        1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1309, 1310, 1312, 1314, 1316, 1318, 1320,
        1322, 1324, 1326, 1328, 1330, 1332, 1342, 1363, 1278, 1289, 1290, 1294, 1296, 1298, 1300,
        1302, 1304, 1306, 1308, 1310, 1311, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328,
        1330, 1332, 1343, 1364, 1279, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308,
        1310, 1312, 1313, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1344, 1365,
        1280, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1315,
        1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1345, 1366, 1281, 1289, 1290, 1294,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1317, 1318, 1320, 1322,
        1324, 1326, 1328, 1330, 1332, 1346, 1367, 1282, 1289, 1290, 1294, 1296, 1298, 1300, 1302,
        1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1319, 1320, 1322, 1324, 1326, 1328, 1330,
        1332, 1347, 1368, 1283, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310,
        1312, 1314, 1316, 1318, 1320, 1321, 1322, 1324, 1326, 1328, 1330, 1332, 1348, 1369, 1284,
        1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318,
        1320, 1322, 1323, 1324, 1326, 1328, 1330, 1332, 1349, 1370, 1285, 1289, 1290, 1294, 1296,
        1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1325,
        1326, 1328, 1330, 1332, 1350, 1371, 1286, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1327, 1328, 1330, 1332,
        1351, 1372, 1287, 1289, 1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312,
        1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1329, 1330, 1332, 1352, 1373, 1288, 1289,
        1290, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320,
        1322, 1324, 1326, 1328, 1330, 1331, 1332, 1353, 1374, 1289, 1290, 1294, 1296, 1298, 1300,
        1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330,
        1332, 1333, 1375, 1269, 1289, 1295, 1297, 1299, 1301, 1303, 1305, 1307, 1309, 1311, 1313,
        1315, 1317, 1319, 1321, 1323, 1325, 1327, 1329, 1331, 1333, 1334, 1290, 1291, 1379, 1380,
        1290, 1292, 1381, 1382, 1290, 1293, 1376, 1378, 1335, 1354, 1336, 1354, 1337, 1354, 1338,
        1354, 1339, 1354, 1340, 1354, 1341, 1354, 1342, 1354, 1343, 1354, 1344, 1354, 1345, 1354,
        1346, 1354, 1347, 1354, 1348, 1354, 1349, 1354, 1350, 1354, 1351, 1354, 1352, 1354, 1353,
        1354, 1354, 1334, 1354, 1379, 1381, 1378, 1335, 1355, 1356, 1384, 1336, 1355, 1357, 1385,
        1337, 1355, 1358, 1386, 1338, 1355, 1359, 1387, 1339, 1355, 1360, 1388, 1340, 1355, 1361,
        1389, 1341, 1355, 1362, 1390, 1342, 1355, 1363, 1391, 1343, 1355, 1364, 1392, 1344, 1355,
        1365, 1393, 1345, 1355, 1366, 1394, 1346, 1355, 1367, 1395, 1347, 1355, 1368, 1396, 1348,
        1355, 1369, 1397, 1349, 1355, 1370, 1398, 1350, 1355, 1371, 1399, 1351, 1355, 1372, 1400,
        1352, 1355, 1373, 1401, 1353, 1355, 1374, 1402, 1355, 1375, 1334, 1355, 1383, 1380, 1382,
        1405, 1376, 1378, 1406, 1384, 1403, 1404, 1432, 1385, 1403, 1404, 1433, 1386, 1403, 1404,
        1434, 1387, 1403, 1404, 1435, 1388, 1403, 1404, 1436, 1389, 1403, 1404, 1437, 1390, 1403,
        1404, 1438, 1391, 1403, 1404, 1439, 1392, 1403, 1404, 1440, 1393, 1403, 1404, 1441, 1394,
        1403, 1404, 1442, 1395, 1403, 1404, 1443, 1396, 1403, 1404, 1444, 1397, 1403, 1404, 1445,
        1398, 1403, 1404, 1446, 1399, 1403, 1404, 1447, 1400, 1403, 1404, 1448, 1401, 1403, 1404,
        1449, 1402, 1403, 1404, 1450, 1403, 1404, 1383, 1403, 1431, 56, 1404, 1404, 1405, 1451,
        1404, 1406, 1476, 62, 85, 86, 1432, 63, 85, 86, 1433, 64, 85, 86, 1434, 65, 85, 86, 1435,
        66, 85, 86, 1436, 67, 85, 86, 1437, 68, 85, 86, 1438, 69, 85, 86, 1439, 70, 85, 86, 1440,
        71, 85, 86, 1441, 72, 85, 86, 1442, 73, 85, 86, 1443, 74, 85, 86, 1444, 75, 85, 86, 1445,
        76, 85, 86, 1446, 77, 85, 86, 1447, 78, 85, 86, 1448, 79, 85, 86, 1449, 80, 85, 86, 1450,
        85, 86, 61, 85, 1431, 86, 1475, 86, 87, 1451, 86, 88, 1476, 416, 435, 436, 442, 443, 444,
        446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480,
        483, 417, 435, 436, 442, 444, 445, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466,
        468, 470, 472, 474, 476, 478, 480, 484, 418, 435, 436, 442, 444, 446, 447, 448, 450, 452,
        454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 485, 419, 435, 436,
        442, 444, 446, 448, 449, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474,
        476, 478, 480, 486, 420, 435, 436, 442, 444, 446, 448, 450, 451, 452, 454, 456, 458, 460,
        462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 487, 421, 435, 436, 442, 444, 446, 448,
        450, 452, 453, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 488,
        422, 435, 436, 442, 444, 446, 448, 450, 452, 454, 455, 456, 458, 460, 462, 464, 466, 468,
        470, 472, 474, 476, 478, 480, 489, 423, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456,
        457, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 490, 424, 435, 436, 442,
        444, 446, 448, 450, 452, 454, 456, 458, 459, 460, 462, 464, 466, 468, 470, 472, 474, 476,
        478, 480, 491, 425, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 461, 462,
        464, 466, 468, 470, 472, 474, 476, 478, 480, 492, 426, 435, 436, 442, 444, 446, 448, 450,
        452, 454, 456, 458, 460, 462, 463, 464, 466, 468, 470, 472, 474, 476, 478, 480, 493, 427,
        435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 465, 466, 468, 470,
        472, 474, 476, 478, 480, 494, 428, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458,
        460, 462, 464, 466, 467, 468, 470, 472, 474, 476, 478, 480, 495, 429, 435, 436, 442, 444,
        446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 469, 470, 472, 474, 476, 478,
        480, 496, 430, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466,
        468, 470, 471, 472, 474, 476, 478, 480, 497, 431, 435, 436, 442, 444, 446, 448, 450, 452,
        454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 473, 474, 476, 478, 480, 498, 432, 435,
        436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474,
        475, 476, 478, 480, 499, 433, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460,
        462, 464, 466, 468, 470, 472, 474, 476, 477, 478, 480, 500, 434, 435, 436, 442, 444, 446,
        448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 479, 480,
        501, 435, 436, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470,
        472, 474, 476, 478, 480, 481, 415, 435, 443, 445, 447, 449, 451, 453, 455, 457, 459, 461,
        463, 465, 467, 469, 471, 473, 475, 477, 479, 481, 482, 30, 436, 439, 442, 444, 446, 448,
        450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 31, 436,
        440, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474,
        476, 478, 480, 506, 436, 441, 507, 416, 437, 438, 442, 443, 444, 446, 448, 450, 452, 454,
        456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 911, 417, 437, 438, 442,
        444, 445, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476,
        478, 480, 912, 418, 437, 438, 442, 444, 446, 447, 448, 450, 452, 454, 456, 458, 460, 462,
        464, 466, 468, 470, 472, 474, 476, 478, 480, 913, 419, 437, 438, 442, 444, 446, 448, 449,
        450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 914, 420,
        437, 438, 442, 444, 446, 448, 450, 451, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470,
        472, 474, 476, 478, 480, 915, 421, 437, 438, 442, 444, 446, 448, 450, 452, 453, 454, 456,
        458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 916, 422, 437, 438, 442, 444,
        446, 448, 450, 452, 454, 455, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478,
        480, 917, 423, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456, 457, 458, 460, 462, 464,
        466, 468, 470, 472, 474, 476, 478, 480, 918, 424, 437, 438, 442, 444, 446, 448, 450, 452,
        454, 456, 458, 459, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480, 919, 425, 437,
        438, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 461, 462, 464, 466, 468, 470, 472,
        474, 476, 478, 480, 920, 426, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460,
        462, 463, 464, 466, 468, 470, 472, 474, 476, 478, 480, 921, 427, 437, 438, 442, 444, 446,
        448, 450, 452, 454, 456, 458, 460, 462, 464, 465, 466, 468, 470, 472, 474, 476, 478, 480,
        922, 428, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 467,
        468, 470, 472, 474, 476, 478, 480, 923, 429, 437, 438, 442, 444, 446, 448, 450, 452, 454,
        456, 458, 460, 462, 464, 466, 468, 469, 470, 472, 474, 476, 478, 480, 924, 430, 437, 438,
        442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 471, 472, 474,
        476, 478, 480, 925, 431, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462,
        464, 466, 468, 470, 472, 473, 474, 476, 478, 480, 926, 432, 437, 438, 442, 444, 446, 448,
        450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 475, 476, 478, 480, 927,
        433, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470,
        472, 474, 476, 477, 478, 480, 928, 434, 437, 438, 442, 444, 446, 448, 450, 452, 454, 456,
        458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 479, 480, 929, 437, 438, 442, 444,
        446, 448, 450, 452, 454, 456, 458, 460, 462, 464, 466, 468, 470, 472, 474, 476, 478, 480,
        481, 415, 437, 443, 445, 447, 449, 451, 453, 455, 457, 459, 461, 463, 465, 467, 469, 471,
        473, 475, 477, 479, 481, 910, 438, 439, 438, 440, 932, 438, 441, 933, 483, 502, 503, 509,
        484, 502, 503, 510, 485, 502, 503, 511, 486, 502, 503, 512, 487, 502, 503, 513, 488, 502,
        503, 514, 489, 502, 503, 515, 490, 502, 503, 516, 491, 502, 503, 517, 492, 502, 503, 518,
        493, 502, 503, 519, 494, 502, 503, 520, 495, 502, 503, 521, 496, 502, 503, 522, 497, 502,
        503, 523, 498, 502, 503, 524, 499, 502, 503, 525, 500, 502, 503, 526, 501, 502, 503, 527,
        502, 503, 482, 502, 508, 503, 503, 506, 503, 507, 536, 509, 528, 529, 578, 604, 605, 606,
        608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642,
        510, 528, 529, 579, 604, 606, 607, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628,
        630, 632, 634, 636, 638, 640, 642, 511, 528, 529, 580, 604, 606, 608, 609, 610, 612, 614,
        616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 512, 528, 529, 581,
        604, 606, 608, 610, 611, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636,
        638, 640, 642, 513, 528, 529, 582, 604, 606, 608, 610, 612, 613, 614, 616, 618, 620, 622,
        624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 514, 528, 529, 583, 604, 606, 608, 610,
        612, 614, 615, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 515,
        528, 529, 584, 604, 606, 608, 610, 612, 614, 616, 617, 618, 620, 622, 624, 626, 628, 630,
        632, 634, 636, 638, 640, 642, 516, 528, 529, 585, 604, 606, 608, 610, 612, 614, 616, 618,
        619, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 517, 528, 529, 586, 604,
        606, 608, 610, 612, 614, 616, 618, 620, 621, 622, 624, 626, 628, 630, 632, 634, 636, 638,
        640, 642, 518, 528, 529, 587, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 623, 624,
        626, 628, 630, 632, 634, 636, 638, 640, 642, 519, 528, 529, 588, 604, 606, 608, 610, 612,
        614, 616, 618, 620, 622, 624, 625, 626, 628, 630, 632, 634, 636, 638, 640, 642, 520, 528,
        529, 589, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 627, 628, 630, 632,
        634, 636, 638, 640, 642, 521, 528, 529, 590, 604, 606, 608, 610, 612, 614, 616, 618, 620,
        622, 624, 626, 628, 629, 630, 632, 634, 636, 638, 640, 642, 522, 528, 529, 591, 604, 606,
        608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 631, 632, 634, 636, 638, 640,
        642, 523, 528, 529, 592, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628,
        630, 632, 633, 634, 636, 638, 640, 642, 524, 528, 529, 593, 604, 606, 608, 610, 612, 614,
        616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 635, 636, 638, 640, 642, 525, 528, 529,
        594, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636,
        637, 638, 640, 642, 526, 528, 529, 595, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622,
        624, 626, 628, 630, 632, 634, 636, 638, 639, 640, 642, 527, 528, 529, 596, 604, 606, 608,
        610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 641, 642,
        528, 529, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634,
        636, 638, 640, 642, 643, 508, 528, 577, 605, 607, 609, 611, 613, 615, 617, 619, 621, 623,
        625, 627, 629, 631, 633, 635, 637, 639, 641, 643, 529, 601, 529, 602, 529, 536, 603, 509,
        530, 531, 537, 538, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565,
        567, 569, 571, 573, 575, 578, 510, 530, 531, 537, 539, 540, 541, 543, 545, 547, 549, 551,
        553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 579, 511, 530, 531, 537, 539,
        541, 542, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573,
        575, 580, 512, 530, 531, 537, 539, 541, 543, 544, 545, 547, 549, 551, 553, 555, 557, 559,
        561, 563, 565, 567, 569, 571, 573, 575, 581, 513, 530, 531, 537, 539, 541, 543, 545, 546,
        547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 582, 514, 530,
        531, 537, 539, 541, 543, 545, 547, 548, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567,
        569, 571, 573, 575, 583, 515, 530, 531, 537, 539, 541, 543, 545, 547, 549, 550, 551, 553,
        555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 584, 516, 530, 531, 537, 539, 541,
        543, 545, 547, 549, 551, 552, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575,
        585, 517, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 554, 555, 557, 559, 561,
        563, 565, 567, 569, 571, 573, 575, 586, 518, 530, 531, 537, 539, 541, 543, 545, 547, 549,
        551, 553, 555, 556, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 587, 519, 530, 531,
        537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 558, 559, 561, 563, 565, 567, 569,
        571, 573, 575, 588, 520, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557,
        559, 560, 561, 563, 565, 567, 569, 571, 573, 575, 589, 521, 530, 531, 537, 539, 541, 543,
        545, 547, 549, 551, 553, 555, 557, 559, 561, 562, 563, 565, 567, 569, 571, 573, 575, 590,
        522, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 564,
        565, 567, 569, 571, 573, 575, 591, 523, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551,
        553, 555, 557, 559, 561, 563, 565, 566, 567, 569, 571, 573, 575, 592, 524, 530, 531, 537,
        539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 568, 569, 571,
        573, 575, 593, 525, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559,
        561, 563, 565, 567, 569, 570, 571, 573, 575, 594, 526, 530, 531, 537, 539, 541, 543, 545,
        547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 572, 573, 575, 595, 527,
        530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567,
        569, 571, 573, 574, 575, 596, 530, 531, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555,
        557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 576, 508, 530, 538, 540, 542, 544, 546,
        548, 550, 552, 554, 556, 558, 560, 562, 564, 566, 568, 570, 572, 574, 576, 577, 32, 531,
        534, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569,
        571, 573, 575, 33, 531, 535, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559,
        561, 563, 565, 567, 569, 571, 573, 575, 531, 536, 603, 509, 532, 533, 537, 538, 539, 541,
        543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 911,
        510, 532, 533, 537, 539, 540, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563,
        565, 567, 569, 571, 573, 575, 912, 511, 532, 533, 537, 539, 541, 542, 543, 545, 547, 549,
        551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 913, 512, 532, 533, 537,
        539, 541, 543, 544, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571,
        573, 575, 914, 513, 532, 533, 537, 539, 541, 543, 545, 546, 547, 549, 551, 553, 555, 557,
        559, 561, 563, 565, 567, 569, 571, 573, 575, 915, 514, 532, 533, 537, 539, 541, 543, 545,
        547, 548, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 916, 515,
        532, 533, 537, 539, 541, 543, 545, 547, 549, 550, 551, 553, 555, 557, 559, 561, 563, 565,
        567, 569, 571, 573, 575, 917, 516, 532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 552,
        553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 575, 918, 517, 532, 533, 537, 539,
        541, 543, 545, 547, 549, 551, 553, 554, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573,
        575, 919, 518, 532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 556, 557, 559,
        561, 563, 565, 567, 569, 571, 573, 575, 920, 519, 532, 533, 537, 539, 541, 543, 545, 547,
        549, 551, 553, 555, 557, 558, 559, 561, 563, 565, 567, 569, 571, 573, 575, 921, 520, 532,
        533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 560, 561, 563, 565, 567,
        569, 571, 573, 575, 922, 521, 532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555,
        557, 559, 561, 562, 563, 565, 567, 569, 571, 573, 575, 923, 522, 532, 533, 537, 539, 541,
        543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 564, 565, 567, 569, 571, 573, 575,
        924, 523, 532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563,
        565, 566, 567, 569, 571, 573, 575, 925, 524, 532, 533, 537, 539, 541, 543, 545, 547, 549,
        551, 553, 555, 557, 559, 561, 563, 565, 567, 568, 569, 571, 573, 575, 926, 525, 532, 533,
        537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 570,
        571, 573, 575, 927, 526, 532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557,
        559, 561, 563, 565, 567, 569, 571, 572, 573, 575, 928, 527, 532, 533, 537, 539, 541, 543,
        545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567, 569, 571, 573, 574, 575, 929,
        532, 533, 537, 539, 541, 543, 545, 547, 549, 551, 553, 555, 557, 559, 561, 563, 565, 567,
        569, 571, 573, 575, 576, 508, 532, 538, 540, 542, 544, 546, 548, 550, 552, 554, 556, 558,
        560, 562, 564, 566, 568, 570, 572, 574, 576, 910, 533, 534, 533, 535, 932, 533, 536, 933,
        935, 954, 955, 1002, 1028, 1029, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046,
        1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 936, 954, 955, 1003, 1028,
        1030, 1031, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056,
        1058, 1060, 1062, 1064, 1066, 937, 954, 955, 1004, 1028, 1030, 1032, 1033, 1034, 1036,
        1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066,
        938, 954, 955, 1005, 1028, 1030, 1032, 1034, 1035, 1036, 1038, 1040, 1042, 1044, 1046,
        1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 939, 954, 955, 1006, 1028,
        1030, 1032, 1034, 1036, 1037, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056,
        1058, 1060, 1062, 1064, 1066, 940, 954, 955, 1007, 1028, 1030, 1032, 1034, 1036, 1038,
        1039, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066,
        941, 954, 955, 1008, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1041, 1042, 1044, 1046,
        1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 942, 954, 955, 1009, 1028,
        1030, 1032, 1034, 1036, 1038, 1040, 1042, 1043, 1044, 1046, 1048, 1050, 1052, 1054, 1056,
        1058, 1060, 1062, 1064, 1066, 943, 954, 955, 1010, 1028, 1030, 1032, 1034, 1036, 1038,
        1040, 1042, 1044, 1045, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066,
        944, 954, 955, 1011, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1047,
        1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 945, 954, 955, 1012, 1028,
        1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1049, 1050, 1052, 1054, 1056,
        1058, 1060, 1062, 1064, 1066, 946, 954, 955, 1013, 1028, 1030, 1032, 1034, 1036, 1038,
        1040, 1042, 1044, 1046, 1048, 1050, 1051, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066,
        947, 954, 955, 1014, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048,
        1050, 1052, 1053, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 948, 954, 955, 1015, 1028,
        1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1055, 1056,
        1058, 1060, 1062, 1064, 1066, 949, 954, 955, 1016, 1028, 1030, 1032, 1034, 1036, 1038,
        1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1057, 1058, 1060, 1062, 1064, 1066,
        950, 954, 955, 1017, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048,
        1050, 1052, 1054, 1056, 1058, 1059, 1060, 1062, 1064, 1066, 951, 954, 955, 1018, 1028,
        1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058,
        1060, 1061, 1062, 1064, 1066, 952, 954, 955, 1019, 1028, 1030, 1032, 1034, 1036, 1038,
        1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1063, 1064, 1066,
        953, 954, 955, 1020, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048,
        1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1065, 1066, 954, 955, 1028, 1030, 1032,
        1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062,
        1064, 1066, 1067, 934, 954, 1001, 1029, 1031, 1033, 1035, 1037, 1039, 1041, 1043, 1045,
        1047, 1049, 1051, 1053, 1055, 1057, 1059, 1061, 1063, 1065, 1067, 46, 955, 1025, 1028,
        1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058,
        1060, 1062, 1064, 1066, 47, 955, 1026, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042,
        1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 955, 960, 1027,
        911, 930, 931, 935, 912, 930, 931, 936, 913, 930, 931, 937, 914, 930, 931, 938, 915, 930,
        931, 939, 916, 930, 931, 940, 917, 930, 931, 941, 918, 930, 931, 942, 919, 930, 931, 943,
        920, 930, 931, 944, 921, 930, 931, 945, 922, 930, 931, 946, 923, 930, 931, 947, 924, 930,
        931, 948, 925, 930, 931, 949, 926, 930, 931, 950, 927, 930, 931, 951, 928, 930, 931, 952,
        929, 930, 931, 953, 930, 931, 910, 930, 934, 931, 931, 932, 931, 933, 960, 483, 504, 505,
        935, 961, 962, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991,
        993, 995, 997, 999, 484, 504, 505, 936, 961, 963, 964, 965, 967, 969, 971, 973, 975, 977,
        979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 485, 504, 505, 937, 961, 963, 965,
        966, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999,
        486, 504, 505, 938, 961, 963, 965, 967, 968, 969, 971, 973, 975, 977, 979, 981, 983, 985,
        987, 989, 991, 993, 995, 997, 999, 487, 504, 505, 939, 961, 963, 965, 967, 969, 970, 971,
        973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 488, 504, 505, 940,
        961, 963, 965, 967, 969, 971, 972, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993,
        995, 997, 999, 489, 504, 505, 941, 961, 963, 965, 967, 969, 971, 973, 974, 975, 977, 979,
        981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 490, 504, 505, 942, 961, 963, 965, 967,
        969, 971, 973, 975, 976, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 491,
        504, 505, 943, 961, 963, 965, 967, 969, 971, 973, 975, 977, 978, 979, 981, 983, 985, 987,
        989, 991, 993, 995, 997, 999, 492, 504, 505, 944, 961, 963, 965, 967, 969, 971, 973, 975,
        977, 979, 980, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 493, 504, 505, 945, 961,
        963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 982, 983, 985, 987, 989, 991, 993, 995,
        997, 999, 494, 504, 505, 946, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983,
        984, 985, 987, 989, 991, 993, 995, 997, 999, 495, 504, 505, 947, 961, 963, 965, 967, 969,
        971, 973, 975, 977, 979, 981, 983, 985, 986, 987, 989, 991, 993, 995, 997, 999, 496, 504,
        505, 948, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 988, 989,
        991, 993, 995, 997, 999, 497, 504, 505, 949, 961, 963, 965, 967, 969, 971, 973, 975, 977,
        979, 981, 983, 985, 987, 989, 990, 991, 993, 995, 997, 999, 498, 504, 505, 950, 961, 963,
        965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 992, 993, 995, 997,
        999, 499, 504, 505, 951, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985,
        987, 989, 991, 993, 994, 995, 997, 999, 500, 504, 505, 952, 961, 963, 965, 967, 969, 971,
        973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 996, 997, 999, 501, 504, 505,
        953, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993,
        995, 997, 998, 999, 504, 505, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983,
        985, 987, 989, 991, 993, 995, 997, 999, 1000, 482, 504, 934, 962, 964, 966, 968, 970, 972,
        974, 976, 978, 980, 982, 984, 986, 988, 990, 992, 994, 996, 998, 1000, 44, 505, 958, 961,
        963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997,
        999, 45, 505, 506, 959, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985,
        987, 989, 991, 993, 995, 997, 999, 505, 507, 960, 935, 956, 957, 961, 962, 963, 965, 967,
        969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1002, 936,
        956, 957, 961, 963, 964, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989,
        991, 993, 995, 997, 999, 1003, 937, 956, 957, 961, 963, 965, 966, 967, 969, 971, 973, 975,
        977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1004, 938, 956, 957, 961, 963,
        965, 967, 968, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997,
        999, 1005, 939, 956, 957, 961, 963, 965, 967, 969, 970, 971, 973, 975, 977, 979, 981, 983,
        985, 987, 989, 991, 993, 995, 997, 999, 1006, 940, 956, 957, 961, 963, 965, 967, 969, 971,
        972, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1007, 941, 956,
        957, 961, 963, 965, 967, 969, 971, 973, 974, 975, 977, 979, 981, 983, 985, 987, 989, 991,
        993, 995, 997, 999, 1008, 942, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975, 976, 977,
        979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1009, 943, 956, 957, 961, 963, 965,
        967, 969, 971, 973, 975, 977, 978, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 999,
        1010, 944, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 980, 981, 983, 985,
        987, 989, 991, 993, 995, 997, 999, 1011, 945, 956, 957, 961, 963, 965, 967, 969, 971, 973,
        975, 977, 979, 981, 982, 983, 985, 987, 989, 991, 993, 995, 997, 999, 1012, 946, 956, 957,
        961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 984, 985, 987, 989, 991, 993,
        995, 997, 999, 1013, 947, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981,
        983, 985, 986, 987, 989, 991, 993, 995, 997, 999, 1014, 948, 956, 957, 961, 963, 965, 967,
        969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 988, 989, 991, 993, 995, 997, 999, 1015,
        949, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989,
        990, 991, 993, 995, 997, 999, 1016, 950, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975,
        977, 979, 981, 983, 985, 987, 989, 991, 992, 993, 995, 997, 999, 1017, 951, 956, 957, 961,
        963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 994, 995,
        997, 999, 1018, 952, 956, 957, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983,
        985, 987, 989, 991, 993, 995, 996, 997, 999, 1019, 953, 956, 957, 961, 963, 965, 967, 969,
        971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993, 995, 997, 998, 999, 1020, 956,
        957, 961, 963, 965, 967, 969, 971, 973, 975, 977, 979, 981, 983, 985, 987, 989, 991, 993,
        995, 997, 999, 1000, 934, 956, 962, 964, 966, 968, 970, 972, 974, 976, 978, 980, 982, 984,
        986, 988, 990, 992, 994, 996, 998, 1000, 1001, 957, 958, 957, 959, 957, 960, 1027, 578,
        599, 600, 604, 605, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632,
        634, 636, 638, 640, 642, 645, 579, 599, 600, 604, 606, 607, 608, 610, 612, 614, 616, 618,
        620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 646, 580, 599, 600, 604, 606,
        608, 609, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640,
        642, 647, 581, 599, 600, 604, 606, 608, 610, 611, 612, 614, 616, 618, 620, 622, 624, 626,
        628, 630, 632, 634, 636, 638, 640, 642, 648, 582, 599, 600, 604, 606, 608, 610, 612, 613,
        614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 649, 583, 599,
        600, 604, 606, 608, 610, 612, 614, 615, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634,
        636, 638, 640, 642, 650, 584, 599, 600, 604, 606, 608, 610, 612, 614, 616, 617, 618, 620,
        622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 651, 585, 599, 600, 604, 606, 608,
        610, 612, 614, 616, 618, 619, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642,
        652, 586, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 621, 622, 624, 626, 628,
        630, 632, 634, 636, 638, 640, 642, 653, 587, 599, 600, 604, 606, 608, 610, 612, 614, 616,
        618, 620, 622, 623, 624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 654, 588, 599, 600,
        604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 625, 626, 628, 630, 632, 634, 636,
        638, 640, 642, 655, 589, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624,
        626, 627, 628, 630, 632, 634, 636, 638, 640, 642, 656, 590, 599, 600, 604, 606, 608, 610,
        612, 614, 616, 618, 620, 622, 624, 626, 628, 629, 630, 632, 634, 636, 638, 640, 642, 657,
        591, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 631,
        632, 634, 636, 638, 640, 642, 658, 592, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618,
        620, 622, 624, 626, 628, 630, 632, 633, 634, 636, 638, 640, 642, 659, 593, 599, 600, 604,
        606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 635, 636, 638,
        640, 642, 660, 594, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626,
        628, 630, 632, 634, 636, 637, 638, 640, 642, 661, 595, 599, 600, 604, 606, 608, 610, 612,
        614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636, 638, 639, 640, 642, 662, 596,
        599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634,
        636, 638, 640, 641, 642, 663, 599, 600, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622,
        624, 626, 628, 630, 632, 634, 636, 638, 640, 642, 643, 577, 599, 605, 607, 609, 611, 613,
        615, 617, 619, 621, 623, 625, 627, 629, 631, 633, 635, 637, 639, 641, 643, 644, 34, 600,
        601, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626, 628, 630, 632, 634, 636,
        638, 640, 642, 35, 600, 602, 604, 606, 608, 610, 612, 614, 616, 618, 620, 622, 624, 626,
        628, 630, 632, 634, 636, 638, 640, 642, 600, 603, 670, 645, 664, 665, 712, 738, 739, 740,
        742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776,
        646, 664, 665, 713, 738, 740, 741, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762,
        764, 766, 768, 770, 772, 774, 776, 647, 664, 665, 714, 738, 740, 742, 743, 744, 746, 748,
        750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 648, 664, 665, 715,
        738, 740, 742, 744, 745, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770,
        772, 774, 776, 649, 664, 665, 716, 738, 740, 742, 744, 746, 747, 748, 750, 752, 754, 756,
        758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 650, 664, 665, 717, 738, 740, 742, 744,
        746, 748, 749, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 651,
        664, 665, 718, 738, 740, 742, 744, 746, 748, 750, 751, 752, 754, 756, 758, 760, 762, 764,
        766, 768, 770, 772, 774, 776, 652, 664, 665, 719, 738, 740, 742, 744, 746, 748, 750, 752,
        753, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 653, 664, 665, 720, 738,
        740, 742, 744, 746, 748, 750, 752, 754, 755, 756, 758, 760, 762, 764, 766, 768, 770, 772,
        774, 776, 654, 664, 665, 721, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 757, 758,
        760, 762, 764, 766, 768, 770, 772, 774, 776, 655, 664, 665, 722, 738, 740, 742, 744, 746,
        748, 750, 752, 754, 756, 758, 759, 760, 762, 764, 766, 768, 770, 772, 774, 776, 656, 664,
        665, 723, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 761, 762, 764, 766,
        768, 770, 772, 774, 776, 657, 664, 665, 724, 738, 740, 742, 744, 746, 748, 750, 752, 754,
        756, 758, 760, 762, 763, 764, 766, 768, 770, 772, 774, 776, 658, 664, 665, 725, 738, 740,
        742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 765, 766, 768, 770, 772, 774,
        776, 659, 664, 665, 726, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762,
        764, 766, 767, 768, 770, 772, 774, 776, 660, 664, 665, 727, 738, 740, 742, 744, 746, 748,
        750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 769, 770, 772, 774, 776, 661, 664, 665,
        728, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770,
        771, 772, 774, 776, 662, 664, 665, 729, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756,
        758, 760, 762, 764, 766, 768, 770, 772, 773, 774, 776, 663, 664, 665, 730, 738, 740, 742,
        744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 775, 776,
        664, 665, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768,
        770, 772, 774, 776, 777, 644, 664, 711, 739, 741, 743, 745, 747, 749, 751, 753, 755, 757,
        759, 761, 763, 765, 767, 769, 771, 773, 775, 777, 665, 735, 665, 736, 665, 670, 737, 645,
        666, 667, 671, 672, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699,
        701, 703, 705, 707, 709, 712, 646, 666, 667, 671, 673, 674, 675, 677, 679, 681, 683, 685,
        687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 713, 647, 666, 667, 671, 673,
        675, 676, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707,
        709, 714, 648, 666, 667, 671, 673, 675, 677, 678, 679, 681, 683, 685, 687, 689, 691, 693,
        695, 697, 699, 701, 703, 705, 707, 709, 715, 649, 666, 667, 671, 673, 675, 677, 679, 680,
        681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 716, 650, 666,
        667, 671, 673, 675, 677, 679, 681, 682, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701,
        703, 705, 707, 709, 717, 651, 666, 667, 671, 673, 675, 677, 679, 681, 683, 684, 685, 687,
        689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 718, 652, 666, 667, 671, 673, 675,
        677, 679, 681, 683, 685, 686, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709,
        719, 653, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 688, 689, 691, 693, 695,
        697, 699, 701, 703, 705, 707, 709, 720, 654, 666, 667, 671, 673, 675, 677, 679, 681, 683,
        685, 687, 689, 690, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 721, 655, 666, 667,
        671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 692, 693, 695, 697, 699, 701, 703,
        705, 707, 709, 722, 656, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691,
        693, 694, 695, 697, 699, 701, 703, 705, 707, 709, 723, 657, 666, 667, 671, 673, 675, 677,
        679, 681, 683, 685, 687, 689, 691, 693, 695, 696, 697, 699, 701, 703, 705, 707, 709, 724,
        658, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 698,
        699, 701, 703, 705, 707, 709, 725, 659, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685,
        687, 689, 691, 693, 695, 697, 699, 700, 701, 703, 705, 707, 709, 726, 660, 666, 667, 671,
        673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 702, 703, 705,
        707, 709, 727, 661, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693,
        695, 697, 699, 701, 703, 704, 705, 707, 709, 728, 662, 666, 667, 671, 673, 675, 677, 679,
        681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 706, 707, 709, 729, 663,
        666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701,
        703, 705, 707, 708, 709, 730, 666, 667, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689,
        691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 710, 644, 666, 672, 674, 676, 678, 680,
        682, 684, 686, 688, 690, 692, 694, 696, 698, 700, 702, 704, 706, 708, 710, 711, 36, 667,
        668, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703,
        705, 707, 709, 37, 667, 669, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693,
        695, 697, 699, 701, 703, 705, 707, 709, 667, 670, 737, 578, 597, 598, 645, 671, 672, 673,
        675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709,
        579, 597, 598, 646, 671, 673, 674, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695,
        697, 699, 701, 703, 705, 707, 709, 580, 597, 598, 647, 671, 673, 675, 676, 677, 679, 681,
        683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 581, 597, 598, 648,
        671, 673, 675, 677, 678, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703,
        705, 707, 709, 582, 597, 598, 649, 671, 673, 675, 677, 679, 680, 681, 683, 685, 687, 689,
        691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 583, 597, 598, 650, 671, 673, 675, 677,
        679, 681, 682, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 584,
        597, 598, 651, 671, 673, 675, 677, 679, 681, 683, 684, 685, 687, 689, 691, 693, 695, 697,
        699, 701, 703, 705, 707, 709, 585, 597, 598, 652, 671, 673, 675, 677, 679, 681, 683, 685,
        686, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 709, 586, 597, 598, 653, 671,
        673, 675, 677, 679, 681, 683, 685, 687, 688, 689, 691, 693, 695, 697, 699, 701, 703, 705,
        707, 709, 587, 597, 598, 654, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 690, 691,
        693, 695, 697, 699, 701, 703, 705, 707, 709, 588, 597, 598, 655, 671, 673, 675, 677, 679,
        681, 683, 685, 687, 689, 691, 692, 693, 695, 697, 699, 701, 703, 705, 707, 709, 589, 597,
        598, 656, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 694, 695, 697, 699,
        701, 703, 705, 707, 709, 590, 597, 598, 657, 671, 673, 675, 677, 679, 681, 683, 685, 687,
        689, 691, 693, 695, 696, 697, 699, 701, 703, 705, 707, 709, 591, 597, 598, 658, 671, 673,
        675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 698, 699, 701, 703, 705, 707,
        709, 592, 597, 598, 659, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695,
        697, 699, 700, 701, 703, 705, 707, 709, 593, 597, 598, 660, 671, 673, 675, 677, 679, 681,
        683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 702, 703, 705, 707, 709, 594, 597, 598,
        661, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703,
        704, 705, 707, 709, 595, 597, 598, 662, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689,
        691, 693, 695, 697, 699, 701, 703, 705, 706, 707, 709, 596, 597, 598, 663, 671, 673, 675,
        677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701, 703, 705, 707, 708, 709,
        597, 598, 671, 673, 675, 677, 679, 681, 683, 685, 687, 689, 691, 693, 695, 697, 699, 701,
        703, 705, 707, 709, 710, 577, 597, 644, 672, 674, 676, 678, 680, 682, 684, 686, 688, 690,
        692, 694, 696, 698, 700, 702, 704, 706, 708, 710, 598, 668, 598, 669, 598, 603, 670, 1069,
        1088, 1089, 1136, 1162, 1163, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1070, 1088, 1089, 1137, 1162, 1164,
        1165, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1071, 1088, 1089, 1138, 1162, 1164, 1166, 1167, 1168, 1170, 1172,
        1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1072,
        1088, 1089, 1139, 1162, 1164, 1166, 1168, 1169, 1170, 1172, 1174, 1176, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1073, 1088, 1089, 1140, 1162, 1164,
        1166, 1168, 1170, 1171, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1074, 1088, 1089, 1141, 1162, 1164, 1166, 1168, 1170, 1172, 1173,
        1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1075,
        1088, 1089, 1142, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1175, 1176, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1076, 1088, 1089, 1143, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1177, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1077, 1088, 1089, 1144, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1179, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1078,
        1088, 1089, 1145, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1181, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1079, 1088, 1089, 1146, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1183, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1080, 1088, 1089, 1147, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1185, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1081,
        1088, 1089, 1148, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184,
        1186, 1187, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1082, 1088, 1089, 1149, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1189, 1190, 1192,
        1194, 1196, 1198, 1200, 1083, 1088, 1089, 1150, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1191, 1192, 1194, 1196, 1198, 1200, 1084,
        1088, 1089, 1151, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184,
        1186, 1188, 1190, 1192, 1193, 1194, 1196, 1198, 1200, 1085, 1088, 1089, 1152, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194,
        1195, 1196, 1198, 1200, 1086, 1088, 1089, 1153, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1197, 1198, 1200, 1087,
        1088, 1089, 1154, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184,
        1186, 1188, 1190, 1192, 1194, 1196, 1198, 1199, 1200, 1088, 1089, 1162, 1164, 1166, 1168,
        1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198,
        1200, 1201, 1068, 1088, 1135, 1163, 1165, 1167, 1169, 1171, 1173, 1175, 1177, 1179, 1181,
        1183, 1185, 1187, 1189, 1191, 1193, 1195, 1197, 1199, 1201, 50, 1089, 1159, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194,
        1196, 1198, 1200, 51, 1089, 1160, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178,
        1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1089, 1094, 1161, 1002,
        1023, 1024, 1028, 1029, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050,
        1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1069, 1003, 1023, 1024, 1028, 1030, 1031,
        1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060,
        1062, 1064, 1066, 1070, 1004, 1023, 1024, 1028, 1030, 1032, 1033, 1034, 1036, 1038, 1040,
        1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1071, 1005,
        1023, 1024, 1028, 1030, 1032, 1034, 1035, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050,
        1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1072, 1006, 1023, 1024, 1028, 1030, 1032,
        1034, 1036, 1037, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060,
        1062, 1064, 1066, 1073, 1007, 1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1039, 1040,
        1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1074, 1008,
        1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1041, 1042, 1044, 1046, 1048, 1050,
        1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1075, 1009, 1023, 1024, 1028, 1030, 1032,
        1034, 1036, 1038, 1040, 1042, 1043, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060,
        1062, 1064, 1066, 1076, 1010, 1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042,
        1044, 1045, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1077, 1011,
        1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1047, 1048, 1050,
        1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1078, 1012, 1023, 1024, 1028, 1030, 1032,
        1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1049, 1050, 1052, 1054, 1056, 1058, 1060,
        1062, 1064, 1066, 1079, 1013, 1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042,
        1044, 1046, 1048, 1050, 1051, 1052, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1080, 1014,
        1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052,
        1053, 1054, 1056, 1058, 1060, 1062, 1064, 1066, 1081, 1015, 1023, 1024, 1028, 1030, 1032,
        1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1055, 1056, 1058, 1060,
        1062, 1064, 1066, 1082, 1016, 1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042,
        1044, 1046, 1048, 1050, 1052, 1054, 1056, 1057, 1058, 1060, 1062, 1064, 1066, 1083, 1017,
        1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052,
        1054, 1056, 1058, 1059, 1060, 1062, 1064, 1066, 1084, 1018, 1023, 1024, 1028, 1030, 1032,
        1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1061,
        1062, 1064, 1066, 1085, 1019, 1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042,
        1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1063, 1064, 1066, 1086, 1020,
        1023, 1024, 1028, 1030, 1032, 1034, 1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052,
        1054, 1056, 1058, 1060, 1062, 1064, 1065, 1066, 1087, 1023, 1024, 1028, 1030, 1032, 1034,
        1036, 1038, 1040, 1042, 1044, 1046, 1048, 1050, 1052, 1054, 1056, 1058, 1060, 1062, 1064,
        1066, 1067, 1001, 1023, 1029, 1031, 1033, 1035, 1037, 1039, 1041, 1043, 1045, 1047, 1049,
        1051, 1053, 1055, 1057, 1059, 1061, 1063, 1065, 1067, 1068, 1024, 1025, 1024, 1026, 1024,
        1027, 1094, 1002, 1021, 1022, 1069, 1095, 1096, 1097, 1099, 1101, 1103, 1105, 1107, 1109,
        1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1003, 1021, 1022,
        1070, 1095, 1097, 1098, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119,
        1121, 1123, 1125, 1127, 1129, 1131, 1133, 1004, 1021, 1022, 1071, 1095, 1097, 1099, 1100,
        1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129,
        1131, 1133, 1005, 1021, 1022, 1072, 1095, 1097, 1099, 1101, 1102, 1103, 1105, 1107, 1109,
        1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1006, 1021, 1022,
        1073, 1095, 1097, 1099, 1101, 1103, 1104, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119,
        1121, 1123, 1125, 1127, 1129, 1131, 1133, 1007, 1021, 1022, 1074, 1095, 1097, 1099, 1101,
        1103, 1105, 1106, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129,
        1131, 1133, 1008, 1021, 1022, 1075, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1108, 1109,
        1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1009, 1021, 1022,
        1076, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1110, 1111, 1113, 1115, 1117, 1119,
        1121, 1123, 1125, 1127, 1129, 1131, 1133, 1010, 1021, 1022, 1077, 1095, 1097, 1099, 1101,
        1103, 1105, 1107, 1109, 1111, 1112, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129,
        1131, 1133, 1011, 1021, 1022, 1078, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111,
        1113, 1114, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1012, 1021, 1022,
        1079, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1116, 1117, 1119,
        1121, 1123, 1125, 1127, 1129, 1131, 1133, 1013, 1021, 1022, 1080, 1095, 1097, 1099, 1101,
        1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1118, 1119, 1121, 1123, 1125, 1127, 1129,
        1131, 1133, 1014, 1021, 1022, 1081, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111,
        1113, 1115, 1117, 1119, 1120, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1015, 1021, 1022,
        1082, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121,
        1122, 1123, 1125, 1127, 1129, 1131, 1133, 1016, 1021, 1022, 1083, 1095, 1097, 1099, 1101,
        1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1124, 1125, 1127, 1129,
        1131, 1133, 1017, 1021, 1022, 1084, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111,
        1113, 1115, 1117, 1119, 1121, 1123, 1125, 1126, 1127, 1129, 1131, 1133, 1018, 1021, 1022,
        1085, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121,
        1123, 1125, 1127, 1128, 1129, 1131, 1133, 1019, 1021, 1022, 1086, 1095, 1097, 1099, 1101,
        1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1130,
        1131, 1133, 1020, 1021, 1022, 1087, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111,
        1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1132, 1133, 1021, 1022, 1095,
        1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125,
        1127, 1129, 1131, 1133, 1134, 1001, 1021, 1068, 1096, 1098, 1100, 1102, 1104, 1106, 1108,
        1110, 1112, 1114, 1116, 1118, 1120, 1122, 1124, 1126, 1128, 1130, 1132, 1134, 48, 1022,
        1092, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121,
        1123, 1125, 1127, 1129, 1131, 1133, 49, 1022, 1093, 1095, 1097, 1099, 1101, 1103, 1105,
        1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1022,
        1027, 1094, 1069, 1090, 1091, 1095, 1096, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111,
        1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1136, 1070, 1090, 1091,
        1095, 1097, 1098, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121,
        1123, 1125, 1127, 1129, 1131, 1133, 1137, 1071, 1090, 1091, 1095, 1097, 1099, 1100, 1101,
        1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131,
        1133, 1138, 1072, 1090, 1091, 1095, 1097, 1099, 1101, 1102, 1103, 1105, 1107, 1109, 1111,
        1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1139, 1073, 1090, 1091,
        1095, 1097, 1099, 1101, 1103, 1104, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121,
        1123, 1125, 1127, 1129, 1131, 1133, 1140, 1074, 1090, 1091, 1095, 1097, 1099, 1101, 1103,
        1105, 1106, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131,
        1133, 1141, 1075, 1090, 1091, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1108, 1109, 1111,
        1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1142, 1076, 1090, 1091,
        1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1110, 1111, 1113, 1115, 1117, 1119, 1121,
        1123, 1125, 1127, 1129, 1131, 1133, 1143, 1077, 1090, 1091, 1095, 1097, 1099, 1101, 1103,
        1105, 1107, 1109, 1111, 1112, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131,
        1133, 1144, 1078, 1090, 1091, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113,
        1114, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1145, 1079, 1090, 1091,
        1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1116, 1117, 1119, 1121,
        1123, 1125, 1127, 1129, 1131, 1133, 1146, 1080, 1090, 1091, 1095, 1097, 1099, 1101, 1103,
        1105, 1107, 1109, 1111, 1113, 1115, 1117, 1118, 1119, 1121, 1123, 1125, 1127, 1129, 1131,
        1133, 1147, 1081, 1090, 1091, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113,
        1115, 1117, 1119, 1120, 1121, 1123, 1125, 1127, 1129, 1131, 1133, 1148, 1082, 1090, 1091,
        1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1122,
        1123, 1125, 1127, 1129, 1131, 1133, 1149, 1083, 1090, 1091, 1095, 1097, 1099, 1101, 1103,
        1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1124, 1125, 1127, 1129, 1131,
        1133, 1150, 1084, 1090, 1091, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113,
        1115, 1117, 1119, 1121, 1123, 1125, 1126, 1127, 1129, 1131, 1133, 1151, 1085, 1090, 1091,
        1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123,
        1125, 1127, 1128, 1129, 1131, 1133, 1152, 1086, 1090, 1091, 1095, 1097, 1099, 1101, 1103,
        1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1130, 1131,
        1133, 1153, 1087, 1090, 1091, 1095, 1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113,
        1115, 1117, 1119, 1121, 1123, 1125, 1127, 1129, 1131, 1132, 1133, 1154, 1090, 1091, 1095,
        1097, 1099, 1101, 1103, 1105, 1107, 1109, 1111, 1113, 1115, 1117, 1119, 1121, 1123, 1125,
        1127, 1129, 1131, 1133, 1134, 1068, 1090, 1096, 1098, 1100, 1102, 1104, 1106, 1108, 1110,
        1112, 1114, 1116, 1118, 1120, 1122, 1124, 1126, 1128, 1130, 1132, 1134, 1135, 1091, 1092,
        1091, 1093, 1091, 1094, 1161, 712, 733, 734, 738, 739, 740, 742, 744, 746, 748, 750, 752,
        754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 779, 713, 733, 734, 738, 740,
        741, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774,
        776, 780, 714, 733, 734, 738, 740, 742, 743, 744, 746, 748, 750, 752, 754, 756, 758, 760,
        762, 764, 766, 768, 770, 772, 774, 776, 781, 715, 733, 734, 738, 740, 742, 744, 745, 746,
        748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 782, 716, 733,
        734, 738, 740, 742, 744, 746, 747, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768,
        770, 772, 774, 776, 783, 717, 733, 734, 738, 740, 742, 744, 746, 748, 749, 750, 752, 754,
        756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 784, 718, 733, 734, 738, 740, 742,
        744, 746, 748, 750, 751, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776,
        785, 719, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 753, 754, 756, 758, 760, 762,
        764, 766, 768, 770, 772, 774, 776, 786, 720, 733, 734, 738, 740, 742, 744, 746, 748, 750,
        752, 754, 755, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 787, 721, 733, 734,
        738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 757, 758, 760, 762, 764, 766, 768, 770,
        772, 774, 776, 788, 722, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758,
        759, 760, 762, 764, 766, 768, 770, 772, 774, 776, 789, 723, 733, 734, 738, 740, 742, 744,
        746, 748, 750, 752, 754, 756, 758, 760, 761, 762, 764, 766, 768, 770, 772, 774, 776, 790,
        724, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 763, 764,
        766, 768, 770, 772, 774, 776, 791, 725, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752,
        754, 756, 758, 760, 762, 764, 765, 766, 768, 770, 772, 774, 776, 792, 726, 733, 734, 738,
        740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 767, 768, 770, 772,
        774, 776, 793, 727, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760,
        762, 764, 766, 768, 769, 770, 772, 774, 776, 794, 728, 733, 734, 738, 740, 742, 744, 746,
        748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 771, 772, 774, 776, 795, 729,
        733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768,
        770, 772, 773, 774, 776, 796, 730, 733, 734, 738, 740, 742, 744, 746, 748, 750, 752, 754,
        756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 775, 776, 797, 733, 734, 738, 740, 742,
        744, 746, 748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 777,
        711, 733, 739, 741, 743, 745, 747, 749, 751, 753, 755, 757, 759, 761, 763, 765, 767, 769,
        771, 773, 775, 777, 778, 38, 734, 735, 738, 740, 742, 744, 746, 748, 750, 752, 754, 756,
        758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 39, 734, 736, 738, 740, 742, 744, 746,
        748, 750, 752, 754, 756, 758, 760, 762, 764, 766, 768, 770, 772, 774, 776, 734, 737, 804,
        779, 798, 799, 846, 870, 871, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894,
        896, 898, 900, 902, 904, 906, 908, 780, 798, 799, 847, 870, 872, 873, 874, 876, 878, 880,
        882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 781, 798, 799, 848,
        870, 872, 874, 875, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902,
        904, 906, 908, 782, 798, 799, 849, 870, 872, 874, 876, 877, 878, 880, 882, 884, 886, 888,
        890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 783, 798, 799, 850, 870, 872, 874, 876,
        878, 879, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 784,
        798, 799, 851, 870, 872, 874, 876, 878, 880, 881, 882, 884, 886, 888, 890, 892, 894, 896,
        898, 900, 902, 904, 906, 908, 785, 798, 799, 852, 870, 872, 874, 876, 878, 880, 882, 883,
        884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 786, 798, 799, 853, 870,
        872, 874, 876, 878, 880, 882, 884, 885, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904,
        906, 908, 787, 798, 799, 854, 870, 872, 874, 876, 878, 880, 882, 884, 886, 887, 888, 890,
        892, 894, 896, 898, 900, 902, 904, 906, 908, 788, 798, 799, 855, 870, 872, 874, 876, 878,
        880, 882, 884, 886, 888, 889, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 789, 798,
        799, 856, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 891, 892, 894, 896, 898,
        900, 902, 904, 906, 908, 790, 798, 799, 857, 870, 872, 874, 876, 878, 880, 882, 884, 886,
        888, 890, 892, 893, 894, 896, 898, 900, 902, 904, 906, 908, 791, 798, 799, 858, 870, 872,
        874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 895, 896, 898, 900, 902, 904, 906,
        908, 792, 798, 799, 859, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894,
        896, 897, 898, 900, 902, 904, 906, 908, 793, 798, 799, 860, 870, 872, 874, 876, 878, 880,
        882, 884, 886, 888, 890, 892, 894, 896, 898, 899, 900, 902, 904, 906, 908, 794, 798, 799,
        861, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 901,
        902, 904, 906, 908, 795, 798, 799, 862, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888,
        890, 892, 894, 896, 898, 900, 902, 903, 904, 906, 908, 796, 798, 799, 863, 870, 872, 874,
        876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 905, 906, 908,
        797, 798, 799, 864, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896,
        898, 900, 902, 904, 906, 907, 908, 798, 799, 870, 872, 874, 876, 878, 880, 882, 884, 886,
        888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 909, 778, 798, 845, 871, 873, 875,
        877, 879, 881, 883, 885, 887, 889, 891, 893, 895, 897, 899, 901, 903, 905, 907, 909, 799,
        867, 799, 868, 799, 804, 869, 779, 800, 801, 805, 806, 807, 809, 811, 813, 815, 817, 819,
        821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 846, 780, 800, 801, 805, 807,
        808, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841,
        843, 847, 781, 800, 801, 805, 807, 809, 810, 811, 813, 815, 817, 819, 821, 823, 825, 827,
        829, 831, 833, 835, 837, 839, 841, 843, 848, 782, 800, 801, 805, 807, 809, 811, 812, 813,
        815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 849, 783, 800,
        801, 805, 807, 809, 811, 813, 814, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835,
        837, 839, 841, 843, 850, 784, 800, 801, 805, 807, 809, 811, 813, 815, 816, 817, 819, 821,
        823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 851, 785, 800, 801, 805, 807, 809,
        811, 813, 815, 817, 818, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843,
        852, 786, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 820, 821, 823, 825, 827, 829,
        831, 833, 835, 837, 839, 841, 843, 853, 787, 800, 801, 805, 807, 809, 811, 813, 815, 817,
        819, 821, 822, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 854, 788, 800, 801,
        805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 824, 825, 827, 829, 831, 833, 835, 837,
        839, 841, 843, 855, 789, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825,
        826, 827, 829, 831, 833, 835, 837, 839, 841, 843, 856, 790, 800, 801, 805, 807, 809, 811,
        813, 815, 817, 819, 821, 823, 825, 827, 828, 829, 831, 833, 835, 837, 839, 841, 843, 857,
        791, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 830, 831,
        833, 835, 837, 839, 841, 843, 858, 792, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819,
        821, 823, 825, 827, 829, 831, 832, 833, 835, 837, 839, 841, 843, 859, 793, 800, 801, 805,
        807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 834, 835, 837, 839,
        841, 843, 860, 794, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827,
        829, 831, 833, 835, 836, 837, 839, 841, 843, 861, 795, 800, 801, 805, 807, 809, 811, 813,
        815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 838, 839, 841, 843, 862, 796,
        800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835,
        837, 839, 840, 841, 843, 863, 797, 800, 801, 805, 807, 809, 811, 813, 815, 817, 819, 821,
        823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 842, 843, 864, 800, 801, 805, 807, 809,
        811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 844,
        778, 800, 806, 808, 810, 812, 814, 816, 818, 820, 822, 824, 826, 828, 830, 832, 834, 836,
        838, 840, 842, 844, 845, 40, 801, 802, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823,
        825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 41, 801, 803, 805, 807, 809, 811, 813,
        815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 801, 804, 869,
        712, 731, 732, 779, 805, 806, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829,
        831, 833, 835, 837, 839, 841, 843, 713, 731, 732, 780, 805, 807, 808, 809, 811, 813, 815,
        817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 714, 731, 732, 781,
        805, 807, 809, 810, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837,
        839, 841, 843, 715, 731, 732, 782, 805, 807, 809, 811, 812, 813, 815, 817, 819, 821, 823,
        825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 716, 731, 732, 783, 805, 807, 809, 811,
        813, 814, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 717,
        731, 732, 784, 805, 807, 809, 811, 813, 815, 816, 817, 819, 821, 823, 825, 827, 829, 831,
        833, 835, 837, 839, 841, 843, 718, 731, 732, 785, 805, 807, 809, 811, 813, 815, 817, 818,
        819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 719, 731, 732, 786, 805,
        807, 809, 811, 813, 815, 817, 819, 820, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839,
        841, 843, 720, 731, 732, 787, 805, 807, 809, 811, 813, 815, 817, 819, 821, 822, 823, 825,
        827, 829, 831, 833, 835, 837, 839, 841, 843, 721, 731, 732, 788, 805, 807, 809, 811, 813,
        815, 817, 819, 821, 823, 824, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 722, 731,
        732, 789, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 826, 827, 829, 831, 833,
        835, 837, 839, 841, 843, 723, 731, 732, 790, 805, 807, 809, 811, 813, 815, 817, 819, 821,
        823, 825, 827, 828, 829, 831, 833, 835, 837, 839, 841, 843, 724, 731, 732, 791, 805, 807,
        809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 830, 831, 833, 835, 837, 839, 841,
        843, 725, 731, 732, 792, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829,
        831, 832, 833, 835, 837, 839, 841, 843, 726, 731, 732, 793, 805, 807, 809, 811, 813, 815,
        817, 819, 821, 823, 825, 827, 829, 831, 833, 834, 835, 837, 839, 841, 843, 727, 731, 732,
        794, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 836,
        837, 839, 841, 843, 728, 731, 732, 795, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823,
        825, 827, 829, 831, 833, 835, 837, 838, 839, 841, 843, 729, 731, 732, 796, 805, 807, 809,
        811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831, 833, 835, 837, 839, 840, 841, 843,
        730, 731, 732, 797, 805, 807, 809, 811, 813, 815, 817, 819, 821, 823, 825, 827, 829, 831,
        833, 835, 837, 839, 841, 842, 843, 731, 732, 805, 807, 809, 811, 813, 815, 817, 819, 821,
        823, 825, 827, 829, 831, 833, 835, 837, 839, 841, 843, 844, 711, 731, 778, 806, 808, 810,
        812, 814, 816, 818, 820, 822, 824, 826, 828, 830, 832, 834, 836, 838, 840, 842, 844, 732,
        802, 732, 803, 732, 737, 804, 1203, 1222, 1223, 1270, 1294, 1295, 1296, 1298, 1300, 1302,
        1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1204, 1222, 1223, 1271, 1294, 1296, 1297, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312,
        1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1205, 1222, 1223, 1272, 1294,
        1296, 1298, 1299, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322,
        1324, 1326, 1328, 1330, 1332, 1206, 1222, 1223, 1273, 1294, 1296, 1298, 1300, 1301, 1302,
        1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1207, 1222, 1223, 1274, 1294, 1296, 1298, 1300, 1302, 1303, 1304, 1306, 1308, 1310, 1312,
        1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1208, 1222, 1223, 1275, 1294,
        1296, 1298, 1300, 1302, 1304, 1305, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322,
        1324, 1326, 1328, 1330, 1332, 1209, 1222, 1223, 1276, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1307, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1210, 1222, 1223, 1277, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1309, 1310, 1312,
        1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1211, 1222, 1223, 1278, 1294,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1311, 1312, 1314, 1316, 1318, 1320, 1322,
        1324, 1326, 1328, 1330, 1332, 1212, 1222, 1223, 1279, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1308, 1310, 1312, 1313, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1213, 1222, 1223, 1280, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314,
        1315, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 1214, 1222, 1223, 1281, 1294,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1317, 1318, 1320, 1322,
        1324, 1326, 1328, 1330, 1332, 1215, 1222, 1223, 1282, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1308, 1310, 1312, 1314, 1316, 1318, 1319, 1320, 1322, 1324, 1326, 1328, 1330, 1332,
        1216, 1222, 1223, 1283, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314,
        1316, 1318, 1320, 1321, 1322, 1324, 1326, 1328, 1330, 1332, 1217, 1222, 1223, 1284, 1294,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1323,
        1324, 1326, 1328, 1330, 1332, 1218, 1222, 1223, 1285, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1325, 1326, 1328, 1330, 1332,
        1219, 1222, 1223, 1286, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314,
        1316, 1318, 1320, 1322, 1324, 1326, 1327, 1328, 1330, 1332, 1220, 1222, 1223, 1287, 1294,
        1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324,
        1326, 1328, 1329, 1330, 1332, 1221, 1222, 1223, 1288, 1294, 1296, 1298, 1300, 1302, 1304,
        1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1331, 1332,
        1222, 1223, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318,
        1320, 1322, 1324, 1326, 1328, 1330, 1332, 1333, 1202, 1222, 1269, 1295, 1297, 1299, 1301,
        1303, 1305, 1307, 1309, 1311, 1313, 1315, 1317, 1319, 1321, 1323, 1325, 1327, 1329, 1331,
        1333, 54, 1223, 1291, 1294, 1296, 1298, 1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314,
        1316, 1318, 1320, 1322, 1324, 1326, 1328, 1330, 1332, 55, 1223, 1292, 1294, 1296, 1298,
        1300, 1302, 1304, 1306, 1308, 1310, 1312, 1314, 1316, 1318, 1320, 1322, 1324, 1326, 1328,
        1330, 1332, 1223, 1228, 1293, 1136, 1157, 1158, 1162, 1163, 1164, 1166, 1168, 1170, 1172,
        1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1203,
        1137, 1157, 1158, 1162, 1164, 1165, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1204, 1138, 1157, 1158, 1162, 1164,
        1166, 1167, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1205, 1139, 1157, 1158, 1162, 1164, 1166, 1168, 1169, 1170, 1172,
        1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1206,
        1140, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1171, 1172, 1174, 1176, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1207, 1141, 1157, 1158, 1162, 1164,
        1166, 1168, 1170, 1172, 1173, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1208, 1142, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1175, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1209,
        1143, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1177, 1178, 1180, 1182,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1210, 1144, 1157, 1158, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1179, 1180, 1182, 1184, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1211, 1145, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1181, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1212,
        1146, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1183,
        1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1213, 1147, 1157, 1158, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1185, 1186, 1188, 1190, 1192,
        1194, 1196, 1198, 1200, 1214, 1148, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1186, 1187, 1188, 1190, 1192, 1194, 1196, 1198, 1200, 1215,
        1149, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184,
        1186, 1188, 1189, 1190, 1192, 1194, 1196, 1198, 1200, 1216, 1150, 1157, 1158, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1191, 1192,
        1194, 1196, 1198, 1200, 1217, 1151, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1193, 1194, 1196, 1198, 1200, 1218,
        1152, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184,
        1186, 1188, 1190, 1192, 1194, 1195, 1196, 1198, 1200, 1219, 1153, 1157, 1158, 1162, 1164,
        1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194,
        1196, 1197, 1198, 1200, 1220, 1154, 1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174,
        1176, 1178, 1180, 1182, 1184, 1186, 1188, 1190, 1192, 1194, 1196, 1198, 1199, 1200, 1221,
        1157, 1158, 1162, 1164, 1166, 1168, 1170, 1172, 1174, 1176, 1178, 1180, 1182, 1184, 1186,
        1188, 1190, 1192, 1194, 1196, 1198, 1200, 1201, 1135, 1157, 1163, 1165, 1167, 1169, 1171,
        1173, 1175, 1177, 1179, 1181, 1183, 1185, 1187, 1189, 1191, 1193, 1195, 1197, 1199, 1201,
        1202, 1158, 1159, 1158, 1160, 1158, 1161, 1228, 1136, 1155, 1156, 1203, 1229, 1230, 1231,
        1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261,
        1263, 1265, 1267, 1137, 1155, 1156, 1204, 1229, 1231, 1232, 1233, 1235, 1237, 1239, 1241,
        1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1138, 1155,
        1156, 1205, 1229, 1231, 1233, 1234, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251,
        1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1139, 1155, 1156, 1206, 1229, 1231, 1233,
        1235, 1236, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261,
        1263, 1265, 1267, 1140, 1155, 1156, 1207, 1229, 1231, 1233, 1235, 1237, 1238, 1239, 1241,
        1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1141, 1155,
        1156, 1208, 1229, 1231, 1233, 1235, 1237, 1239, 1240, 1241, 1243, 1245, 1247, 1249, 1251,
        1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1142, 1155, 1156, 1209, 1229, 1231, 1233,
        1235, 1237, 1239, 1241, 1242, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261,
        1263, 1265, 1267, 1143, 1155, 1156, 1210, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243,
        1244, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1144, 1155,
        1156, 1211, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1246, 1247, 1249, 1251,
        1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1145, 1155, 1156, 1212, 1229, 1231, 1233,
        1235, 1237, 1239, 1241, 1243, 1245, 1247, 1248, 1249, 1251, 1253, 1255, 1257, 1259, 1261,
        1263, 1265, 1267, 1146, 1155, 1156, 1213, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243,
        1245, 1247, 1249, 1250, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1147, 1155,
        1156, 1214, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1252,
        1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1148, 1155, 1156, 1215, 1229, 1231, 1233,
        1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1254, 1255, 1257, 1259, 1261,
        1263, 1265, 1267, 1149, 1155, 1156, 1216, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243,
        1245, 1247, 1249, 1251, 1253, 1255, 1256, 1257, 1259, 1261, 1263, 1265, 1267, 1150, 1155,
        1156, 1217, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253,
        1255, 1257, 1258, 1259, 1261, 1263, 1265, 1267, 1151, 1155, 1156, 1218, 1229, 1231, 1233,
        1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1260, 1261,
        1263, 1265, 1267, 1152, 1155, 1156, 1219, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243,
        1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1262, 1263, 1265, 1267, 1153, 1155,
        1156, 1220, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253,
        1255, 1257, 1259, 1261, 1263, 1264, 1265, 1267, 1154, 1155, 1156, 1221, 1229, 1231, 1233,
        1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263,
        1265, 1266, 1267, 1155, 1156, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247,
        1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1268, 1135, 1155, 1202, 1230,
        1232, 1234, 1236, 1238, 1240, 1242, 1244, 1246, 1248, 1250, 1252, 1254, 1256, 1258, 1260,
        1262, 1264, 1266, 1268, 52, 1156, 1226, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243,
        1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 53, 1156, 1227,
        1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257,
        1259, 1261, 1263, 1265, 1267, 1156, 1161, 1228, 1203, 1224, 1225, 1229, 1230, 1231, 1233,
        1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263,
        1265, 1267, 1270, 1204, 1224, 1225, 1229, 1231, 1232, 1233, 1235, 1237, 1239, 1241, 1243,
        1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1271, 1205, 1224,
        1225, 1229, 1231, 1233, 1234, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253,
        1255, 1257, 1259, 1261, 1263, 1265, 1267, 1272, 1206, 1224, 1225, 1229, 1231, 1233, 1235,
        1236, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263,
        1265, 1267, 1273, 1207, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1238, 1239, 1241, 1243,
        1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1274, 1208, 1224,
        1225, 1229, 1231, 1233, 1235, 1237, 1239, 1240, 1241, 1243, 1245, 1247, 1249, 1251, 1253,
        1255, 1257, 1259, 1261, 1263, 1265, 1267, 1275, 1209, 1224, 1225, 1229, 1231, 1233, 1235,
        1237, 1239, 1241, 1242, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263,
        1265, 1267, 1276, 1210, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1244,
        1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1277, 1211, 1224,
        1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1246, 1247, 1249, 1251, 1253,
        1255, 1257, 1259, 1261, 1263, 1265, 1267, 1278, 1212, 1224, 1225, 1229, 1231, 1233, 1235,
        1237, 1239, 1241, 1243, 1245, 1247, 1248, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263,
        1265, 1267, 1279, 1213, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245,
        1247, 1249, 1250, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1280, 1214, 1224,
        1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1252, 1253,
        1255, 1257, 1259, 1261, 1263, 1265, 1267, 1281, 1215, 1224, 1225, 1229, 1231, 1233, 1235,
        1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1254, 1255, 1257, 1259, 1261, 1263,
        1265, 1267, 1282, 1216, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245,
        1247, 1249, 1251, 1253, 1255, 1256, 1257, 1259, 1261, 1263, 1265, 1267, 1283, 1217, 1224,
        1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255,
        1257, 1258, 1259, 1261, 1263, 1265, 1267, 1284, 1218, 1224, 1225, 1229, 1231, 1233, 1235,
        1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1260, 1261, 1263,
        1265, 1267, 1285, 1219, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245,
        1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1262, 1263, 1265, 1267, 1286, 1220, 1224,
        1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255,
        1257, 1259, 1261, 1263, 1264, 1265, 1267, 1287, 1221, 1224, 1225, 1229, 1231, 1233, 1235,
        1237, 1239, 1241, 1243, 1245, 1247, 1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265,
        1266, 1267, 1288, 1224, 1225, 1229, 1231, 1233, 1235, 1237, 1239, 1241, 1243, 1245, 1247,
        1249, 1251, 1253, 1255, 1257, 1259, 1261, 1263, 1265, 1267, 1268, 1202, 1224, 1230, 1232,
        1234, 1236, 1238, 1240, 1242, 1244, 1246, 1248, 1250, 1252, 1254, 1256, 1258, 1260, 1262,
        1264, 1266, 1268, 1269, 1225, 1226, 1225, 1227, 1225, 1228, 1293, 846, 865, 866, 870, 871,
        872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906,
        908, 847, 865, 866, 870, 872, 873, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894,
        896, 898, 900, 902, 904, 906, 908, 848, 865, 866, 870, 872, 874, 875, 876, 878, 880, 882,
        884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 849, 865, 866, 870, 872,
        874, 876, 877, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906,
        908, 850, 865, 866, 870, 872, 874, 876, 878, 879, 880, 882, 884, 886, 888, 890, 892, 894,
        896, 898, 900, 902, 904, 906, 908, 851, 865, 866, 870, 872, 874, 876, 878, 880, 881, 882,
        884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 852, 865, 866, 870, 872,
        874, 876, 878, 880, 882, 883, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906,
        908, 853, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884, 885, 886, 888, 890, 892, 894,
        896, 898, 900, 902, 904, 906, 908, 854, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884,
        886, 887, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 855, 865, 866, 870, 872,
        874, 876, 878, 880, 882, 884, 886, 888, 889, 890, 892, 894, 896, 898, 900, 902, 904, 906,
        908, 856, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 891, 892, 894,
        896, 898, 900, 902, 904, 906, 908, 857, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884,
        886, 888, 890, 892, 893, 894, 896, 898, 900, 902, 904, 906, 908, 858, 865, 866, 870, 872,
        874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 895, 896, 898, 900, 902, 904, 906,
        908, 859, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896,
        897, 898, 900, 902, 904, 906, 908, 860, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884,
        886, 888, 890, 892, 894, 896, 898, 899, 900, 902, 904, 906, 908, 861, 865, 866, 870, 872,
        874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 901, 902, 904, 906,
        908, 862, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896,
        898, 900, 902, 903, 904, 906, 908, 863, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884,
        886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 905, 906, 908, 864, 865, 866, 870, 872,
        874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 907,
        908, 865, 866, 870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898,
        900, 902, 904, 906, 908, 909, 845, 865, 871, 873, 875, 877, 879, 881, 883, 885, 887, 889,
        891, 893, 895, 897, 899, 901, 903, 905, 907, 909, 42, 866, 867, 870, 872, 874, 876, 878,
        880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904, 906, 908, 43, 866, 868,
        870, 872, 874, 876, 878, 880, 882, 884, 886, 888, 890, 892, 894, 896, 898, 900, 902, 904,
        906, 908, 866, 869, 112, 57, 161, 185, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220,
        221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233, 234, 235, 236, 237, 238,
        239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 58, 300, 324, 350, 351,
        352, 353, 354, 355, 356, 357, 358, 359, 360, 361, 362, 363, 364, 365, 366, 367, 368, 369,
        370, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386, 387,
        388, 389, 390, 413, 441, 442, 443, 444, 445, 446, 447, 448, 449, 450, 451, 452, 453, 454,
        455, 456, 457, 458, 459, 460, 461, 462, 463, 464, 465, 466, 467, 468, 469, 470, 471, 472,
        473, 474, 475, 476, 477, 478, 479, 480, 481, 536, 537, 538, 539, 540, 541, 542, 543, 544,
        545, 546, 547, 548, 549, 550, 551, 552, 553, 554, 555, 556, 557, 558, 559, 560, 561, 562,
        563, 564, 565, 566, 567, 568, 569, 570, 571, 572, 573, 574, 575, 576, 603, 604, 605, 606,
        607, 608, 609, 610, 611, 612, 613, 614, 615, 616, 617, 618, 619, 620, 621, 622, 623, 624,
        625, 626, 627, 628, 629, 630, 631, 632, 633, 634, 635, 636, 637, 638, 639, 640, 641, 642,
        643, 670, 671, 672, 673, 674, 675, 676, 677, 678, 679, 680, 681, 682, 683, 684, 685, 686,
        687, 688, 689, 690, 691, 692, 693, 694, 695, 696, 697, 698, 699, 700, 701, 702, 703, 704,
        705, 706, 707, 708, 709, 710, 737, 738, 739, 740, 741, 742, 743, 744, 745, 746, 747, 748,
        749, 750, 751, 752, 753, 754, 755, 756, 757, 758, 759, 760, 761, 762, 763, 764, 765, 766,
        767, 768, 769, 770, 771, 772, 773, 774, 775, 776, 777, 804, 805, 806, 807, 808, 809, 810,
        811, 812, 813, 814, 815, 816, 817, 818, 819, 820, 821, 822, 823, 824, 825, 826, 827, 828,
        829, 830, 831, 832, 833, 834, 835, 836, 837, 838, 839, 840, 841, 842, 843, 844, 869, 870,
        871, 872, 873, 874, 875, 876, 877, 878, 879, 880, 881, 882, 883, 884, 885, 886, 887, 888,
        889, 890, 891, 892, 893, 894, 895, 896, 897, 898, 899, 900, 901, 902, 903, 904, 905, 906,
        907, 908, 909, 960, 961, 962, 963, 964, 965, 966, 967, 968, 969, 970, 971, 972, 973, 974,
        975, 976, 977, 978, 979, 980, 981, 982, 983, 984, 985, 986, 987, 988, 989, 990, 991, 992,
        993, 994, 995, 996, 997, 998, 999, 1000, 1027, 1028, 1029, 1030, 1031, 1032, 1033, 1034,
        1035, 1036, 1037, 1038, 1039, 1040, 1041, 1042, 1043, 1044, 1045, 1046, 1047, 1048, 1049,
        1050, 1051, 1052, 1053, 1054, 1055, 1056, 1057, 1058, 1059, 1060, 1061, 1062, 1063, 1064,
        1065, 1066, 1067, 1094, 1095, 1096, 1097, 1098, 1099, 1100, 1101, 1102, 1103, 1104, 1105,
        1106, 1107, 1108, 1109, 1110, 1111, 1112, 1113, 1114, 1115, 1116, 1117, 1118, 1119, 1120,
        1121, 1122, 1123, 1124, 1125, 1126, 1127, 1128, 1129, 1130, 1131, 1132, 1133, 1134, 1161,
        1162, 1163, 1164, 1165, 1166, 1167, 1168, 1169, 1170, 1171, 1172, 1173, 1174, 1175, 1176,
        1177, 1178, 1179, 1180, 1181, 1182, 1183, 1184, 1185, 1186, 1187, 1188, 1189, 1190, 1191,
        1192, 1193, 1194, 1195, 1196, 1197, 1198, 1199, 1200, 1201, 1228, 1229, 1230, 1231, 1232,
        1233, 1234, 1235, 1236, 1237, 1238, 1239, 1240, 1241, 1242, 1243, 1244, 1245, 1246, 1247,
        1248, 1249, 1250, 1251, 1252, 1253, 1254, 1255, 1256, 1257, 1258, 1259, 1260, 1261, 1262,
        1263, 1264, 1265, 1266, 1267, 1268, 1293, 1294, 1295, 1296, 1297, 1298, 1299, 1300, 1301,
        1302, 1303, 1304, 1305, 1306, 1307, 1308, 1309, 1310, 1311, 1312, 1313, 1314, 1315, 1316,
        1317, 1318, 1319, 1320, 1321, 1322, 1323, 1324, 1325, 1326, 1327, 1328, 1329, 1330, 1331,
        1332, 1333, 59, 1377, 1356, 1357, 1358, 1359, 1360, 1361, 1362, 1363, 1364, 1365, 1366,
        1367, 1368, 1369, 1370, 1371, 1372, 1373, 1374, 1375, 1376, 1377, 1406, 60, 1475,
    ];

    let colst = vec![
        0, 4, 8, 12, 16, 20, 24, 28, 32, 36, 40, 44, 48, 52, 56, 60, 64, 68, 72, 76, 79, 81, 83,
        86, 88, 92, 96, 100, 104, 108, 112, 116, 120, 124, 128, 132, 136, 140, 144, 148, 152, 156,
        160, 164, 166, 169, 170, 173, 176, 180, 184, 188, 192, 196, 200, 204, 208, 212, 216, 220,
        224, 228, 232, 236, 240, 244, 248, 252, 254, 257, 259, 262, 265, 269, 273, 277, 281, 285,
        289, 293, 297, 301, 305, 309, 313, 317, 321, 325, 329, 333, 337, 341, 343, 346, 349, 352,
        355, 358, 361, 364, 367, 370, 373, 376, 379, 382, 385, 388, 391, 394, 397, 400, 403, 406,
        409, 412, 413, 416, 417, 419, 421, 446, 471, 496, 521, 546, 571, 596, 621, 646, 671, 696,
        721, 746, 771, 796, 821, 846, 871, 896, 919, 942, 965, 989, 992, 1017, 1042, 1067, 1092,
        1117, 1142, 1167, 1192, 1217, 1242, 1267, 1292, 1317, 1342, 1367, 1392, 1417, 1442, 1467,
        1490, 1513, 1515, 1518, 1521, 1525, 1529, 1533, 1537, 1541, 1545, 1549, 1553, 1557, 1561,
        1565, 1569, 1573, 1577, 1581, 1585, 1589, 1593, 1597, 1599, 1602, 1605, 1608, 1611, 1614,
        1617, 1620, 1623, 1626, 1629, 1632, 1635, 1638, 1641, 1644, 1647, 1650, 1653, 1656, 1659,
        1662, 1665, 1668, 1669, 1672, 1673, 1675, 1677, 1702, 1727, 1752, 1777, 1802, 1827, 1852,
        1877, 1902, 1927, 1952, 1977, 2002, 2027, 2052, 2077, 2102, 2127, 2152, 2175, 2198, 2221,
        2245, 2248, 2273, 2298, 2323, 2348, 2373, 2398, 2423, 2448, 2473, 2498, 2523, 2548, 2573,
        2598, 2623, 2648, 2673, 2698, 2723, 2746, 2769, 2772, 2775, 2778, 2781, 2784, 2787, 2790,
        2793, 2796, 2799, 2802, 2805, 2808, 2811, 2814, 2817, 2820, 2823, 2826, 2829, 2832, 2835,
        2836, 2839, 2840, 2842, 2844, 2848, 2852, 2856, 2860, 2864, 2868, 2872, 2876, 2880, 2884,
        2888, 2892, 2896, 2900, 2904, 2908, 2912, 2916, 2920, 2922, 2925, 2927, 2930, 2933, 2937,
        2941, 2945, 2949, 2953, 2957, 2961, 2965, 2969, 2973, 2977, 2981, 2985, 2989, 2993, 2997,
        3001, 3005, 3009, 3011, 3014, 3016, 3019, 3022, 3026, 3030, 3034, 3038, 3042, 3046, 3050,
        3054, 3058, 3062, 3066, 3070, 3074, 3078, 3082, 3086, 3090, 3094, 3098, 3100, 3103, 3104,
        3107, 3110, 3113, 3116, 3119, 3122, 3125, 3128, 3131, 3134, 3137, 3140, 3143, 3146, 3149,
        3152, 3155, 3158, 3161, 3164, 3167, 3169, 3171, 3173, 3175, 3177, 3203, 3229, 3255, 3281,
        3307, 3333, 3359, 3385, 3411, 3437, 3463, 3489, 3515, 3541, 3567, 3593, 3619, 3645, 3671,
        3695, 3718, 3722, 3726, 3730, 3732, 3734, 3736, 3738, 3740, 3742, 3744, 3746, 3748, 3750,
        3752, 3754, 3756, 3758, 3760, 3762, 3764, 3766, 3768, 3769, 3771, 3772, 3773, 3774, 3778,
        3782, 3786, 3790, 3794, 3798, 3802, 3806, 3810, 3814, 3818, 3822, 3826, 3830, 3834, 3838,
        3842, 3846, 3850, 3852, 3855, 3856, 3858, 3861, 3865, 3869, 3873, 3877, 3881, 3885, 3889,
        3893, 3897, 3901, 3905, 3909, 3913, 3917, 3921, 3925, 3929, 3933, 3937, 3939, 3942, 3944,
        3947, 3950, 3954, 3958, 3962, 3966, 3970, 3974, 3978, 3982, 3986, 3990, 3994, 3998, 4002,
        4006, 4010, 4014, 4018, 4022, 4026, 4028, 4031, 4033, 4036, 4039, 4064, 4089, 4114, 4139,
        4164, 4189, 4214, 4239, 4264, 4289, 4314, 4339, 4364, 4389, 4414, 4439, 4464, 4489, 4514,
        4537, 4560, 4583, 4607, 4610, 4635, 4660, 4685, 4710, 4735, 4760, 4785, 4810, 4835, 4860,
        4885, 4910, 4935, 4960, 4985, 5010, 5035, 5060, 5085, 5108, 5131, 5133, 5136, 5139, 5143,
        5147, 5151, 5155, 5159, 5163, 5167, 5171, 5175, 5179, 5183, 5187, 5191, 5195, 5199, 5203,
        5207, 5211, 5215, 5217, 5220, 5221, 5223, 5226, 5251, 5276, 5301, 5326, 5351, 5376, 5401,
        5426, 5451, 5476, 5501, 5526, 5551, 5576, 5601, 5626, 5651, 5676, 5701, 5724, 5747, 5749,
        5751, 5754, 5779, 5804, 5829, 5854, 5879, 5904, 5929, 5954, 5979, 6004, 6029, 6054, 6079,
        6104, 6129, 6154, 6179, 6204, 6229, 6252, 6275, 6298, 6321, 6324, 6349, 6374, 6399, 6424,
        6449, 6474, 6499, 6524, 6549, 6574, 6599, 6624, 6649, 6674, 6699, 6724, 6749, 6774, 6799,
        6822, 6845, 6847, 6850, 6853, 6878, 6903, 6928, 6953, 6978, 7003, 7028, 7053, 7078, 7103,
        7128, 7153, 7178, 7203, 7228, 7253, 7278, 7303, 7328, 7351, 7374, 7397, 7420, 7423, 7427,
        7431, 7435, 7439, 7443, 7447, 7451, 7455, 7459, 7463, 7467, 7471, 7475, 7479, 7483, 7487,
        7491, 7495, 7499, 7501, 7504, 7505, 7507, 7510, 7535, 7560, 7585, 7610, 7635, 7660, 7685,
        7710, 7735, 7760, 7785, 7810, 7835, 7860, 7885, 7910, 7935, 7960, 7985, 8008, 8031, 8054,
        8078, 8081, 8106, 8131, 8156, 8181, 8206, 8231, 8256, 8281, 8306, 8331, 8356, 8381, 8406,
        8431, 8456, 8481, 8506, 8531, 8556, 8579, 8602, 8604, 8606, 8609, 8634, 8659, 8684, 8709,
        8734, 8759, 8784, 8809, 8834, 8859, 8884, 8909, 8934, 8959, 8984, 9009, 9034, 9059, 9084,
        9107, 9130, 9153, 9176, 9179, 9204, 9229, 9254, 9279, 9304, 9329, 9354, 9379, 9404, 9429,
        9454, 9479, 9504, 9529, 9554, 9579, 9604, 9629, 9654, 9677, 9700, 9702, 9704, 9707, 9732,
        9757, 9782, 9807, 9832, 9857, 9882, 9907, 9932, 9957, 9982, 10007, 10032, 10057, 10082,
        10107, 10132, 10157, 10182, 10205, 10228, 10251, 10274, 10277, 10302, 10327, 10352, 10377,
        10402, 10427, 10452, 10477, 10502, 10527, 10552, 10577, 10602, 10627, 10652, 10677, 10702,
        10727, 10752, 10775, 10798, 10800, 10802, 10805, 10830, 10855, 10880, 10905, 10930, 10955,
        10980, 11005, 11030, 11055, 11080, 11105, 11130, 11155, 11180, 11205, 11230, 11255, 11280,
        11303, 11326, 11349, 11372, 11375, 11400, 11425, 11450, 11475, 11500, 11525, 11550, 11575,
        11600, 11625, 11650, 11675, 11700, 11725, 11750, 11775, 11800, 11825, 11850, 11873, 11896,
        11898, 11900, 11903, 11928, 11953, 11978, 12003, 12028, 12053, 12078, 12103, 12128, 12153,
        12178, 12203, 12228, 12253, 12278, 12303, 12328, 12353, 12378, 12401, 12424, 12447, 12470,
        12473, 12498, 12523, 12548, 12573, 12598, 12623, 12648, 12673, 12698, 12723, 12748, 12773,
        12798, 12823, 12848, 12873, 12898, 12923, 12948, 12971, 12994, 12996, 12998, 13001, 13026,
        13051, 13076, 13101, 13126, 13151, 13176, 13201, 13226, 13251, 13276, 13301, 13326, 13351,
        13376, 13401, 13426, 13451, 13476, 13499, 13522, 13545, 13568, 13571, 13596, 13621, 13646,
        13671, 13696, 13721, 13746, 13771, 13796, 13821, 13846, 13871, 13896, 13921, 13946, 13971,
        13996, 14021, 14046, 14069, 14092, 14094, 14096, 14099, 14124, 14149, 14174, 14199, 14224,
        14249, 14274, 14299, 14324, 14349, 14374, 14399, 14424, 14449, 14474, 14499, 14524, 14549,
        14574, 14597, 14620, 14643, 14666, 14669, 14694, 14719, 14744, 14769, 14794, 14819, 14844,
        14869, 14894, 14919, 14944, 14969, 14994, 15019, 15044, 15069, 15094, 15119, 15144, 15167,
        15190, 15192, 15194, 15197, 15222, 15247, 15272, 15297, 15322, 15347, 15372, 15397, 15422,
        15447, 15472, 15497, 15522, 15547, 15572, 15597, 15622, 15647, 15672, 15695, 15718, 15741,
        15764, 15767, 15792, 15817, 15842, 15867, 15892, 15917, 15942, 15967, 15992, 16017, 16042,
        16067, 16092, 16117, 16142, 16167, 16192, 16217, 16242, 16265, 16288, 16290, 16292, 16295,
        16320, 16345, 16370, 16395, 16420, 16445, 16470, 16495, 16520, 16545, 16570, 16595, 16620,
        16645, 16670, 16695, 16720, 16745, 16770, 16793, 16816, 16839, 16862, 16865, 16890, 16915,
        16940, 16965, 16990, 17015, 17040, 17065, 17090, 17115, 17140, 17165, 17190, 17215, 17240,
        17265, 17290, 17315, 17340, 17363, 17386, 17388, 17390, 17393, 17417, 17441, 17465, 17489,
        17513, 17537, 17561, 17585, 17609, 17633, 17657, 17681, 17705, 17729, 17753, 17777, 17801,
        17825, 17849, 17872, 17894, 17917, 17940, 17942, 17943, 17945, 17946, 17947, 17949, 17951,
        17953, 17955, 17957, 17959, 17961, 17963, 17965, 17967, 17969, 17971, 17973, 17975, 17977,
        17979, 17981, 17983, 17985, 17987, 17989, 17990, 17991, 17993, 17995, 17997, 17999, 18001,
        18003, 18005, 18007, 18009, 18011, 18013, 18015, 18017, 18019, 18021, 18023, 18025, 18027,
        18029, 18031, 18032, 18033, 18035, 18037, 18039, 18041, 18043, 18045, 18047, 18049, 18051,
        18053, 18055, 18057, 18059, 18061, 18063, 18065, 18067, 18069, 18071, 18073, 18074, 18076,
        18078, 18080, 18082, 18084, 18086, 18088, 18090, 18092, 18094, 18096, 18098, 18100, 18102,
        18104, 18106, 18108, 18110, 18112, 18114, 18115, 18117, 18119, 18121, 18123, 18125, 18127,
        18129, 18131, 18133, 18135, 18137, 18139, 18141, 18143, 18145, 18147, 18149, 18151, 18153,
        18155, 18156, 18158, 18160, 18162, 18164, 18166, 18168, 18170, 18172, 18174, 18176, 18178,
        18180, 18182, 18184, 18186, 18188, 18190, 18192, 18194, 18196, 18197, 18199, 18201, 18203,
        18205, 18207, 18209, 18211, 18213, 18215, 18217, 18219, 18221, 18223, 18225, 18227, 18229,
        18231, 18233, 18235, 18237, 18238, 18240, 18242, 18244, 18246, 18248, 18250, 18252, 18254,
        18256, 18258, 18260, 18262, 18264, 18266, 18268, 18270, 18272, 18274, 18276, 18278, 18279,
        18281, 18283, 18285, 18287, 18289, 18291, 18293, 18295, 18297, 18299, 18301, 18303, 18305,
        18307, 18309, 18311, 18313, 18315, 18317, 18319, 18320, 18322, 18324, 18326, 18328, 18330,
        18332, 18334, 18336, 18338, 18340, 18342, 18344, 18346, 18348, 18350, 18352, 18354, 18356,
        18358, 18360, 18361, 18363, 18365, 18367, 18369, 18371, 18373, 18375, 18377, 18379, 18381,
        18383, 18385, 18387, 18389, 18391, 18393, 18395, 18397, 18399, 18401, 18402, 18404, 18406,
        18408, 18410, 18412, 18414, 18416, 18418, 18420, 18422, 18424, 18426, 18428, 18430, 18432,
        18434, 18436, 18438, 18440, 18442, 18443, 18445, 18447, 18449, 18451, 18453, 18455, 18457,
        18459, 18461, 18463, 18465, 18467, 18469, 18471, 18473, 18475, 18477, 18479, 18481, 18483,
        18484, 18486, 18488, 18490, 18492, 18494, 18496, 18498, 18500, 18502, 18504, 18506, 18508,
        18510, 18512, 18514, 18516, 18518, 18520, 18522, 18524, 18525, 18527, 18529, 18531, 18533,
        18535, 18537, 18539, 18541, 18543, 18545, 18547, 18549, 18551, 18553, 18555, 18557, 18559,
        18561, 18563, 18565, 18567, 18589, 18590, 18592,
    ];

    let nz = vec![
        -1.0,
        0.947976098819,
        0.999999957331,
        0.0026183635903,
        -1.0,
        0.682263602037,
        0.999999954206,
        0.00111738665827,
        -1.0,
        1.0,
        0.999999948593,
        0.00227761242926,
        -1.0,
        1.0,
        0.999999951654,
        0.00176304741968,
        -1.0,
        0.999999994608,
        0.99999995951,
        0.00193211057024,
        -1.0,
        1.0,
        0.999999976443,
        0.00269190757248,
        -1.0,
        1.0,
        0.999999950105,
        0.00247357613436,
        -1.0,
        1.0,
        0.99999995699,
        0.00283913750398,
        -1.0,
        1.0,
        0.999999945541,
        0.00263093952491,
        -1.0,
        0.999999999409,
        1.0,
        0.00318317131112,
        -1.0,
        1.0,
        0.999999996341,
        0.00284550808338,
        -1.0,
        1.0,
        0.999999996341,
        0.00343711581928,
        -1.0,
        1.0,
        0.999999414265,
        0.0015011431945,
        -1.0,
        1.0,
        0.999999996341,
        0.00386725123146,
        -1.0,
        1.0,
        0.999999996341,
        0.00411744690822,
        -1.0,
        1.0,
        0.999999996341,
        0.00336949655326,
        -1.0,
        1.0,
        0.999999996341,
        0.00330517639608,
        -1.0,
        1.0,
        0.999999414265,
        0.00365173240578,
        -1.0,
        1.0,
        0.999999996341,
        0.00363622695039,
        -1.0,
        0.999999996341,
        0.00365618780537,
        0.999999964667,
        -0.999999948593,
        -1.0,
        1.0,
        -1.0,
        -0.0406042553141,
        1.0,
        -0.00815807963122,
        1.0,
        -0.961044170577,
        0.99999987073,
        0.00259490828773,
        -1.0,
        -0.790653591723,
        0.999999886521,
        0.00236342378223,
        -1.0,
        -0.999714058969,
        0.99999988556,
        0.00275050957663,
        -1.0,
        -0.999999991474,
        0.999999885572,
        0.00336108768431,
        -1.0,
        -0.999999994433,
        0.999999884483,
        0.00447118451674,
        -1.0,
        -0.999999999594,
        0.999999866774,
        0.0059694093261,
        -1.0,
        -0.999999999968,
        0.999999880012,
        0.005925618852,
        -1.0,
        -0.999999999786,
        0.999999915836,
        0.0070649840293,
        -1.0,
        -0.999999999913,
        0.999999897077,
        0.00705624003659,
        -1.0,
        -0.999999999599,
        0.999999831988,
        0.00955238711615,
        -1.0,
        -0.999999999849,
        0.999999762957,
        0.00827939936963,
        -1.0,
        -0.999999999846,
        0.999999697253,
        0.0107878283521,
        -1.0,
        -0.999999999771,
        0.99999977798,
        0.0098865930499,
        -1.0,
        -0.999999999696,
        0.999999912962,
        0.0132204157743,
        -1.0,
        -0.999999999924,
        0.999999913575,
        0.014458349284,
        -1.0,
        -0.999999999704,
        0.999999762657,
        0.00854500825252,
        -1.0,
        -0.999999999626,
        0.999999600854,
        0.00847326055559,
        -1.0,
        -0.999999999846,
        1.00000000134,
        0.00977135073037,
        -1.0,
        -0.999999999768,
        0.999999780397,
        0.00972357942446,
        -1.0,
        0.999999813607,
        0.0107729711352,
        -0.999999969212,
        -0.999999849381,
        -1.0,
        0.990356830509,
        -0.0430052581219,
        -1.0,
        -1.0,
        -0.00815018194471,
        -1.00000000002,
        -1.00000051642,
        1.0,
        0.99999987073,
        -0.00254532416949,
        -1.0,
        1.0,
        0.999999886521,
        -0.00574398410657,
        -1.0,
        1.0,
        0.99999988556,
        -0.00339057041974,
        -1.0,
        1.0,
        0.999999885572,
        -0.00765994246503,
        -1.0,
        1.0,
        0.999999884483,
        -0.0216995879727,
        -1.0,
        1.0,
        0.999999866774,
        -0.047090992766,
        -1.0,
        1.0,
        0.999999880012,
        -0.0608605827996,
        -1.0,
        1.0,
        0.999999915836,
        -0.0943784049719,
        -1.0,
        1.0,
        0.999999897077,
        -0.103374763658,
        -1.0,
        1.0,
        0.999999831988,
        -0.148975424092,
        -1.0,
        1.0,
        0.999999762957,
        -0.127515296229,
        -1.0,
        1.0,
        0.999999697253,
        -0.162205052365,
        -1.0,
        1.0,
        0.99999977798,
        -0.176172045763,
        -1.0,
        1.0,
        0.999999912962,
        -0.194340029828,
        -1.0,
        1.0,
        0.999999913575,
        -0.210090414121,
        -1.0,
        1.0,
        0.999999762657,
        -0.121961150217,
        -1.0,
        1.0,
        0.999999600854,
        -0.123607354392,
        -1.0,
        1.0,
        1.00000000134,
        -0.140873540329,
        -1.0,
        1.0,
        0.999999780397,
        -0.141615562759,
        -1.0,
        0.999999813607,
        -0.156780300721,
        1.0,
        -0.999999849381,
        -1.0,
        -1.0,
        1.0,
        -0.0718467208838,
        1.0,
        -1.0,
        -0.00667289042536,
        0.999999877904,
        -0.999999998956,
        1.0,
        0.99999987073,
        0.000504155542341,
        -1.0,
        1.0,
        0.999999886521,
        -0.000582480377642,
        -1.0,
        1.0,
        0.99999988556,
        0.000183238262873,
        -1.0,
        1.0,
        0.999999885572,
        -0.000929796196577,
        -1.0,
        1.0,
        0.999999884483,
        -0.00339859772938,
        -1.0,
        1.0,
        0.999999866774,
        -0.00827918843461,
        -1.0,
        1.0,
        0.999999880012,
        -0.0118415116568,
        -1.0,
        1.0,
        0.999999915836,
        -0.0290948503221,
        -1.0,
        1.0,
        0.999999897077,
        -0.0375580467748,
        -1.0,
        1.0,
        0.999999831988,
        -0.121324541594,
        -1.0,
        1.0,
        0.999999762957,
        -0.081101207495,
        -1.0,
        1.0,
        0.999999697253,
        -0.145530612727,
        -1.0,
        1.0,
        0.99999977798,
        -0.163411503501,
        -1.0,
        1.0,
        0.999999912962,
        -0.177869863467,
        -1.0,
        1.0,
        0.999999913575,
        -0.191426442607,
        -1.0,
        1.0,
        0.999999762657,
        -0.0681211965878,
        -1.0,
        1.0,
        0.999999600854,
        -0.0725062564001,
        -1.0,
        1.0,
        1.00000000134,
        -0.109937739407,
        -1.0,
        1.0,
        0.999999780397,
        -0.111882678136,
        -1.0,
        0.999999813607,
        -0.138532050578,
        1.0,
        -0.999999849381,
        -1.0,
        0.976579125388,
        -1.0,
        -1.0,
        -0.0713191002836,
        1.0,
        0.999999989078,
        -0.00683888910038,
        0.999999999027,
        -0.999999477315,
        1.0,
        0.99999987073,
        -1.0,
        1.0,
        0.999999886521,
        -1.00000000007,
        1.0,
        0.99999988556,
        -1.00000000003,
        1.0,
        0.999999885572,
        -1.0,
        1.0,
        0.999999884483,
        -1.0,
        1.0,
        0.999999866774,
        -1.00000000022,
        1.0,
        0.999999880012,
        -1.00000000005,
        1.0,
        0.999999915836,
        -1.00000000032,
        1.0,
        0.999999897077,
        -1.00000000043,
        1.0,
        0.999999831988,
        -1.00000000062,
        1.0,
        0.999999762957,
        -1.00000000044,
        1.0,
        0.999999697253,
        -1.00000000068,
        1.0,
        0.99999977798,
        -1.00000000002,
        1.0,
        0.999999912962,
        -1.00000000197,
        1.0,
        0.999999913575,
        -1.00000000176,
        1.0,
        0.999999762657,
        -1.00000000041,
        1.0,
        0.999999600854,
        -1.00000000057,
        1.0,
        1.00000000134,
        -1.00000000098,
        1.0,
        0.999999780397,
        -1.00000000058,
        0.999999813607,
        1.0,
        -0.999999849381,
        -0.999999998947,
        1.0,
        -1.0,
        -1.0,
        0.999999637966,
        -0.999999869904,
        -1.0,
        0.975817055702,
        -0.0067739599464,
        0.999999981163,
        0.000122725589311,
        0.00211142942964,
        -2.93357010401e-06,
        4.18084934026e-06,
        -4.01049287839e-06,
        -1.20289473283e-06,
        -4.72037147427e-07,
        -2.89877197534e-07,
        -9.46928079629e-08,
        -6.31173709862e-08,
        -2.98396535656e-09,
        -1.43466368711e-08,
        -6.8759137363e-10,
        -1.44726165262e-10,
        -3.1893066328e-11,
        -6.98293476294e-12,
        -2.22864235073e-08,
        -1.8833147576e-08,
        -5.0549309362e-09,
        -4.56408457303e-09,
        -1.13925826591e-09,
        -1.0,
        0.975817063624,
        -0.0106698920909,
        0.999999998643,
        -0.000127149811329,
        3.03973588463e-06,
        0.00793063631169,
        -4.33140835182e-06,
        4.15340904762e-06,
        1.24587777999e-06,
        4.88852045331e-07,
        3.00369538026e-07,
        9.81171793479e-08,
        6.54060558828e-08,
        3.09214116356e-09,
        1.48661678891e-08,
        7.12477779607e-10,
        1.49965742545e-10,
        3.30436711714e-11,
        7.23409374121e-12,
        2.30919719612e-08,
        1.95162153488e-08,
        5.2376553146e-09,
        4.7295291671e-09,
        1.18035471672e-09,
        -1.0,
        0.97581705527,
        -0.00854744472896,
        0.999999992557,
        1.75894524867e-05,
        -4.20416265932e-07,
        5.99252175344e-07,
        0.00131551239087,
        -5.74923671308e-07,
        -1.72432569341e-07,
        -6.76696796577e-08,
        -4.15430449051e-08,
        -1.35709203865e-08,
        -9.04523260669e-09,
        -4.27627308748e-10,
        -2.05604600221e-09,
        -9.85410706236e-11,
        -2.07411104531e-11,
        -4.57098307304e-12,
        -1.00087130881e-12,
        -3.19401682765e-09,
        -2.69893491877e-09,
        -7.24455567557e-10,
        -6.54074955309e-10,
        -1.63280997485e-10,
        -1.0,
        0.975817063094,
        -0.0138042980335,
        0.999999999664,
        -0.00021948141388,
        5.24917360381e-06,
        -7.47634423119e-06,
        7.15571950509e-06,
        0.00195550034645,
        2.1476385101e-06,
        8.42260067391e-07,
        5.19008578887e-07,
        1.69512693926e-07,
        1.13052749193e-07,
        5.34470060048e-09,
        2.56901986624e-08,
        1.23114528939e-09,
        2.59150696808e-10,
        5.70664649035e-11,
        1.24862918464e-11,
        3.98921949311e-08,
        3.37351321884e-08,
        9.04848418271e-09,
        8.17467257945e-09,
        2.03842748014e-09,
        -1.0,
        0.975817042832,
        -0.0196710586703,
        1.0,
        -0.00042581541483,
        1.01954320269e-05,
        -1.45007151955e-05,
        1.38170346058e-05,
        4.1522580774e-06,
        0.000829883298777,
        1.62639723827e-06,
        1.00922948118e-06,
        3.29505605612e-07,
        2.2000904594e-07,
        1.04010446307e-08,
        4.99680135484e-08,
        2.39416905539e-09,
        5.04023437918e-10,
        1.10822971892e-10,
        2.42154725269e-11,
        7.75301030869e-08,
        6.56591384761e-08,
        1.75866622577e-08,
        1.59072683321e-08,
        3.95842503266e-09,
        -1.0,
        0.975817198385,
        -0.0259173900306,
        0.999999998734,
        -0.000595395957916,
        1.42382202979e-05,
        -2.02820780795e-05,
        1.94200229601e-05,
        5.82789789026e-06,
        2.28583237573e-06,
        0.00179593768152,
        1.40769785601e-06,
        4.59780391411e-07,
        3.06610190884e-07,
        1.44953856552e-08,
        6.96776648233e-08,
        3.33920253671e-09,
        7.02881548745e-10,
        1.54798841913e-10,
        3.38744282275e-11,
        1.08204368059e-07,
        9.14921353032e-08,
        2.45431169665e-08,
        2.21707372114e-08,
        5.52946079842e-09,
        -1.0,
        0.975816996938,
        -0.0266491106946,
        0.999999999636,
        -0.000629380734847,
        1.50517067725e-05,
        -2.14410188027e-05,
        2.05214807591e-05,
        6.15898299186e-06,
        2.41548608953e-06,
        1.48824019701e-06,
        0.00104645704781,
        4.86077142709e-07,
        3.24171196671e-07,
        1.53256159001e-08,
        7.36658399464e-08,
        3.53028298983e-09,
        7.43109747608e-10,
        1.63641626792e-10,
        3.58062116145e-11,
        1.14391409818e-07,
        9.67331753472e-08,
        2.59466373803e-08,
        2.34404198026e-08,
        5.84531754267e-09,
        -1.0,
        0.97581792643,
        -0.0318108598872,
        0.999999991467,
        -0.000802321657135,
        1.91797586827e-05,
        -2.73362181793e-05,
        2.62090850061e-05,
        7.86214314724e-06,
        3.08494477965e-06,
        1.89555765949e-06,
        6.19203933966e-07,
        0.00207866806737,
        4.12768187559e-07,
        1.95144484908e-08,
        9.38180720656e-08,
        4.49646636971e-09,
        9.46428722018e-10,
        2.08537228935e-10,
        4.56548167438e-11,
        1.45732005381e-07,
        1.23166949068e-07,
        3.30541406066e-08,
        2.98478879332e-08,
        7.44903806286e-09,
        -1.0,
        0.975818381952,
        -0.0326829894697,
        0.999999982536,
        -0.000842076494622,
        2.01302415489e-05,
        -2.86964861653e-05,
        2.75109789556e-05,
        8.25239430219e-06,
        3.23815990037e-06,
        1.98921360758e-06,
        6.49818623512e-07,
        4.33159985485e-07,
        0.00235163286661,
        2.0478359796e-08,
        9.845591655e-08,
        4.71864334891e-09,
        9.9321415198e-10,
        2.18854647664e-10,
        4.79148542145e-11,
        1.52937341611e-07,
        1.29250768936e-07,
        3.4687642126e-08,
        3.13224684916e-08,
        7.81750374502e-09,
        -1.0,
        0.975837106449,
        -0.0457106314641,
        0.9999992065,
        -0.00126964819274,
        3.03462258495e-05,
        -4.37196361663e-05,
        4.14973604535e-05,
        1.24542713002e-05,
        4.8851875372e-06,
        2.99805048759e-06,
        9.79476278068e-07,
        6.53016182293e-07,
        3.08745381924e-08,
        0.00161619692726,
        1.48414411509e-07,
        7.1135298683e-09,
        1.4979606493e-09,
        3.30166272025e-10,
        7.23188099349e-11,
        2.30615424432e-07,
        1.94950378563e-07,
        5.23006833556e-08,
        4.72406073022e-08,
        1.17910690641e-08,
        -1.0,
        0.975868011835,
        -0.0392721023417,
        0.999999901763,
        -0.00106386077005,
        2.54142801296e-05,
        -3.66118937848e-05,
        3.4761094144e-05,
        1.04308256906e-05,
        4.09258402123e-06,
        2.51301221575e-06,
        8.20770300882e-07,
        5.47183195099e-07,
        2.58648641517e-08,
        1.24379255199e-07,
        0.00640371290566,
        5.96250858322e-09,
        1.25518190554e-09,
        2.76529134943e-10,
        6.05587611048e-11,
        1.93251709888e-07,
        1.63353570264e-07,
        4.38198981652e-08,
        3.95902788296e-08,
        9.88091631346e-09,
        -1.0,
        0.978102229716,
        -0.0521351100738,
        0.999996080026,
        -0.00144622810549,
        3.58250220479e-05,
        -4.85015483538e-05,
        4.85014876808e-05,
        1.44677786111e-05,
        5.70099169838e-06,
        3.44471470509e-06,
        1.12599100755e-06,
        7.57837409932e-07,
        3.57927468071e-08,
        1.71697511505e-07,
        8.19128953002e-09,
        0.00602016675867,
        1.74716681266e-09,
        3.8422936187e-10,
        8.46250391664e-11,
        2.6804182472e-07,
        2.26597699746e-07,
        6.06861770919e-08,
        5.51692707838e-08,
        1.35736509741e-08,
        -1.0,
        0.951483463508,
        -0.0512516428133,
        0.999999359791,
        -0.00130928919534,
        2.97565742287e-05,
        -0.000119026289649,
        4.09152779534e-05,
        1.44133370721e-05,
        5.11440968765e-06,
        3.02215224632e-06,
        1.10424793151e-06,
        6.97419783739e-07,
        3.54158488405e-08,
        1.50744374901e-07,
        7.77559464888e-09,
        1.74525211275e-09,
        0.00591234785639,
        3.68915047907e-10,
        8.51342354807e-11,
        2.47002822864e-07,
        2.25208476766e-07,
        5.44859168009e-08,
        5.63021188837e-08,
        1.32809416959e-08,
        -1.0,
        0.986780496814,
        -0.0573921063377,
        0.999713891661,
        0.0,
        6.17208957481e-05,
        0.0,
        4.11472608525e-05,
        1.54302228965e-05,
        3.85755573903e-06,
        2.57170396875e-06,
        1.20548624187e-06,
        8.03657505727e-07,
        2.51142966967e-08,
        1.40640061447e-07,
        9.41786133629e-09,
        1.72660790614e-09,
        3.33549248334e-10,
        0.00588412646668,
        1.34891238328e-10,
        2.21005809279e-07,
        2.81280129516e-07,
        4.52057336497e-08,
        7.78543218672e-08,
        8.16214627501e-09,
        -1.0,
        1.0,
        -0.0211494419308,
        0.998762375742,
        0.0160122222973,
        0.000125095489545,
        0.0,
        0.0,
        0.0,
        0.0,
        -1.36823186561e-05,
        -3.90923392519e-06,
        -2.4432711847e-06,
        -1.37434003472e-07,
        -8.55144907957e-07,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0058353369435,
        -6.10817805242e-07,
        0.0,
        -2.44327119831e-07,
        0.0,
        -4.96289466892e-08,
        -1.0,
        0.97584422449,
        -0.0387998412437,
        0.999999938904,
        -0.00101369634517,
        2.42133960547e-05,
        -3.48609020246e-05,
        3.31017464345e-05,
        9.93136426375e-06,
        3.89639973089e-06,
        2.39270097579e-06,
        7.81746851541e-07,
        5.2117834614e-07,
        2.46416236958e-08,
        1.18445167522e-07,
        5.67688623747e-09,
        1.19516989093e-09,
        2.63353425157e-10,
        5.76636138578e-11,
        1.84031642792e-07,
        0.0066257350014,
        1.55509914108e-07,
        4.17301858692e-08,
        3.76849417996e-08,
        9.40561271039e-09,
        -1.0,
        0.975863163173,
        -0.0387067097915,
        0.999999903794,
        -0.00101103404356,
        2.40927098515e-05,
        -3.48683003761e-05,
        3.29571258332e-05,
        9.89462867995e-06,
        3.88288385851e-06,
        2.38375527225e-06,
        7.78982557913e-07,
        5.19180238267e-07,
        2.45505270687e-08,
        1.18007901427e-07,
        5.65677511515e-09,
        1.19097161186e-09,
        2.62391082734e-10,
        5.74675245679e-11,
        1.83361740606e-07,
        1.54953603425e-07,
        0.00654192250722,
        4.15817507589e-08,
        3.75497153076e-08,
        9.37260079246e-09,
        -1.0,
        0.975853221892,
        -0.0447173056581,
        0.999999293844,
        -0.00121583730067,
        2.92306220217e-05,
        -4.16065779181e-05,
        3.98726566704e-05,
        1.19666396874e-05,
        4.69063556013e-06,
        2.88183452634e-06,
        9.40922133647e-07,
        6.27533521979e-07,
        2.9656648385e-08,
        1.42445832308e-07,
        6.8321894749e-09,
        1.43874273083e-09,
        3.17250010516e-10,
        6.94097536528e-11,
        2.21498491942e-07,
        1.87310144542e-07,
        5.02368976909e-08,
        0.0062481959526,
        4.53814270514e-08,
        1.13247405285e-08,
        -1.0,
        0.976011253422,
        -0.0449724993033,
        0.999999538006,
        -0.00121015678909,
        2.88083204115e-05,
        -4.23219455762e-05,
        3.94932209366e-05,
        1.18637037486e-05,
        4.65348280499e-06,
        2.85546624931e-06,
        9.33407111392e-07,
        6.22408952263e-07,
        2.94183875602e-08,
        1.41458782688e-07,
        6.7863303372e-09,
        1.42783906125e-09,
        3.15048961711e-10,
        6.89901787774e-11,
        2.19796646759e-07,
        1.8583269007e-07,
        4.98468354042e-08,
        4.51090426848e-08,
        0.00625350104443,
        1.12372615527e-08,
        0.975600742664,
        -0.0498935085622,
        -0.00142274406025,
        3.46137576095e-05,
        -5.25173915219e-05,
        4.68478077917e-05,
        1.4099109356e-05,
        5.52028316609e-06,
        3.38491386889e-06,
        1.10499236676e-06,
        7.35496349954e-07,
        3.48588552398e-08,
        1.6770128713e-07,
        8.03629310229e-09,
        1.68807695264e-09,
        3.74211011527e-10,
        8.19208589683e-11,
        2.60512420308e-07,
        2.20590638615e-07,
        5.89358369403e-08,
        5.35449488333e-08,
        1.32769494276e-08,
        0.00608411590686,
        -1.0,
        -0.975817130273,
        0.999999998785,
        -2.1121084913e-05,
        -0.000111960230515,
        -0.00115300456145,
        -8.86920891008e-05,
        -1.3741684493e-05,
        -3.85919998166e-06,
        -2.24258375143e-06,
        -5.74596675491e-07,
        -3.6490823622e-07,
        -1.14275860761e-08,
        -6.55718146116e-08,
        -2.24890855179e-09,
        -4.15125881862e-10,
        -8.04485436996e-11,
        -1.5824384092e-11,
        -1.06982102863e-07,
        -9.06889486821e-08,
        -2.01274359039e-08,
        -1.82936351931e-08,
        -3.84787959284e-09,
        -1.0,
        1.0,
        -1.0,
        0.185715570786,
        0.0237689194951,
        0.104370429299,
        0.0176475791535,
        0.0037638114886,
        0.00123393589806,
        0.000773577161474,
        0.000228367632135,
        0.000153189109985,
        6.40759461005e-06,
        3.19365200833e-05,
        1.40596221274e-06,
        2.88930590654e-07,
        6.12563343665e-08,
        1.31012618043e-08,
        4.97957198027e-05,
        4.28602705216e-05,
        1.07481987265e-05,
        9.83894718947e-06,
        2.31144922181e-06,
        -1.0,
        -0.0671867027997,
        -1.0,
        -1.0,
        -0.190064058051,
        -0.0038136666612,
        -0.0426817393447,
        -0.00284355414398,
        -0.000380350551705,
        -9.28692777774e-05,
        -5.24094286504e-05,
        -1.13322489789e-05,
        -6.89620532727e-06,
        -1.12855741219e-07,
        -9.31178073251e-07,
        -1.19077582433e-08,
        -3.75530312372e-10,
        3.2308730928e-10,
        1.36422599241e-10,
        -1.63309159422e-06,
        -1.39216620472e-06,
        -2.18983827979e-07,
        -2.02424096145e-07,
        -2.41244770396e-08,
        -0.0103118272757,
        -0.999999998921,
        0.999999967288,
        0.999999981163,
        0.99999977881,
        -0.0572808807569,
        -0.0487304289763,
        -1.00419226105,
        0.000203348549212,
        -0.00697201020168,
        -0.000134489367266,
        -3.31267899885e-08,
        -2.02270942911e-06,
        2.88658680796e-06,
        6.84694925423e-07,
        7.68475662246e-07,
        2.81434876173e-08,
        1.13674114815e-07,
        4.04437631056e-09,
        8.32605819955e-10,
        5.02475023089e-11,
        -1.40956841192e-12,
        1.25552799805e-07,
        2.21072639281e-07,
        2.54916897402e-08,
        4.25495645694e-08,
        2.75183146627e-09,
        -1.0,
        0.999999998561,
        0.999999996348,
        -0.230752555681,
        0.00883219921804,
        -8.73925776682e-05,
        -0.0857246769039,
        0.00110307153365,
        4.49645767839e-06,
        -2.38057604846e-06,
        1.98804721185e-07,
        -7.79372458044e-07,
        -1.2612354473e-07,
        -1.48591894017e-07,
        -2.5987311037e-09,
        -1.20939558601e-08,
        1.33347818697e-10,
        5.7532876024e-11,
        4.83493524851e-11,
        1.42385748298e-11,
        -9.18424038844e-09,
        -3.45341646254e-08,
        4.04572733136e-10,
        -4.21182883305e-09,
        1.08236337951e-09,
        -1.0,
        0.999999992528,
        0.99999998217,
        -0.111285311239,
        -0.0284878788914,
        0.000105024679133,
        -0.00412067558481,
        -0.148359159776,
        -8.45233303739e-05,
        -7.59302484921e-07,
        -1.26540481955e-06,
        1.65004635638e-06,
        4.10062337406e-07,
        4.5596585273e-07,
        1.75906740605e-08,
        7.07007625687e-08,
        2.69097046324e-09,
        5.63453752397e-10,
        4.78785109195e-11,
        3.45345929219e-12,
        7.95357556697e-08,
        1.33658234428e-07,
        1.68440786667e-08,
        2.6510899137e-08,
        2.14189697194e-09,
        -1.0,
        0.999999999663,
        0.999999981967,
        -0.228580524771,
        -0.00731211039753,
        5.67116684867e-06,
        -0.00112457913196,
        -2.98981191383e-05,
        -0.0167054850718,
        -1.19241637204e-06,
        -3.88390290992e-07,
        3.07702475616e-07,
        1.02148825336e-07,
        1.11083121826e-07,
        5.47665804728e-09,
        2.13419879216e-08,
        1.03904771626e-09,
        2.28932532551e-10,
        3.61153505247e-11,
        6.7366561752e-12,
        2.56564837615e-08,
        3.6019412682e-08,
        6.33458126383e-09,
        8.13911670424e-09,
        1.20068363349e-09,
        -1.0,
        1.0,
        0.99999998369,
        -0.317252517592,
        -1.30096927913e-05,
        -1.95503513701e-05,
        -6.377180916e-05,
        -7.67721794876e-06,
        -9.23616423998e-07,
        -0.00106893486638,
        -6.52683158369e-08,
        -9.88156235988e-08,
        -7.03923933855e-10,
        -3.49005348509e-09,
        9.71150591447e-10,
        3.30584293495e-09,
        3.35582352341e-10,
        8.07453323308e-11,
        2.2239816844e-11,
        5.41353164636e-12,
        5.26684339937e-09,
        2.17177126233e-09,
        1.9468380564e-09,
        1.35851855959e-09,
        6.12377162834e-10,
        -1.0,
        0.999999999651,
        0.999999983357,
        -0.380241231459,
        -0.00254774705322,
        5.78663239029e-06,
        -0.000379895856172,
        -8.93794561996e-06,
        -2.32469702243e-07,
        -1.22089561645e-07,
        -0.000675676269638,
        1.26290083581e-07,
        3.56186239791e-08,
        3.92968936796e-08,
        1.71289809602e-09,
        6.76595703681e-09,
        2.95051746943e-10,
        6.36764547423e-11,
        8.16950293897e-12,
        1.27484268204e-12,
        7.87652116236e-09,
        1.20953605852e-08,
        1.81772186657e-09,
        2.56210650448e-09,
        2.96565013475e-10,
        -1.0,
        0.999999999938,
        0.999999983019,
        -0.406055203408,
        0.0063091361713,
        -3.9286794616e-05,
        0.000861288416182,
        1.22774588628e-05,
        -6.05336483263e-07,
        2.17603834953e-07,
        -4.37315451187e-07,
        -0.000234487436922,
        -8.86493033024e-08,
        -1.01410289093e-07,
        -2.9836394721e-09,
        -1.24568837965e-08,
        -2.98769407635e-10,
        -5.38472703881e-11,
        8.23320771912e-12,
        3.75782996102e-12,
        -1.26843131464e-08,
        -2.70674145331e-08,
        -1.99426852368e-09,
        -4.58282558337e-09,
        5.03086426842e-11,
        -1.0,
        0.999999999649,
        0.999999981845,
        -0.47145352525,
        0.00581901803045,
        -2.46955889176e-05,
        0.000831180484629,
        1.58808724835e-05,
        -1.6249928888e-08,
        2.39536322451e-07,
        -3.45898453279e-07,
        -8.15148030057e-08,
        -0.000121846215484,
        -9.16072828365e-08,
        -3.32875366553e-09,
        -1.34577269234e-08,
        -4.73704188119e-10,
        -9.73086350914e-11,
        -5.46817652697e-12,
        2.92062209509e-13,
        -1.48232567433e-08,
        -2.62803920416e-08,
        -2.98892769034e-09,
        -5.03440006116e-09,
        -3.13645472287e-10,
        -1.0,
        0.999999999545,
        0.999999987495,
        -0.495006382081,
        0.0103029351013,
        -4.58700900467e-05,
        0.00146477542148,
        2.7258574447e-05,
        -1.30906058629e-07,
        4.16278629098e-07,
        -6.22538905147e-07,
        -1.44194032654e-07,
        -1.62395609376e-07,
        -8.79876357412e-05,
        -5.77869573071e-09,
        -2.3429208807e-08,
        -8.00347682019e-10,
        -1.63104004871e-10,
        -7.20701198404e-12,
        1.11426474802e-12,
        -2.56233339155e-08,
        -4.62368610168e-08,
        -5.06876497545e-09,
        -8.75187848435e-09,
        -4.86725401843e-10,
        -1.0,
        0.999999999375,
        0.999999983319,
        -0.667692398991,
        0.0120977899095,
        -2.56010763672e-05,
        0.00181046997688,
        4.32086084904e-05,
        1.17206638398e-06,
        5.79778761842e-07,
        -5.85320599859e-07,
        -1.67358399775e-07,
        -1.84740215051e-07,
        -8.1475394429e-09,
        -1.90609293852e-06,
        -3.21100053607e-08,
        -1.41636277487e-09,
        -3.06416604553e-10,
        -4.02827203158e-11,
        -6.44926503983e-12,
        -3.74687378766e-08,
        -5.71538745802e-08,
        -8.71016598434e-09,
        -1.21772524495e-08,
        -1.44559390048e-09,
        -1.0,
        0.999999999552,
        0.999999978774,
        -0.578606640501,
        0.00845099309846,
        -2.0711648335e-05,
        0.0012554090386,
        2.90521456292e-05,
        6.95033993379e-07,
        3.98687393467e-07,
        -4.25612202458e-07,
        -1.1785538692e-07,
        -1.30313320121e-07,
        -5.58997069591e-09,
        -2.21169513203e-08,
        -4.33118997522e-05,
        -9.4907683554e-10,
        -2.041588525e-10,
        -2.52319613144e-11,
        -3.78160877604e-12,
        -2.56256035636e-08,
        -3.98528857893e-08,
        -5.85550164104e-09,
        -8.36829301754e-09,
        -9.31615116273e-10,
        -1.0,
        0.999999999319,
        0.999999991014,
        -0.748470411301,
        0.00876478404106,
        6.67525157999e-06,
        0.00139185768241,
        4.12746872577e-05,
        2.05654113461e-06,
        5.06944717267e-07,
        -2.97537579222e-07,
        -1.20924747302e-07,
        -1.29790681312e-07,
        -7.18980555942e-09,
        -2.76672399599e-08,
        -1.46770683359e-09,
        -1.43167694817e-06,
        -3.28129267017e-10,
        -5.82745845918e-11,
        -1.17366235183e-11,
        -3.41309303212e-08,
        -4.43787690336e-08,
        -8.87451781027e-09,
        -1.06250678458e-08,
        -1.84953773095e-09,
        -1.0,
        0.999999999978,
        0.99999997703,
        -0.819893937447,
        0.00978164542418,
        1.55743490101e-05,
        0.00157936365104,
        4.92967103625e-05,
        2.68097989931e-06,
        5.92704514677e-07,
        -2.90434283235e-07,
        -1.34546270047e-07,
        -1.43274784182e-07,
        -8.42533955302e-09,
        -3.22389880677e-08,
        -1.77744750941e-09,
        -3.99790969874e-10,
        -2.6402037475e-07,
        -7.43035231716e-11,
        -1.53540137536e-11,
        -4.02683681128e-08,
        -5.04022162883e-08,
        -1.07124439245e-08,
        -1.24188297178e-08,
        -2.31898901428e-09,
        -1.0,
        0.999999998032,
        0.999999987164,
        -0.917169671046,
        0.00303637989671,
        6.73402816962e-05,
        0.000689714813423,
        4.00721339898e-05,
        3.80880488365e-06,
        3.92435890827e-07,
        2.29667736887e-07,
        -3.89770513074e-08,
        -3.2642607798e-08,
        -5.72012744597e-09,
        -2.0558610139e-08,
        -1.62920544708e-09,
        -3.83521702308e-10,
        -9.46136544334e-11,
        -5.11202786243e-08,
        -2.2178036158e-11,
        -2.9349935935e-08,
        -2.245918551e-08,
        -9.57177639835e-09,
        -8.19496260179e-09,
        -2.68851205056e-09,
        -1.0,
        0.999999998238,
        0.999999980839,
        -1.00000002549,
        -0.000438403722483,
        0.000100668488835,
        0.000252750518439,
        3.79789858185e-05,
        4.71113557653e-06,
        3.11489319543e-07,
        5.32402125857e-07,
        1.06734942017e-08,
        2.57223082747e-08,
        -4.65654469096e-09,
        -1.56588311205e-08,
        -1.66822857673e-09,
        -4.02950035769e-10,
        -1.1277524345e-10,
        -2.75682667265e-11,
        -1.0076330764e-08,
        -2.55184003035e-08,
        -8.76729934916e-09,
        -9.65259264439e-09,
        -6.47993717541e-09,
        -3.09035395827e-09,
        -1.0,
        0.999999999581,
        0.99999998274,
        -0.557081628193,
        0.00571796993882,
        -9.63068492168e-06,
        0.000863355222718,
        2.13864039641e-05,
        6.78656881532e-07,
        2.84516087687e-07,
        -2.65687236405e-07,
        -7.95745200998e-08,
        -8.73711396539e-08,
        -4.00054803709e-09,
        -1.5708163161e-08,
        -7.17704867195e-10,
        -1.56323372452e-10,
        -2.20809952206e-11,
        -3.77730066454e-12,
        -1.85233179403e-08,
        -7.18583604279e-05,
        -2.744861986e-08,
        -4.3998659669e-09,
        -5.96745480524e-09,
        -7.68243789011e-10,
        -1.0,
        0.999999999427,
        0.999999994736,
        -0.5634104611,
        0.011891611026,
        -4.27927797166e-05,
        0.00172304496428,
        3.54893700382e-05,
        3.30536653581e-07,
        5.15383530362e-07,
        -6.6821710347e-07,
        -1.66368751784e-07,
        -1.8587457761e-07,
        -7.18719444873e-09,
        -2.88129809319e-08,
        -1.10057581936e-09,
        -2.30744742782e-10,
        -1.99326340205e-11,
        -1.53213834458e-12,
        -3.23779485931e-08,
        -5.45742103352e-08,
        -6.04407345533e-05,
        -6.87704213795e-09,
        -1.08264948812e-08,
        -8.82230932562e-10,
        -1.0,
        0.999999999013,
        0.999999983385,
        -0.639678634772,
        0.00616951525021,
        2.27030898552e-06,
        0.000971894498371,
        2.80802458244e-05,
        1.3328001849e-06,
        3.48913891593e-07,
        -2.22002173888e-07,
        -8.52764821896e-08,
        -9.18633381978e-08,
        -4.94279123294e-09,
        -1.90762522014e-08,
        -9.91861796042e-10,
        -2.21026475275e-10,
        -3.82681259588e-11,
        -7.59140495399e-12,
        -2.33858949738e-08,
        -3.09883748975e-08,
        -6.00810891608e-09,
        -1.29934617982e-05,
        -7.31414674711e-09,
        -1.2262953328e-09,
        -1.0,
        0.99999999942,
        0.999999991783,
        -0.644101362984,
        0.0113428904893,
        -2.58588150182e-05,
        0.00169121781352,
        3.97610348833e-05,
        1.02359940093e-06,
        5.41175469892e-07,
        -5.61061222314e-07,
        -1.58004563246e-07,
        -1.74454198972e-07,
        -7.59422685394e-09,
        -2.99927010749e-08,
        -1.30620566793e-09,
        -2.81836734302e-10,
        -3.60420783671e-11,
        -5.60680993136e-12,
        -3.48920650244e-08,
        -5.3673113206e-08,
        -8.04612637423e-09,
        -1.13594378935e-08,
        -1.17926140008e-05,
        -1.31007521894e-09,
        -1.0,
        0.999999999968,
        -0.718780092487,
        0.00348287116799,
        3.15710679873e-05,
        0.000645207943494,
        2.78321352211e-05,
        2.19246651282e-06,
        2.97854711275e-07,
        2.94992673072e-08,
        -4.67824148592e-08,
        -4.6124132279e-08,
        -4.29221835589e-09,
        -1.58728597858e-08,
        -1.08115640904e-09,
        -2.50276096591e-10,
        -5.62191475245e-11,
        -1.27101042573e-11,
        -2.13525431847e-08,
        -2.07888619878e-08,
        -6.41359137231e-09,
        -6.22832876972e-09,
        -1.6446971856e-09,
        -2.45089017027e-06,
        0.999999998878,
        -0.999999984184,
        0.000254747201237,
        0.00150709562496,
        0.0182962442532,
        0.00121875232866,
        0.000164110954316,
        4.11498271016e-05,
        2.2701030198e-05,
        5.35742030114e-06,
        3.26002872977e-06,
        7.62380749866e-08,
        4.84264091982e-07,
        1.15648874549e-08,
        1.69217308719e-09,
        3.04647552587e-10,
        4.98225281548e-11,
        9.09953124718e-07,
        7.36421299234e-07,
        1.3619712434e-07,
        1.25116425447e-07,
        2.14806754431e-08,
        -1.0,
        0.739666052143,
        1.0,
        0.00294087073725,
        1.0,
        -1.0,
        -0.033358870275,
        1.00000062844,
        -0.999999888442,
        1.0,
        0.99999977881,
        -0.0156111729625,
        -1.0,
        1.0,
        0.999999996348,
        -0.210247115878,
        -1.0,
        1.0,
        0.99999998217,
        -0.0790560150187,
        -1.0,
        1.0,
        0.999999981967,
        -0.205751778036,
        -1.0,
        1.0,
        0.99999998369,
        -0.301996137335,
        -1.0,
        1.0,
        0.999999983357,
        -0.353462586602,
        -1.0,
        1.0,
        0.999999983019,
        -0.386513092844,
        -1.0,
        1.0,
        0.999999981845,
        -0.445624111347,
        -1.0,
        1.0,
        0.999999987495,
        -0.470128160839,
        -1.0,
        1.0,
        0.999999983319,
        -0.64917416891,
        -1.0,
        1.0,
        0.999999978774,
        -0.546710777795,
        -1.0,
        1.0,
        0.999999991014,
        -0.71130144443,
        -1.0,
        1.0,
        0.99999997703,
        -0.795152592678,
        -1.0,
        1.0,
        0.999999987164,
        -0.874383082601,
        -1.0,
        1.0,
        0.999999980839,
        -0.954206758269,
        -1.0,
        1.0,
        0.99999998274,
        -0.521707586951,
        -1.0,
        1.0,
        0.999999994736,
        -0.529939483818,
        -1.0,
        1.0,
        0.999999983385,
        -0.603254723829,
        -1.0,
        1.0,
        0.999999991783,
        -0.607737383593,
        -1.0,
        0.999999999968,
        -0.683276219788,
        1.0,
        -0.999999984184,
        -1.0,
        0.823804434076,
        -1.0,
        -1.0,
        0.00271702944451,
        1.0,
        1.0,
        -0.0335994648709,
        0.999999889179,
        -1.000000182,
        1.0,
        0.99999977881,
        -0.999999999553,
        1.0,
        0.999999996348,
        -0.99999999975,
        1.0,
        0.99999998217,
        -0.999999999709,
        1.0,
        0.999999981967,
        -0.99999999994,
        1.0,
        0.99999998369,
        -0.999999999993,
        1.0,
        0.999999983357,
        -0.999999999973,
        1.0,
        0.999999983019,
        -0.999999999999,
        1.0,
        0.999999981845,
        -1.00000000002,
        1.0,
        0.999999987495,
        -1.00000000001,
        1.0,
        0.999999983319,
        -1.00000000009,
        1.0,
        0.999999978774,
        -1.00000000012,
        1.0,
        0.999999991014,
        -1.00000000018,
        1.0,
        0.99999997703,
        -1.00000000046,
        1.0,
        0.999999987164,
        -1.00000000049,
        1.0,
        0.999999980839,
        -1.00000000059,
        1.0,
        0.99999998274,
        -1.00000000003,
        1.0,
        0.999999994736,
        -1.0,
        1.0,
        0.999999983385,
        -1.00000000012,
        1.0,
        0.999999991783,
        -1.00000000005,
        0.999999999968,
        1.0,
        -0.999999984184,
        -0.99999999793,
        1.0,
        -1.0,
        -0.999999999545,
        0.999999969503,
        -0.999999772979,
        -1.0,
        0.999999880318,
        -0.102404304715,
        0.999999999787,
        0.0359129785539,
        0.392062902835,
        -0.000715147246002,
        0.00178914787423,
        -0.00100818556579,
        -0.00028462648993,
        -0.000106324477263,
        -6.38380493636e-05,
        -1.96952011893e-05,
        -1.28608522187e-05,
        -5.21840568444e-07,
        -2.71256427254e-06,
        -1.11767539666e-07,
        -2.17862366493e-08,
        -4.45006225719e-09,
        -9.02745592773e-10,
        -4.30613337488e-06,
        -3.61843503095e-06,
        -9.06608219775e-07,
        -8.16515256095e-07,
        -1.89611469265e-07,
        -1.0,
        0.999999985798,
        -0.145275802639,
        0.999999985841,
        -0.0325931597321,
        0.000648607644369,
        1.01079527798,
        -0.00162392989264,
        0.000914124777175,
        0.000257333075039,
        9.61289289302e-05,
        5.76979277787e-05,
        1.77988206486e-05,
        1.16239824204e-05,
        4.71591728031e-07,
        2.45130238936e-06,
        1.00998454814e-07,
        1.96871266537e-08,
        4.02145688842e-09,
        8.1939108064e-10,
        3.89134498909e-06,
        3.27028027139e-06,
        8.19268626247e-07,
        7.3788834596e-07,
        1.71346780372e-07,
        -1.0,
        0.999999998383,
        -0.123042154252,
        0.999999999517,
        0.00713804093585,
        -0.00014281800027,
        0.000355360605297,
        0.604548296448,
        -0.000201690401995,
        -5.80684719238e-05,
        -2.16908000559e-05,
        -1.30524072496e-05,
        -4.02998268248e-06,
        -2.62941494397e-06,
        -1.06781773272e-07,
        -5.5513947978e-07,
        -2.28793487411e-08,
        -4.459941254e-09,
        -9.10722151858e-10,
        -1.79252562023e-10,
        -8.81354516724e-07,
        -7.39978474141e-07,
        -1.85578987489e-07,
        -1.67080999296e-07,
        -3.88068552944e-08,
        -1.0,
        0.999999995936,
        -0.188146070978,
        0.999999996054,
        -0.0582719017767,
        0.00115677585266,
        -0.002904468641,
        0.00162896629067,
        0.585462984635,
        0.000453899169325,
        0.000169563392462,
        0.000101653153005,
        3.13456158544e-05,
        2.04800133936e-05,
        8.30510786607e-07,
        4.31659127518e-06,
        1.77829409209e-07,
        3.46626598807e-08,
        7.08159732726e-09,
        1.46575629739e-09,
        6.85208290349e-06,
        5.76104968664e-06,
        1.44253113255e-06,
        1.29947187586e-06,
        3.01723749272e-07,
        -1.0,
        0.9999999946,
        -0.264812941992,
        0.999999994621,
        -0.115365415507,
        0.00229447323386,
        -0.00574856057044,
        0.00323331366318,
        0.000908164906613,
        0.254966564292,
        0.000339256825164,
        0.000203572767533,
        6.27933806014e-05,
        4.10127668767e-05,
        1.66375273478e-06,
        8.64791701899e-06,
        3.56302442975e-07,
        6.94518615408e-08,
        1.4187328121e-08,
        2.90074683706e-09,
        1.37280861909e-05,
        1.15381792069e-05,
        2.89022226617e-06,
        2.60322982604e-06,
        6.0449084081e-07,
        -1.0,
        0.999999753168,
        -0.3501847045,
        0.999999753169,
        -0.162506423781,
        0.00323514432157,
        -0.00809615931618,
        0.00456053900976,
        0.00128620258282,
        0.000480471943262,
        0.585025769665,
        0.000288446534033,
        8.8987371098e-05,
        5.8111080859e-05,
        2.35780723786e-06,
        1.22558549772e-05,
        5.04978862524e-07,
        9.84331592626e-08,
        2.01062910542e-08,
        4.08521321668e-09,
        1.94558464056e-05,
        1.63493286282e-05,
        4.09620097776e-06,
        3.68918623075e-06,
        8.56692714779e-07,
        -1.0,
        0.999999690465,
        -0.359295518188,
        0.999999690465,
        -0.171803117201,
        0.00342041194605,
        -0.00855953117519,
        0.00482178446821,
        0.00136014131979,
        0.000508092357891,
        0.00030503447138,
        0.346564325671,
        9.41056673327e-05,
        6.14528257403e-05,
        2.49342213028e-06,
        1.29608129361e-05,
        5.34025240108e-07,
        1.04095326207e-07,
        2.12627760092e-08,
        4.31889996235e-09,
        2.05749829621e-05,
        1.72895661305e-05,
        4.33181369338e-06,
        3.9013757568e-06,
        9.05969238852e-07,
        -1.0,
        0.999998429802,
        -0.427266767094,
        0.999998429802,
        -0.219836386689,
        0.00437668088214,
        -0.0109509876972,
        0.00616991809656,
        0.0017404394653,
        0.000650161587727,
        0.000390332621015,
        0.000120419043746,
        0.696278174055,
        7.86370009726e-05,
        3.19063992317e-06,
        1.65848653677e-05,
        6.83341148877e-07,
        1.33200601653e-07,
        2.72080763778e-08,
        5.52628328638e-09,
        2.63282312235e-05,
        2.21240464691e-05,
        5.54318894551e-06,
        4.99225470524e-06,
        1.15927735282e-06,
        -1.0,
        0.999998717469,
        -0.438040386527,
        0.999998717469,
        -0.230840949479,
        0.00459469902106,
        -0.0115003526783,
        0.00647786179332,
        0.00182733432072,
        0.000682629484621,
        0.000409838399927,
        0.000126433487855,
        8.25666132188e-05,
        0.787696217947,
        3.35001464218e-06,
        1.74128312715e-05,
        7.17473274647e-07,
        1.39856356688e-07,
        2.85672324784e-08,
        5.80245011803e-09,
        2.76432711931e-05,
        2.32285345748e-05,
        5.82011615226e-06,
        5.24157576335e-06,
        1.21718713918e-06,
        -1.0,
        0.999830595152,
        -0.610752906071,
        0.999830595152,
        -0.349500533073,
        0.00700160393019,
        -0.0172415079747,
        0.00985308780974,
        0.00277688606082,
        0.00103666958272,
        0.000622899911688,
        0.000192033479885,
        0.000125417537292,
        5.08969134046e-06,
        0.532839582851,
        2.64542530222e-05,
        1.08902197605e-06,
        2.12302368818e-07,
        4.33732607725e-08,
        8.80703379231e-09,
        4.20019285167e-05,
        3.52669375849e-05,
        8.84506767706e-06,
        7.95462427138e-06,
        1.84598284874e-06,
        -1.0,
        0.99995850959,
        -0.525076954685,
        0.99995850959,
        -0.292684649329,
        0.00583256431814,
        -0.0145083271251,
        0.00822072892935,
        0.00231883294891,
        0.00086566994692,
        0.000519817315843,
        0.000160392929582,
        0.000104731563067,
        4.25082784827e-06,
        2.2082896996e-05,
        0.960867493137,
        9.09545025436e-07,
        1.77426154785e-07,
        3.6231526736e-08,
        7.35484654241e-09,
        3.50640598897e-05,
        2.94622404723e-05,
        7.38705039126e-06,
        6.6474935638e-06,
        1.54287133191e-06,
        -1.0,
        0.99893004564,
        -0.69437909588,
        0.99893004564,
        -0.403055195503,
        0.00801893438108,
        -0.0183289917167,
        0.011598816057,
        0.00325769292225,
        0.00121268513791,
        0.000733875898449,
        0.000228217507762,
        0.000149907577442,
        6.06552353828e-06,
        3.10442932665e-05,
        1.27384733958e-06,
        0.905104889451,
        2.50727243218e-07,
        5.08692252412e-08,
        1.02250520198e-08,
        5.02022584223e-05,
        4.09728730259e-05,
        1.06802158959e-05,
        9.43914324118e-06,
        2.1325471607e-06,
        -1.0,
        0.971385257077,
        -0.707182848694,
        0.987575011361,
        -0.523121561089,
        0.010118596621,
        0.0,
        0.0151778950427,
        0.00404743894117,
        0.00142292774486,
        0.000916997892882,
        0.000292490709002,
        0.000177865969908,
        7.53460016841e-06,
        4.00198436289e-05,
        1.37413812602e-06,
        3.0107521075e-07,
        0.867226828267,
        5.98290480708e-08,
        1.09766992919e-08,
        6.126494559e-05,
        4.64427807479e-05,
        1.35869839804e-05,
        1.04990329033e-05,
        2.37772215348e-06,
        -1.0,
        0.928450796754,
        -1.0,
        0.928450796754,
        -1.00000000121,
        0.0174084525452,
        0.0,
        0.0232112700929,
        0.00507746540283,
        0.00199471855322,
        0.00145070441248,
        0.000408010615241,
        0.000340008848282,
        1.27503317896e-05,
        3.68342913096e-05,
        2.21359925621e-06,
        3.98447865318e-07,
        1.02378966013e-07,
        0.86277850193,
        1.31432452338e-08,
        9.63358399478e-05,
        5.10013264397e-05,
        2.19589047508e-05,
        9.91692456364e-06,
        3.89593469412e-06,
        -1.0,
        0.0,
        -0.880113922694,
        0.588342580387,
        0.0,
        0.0,
        0.0,
        0.0735428229173,
        0.0183857059364,
        0.00459642645925,
        0.0045964264995,
        0.00143638328221,
        0.000718191637361,
        4.0398279864e-05,
        0.000233412284252,
        7.29413384127e-06,
        1.26244623916e-06,
        2.10407705862e-07,
        0.0,
        0.840138657783,
        0.000395005404782,
        0.000143638326855,
        7.18191642575e-05,
        6.28417686699e-05,
        0.0,
        -1.0,
        0.999955292929,
        -0.52105892077,
        0.999983794676,
        -0.278739468397,
        0.00554002935479,
        -0.0138803510828,
        0.00781838966438,
        0.00220554966553,
        0.000823880235367,
        0.000494829230901,
        0.000152612713655,
        9.96658656978e-05,
        4.04481621587e-06,
        2.10153874673e-05,
        8.66055171301e-07,
        1.68830745042e-07,
        3.44907036517e-08,
        7.00302879271e-09,
        3.33709847069e-05,
        0.988851055334,
        2.80338521149e-05,
        7.02738180618e-06,
        6.32697019721e-06,
        1.46899394244e-06,
        -1.0,
        0.999998856527,
        -0.520016463169,
        0.999998856527,
        -0.277465011706,
        0.00554339896057,
        -0.0137556724293,
        0.00780538706899,
        0.00220019152272,
        0.000821790397362,
        0.000493511872065,
        0.000152227129102,
        9.9425311666e-05,
        4.03394630155e-06,
        2.0959643962e-05,
        8.63449067766e-07,
        1.68311983555e-07,
        3.43806987441e-08,
        6.98220569828e-09,
        3.32777879065e-05,
        2.79625684749e-05,
        0.980640189529,
        7.00693238632e-06,
        6.30914969598e-06,
        1.46452869053e-06,
        -1.0,
        0.99987541278,
        -0.598067500412,
        0.99987541278,
        -0.336031132967,
        0.00672429654354,
        -0.0162612115584,
        0.00946601409223,
        0.0026649043743,
        0.000996015393985,
        0.000598274057579,
        0.000184467841841,
        0.000120531910469,
        4.89184851829e-06,
        2.54127997972e-05,
        1.04678491643e-06,
        2.0381199862e-07,
        4.16831821823e-08,
        8.45975082462e-09,
        4.03696522375e-05,
        3.3883731675e-05,
        8.49690011786e-06,
        0.953191102327,
        7.6443247599e-06,
        1.77367378299e-06,
        -1.0,
        0.999936809426,
        -0.602949170331,
        0.999936809426,
        -0.333281692767,
        0.00666384571561,
        -0.0166724747983,
        0.00940399921752,
        0.00264688720819,
        0.000990170593826,
        0.000595388876116,
        0.000183219776397,
        0.000119801407388,
        4.86190713964e-06,
        2.52957367058e-05,
        1.03998645214e-06,
        2.02726691267e-07,
        4.14065948834e-08,
        8.40246017743e-09,
        4.01390012745e-05,
        3.36690200428e-05,
        8.46279653039e-06,
        7.58495200352e-06,
        0.956661728734,
        1.76511100427e-06,
        0.997694043338,
        -0.663930249194,
        -0.408234804837,
        0.00748881932765,
        -0.019562628391,
        0.0109275632409,
        0.0030853182508,
        0.00116296439081,
        0.000704465112502,
        0.000216712575265,
        0.00014208703111,
        5.74616677396e-06,
        2.96263403335e-05,
        1.22665736138e-06,
        2.39909460741e-07,
        4.89001358384e-08,
        9.92030939995e-09,
        4.73125940999e-05,
        3.95515369874e-05,
        1.00837765447e-05,
        8.88043950613e-06,
        2.07085880344e-06,
        0.922715743751,
        -1.0,
        -1.0000000018,
        0.999999999998,
        -0.00206079379964,
        -0.0188469054658,
        -0.523076923432,
        -0.0355156954151,
        -0.00490762335666,
        -0.00125565887432,
        -0.000700891768406,
        -0.00016670189339,
        -0.000102962780482,
        -2.69193068948e-06,
        -7.60957572229e-06,
        -2.26488819542e-07,
        -3.85584651442e-08,
        -6.9320158049e-09,
        -1.26383589808e-09,
        -1.28019246168e-05,
        -1.07811893312e-05,
        -2.24986035145e-06,
        -2.04737707548e-06,
        -3.98338860554e-07,
        -1.0,
        0.0470865811941,
        -1.0,
        0.288549001405,
        0.0339081814496,
        0.166990365839,
        0.0251777581242,
        0.00495322650348,
        0.00153831712287,
        0.000941498494304,
        0.000261870777465,
        0.000171955355991,
        6.1567228308e-06,
        3.32151897223e-05,
        1.25508697703e-06,
        2.38734441901e-07,
        4.69143165573e-08,
        9.30733428098e-09,
        5.29521012879e-05,
        4.53066324978e-05,
        1.0597564701e-05,
        9.67439290057e-06,
        2.11372104763e-06,
        -1.0,
        -0.00338588720523,
        -1.0,
        -1.0,
        -0.435217146126,
        -0.0087529076773,
        -0.10536238838,
        -0.0066504760542,
        -0.000871753494055,
        -0.000214441148115,
        -0.000119959999631,
        -2.65835402369e-05,
        -1.62048382392e-05,
        -3.37389217777e-07,
        -2.38299855603e-06,
        -5.26905874207e-08,
        -7.53656929805e-09,
        -1.05781006268e-09,
        -1.42644273928e-10,
        -4.09220318498e-06,
        -3.45963095758e-06,
        -6.28798228488e-07,
        -5.73609718462e-07,
        -9.56314327013e-08,
        -0.149592468698,
        -0.999999888352,
        1.00000000106,
        0.99999999919,
        0.999999119609,
        -0.0597792423732,
        -0.0698441522712,
        -0.962149080791,
        0.000289117004319,
        -0.0105591482599,
        -0.000188465768226,
        -2.28954411244e-06,
        -3.44373670452e-06,
        2.77019714333e-06,
        5.5272821473e-07,
        6.97715282388e-07,
        2.02809270691e-08,
        8.46337766554e-08,
        2.21911717133e-09,
        4.17595437253e-10,
        -1.34578537145e-11,
        -1.12818700542e-11,
        8.22063661161e-08,
        1.85804303332e-07,
        1.42424208271e-08,
        3.12309690311e-08,
        3.09788762102e-10,
        -1.0,
        1.00000000024,
        0.999999984472,
        -0.231496678005,
        0.013182464527,
        -0.000126099175012,
        -0.0505382561678,
        0.00174209493109,
        7.51538418443e-06,
        -2.71801755682e-06,
        4.19819533922e-07,
        -8.45155480652e-07,
        -1.1140156258e-07,
        -1.45451693429e-07,
        -1.61586923768e-09,
        -7.98630426261e-09,
        3.06853055432e-10,
        8.38833147385e-11,
        4.42732260015e-11,
        1.15732386369e-11,
        -2.47441013988e-09,
        -3.02813592009e-08,
        1.92753812943e-09,
        -2.73997604232e-09,
        1.30551845111e-09,
        -1.0,
        0.999999999819,
        0.999999997757,
        -0.112984384892,
        -0.0416433429075,
        0.000150443152542,
        -0.00638269666636,
        -0.362010565083,
        -0.000123433098063,
        -2.40228968974e-06,
        -2.16790614413e-06,
        1.55403334969e-06,
        3.27182905455e-07,
        4.11922546514e-07,
        1.27729274929e-08,
        5.28352903447e-08,
        1.54680718778e-09,
        2.97890645679e-10,
        4.33580106954e-12,
        -3.95161574774e-12,
        5.29446647784e-08,
        1.12191163738e-07,
        9.86966867761e-09,
        1.95852076812e-08,
        5.85862504036e-10,
        -1.0,
        1.00000000006,
        0.999999996872,
        -0.229405593376,
        -0.0108812207299,
        9.52797773627e-06,
        -0.00177066219011,
        -4.37308542452e-05,
        -0.0233718179063,
        -1.95515029829e-06,
        -6.5866661457e-07,
        2.69503302539e-07,
        8.18885754011e-08,
        1.01659610476e-07,
        4.24695118799e-09,
        1.70531534616e-08,
        7.0507047878e-10,
        1.45246308596e-10,
        1.86041103008e-11,
        2.93745114206e-12,
        1.91743893903e-08,
        3.1230640332e-08,
        4.48628508519e-09,
        6.41136593226e-09,
        7.2217430224e-10,
        -1.0,
        1.00000000001,
        1.00000000001,
        -0.317751111268,
        -0.000958957355059,
        -2.4300483609e-05,
        -0.000243775037504,
        -1.37192819384e-05,
        -1.28558286791e-06,
        -0.00137475549746,
        -1.39478533536e-07,
        -9.00027123433e-08,
        4.47119006444e-09,
        4.05248897543e-09,
        1.14326822552e-09,
        4.26092310585e-09,
        3.15205004301e-10,
        6.94387632858e-11,
        1.62145886111e-11,
        3.56849965304e-12,
        6.20945831001e-09,
        4.39383985301e-09,
        2.00031031447e-09,
        1.66013973824e-09,
        5.39345118084e-10,
        -1.0,
        1.00000000003,
        0.999999998222,
        -0.380870866253,
        -0.00528643707602,
        1.40830631921e-05,
        -0.000826860402006,
        -1.74022045483e-05,
        -5.25223198236e-07,
        -2.87891074657e-07,
        -0.000814582806826,
        1.73754427019e-07,
        4.07720690567e-08,
        5.122011784e-08,
        1.7711176869e-09,
        7.23343247134e-09,
        2.46275777824e-10,
        4.90269436796e-11,
        3.49093998872e-12,
        1.69884114967e-13,
        7.59110261331e-09,
        1.45464633485e-08,
        1.5685310322e-09,
        2.69853066524e-09,
        1.69819071772e-10,
        -1.0,
        1.0,
        0.999999997264,
        -0.406501350782,
        0.00746521687272,
        -5.02057202595e-05,
        0.00106069726795,
        1.25589215104e-05,
        -5.969543409e-07,
        3.05852207193e-07,
        -3.82021855161e-07,
        -0.000272907584827,
        -6.07012599257e-08,
        -7.8184461282e-08,
        -1.56691606686e-09,
        -6.85764871071e-09,
        -4.1171338564e-11,
        -6.25338555368e-13,
        1.26585389802e-11,
        3.74102952341e-12,
        -5.23016583168e-09,
        -1.85343712528e-08,
        -2.70932771464e-10,
        -2.47976923674e-09,
        3.34551788929e-10,
        -1.0,
        0.999999999984,
        0.999999997868,
        -0.471752096016,
        0.0061491727747,
        -2.73332424059e-05,
        0.000922962348115,
        1.58369985e-05,
        1.23887080626e-07,
        2.97892201608e-07,
        -2.51568800965e-07,
        -4.85331829829e-08,
        -0.000131782569117,
        -6.16895807478e-08,
        -1.72064163439e-09,
        -7.1918835808e-09,
        -1.74944807549e-10,
        -3.20717084015e-11,
        2.34397143628e-12,
        1.25014005796e-12,
        -6.83063959458e-09,
        -1.61900678108e-08,
        -1.11697480602e-09,
        -2.65506918469e-09,
        1.15793309602e-11,
        -1.0,
        0.999999999988,
        0.999999998574,
        -0.495206984804,
        0.0125509659338,
        -5.74436887467e-05,
        0.00187777910139,
        3.16256859481e-05,
        1.80355494582e-07,
        6.01704120776e-07,
        -5.2071365207e-07,
        -9.91811040169e-08,
        -1.26212927514e-07,
        -9.24416854757e-05,
        -3.46023539913e-09,
        -1.44910183204e-08,
        -3.40281879851e-10,
        -6.1709751355e-11,
        5.72831130353e-12,
        2.7639878679e-12,
        -1.36369571995e-08,
        -3.29306067609e-08,
        -2.17330707546e-09,
        -5.34528804009e-09,
        5.45808509198e-11,
        -1.0,
        0.999999999915,
        0.999999997638,
        -0.667732223618,
        0.0136339278833,
        -2.38184795291e-05,
        0.00217487555874,
        4.92297880759e-05,
        1.89168860899e-06,
        7.76239417273e-07,
        -3.90391189296e-07,
        -1.03422564693e-07,
        -1.29495879844e-07,
        -4.9437614001e-09,
        -1.64013455571e-06,
        -1.99799487083e-08,
        -7.59365939895e-10,
        -1.54305320346e-10,
        -1.62127491073e-11,
        -2.07407841423e-12,
        -2.1752545043e-08,
        -3.82951810105e-08,
        -4.82902628714e-09,
        -7.49326411198e-09,
        -6.72433082924e-10,
        -1.0,
        0.99999999988,
        0.999999995466,
        -0.578729366171,
        0.00909866163236,
        -1.89188724251e-05,
        0.0014414293647,
        3.1915759134e-05,
        1.13671789738e-06,
        5.11239692736e-07,
        -2.74619165226e-07,
        -6.94873142644e-08,
        -8.70700388046e-08,
        -3.20975857066e-09,
        -1.30220877052e-08,
        -1.84969393136e-05,
        -4.77180407609e-10,
        -9.63297634504e-11,
        -9.09767903442e-12,
        -9.93451168379e-13,
        -1.40047627492e-08,
        -2.53751099865e-08,
        -3.03656559486e-09,
        -4.87380239366e-09,
        -3.9277912253e-10,
        -1.0,
        0.999999999825,
        1.00000000008,
        -0.748491005068,
        0.00811973784998,
        2.47393207252e-05,
        0.0014328616024,
        4.48581644106e-05,
        2.85971795686e-06,
        5.91907051264e-07,
        -5.58860296177e-08,
        -5.74538608414e-08,
        -6.95293243987e-08,
        -4.14920406196e-09,
        -1.62254849461e-08,
        -8.48786930631e-10,
        -5.194871508e-07,
        -1.80655963491e-10,
        -3.24813051392e-11,
        -6.39610769988e-12,
        -2.0040885755e-08,
        -2.53859789938e-08,
        -5.38983550908e-09,
        -6.17879158175e-09,
        -1.14509487541e-09,
        -1.0,
        0.99999999954,
        0.999999992467,
        -0.819154048314,
        0.00890651458117,
        3.95704981998e-05,
        0.00161555853084,
        5.40915135836e-05,
        3.68790041894e-06,
        6.89799011514e-07,
        -4.71233973682e-09,
        -6.16543512547e-08,
        -7.38243817801e-08,
        -4.93601464593e-09,
        -1.91768408464e-08,
        -1.05774392369e-09,
        -2.26523289214e-10,
        -8.76023085655e-08,
        -4.29272359026e-11,
        -8.66429288129e-12,
        -2.42415644792e-08,
        -2.86669902213e-08,
        -6.7148463428e-09,
        -7.32554743533e-09,
        -1.49403018914e-09,
        -1.0,
        0.999999999511,
        0.999999990583,
        -0.917157833343,
        -0.00168177249751,
        0.000116174687488,
        0.000130779620272,
        3.85897637464e-05,
        4.79825749743e-06,
        2.74052626965e-07,
        5.63706195333e-07,
        2.52144783161e-08,
        3.82461607804e-08,
        -2.8919862907e-09,
        -1.00862067365e-08,
        -1.05980531789e-09,
        -2.39230457289e-10,
        -6.44983944602e-11,
        -1.56995797364e-08,
        -1.47751287745e-11,
        -1.78825836366e-08,
        -2.74435693713e-09,
        -6.71352751917e-09,
        -4.06182654692e-09,
        -2.08522423101e-09,
        -1.0,
        0.999999999408,
        0.999999998808,
        -1.00000000122,
        -0.00746203271149,
        0.000166132862327,
        -0.000650957120524,
        3.33672043743e-05,
        5.77099820348e-06,
        7.37233726246e-08,
        9.11557579488e-07,
        7.35782462943e-08,
        1.01061212932e-07,
        -2.03046523952e-09,
        -6.03394548567e-09,
        -1.14506019422e-09,
        -2.65015774995e-10,
        -8.11555342835e-11,
        -1.92112742858e-11,
        -2.85701041597e-09,
        -1.59064558212e-08,
        1.08696402948e-08,
        -7.24618596417e-09,
        -2.64195836015e-09,
        -2.56684586285e-09,
        -1.0,
        0.999999999972,
        0.999999987349,
        -0.557408362755,
        0.0052879581187,
        -3.50294831091e-06,
        0.000864105093849,
        2.15099091766e-05,
        9.93297668915e-07,
        3.21710359358e-07,
        -1.25569992705e-07,
        -3.95679124546e-08,
        -4.91370811994e-08,
        -2.09646936531e-09,
        -8.39682650545e-09,
        -3.53415210209e-10,
        -7.30144222169e-11,
        -9.668236275e-12,
        -1.56866576549e-12,
        -9.49653874535e-09,
        -3.18697348604e-05,
        -1.52413546458e-08,
        -2.24682669879e-09,
        -3.16150898102e-09,
        -3.71207203132e-10,
        -1.0,
        0.999999999997,
        0.999999986101,
        -0.563630978237,
        0.0142572239092,
        -5.10251155701e-05,
        0.00218338439852,
        4.15791590944e-05,
        8.33556893311e-07,
        7.31028259024e-07,
        -5.27313733269e-07,
        -1.11195184515e-07,
        -1.40619075456e-07,
        -4.36989614662e-09,
        -1.80389659364e-08,
        -5.30730469216e-10,
        -1.02356964906e-10,
        -1.76174502702e-12,
        1.26975680542e-12,
        -1.8071579171e-08,
        -3.83519170509e-08,
        -2.65670583186e-05,
        -3.38268133373e-09,
        -6.69772151665e-09,
        -2.07957460493e-10,
        -1.0,
        0.999999999883,
        0.999999985202,
        -0.639787226138,
        0.00523888216171,
        1.57214804067e-05,
        0.000923615433188,
        2.88637613701e-05,
        1.83525918019e-06,
        3.81264906647e-07,
        -3.71917269058e-08,
        -3.70966353023e-08,
        -4.49176489871e-08,
        -2.66936972187e-09,
        -1.04418555489e-08,
        -5.45114915492e-10,
        -1.1599063744e-10,
        -2.0811074254e-11,
        -4.09495538195e-12,
        -1.2886754177e-08,
        -1.63639081373e-08,
        -3.4616561695e-09,
        -5.31159115839e-06,
        -3.97555974261e-09,
        -7.34090897955e-10,
        -1.0,
        0.999999999949,
        0.999999983928,
        -0.644277956756,
        0.0127664511016,
        -2.46136758878e-05,
        0.00202918571287,
        4.55087692437e-05,
        1.67964267269e-06,
        7.2325925444e-07,
        -3.76466640009e-07,
        -9.72554792205e-08,
        -1.2177867416e-07,
        -4.56222504452e-09,
        -1.84791982769e-08,
        -6.88926148495e-10,
        -1.39498082146e-10,
        -1.38827810188e-11,
        -1.64707072934e-12,
        -1.99927506133e-08,
        -3.57265167479e-08,
        -4.38308916059e-09,
        -6.92177332534e-09,
        -4.82838797639e-06,
        -5.87585400807e-10,
        -1.0,
        0.999999987312,
        -0.718541967414,
        0.000588539612823,
        6.00159819816e-05,
        0.000308853683826,
        2.62173042092e-05,
        2.79303102981e-06,
        2.33197421411e-07,
        2.60761128066e-07,
        2.22165617772e-09,
        6.39393094837e-09,
        -2.09866032359e-09,
        -7.62133056987e-09,
        -6.53627113239e-10,
        -1.45650789743e-10,
        -3.6477576929e-11,
        -8.18109121054e-12,
        -1.20128407905e-08,
        -5.67547970216e-09,
        -4.14304376721e-09,
        -3.00800934e-09,
        -1.19577223686e-09,
        -9.16940824279e-07,
        0.99999999784,
        -0.999999996109,
        8.76061106996e-05,
        0.000933516408752,
        0.0285935045768,
        0.00180993161394,
        0.000230681038858,
        5.67409837896e-05,
        3.11022796802e-05,
        7.19872185998e-06,
        4.34555793323e-06,
        9.52297676017e-08,
        2.84570313809e-07,
        6.54704127129e-09,
        9.1754611718e-10,
        1.59710237637e-10,
        2.50080737529e-11,
        5.28034915178e-07,
        4.30905705549e-07,
        7.98457437089e-08,
        7.33294572326e-08,
        1.22964825949e-08,
        -1.0,
        0.705862613142,
        1.0,
        -1.0,
        0.0029091836193,
        1.0,
        0.999999999545,
        -0.0329287854292,
        1.00000030599,
        -1.00000038296,
        1.0,
        0.999999119609,
        -0.999999999967,
        1.0,
        0.999999984472,
        -1.00000000073,
        1.0,
        0.999999997757,
        -1.00000000007,
        1.0,
        0.999999996872,
        -0.999999999745,
        1.0,
        1.00000000001,
        -0.999999999619,
        1.0,
        0.999999998222,
        -0.999999999607,
        1.0,
        0.999999997264,
        -0.999999999851,
        1.0,
        0.999999997868,
        -1.0,
        1.0,
        0.999999998574,
        -0.999999999903,
        1.0,
        0.999999997638,
        -0.999999999938,
        1.0,
        0.999999995466,
        -0.999999999981,
        1.0,
        1.00000000008,
        -1.0,
        1.0,
        0.999999992467,
        -1.00000000004,
        1.0,
        0.999999990583,
        -1.00000000001,
        1.0,
        0.999999998808,
        -1.00000000034,
        1.0,
        0.999999987349,
        -0.999999999862,
        1.0,
        0.999999986101,
        -0.999999999816,
        1.0,
        0.999999985202,
        -0.999999999912,
        1.0,
        0.999999983928,
        -0.99999999995,
        0.999999987312,
        1.0,
        -0.999999996109,
        -0.999999999816,
        1.0,
        -1.0,
        -1.0,
        0.999999847963,
        -0.999999683297,
        1.0,
        0.975817055702,
        -0.000276840452676,
        0.999999928517,
        1.0,
        0.975817063624,
        -0.00210536516119,
        0.999999984222,
        1.0,
        0.97581705527,
        -0.000991485547322,
        0.999999994713,
        1.0,
        0.975817063094,
        -0.00268579585509,
        0.999999998517,
        1.0,
        0.975817042832,
        -0.00415256447901,
        0.999999999991,
        1.0,
        0.975817198385,
        -0.0053491175376,
        0.999999999973,
        1.0,
        0.975816996938,
        -0.00559910821519,
        1.0,
        1.0,
        0.97581792643,
        -0.00682604298098,
        1.0,
        1.0,
        0.975818381952,
        -0.00712214406663,
        1.00000000001,
        1.0,
        0.975837106449,
        -0.0101859666316,
        1.00000000005,
        1.0,
        0.975868011835,
        -0.00870680353818,
        1.00000000002,
        1.0,
        0.978102229716,
        -0.0118491783215,
        1.00000000002,
        1.0,
        0.951483463508,
        -0.0139411655754,
        1.00000000003,
        1.0,
        0.986780496814,
        -0.015216598367,
        1.00000000001,
        1.0,
        1.0,
        -0.00875214675442,
        1.00000000004,
        1.0,
        0.97584422449,
        -0.00834030957964,
        1.00000000001,
        1.0,
        0.975863163173,
        -0.00831770764936,
        1.00000000001,
        1.0,
        0.975853221892,
        -0.00986344490786,
        1.00000000001,
        1.0,
        0.976011253422,
        -0.00976774216727,
        1.00000000002,
        0.975600742664,
        -0.0112913527143,
        1.0,
        -0.975817130273,
        0.999999910533,
        0.992948170197,
        1.0,
        -0.0449230386191,
        1.0,
        0.0,
        -0.00982868441411,
        0.999999998739,
        0.999999889054,
        1.0,
        0.999999880318,
        -0.017894118367,
        0.999999965394,
        1.0,
        0.999999985798,
        -0.13548205804,
        0.999999999991,
        1.0,
        0.999999998383,
        -0.0641399413735,
        0.999999984855,
        1.0,
        0.999999995936,
        -0.173637960602,
        0.999999905851,
        1.0,
        0.9999999946,
        -0.268799925101,
        0.999999994621,
        1.0,
        0.999999753168,
        -0.346724601222,
        0.999999904338,
        1.0,
        0.999999690465,
        -0.362786501186,
        0.9999999407,
        1.0,
        0.999998429802,
        -0.442582954312,
        0.999999994965,
        1.0,
        0.999998717469,
        -0.461723335083,
        0.999999935432,
        1.0,
        0.999830595152,
        -0.660824921098,
        0.999999832705,
        1.0,
        0.99995850959,
        -0.564462859266,
        0.999999924585,
        1.0,
        0.99893004564,
        -0.726286288271,
        0.999999954662,
        1.0,
        0.971385257077,
        -0.653661328055,
        0.999999936311,
        1.0,
        0.928450796754,
        -0.47583103322,
        0.999999989624,
        1.0,
        0.0,
        -0.367714112727,
        0.999999921097,
        1.0,
        0.999955292929,
        -0.540720861952,
        0.999999938244,
        1.0,
        0.999998856527,
        -0.538868783771,
        0.99999995033,
        1.0,
        0.99987541278,
        -0.63664530502,
        0.999999960689,
        1.0,
        0.999936809426,
        -0.630595162339,
        0.999999926192,
        0.997694043338,
        -0.729166393798,
        1.0,
        -1.0000000018,
        0.99999999944,
        0.283206441026,
        1.0,
        -0.0159507882569,
        1.0,
        1.0,
        -1.119451156,
        0.999999999997,
        1.0,
        -0.999999874289,
        0.979530370654,
        -0.000191431417132,
        -1.0,
        -0.999999983686,
        0.979530394631,
        -0.00132283122494,
        -1.0,
        -1.00000000103,
        0.979530394103,
        -0.000633893461507,
        -1.0,
        -1.00000000138,
        0.979530414827,
        -0.00168830965093,
        -1.0,
        -1.00000000001,
        0.979530392246,
        -0.00260427031459,
        -1.0,
        -1.00000000003,
        0.979530355301,
        -0.00335419986083,
        -1.0,
        -1.0,
        0.979530229059,
        -0.00350881510816,
        -1.0,
        -0.999999999999,
        0.979528096878,
        -0.00427619346353,
        -1.0,
        -0.999999999988,
        0.979530896008,
        -0.00446010866183,
        -1.0,
        -0.999999999947,
        0.979526455442,
        -0.00636614911279,
        -1.0,
        -0.999999999976,
        0.979494170997,
        -0.0054463602111,
        -1.0,
        -0.999999999979,
        0.977584878461,
        -0.0073681650907,
        -1.0,
        -0.999999999966,
        0.980906303231,
        -0.00629665345904,
        -1.0,
        -0.999999999995,
        0.986619266006,
        -0.00342112711326,
        -1.0,
        -0.99999999996,
        1.0,
        0.0245879055745,
        -1.0,
        -0.999999999992,
        0.97951652857,
        -0.00522796103463,
        -1.0,
        -0.99999999999,
        0.979476456553,
        -0.00520592379676,
        -1.0,
        -0.999999999987,
        0.979639520903,
        -0.00615100762252,
        -1.0,
        -0.999999999977,
        0.979792676757,
        -0.00608434968761,
        -1.0,
        0.979781864216,
        -0.00718259755621,
        -0.999999900311,
        -0.979530328249,
        -1.0,
        0.987865167215,
        -0.0452017857855,
        -1.0,
        -1.0,
        -0.00999815070733,
        -0.999999889249,
        -0.999999999983,
        1.0,
        0.979530370654,
        -0.000191431417132,
        1.0,
        0.979530394631,
        -0.00132283124912,
        1.0,
        0.979530394103,
        -0.000633893466375,
        1.0,
        0.979530414827,
        -0.001688309796,
        1.0,
        0.979530392246,
        -0.0026042708871,
        1.0,
        0.979530355301,
        -0.00335420004644,
        1.0,
        0.979530229059,
        -0.0035088153325,
        1.0,
        0.979528096878,
        -0.00427619351061,
        1.0,
        0.979530896008,
        -0.00446010869151,
        1.0,
        0.979526455442,
        -0.00636614911596,
        1.0,
        0.979494170997,
        -0.00544636021432,
        1.0,
        0.977584878461,
        -0.00736816509095,
        1.0,
        0.980906303231,
        -0.00629665345908,
        1.0,
        0.986619266006,
        -0.00342112711327,
        1.0,
        1.0,
        0.0245879055745,
        1.0,
        0.97951652857,
        -0.0052279610389,
        1.0,
        0.979476456553,
        -0.00520592380051,
        1.0,
        0.979639520903,
        -0.00615100762386,
        1.0,
        0.979792676757,
        -0.00608434968879,
        0.979781864216,
        -0.0071825975566,
        1.0,
        -0.979530328249,
        0.988620838357,
        1.0,
        -0.0452017857855,
        1.0,
        -0.00999101909666,
        1.00000000002,
        0.999999999582,
        0.850561032911,
        0.295757233996,
        -0.0540463285299,
        -0.778565635795,
        -0.0143847128538,
        -0.0431541400097,
        -0.0119872594242,
        -0.00239745227353,
        -0.00179808869367,
        -0.00779170438769,
        -0.00209776890074,
        -0.00149840551365,
        -0.000524442361034,
        0.000899045122278,
        0.000299681631929,
        -0.000187300886227,
        9.36504941466e-05,
        0.000121745667058,
        0.0,
        0.0,
        0.000599363158332,
        0.000599363346278,
        0.000299681627052,
        -1.0,
        -0.2,
        0.999999999142,
        0.999972867437,
        -0.0451999050839,
        -0.0695900224209,
        -0.0553959491141,
        -0.246766479066,
        -0.11128489776,
        -0.0867988692419,
        0.0,
        -0.0203275116886,
        -0.0122179921401,
        -0.00178880680179,
        -0.00155948175108,
        -0.00253915536235,
        -0.00141825435974,
        -0.00216644551558,
        -0.00139179332583,
        -0.00101701259196,
        -0.000670898139711,
        -0.00236320734805,
        -0.0013686625306,
        0.0,
        -0.00230799971017,
        -0.00218639150376,
        -0.999999999919,
        -0.199999999959,
        1.0,
        0.852817925131,
        0.10822277079,
        -0.0385104203375,
        -0.0118246275385,
        -0.035363936246,
        -0.448676866347,
        -0.00938505678137,
        -0.00332276757487,
        -0.00157075964143,
        -0.000888268759923,
        -0.000267640030307,
        -0.000112357028593,
        0.00015968554574,
        9.29022879412e-05,
        0.000132518837028,
        0.000105944881269,
        6.98772550484e-05,
        4.77546282519e-05,
        3.01815860636e-05,
        0.000123950044002,
        0.000122214795459,
        0.000164406211487,
        0.000119109248524,
        -0.999999999827,
        -0.19999999999,
        1.00000000005,
        0.852744687531,
        0.0275346303853,
        -0.0243473038137,
        -0.00751130447638,
        -0.0223371077253,
        -0.00583848199098,
        -0.190659914786,
        -0.00214060638491,
        -0.000956605203447,
        -0.000542196726116,
        -0.000167840380786,
        -7.2642601105e-05,
        0.000103276135882,
        5.85555802203e-05,
        8.65663433478e-05,
        6.92505301022e-05,
        4.59929217713e-05,
        3.14718184534e-05,
        1.87985089895e-05,
        7.7068832614e-05,
        7.80171464512e-05,
        0.000105664507947,
        7.79807753344e-05,
        -0.999999999938,
        -0.117370234964,
        1.0,
        0.852795622045,
        -0.00826957146756,
        -0.015995727016,
        -0.00501540300229,
        -0.014764875986,
        -0.00394293701748,
        -0.00140703491565,
        -0.0910008047842,
        -0.000658909070636,
        -0.000378338514651,
        -0.00011492171147,
        -5.05309550211e-05,
        6.72009308612e-05,
        3.85480267731e-05,
        5.63983092298e-05,
        4.5171310539e-05,
        3.00820755167e-05,
        2.06226578226e-05,
        1.24546927772e-05,
        5.01237660992e-05,
        5.19351752762e-05,
        6.88094539678e-05,
        5.11761788519e-05,
        -0.999999999872,
        -0.0075881660042,
        1.0000000001,
        0.85291243671,
        -0.0103339271248,
        -0.0120984564068,
        -0.00379818077978,
        -0.0111937849051,
        -0.00303866567877,
        -0.00105049972319,
        -0.000517938700937,
        -0.0590865452346,
        -0.000297795568982,
        -8.85087952624e-05,
        -3.83486410063e-05,
        4.98101082664e-05,
        2.9151837513e-05,
        4.15214280616e-05,
        3.32668829147e-05,
        2.2070342945e-05,
        1.5121490323e-05,
        9.47081114123e-06,
        3.81739746169e-05,
        3.90994089786e-05,
        5.12189821904e-05,
        3.76929626524e-05,
        -0.999999999757,
        -0.00958350431632,
        0.999999999997,
        0.852993719675,
        -0.0373226160784,
        -0.00800996989589,
        -0.00259112391125,
        -0.00747206839933,
        -0.00205856457922,
        -0.0007109639868,
        -0.000352352996707,
        -0.000207336253863,
        -0.0337634009333,
        -6.1676780106e-05,
        -2.82332869757e-05,
        3.34264456768e-05,
        1.93835983057e-05,
        2.82245161099e-05,
        2.26552721248e-05,
        1.51966208149e-05,
        1.04587312662e-05,
        6.31978884519e-06,
        2.44834603675e-05,
        2.67711084718e-05,
        3.41967364449e-05,
        2.59722303875e-05,
        -0.999999999784,
        -0.00510221180274,
        0.999999999962,
        0.85296412145,
        -0.0487480307347,
        -0.00389719702636,
        -0.00126420581658,
        -0.00363443839102,
        -0.000994042085184,
        -0.000349336950644,
        -0.000168421499091,
        -9.92535790549e-05,
        -2.98769676284e-05,
        -0.0291913659357,
        -1.38481248813e-05,
        1.64751727868e-05,
        9.43669997112e-06,
        1.39859776739e-05,
        1.12265037057e-05,
        7.55608147037e-06,
        5.20452917205e-06,
        3.06875175257e-06,
        1.18171221771e-05,
        1.3109630128e-05,
        1.68041485014e-05,
        1.28860406661e-05,
        -0.999999999785,
        -0.00848322210113,
        0.999999999971,
        0.852966578926,
        -0.0682513277622,
        -0.0018225008626,
        -0.00063743649426,
        -0.00173426448451,
        -0.000486703543509,
        -0.000174583386737,
        -8.21694027378e-05,
        -5.10549167431e-05,
        -1.56266917381e-05,
        -8.14898301206e-06,
        -0.0256117642626,
        8.10510380198e-06,
        4.46496643803e-06,
        7.13786236983e-06,
        5.75340649149e-06,
        3.98347350948e-06,
        2.77258036702e-06,
        1.45975345549e-06,
        5.00614174329e-06,
        6.69556182642e-06,
        8.13302691365e-06,
        6.78489773731e-06,
        -0.999999999921,
        -0.00935475150056,
        0.999999999598,
        0.853184590808,
        -0.118495484331,
        0.0085857946809,
        0.00263204718967,
        0.00785470819201,
        0.00202823572834,
        0.000757011898012,
        0.000328042300244,
        0.000184897710626,
        5.80069063111e-05,
        2.50471887689e-05,
        -3.67169113224e-05,
        -0.00631784029793,
        -2.06332795093e-05,
        -3.08089469996e-05,
        -2.46338747394e-05,
        -1.63585707371e-05,
        -1.11876888597e-05,
        -6.59921020349e-06,
        -2.7233479778e-05,
        -2.74071200665e-05,
        -3.75198128848e-05,
        -2.7677930362e-05,
        -0.999999999551,
        -0.00589290257333,
        0.999999999862,
        0.852908547355,
        -0.0919955432225,
        0.00279070709155,
        0.000854327433286,
        0.00256121635754,
        0.000680111293965,
        0.000239897917718,
        0.000114137655443,
        6.43755024081e-05,
        1.9368757468e-05,
        8.07670762955e-06,
        -1.1505680557e-05,
        -6.70156301188e-06,
        -0.01415829357,
        -9.52637033785e-06,
        -7.61899403211e-06,
        -5.01835405239e-06,
        -3.42616754567e-06,
        -2.16351289711e-06,
        -8.98981253434e-06,
        -8.79901324642e-06,
        -1.18506996849e-05,
        -8.54607966364e-06,
        -0.999999999899,
        -0.0238050706378,
        0.999999999605,
        0.852925838661,
        -0.137360194588,
        0.0121414543145,
        0.00383124242113,
        0.0112170826202,
        0.00298329444174,
        0.0010798123809,
        0.000494437844447,
        0.000285049621426,
        8.750496932e-05,
        3.91888058292e-05,
        -5.16348216784e-05,
        -2.9291049837e-05,
        -4.36018772395e-05,
        -0.00456219050047,
        -3.49328255556e-05,
        -2.33658927423e-05,
        -1.60398429444e-05,
        -9.4482489229e-06,
        -3.76436261954e-05,
        -3.98129283412e-05,
        -5.2705719345e-05,
        -3.96863131171e-05,
        -0.999999999905,
        -0.02236706231,
        0.999999999019,
        0.852927033651,
        -0.216649369566,
        0.0168573760321,
        0.00531760456041,
        0.0155636776531,
        0.00411870121177,
        0.00150510382205,
        0.000678341277587,
        0.000390737692792,
        0.000120798411677,
        5.43049912917e-05,
        -7.2119411773e-05,
        -4.06704236389e-05,
        -6.10254988682e-05,
        -4.88880796219e-05,
        -0.00263518223646,
        -3.27388266618e-05,
        -2.24784357452e-05,
        -1.30976283225e-05,
        -5.21479053282e-05,
        -5.53572118138e-05,
        -7.35222975784e-05,
        -5.55428278378e-05,
        -0.999999999651,
        -0.0222796261454,
        0.999999999474,
        0.852927475509,
        -0.179129845424,
        0.0192362605996,
        0.0061557883126,
        0.0178226121645,
        0.00472955119662,
        0.00174249701206,
        0.000776253505635,
        0.000451408643987,
        0.00014050785294,
        6.49139985428e-05,
        -8.33033306016e-05,
        -4.65125342608e-05,
        -7.10823688551e-05,
        -5.69923184483e-05,
        -3.84172969545e-05,
        -0.00151904890428,
        -2.64398023311e-05,
        -1.49841973398e-05,
        -5.84319014117e-05,
        -6.43004411301e-05,
        -8.46062743714e-05,
        -6.51247739398e-05,
        -0.999999999812,
        -0.0222488144268,
        0.999999999294,
        0.852927656625,
        -0.197748238634,
        0.0224781047707,
        0.00722487216896,
        0.0208429622905,
        0.00552145880013,
        0.00204972172173,
        0.000902204349505,
        0.000525844498704,
        0.000164658443105,
        7.68221290674e-05,
        -9.80215595324e-05,
        -5.43917743497e-05,
        -8.39549503104e-05,
        -6.73284262417e-05,
        -4.55066222664e-05,
        -3.13452173979e-05,
        -0.000889173806253,
        -1.7510215797e-05,
        -6.77909089273e-05,
        -7.56180455251e-05,
        -9.9370172565e-05,
        -7.7077363057e-05,
        -0.999999999746,
        -0.0222355908899,
        0.999999999746,
        0.852900998727,
        -0.0657544711661,
        0.000772700662733,
        0.00023750936355,
        0.000710047524804,
        0.000189149011781,
        6.65528278471e-05,
        3.18097423265e-05,
        1.79957462926e-05,
        5.40642595855e-06,
        2.26935440948e-06,
        -3.18651652417e-06,
        -1.85672303667e-06,
        -2.64193204423e-06,
        -2.11350742532e-06,
        -1.3939027549e-06,
        -9.52215747979e-07,
        -6.00222774291e-07,
        -0.0165108092783,
        -2.48050122163e-06,
        -2.44623969757e-06,
        -3.28099919339e-06,
        -2.37462798994e-06,
        -0.999999998465,
        -0.0245284807032,
        0.999999999785,
        0.852904579607,
        -0.0787211827438,
        0.00343270415798,
        0.00100373688034,
        0.00311425132828,
        0.000812647630807,
        0.000284065547487,
        0.000136341295361,
        7.4189587316e-05,
        2.2109511859e-05,
        8.25195958515e-06,
        -1.37752073634e-05,
        -8.19047065447e-06,
        -1.11411432113e-05,
        -8.88400177188e-06,
        -5.73261765997e-06,
        -3.88172577403e-06,
        -2.63447300048e-06,
        -1.15804039236e-05,
        -0.0155448274973,
        -1.02542312286e-05,
        -1.43229516223e-05,
        -9.76646959004e-06,
        -0.99999999954,
        -0.0241902066197,
        0.999999999644,
        0.852919320387,
        -0.0844620429311,
        0.00545277599104,
        0.00173152262994,
        0.00505022651765,
        0.0013557781519,
        0.000484591630219,
        0.000226702403989,
        0.000131390451188,
        3.99818679916e-05,
        1.79992412581e-05,
        -2.30712661309e-05,
        -1.31648336091e-05,
        -1.94804358897e-05,
        -1.56146751946e-05,
        -1.04514673068e-05,
        -7.17917895727e-06,
        -4.25856161524e-06,
        -1.68478545447e-05,
        -1.79650677908e-05,
        -0.0093910312931,
        -2.35645079953e-05,
        -1.77800201687e-05,
        -0.999999999903,
        -0.0228746488089,
        0.999999999922,
        0.852919924144,
        -0.0976437005566,
        0.0076653243025,
        0.00236237295985,
        0.00704267651489,
        0.00186584474943,
        0.000665407332515,
        0.000311232888121,
        0.000176426261167,
        5.35148653413e-05,
        2.27663706278e-05,
        -3.19285617798e-05,
        -1.84270317893e-05,
        -2.65830416386e-05,
        -2.12681226478e-05,
        -1.40669461189e-05,
        -9.61704268394e-06,
        -5.94227649717e-06,
        -2.44595178726e-05,
        -2.44041623522e-05,
        -3.27970462325e-05,
        -0.00906104195264,
        -2.39268988463e-05,
        -0.999999999662,
        -0.0228264679036,
        0.852929338816,
        -0.105060462186,
        0.00920448274867,
        0.00296021906186,
        0.00854908999023,
        0.00229570050038,
        0.00083010840916,
        0.000381666081039,
        0.000223102722093,
        6.84986600974e-05,
        3.16676316221e-05,
        -3.94794165719e-05,
        -2.22672949597e-05,
        -3.36147661427e-05,
        -2.69629473033e-05,
        -1.81619319317e-05,
        -1.25026896466e-05,
        -7.19975743584e-06,
        -2.79519838019e-05,
        -3.08304293879e-05,
        -4.01658246961e-05,
        -3.08592432366e-05,
        -0.0053955333718,
        -0.0224237445936,
        0.999999999815,
        -0.851211728231,
        9.33510725801e-09,
        4.9223990892e-09,
        6.71992635088e-05,
        0.00242222819957,
        0.01299566037,
        0.00539288774382,
        0.00567385234492,
        0.00260239552214,
        0.00198181686202,
        0.000517533077794,
        0.000344659759498,
        8.33864591119e-05,
        4.14287574247e-05,
        2.06009880177e-05,
        1.03904684005e-05,
        0.000419831856977,
        0.00038288429184,
        0.00020838419925,
        0.000198711138639,
        0.000103020700574,
        -0.999999999947,
        1.0,
        1.0,
        -1.0,
        -1.0,
        -0.00111426067377,
        1.0,
        -1.0,
        -1.0,
        -0.0378879515982,
        0.999999992089,
        -0.000395422028644,
        -1.00000001129,
        1.0,
        0.719705488722,
        1.00000000007,
        0.999908674466,
        1.00000000014,
        0.682241885467,
        1.00000000005,
        0.682080877433,
        1.0000000001,
        0.682190614257,
        1.0000000002,
        0.682442258611,
        0.99999999981,
        0.682617352717,
        1.00000000017,
        0.682553589438,
        1.00000000006,
        0.682558881878,
        0.99999999994,
        0.683028523843,
        1.00000000008,
        0.682433886622,
        1.00000000008,
        0.682471128947,
        0.999999999962,
        0.682473701056,
        1.00000000015,
        0.682474632042,
        0.999999999887,
        0.682475055539,
        0.999999998091,
        0.682417622654,
        0.999999999939,
        0.682425332157,
        1.00000000008,
        0.682457085443,
        0.999999999865,
        0.682458382188,
        0.682494244848,
        1.00000000004,
        -0.678775513806,
        1.0,
        1.0,
        0.999999999641,
        1.0,
        0.719705488717,
        1.0,
        -1.0,
        0.999999999334,
        0.999907887696,
        0.999999999795,
        -1.0,
        0.999999998584,
        0.682241699979,
        0.999999999952,
        -1.0,
        0.999999999491,
        0.682079303422,
        0.586851174817,
        -1.0,
        0.999999998947,
        0.682189985461,
        0.037940830021,
        -1.0,
        0.999999998006,
        0.682443797733,
        0.0479175215816,
        -1.0,
        0.999999999681,
        0.682620402263,
        0.0255110590137,
        -1.0,
        0.999999998235,
        0.682556087691,
        0.0424161105057,
        -1.0,
        0.999999999351,
        0.682561429365,
        0.0467737575028,
        -1.0,
        0.999999997996,
        0.683035113638,
        0.0294645128667,
        -1.0,
        0.999999999174,
        0.68243535357,
        0.119025353189,
        -1.0,
        0.999999999224,
        0.682472916759,
        0.11183531155,
        -1.0,
        0.999999998407,
        0.682475511086,
        0.111398130727,
        -1.0,
        0.999999998457,
        0.682476458765,
        0.111244072134,
        -1.0,
        0.999999999182,
        0.682476868511,
        0.111177954449,
        -1.0,
        0.999999999958,
        0.68241894949,
        0.122642403516,
        -1.0,
        0.999999997944,
        0.682426716008,
        0.120951033099,
        -1.0,
        0.999999999207,
        0.682458752307,
        0.114373244044,
        -1.0,
        0.999999998851,
        0.682460060226,
        0.114132339518,
        -1.0,
        0.682480511784,
        0.112118722968,
        0.999999999563,
        -0.678761175209,
        -1.0,
        1.0,
        1.0,
        -1.0,
        0.00197711012076,
        0.999999980964,
        -0.999999979591,
        1.0,
        0.719705488717,
        -0.261711086787,
        -1.0,
        1.0,
        0.999907887696,
        -0.999722940124,
        -1.0,
        1.0,
        0.682241699979,
        -0.0761345993822,
        -1.0,
        1.0,
        0.682079303422,
        -0.203028262128,
        -1.0,
        1.0,
        0.682189985461,
        -0.29061652278,
        -1.0,
        1.0,
        0.682443797733,
        -0.34413519135,
        -1.0,
        1.0,
        0.682620402263,
        -0.373555228242,
        -1.0,
        1.0,
        0.682556087691,
        -0.43469759657,
        -1.0,
        1.0,
        0.682561429365,
        -0.460102794831,
        -1.0,
        1.0,
        0.683035113638,
        -0.623819371918,
        -1.0,
        1.0,
        0.68243535357,
        -0.538396084045,
        -1.0,
        1.0,
        0.682472916759,
        -0.697162118198,
        -1.0,
        1.0,
        0.682475511086,
        -0.784679414973,
        -1.0,
        1.0,
        0.682476458765,
        -0.854026415333,
        -1.0,
        1.0,
        0.682476868511,
        -0.930431888222,
        -1.0,
        1.0,
        0.68241894949,
        -0.51220833114,
        -1.0,
        1.0,
        0.682426716008,
        -0.520825360073,
        -1.0,
        1.0,
        0.682458752307,
        -0.591095073338,
        -1.0,
        1.0,
        0.682460060226,
        -0.595761613231,
        -1.0,
        0.682480511784,
        -0.668996795391,
        1.0,
        -0.678761175209,
        -1.0,
        -1.0,
        0.164505890395,
        0.000522484569277,
        1.0,
        -1.0,
        -0.0420365306993,
        1.00000000862,
        -0.999999999014,
        0.951929311864,
        0.719705488717,
        -0.0785133253137,
        1.0,
        0.841131408362,
        0.999907887696,
        -0.989340194897,
        1.0,
        0.999755028305,
        0.682241699979,
        -0.0377898560453,
        1.0,
        0.999999999547,
        0.682079303422,
        -0.0903704450357,
        1.0,
        0.999999999994,
        0.682189985461,
        -0.128035265648,
        1.0,
        0.999999995268,
        0.682443797733,
        -0.153431765448,
        1.0,
        0.99999999964,
        0.682620402263,
        -0.166703990132,
        1.0,
        0.999999998217,
        0.682556087691,
        -0.193769785363,
        1.0,
        0.999999999347,
        0.682561429365,
        -0.204644105642,
        1.0,
        0.999999997986,
        0.683035113638,
        -0.286527500075,
        1.0,
        0.999999999182,
        0.68243535357,
        -0.238189018989,
        1.0,
        0.999999999236,
        0.682472916759,
        -0.309854255886,
        1.0,
        0.999999998859,
        0.682475511086,
        -0.346303465138,
        1.0,
        0.99999999848,
        0.682476458765,
        -0.380907140116,
        1.0,
        0.99999999962,
        0.682476868511,
        -0.415758824866,
        1.0,
        0.999999998309,
        0.68241894949,
        -0.227281327998,
        1.0,
        0.999999997919,
        0.682426716008,
        -0.230729833983,
        1.0,
        0.999999999218,
        0.682458752307,
        -0.262904840402,
        1.0,
        0.999999998829,
        0.682460060226,
        -0.264701195467,
        1.0,
        0.682480511784,
        -0.297936517375,
        0.999999999405,
        -0.678761175209,
        1.0,
        0.0463098902535,
        -1.0,
        0.000220938695323,
        0.0,
        1.0,
        -0.0163522907215,
        0.99999999891,
        0.999999999612,
        1.00000000003,
        0.999999999999,
        0.23727221428,
        0.0425402632953,
        0.0903494405828,
        0.00386833597763,
        0.0151178147774,
        0.00138796668588,
        -0.000667799553981,
        -0.000929641281331,
        -0.000832329236768,
        -0.000684837302699,
        -0.000617708593325,
        -0.000236638491723,
        -0.000399775773202,
        -0.000138955570359,
        -7.80457871628e-05,
        -4.36671350919e-05,
        -2.41316342378e-05,
        -0.000459577783064,
        -0.000424949707208,
        -0.000283060574441,
        -0.000267799133812,
        -0.000167079910676,
        1.0,
        0.99999999902,
        0.999999272251,
        0.207399288205,
        0.0193758123412,
        0.00176657158663,
        0.359908148721,
        0.00688704138674,
        0.000627508699197,
        -0.000293708968393,
        -0.000414935570781,
        -0.000375757568281,
        -0.000311487908912,
        -0.000281599204583,
        -0.000108257863105,
        -0.000182601988581,
        -6.35840588196e-05,
        -3.57304582069e-05,
        -1.99870758793e-05,
        -1.10460959092e-05,
        -0.000209699138333,
        -0.000194227772834,
        -0.000129354237449,
        -0.000122500056371,
        -7.6390436122e-05,
        0.999999992073,
        0.999999999927,
        0.999999268462,
        0.247031197057,
        0.0335826966865,
        0.00308121790399,
        0.0119406806797,
        0.171463262959,
        0.00106476794986,
        -0.00045963087954,
        -0.00067880310621,
        -0.000635343630038,
        -0.000537657314803,
        -0.000489126588522,
        -0.000189811390907,
        -0.000318815835317,
        -0.000111544868919,
        -6.27666841211e-05,
        -3.5088383323e-05,
        -1.93950042832e-05,
        -0.000365093034992,
        -0.000339729560357,
        -0.00022613224301,
        -0.000214725988048,
        -0.000133714580522,
        0.999999996414,
        0.999999999783,
        0.999999268541,
        0.283424531433,
        0.0091839890568,
        0.000837674404348,
        0.00326443104912,
        0.000296951698462,
        0.224962057172,
        -0.000138200994972,
        -0.000195841526407,
        -0.00017777520213,
        -0.000147592264116,
        -0.000133493103284,
        -5.1355944562e-05,
        -8.65962455803e-05,
        -3.01644996364e-05,
        -1.69523798236e-05,
        -9.48241431898e-06,
        -5.24062234608e-06,
        -9.94253933311e-05,
        -9.2122564452e-05,
        -6.13499259743e-05,
        -5.8111124704e-05,
        -3.62337470753e-05,
        1.00000000001,
        0.999999999385,
        0.999999267923,
        0.356950921302,
        -0.00969759289877,
        -0.000877487707741,
        -0.00344538250084,
        -0.000321420064861,
        0.000163085805571,
        0.0556251585093,
        0.000220777037249,
        0.000193225948121,
        0.000156592751909,
        0.000140572411783,
        5.34623548811e-05,
        9.06151955915e-05,
        3.13799586823e-05,
        1.76058480225e-05,
        9.85553549163e-06,
        5.44575579424e-06,
        0.00010439795417,
        9.61861315165e-05,
        6.40972995025e-05,
        6.05145154884e-05,
        3.7796243978e-05,
        1.00000000001,
        0.999999998819,
        0.99999926754,
        0.457014252086,
        -0.0253316372368,
        -0.00229792845881,
        -0.00900126644019,
        -0.000833255071305,
        0.000412112188988,
        0.000565390330121,
        0.101405673935,
        0.000500281611476,
        0.000408448083917,
        0.000367516626288,
        0.000140275322317,
        0.000237373566226,
        8.23532432863e-05,
        4.62289786594e-05,
        2.58721960605e-05,
        1.42967914658e-05,
        0.000273185918349,
        0.000252139674142,
        0.000167989824076,
        0.000158762434564,
        9.91079117506e-05,
        0.999999999972,
        0.999999998995,
        0.999999269261,
        0.448149449636,
        -0.0282495989133,
        -0.00255997437664,
        -0.0100374761284,
        -0.000932111968343,
        0.000465887204192,
        0.000635648845448,
        0.000559928831553,
        0.0574088471102,
        0.000455766032621,
        0.000409704257015,
        0.000156150015581,
        0.000264410179594,
        9.16648161323e-05,
        5.14450157283e-05,
        2.87941619628e-05,
        1.59110387449e-05,
        0.00030443392227,
        0.000280780159353,
        0.000187086539747,
        0.000176736739696,
        0.000110351901764,
        0.999999999915,
        1.0,
        0.999999276599,
        0.528774109887,
        -0.044538003054,
        -0.0040460932616,
        -0.0158272549325,
        -0.00145826522038,
        0.000709882304505,
        0.00098207901619,
        0.000874866739017,
        0.000717481508049,
        0.105242686248,
        0.000646487503859,
        0.000247282540924,
        0.000418047293395,
        0.00014519395921,
        8.15301693832e-05,
        4.56219811269e-05,
        2.52112955165e-05,
        0.00048080903745,
        0.000444235699749,
        0.000295938678359,
        0.00027985523373,
        0.000174644694569,
        1.0,
        0.999999997499,
        0.999999282997,
        0.523317819002,
        -0.0479962602893,
        -0.00436146082571,
        -0.0170564349323,
        -0.00157008308735,
        0.000761951152664,
        0.00105584219799,
        0.000941814799587,
        0.000773055317598,
        0.000696750975695,
        0.11758059945,
        0.000266617963738,
        0.000450651508948,
        0.000156550741566,
        8.79125979107e-05,
        4.91920093476e-05,
        2.71843261816e-05,
        0.000518244058261,
        0.000478920539781,
        0.000319037076815,
        0.000301734051768,
        0.000188286611158,
        0.999999999565,
        0.999999972607,
        0.999999240925,
        0.684295765461,
        -0.0868339830579,
        -0.00789575117597,
        -0.0308594310251,
        -0.00283506940759,
        0.00136644043276,
        0.00190038562698,
        0.00170004569604,
        0.0013980811672,
        0.00126082801955,
        0.000482902724745,
        0.0751970820571,
        0.000815895848558,
        0.000283562732959,
        0.000159258434165,
        8.91086851854e-05,
        4.92437398282e-05,
        0.000938018080918,
        0.000867225546829,
        0.000577681787304,
        0.000546491212014,
        0.000340974046575,
        0.99999999989,
        0.999999998193,
        0.999999305073,
        0.601269432904,
        -0.0684562428782,
        -0.00622475452626,
        -0.0243282695314,
        -0.00223490788358,
        0.00107695187121,
        0.00149794353789,
        0.00134014893972,
        0.00110217244944,
        0.000993988307982,
        0.000380712262984,
        0.000643230459969,
        0.306920627732,
        0.000223556310702,
        0.000125557354854,
        7.0252017083e-05,
        3.88230660882e-05,
        0.000739502590341,
        0.000683700745798,
        0.000455430557414,
        0.000430844108203,
        0.000268816678503,
        0.999999999133,
        0.999999936471,
        0.999999313214,
        0.763751608137,
        -0.106441054675,
        -0.00968004868557,
        -0.0378279267486,
        -0.0034737751384,
        0.00167162182747,
        0.00232675529631,
        0.0020828339696,
        0.00171361529355,
        0.00154559629854,
        0.000592090882556,
        0.00100028746973,
        0.000347683633526,
        0.278525588544,
        0.000195276623512,
        0.000109260192365,
        6.03803459576e-05,
        0.00114993841244,
        0.00106325546651,
        0.000708256668298,
        0.000670054210017,
        0.000418057098353,
        1.0,
        0.999999919476,
        0.99999996083,
        0.688069871968,
        -0.125006723524,
        -0.0113686642685,
        -0.0444255773155,
        -0.00407947287454,
        0.0019629201092,
        0.00273238140996,
        0.00244603909267,
        0.00201251166202,
        0.00181519731412,
        0.00069538549753,
        0.00117477514578,
        0.000408336968732,
        0.000229343868108,
        0.270547411784,
        0.000128321275892,
        7.09140450951e-05,
        0.00135053171369,
        0.00124873660203,
        0.000831809107491,
        0.000786947803126,
        0.000490987768223,
        1.0,
        0.999999985251,
        0.999998649465,
        0.920094593368,
        -0.145656098364,
        -0.013245786317,
        -0.0517645938859,
        -0.00475292097032,
        0.00228744095511,
        0.00318393349013,
        0.00285024044036,
        0.00234503687551,
        0.00211513995034,
        0.000810277216225,
        0.0013688776906,
        0.000475803799786,
        0.000267236853094,
        0.000149521831602,
        0.264409154013,
        8.26304080265e-05,
        0.00157369281762,
        0.00145506134327,
        0.000969245447313,
        0.000916970094059,
        0.000572110752645,
        0.999999999421,
        0.999998769831,
        0.999995210558,
        0.999999984104,
        -0.165002013645,
        -0.0150058430525,
        -0.0586386625629,
        -0.00538539029604,
        0.00259081798167,
        0.00360649989339,
        0.00322853270066,
        0.00265630411029,
        0.00239585912482,
        0.000917844935559,
        0.00155057685911,
        0.000538975143474,
        0.000302713941818,
        0.000169375395768,
        9.36006867576e-05,
        0.258576574778,
        0.00178262849832,
        0.00164816304869,
        0.00109791664125,
        0.0010386906564,
        0.000648066132904,
        1.0,
        0.999999991542,
        0.999999289944,
        0.62569402351,
        -0.0638916876749,
        -0.00580946393577,
        -0.0227061023385,
        -0.002086219972,
        0.00100582364564,
        0.00139862267158,
        0.00125100859306,
        0.00102871165722,
        0.000927695434901,
        0.000355296600409,
        0.000600308197477,
        0.00020863141441,
        0.000117173668932,
        6.5561538487e-05,
        3.62309416942e-05,
        0.000690170413295,
        0.315749586701,
        0.000638069079411,
        0.000425035986555,
        0.000402082441589,
        0.0002508741673,
        0.999999999678,
        0.999999985901,
        0.99999927093,
        0.617971428049,
        -0.0639449093084,
        -0.00581437939462,
        -0.022724921652,
        -0.00208773495543,
        0.00100623157059,
        0.00139942616223,
        0.00125190670808,
        0.00102954290925,
        0.000928471945323,
        0.00035560875928,
        0.000600824501491,
        0.000208815128356,
        0.000117277599055,
        6.56194373222e-05,
        3.62629750315e-05,
        0.000690754268001,
        0.000638623981487,
        0.311611311178,
        0.000425403440625,
        0.000402435548083,
        0.000251092507067,
        0.999999999442,
        0.999999975738,
        0.999999420514,
        0.71260073101,
        -0.0826212277592,
        -0.00751338022637,
        -0.029362446655,
        -0.00269682392036,
        0.00129856938015,
        0.00180690634697,
        0.00161706316015,
        0.00133018487371,
        0.00119969508413,
        0.000459545289409,
        0.000776388959434,
        0.000269849505014,
        0.000151559358424,
        8.48002741954e-05,
        4.68629025411e-05,
        0.00089256586064,
        0.000825252871824,
        0.000549719102078,
        0.293285030127,
        0.000520056685165,
        0.00032447484743,
        0.999999999883,
        0.999999984876,
        0.999999290424,
        0.700359901571,
        -0.0822213226157,
        -0.00747695784739,
        -0.0292202551422,
        -0.00268371125641,
        0.00129220153961,
        0.00179808662126,
        0.00160920207646,
        0.00132373800904,
        0.00119388511612,
        0.000457322797368,
        0.000772631887656,
        0.000268544485915,
        0.000150826263452,
        8.43899445322e-05,
        4.66361892749e-05,
        0.000888243922447,
        0.000821259665917,
        0.000547059245111,
        0.00051754116019,
        0.292434193911,
        0.000322904807523,
        0.999999999883,
        0.999999551147,
        0.785552946855,
        -0.102197983046,
        -0.00929407754641,
        -0.0363198347447,
        -0.0033353683379,
        0.00160525252629,
        0.00223422752899,
        0.00199990214798,
        0.00164533800933,
        0.00148399501288,
        0.00056848330133,
        0.00096040958945,
        0.000333820535771,
        0.00018749019664,
        0.000104903538692,
        5.79726020662e-05,
        0.00110410032504,
        0.00102086500022,
        0.000680019326409,
        0.000643338302403,
        0.000401389188133,
        0.282065902193,
        0.999999999292,
        -0.999999269587,
        -2.08620068222e-05,
        -0.00205685159966,
        -0.0271604575781,
        -0.0337798305665,
        -0.0140353854871,
        -0.00733507943767,
        -0.00578215922147,
        -0.0029556455049,
        -0.00245560367223,
        -0.00048778091712,
        -0.00107783008411,
        -0.000227708082634,
        -0.000106955059416,
        -5.07509532593e-05,
        -2.45458653501e-05,
        -0.001348135091,
        -0.00123455167853,
        -0.00061775761247,
        -0.000586991371562,
        -0.000287660826654,
        1.0,
        -1.0,
        0.724188956561,
        -1.0,
        -0.0391807361086,
        0.0391795250796,
        0.0208210703611,
        0.0289209500752,
        0.0188861354183,
        0.0124425646854,
        0.0108775677954,
        0.00681681801091,
        0.00606221999166,
        0.00176819895191,
        0.00325834868971,
        0.000951704206482,
        0.000509171065546,
        0.000267952190212,
        0.000142413491593,
        0.00381877650337,
        0.00360978351266,
        0.00211050505105,
        0.00203223178411,
        0.00114085237475,
        -1.0,
        -0.0463802744399,
        -1.0,
        -0.102758977378,
        -0.0184556459694,
        -0.0469909656387,
        -0.0131278196488,
        -0.00491917843393,
        -0.00235306112886,
        -0.00182733883097,
        -0.000836825026024,
        -0.000678644988293,
        -8.55031373435e-05,
        -0.000243169767576,
        -2.66419515927e-05,
        -6.24762293698e-06,
        5.13593657082e-07,
        1.84901927714e-06,
        -0.00031737792219,
        -0.000293993886106,
        -0.000115433717655,
        -0.000110838468221,
        -3.70887188179e-05,
        1.0,
        -0.17694608476,
        0.999999996945,
        0.999999999254,
        0.999999997046,
        1.0,
        0.0830688120877,
        -0.0895190361875,
        -1.02523032396,
        -0.00611348400664,
        -0.0312498637248,
        -0.00554652405486,
        -0.00142732788567,
        -0.000604457325823,
        -0.000225991041079,
        -2.9496802913e-05,
        3.29758935014e-05,
        4.91612487631e-05,
        5.13168145712e-05,
        2.90051361501e-05,
        1.83054699022e-05,
        9.12319290514e-06,
        4.94803234336e-06,
        3.49801563849e-05,
        7.24183250644e-05,
        4.05580520539e-05,
        5.47769777088e-05,
        2.66510798476e-05,
        0.999999998514,
        0.999999993908,
        0.999991411968,
        -0.0966194109297,
        -0.0302684715524,
        -0.00233567530636,
        -0.85856511543,
        -0.010814837224,
        -0.00200300090444,
        -0.000528049825408,
        -0.000215947815728,
        -9.19620681818e-05,
        -1.37306544407e-05,
        7.05487600619e-06,
        1.75201605507e-05,
        1.77833306882e-05,
        1.0874117181e-05,
        6.88676608644e-06,
        3.60004107583e-06,
        1.99406941486e-06,
        1.25819942068e-05,
        2.35969681199e-05,
        1.51622749341e-05,
        1.92640931581e-05,
        1.04583555361e-05,
        1.00000000011,
        0.999999998529,
        0.999991472425,
        0.0240364426052,
        -0.0653221003927,
        -0.00455763464773,
        -0.0228951629199,
        -0.920115754824,
        -0.00410636540033,
        -0.00105732406048,
        -0.000441252154886,
        -0.000169332604766,
        -2.25591558478e-05,
        2.19639594932e-05,
        3.57354297807e-05,
        3.68589073701e-05,
        2.13577866165e-05,
        1.34524487884e-05,
        6.7613256382e-06,
        3.6801535868e-06,
        2.55438498789e-05,
        5.17061342225e-05,
        2.97174808129e-05,
        3.97172222434e-05,
        1.97593926219e-05,
        0.99999999551,
        0.999999999469,
        0.999991475067,
        -0.0751976101006,
        -0.0353642796985,
        -0.00257811584831,
        -0.0124907187413,
        -0.00225867639915,
        -0.385802417813,
        -0.000592888258191,
        -0.000250390290545,
        -0.000100620000935,
        -1.40886347767e-05,
        1.07075719028e-05,
        2.02335923805e-05,
        2.07606995199e-05,
        1.22823640533e-05,
        7.75017723902e-06,
        3.96218545424e-06,
        2.17289611462e-06,
        1.45180446507e-05,
        2.84694884513e-05,
        1.71243065027e-05,
        2.23909320209e-05,
        1.15607971545e-05,
        0.999999999775,
        0.999999999937,
        0.999991468046,
        -0.13194916422,
        -0.0202640582654,
        -0.00152189885045,
        -0.00719075682101,
        -0.00129749188441,
        -0.000349244818446,
        -0.0379907431985,
        -0.000152602469351,
        -6.32941112085e-05,
        -9.10331621687e-06,
        5.83540637824e-06,
        1.22713028256e-05,
        1.25696040919e-05,
        7.52104744024e-06,
        4.74814212156e-06,
        2.45393045227e-06,
        1.35187848972e-06,
        8.84934832067e-06,
        1.69690741885e-05,
        1.05147904909e-05,
        1.354346661e-05,
        7.162083682e-06,
        0.99999999799,
        0.999999999952,
        0.999991464233,
        -0.15458124323,
        -0.016369145446,
        -0.00118987633699,
        -0.00577645994965,
        -0.00103946897735,
        -0.00027435447601,
        -0.000117570064425,
        -0.035040098392,
        -4.70769067294e-05,
        -6.54843856654e-06,
        5.12334499393e-06,
        9.48200225511e-06,
        9.74531411513e-06,
        5.74487083949e-06,
        3.62291836289e-06,
        1.84909370458e-06,
        1.01321210326e-06,
        6.8116819611e-06,
        1.33938454249e-05,
        8.0149389387e-06,
        1.04989762303e-05,
        5.40005630438e-06,
        1.0,
        1.00000000011,
        0.999991464828,
        -0.181826554337,
        -0.00773660442916,
        -0.000637759916555,
        -0.00279991954135,
        -0.000526094509106,
        -0.000142397819961,
        -5.8954817686e-05,
        -2.6845300471e-05,
        -0.0155979891052,
        -4.24451337607e-06,
        1.25144683585e-06,
        4.77077379415e-06,
        4.79740523711e-06,
        3.03341297629e-06,
        1.92633980222e-06,
        1.03116540271e-06,
        5.76805570176e-07,
        3.44248770298e-06,
        6.11570348211e-06,
        4.24127157321e-06,
        5.20825511636e-06,
        2.98971324255e-06,
        0.999999999766,
        1.0,
        0.999991468427,
        -0.213893585621,
        -0.00191267469074,
        -0.000182621805427,
        -0.000715609222019,
        -0.000142264270553,
        -3.93362568291e-05,
        -1.54346798352e-05,
        -7.97036657367e-06,
        -1.39443324301e-06,
        -0.014591420108,
        -6.27934819616e-08,
        1.25635755753e-06,
        1.23079719847e-06,
        8.41460153166e-07,
        5.3836687303e-07,
        3.01627495459e-07,
        1.71887370583e-07,
        9.09752056689e-07,
        1.42470627362e-06,
        1.17801287476e-06,
        1.34945054589e-06,
        8.67962570941e-07,
        0.999999999157,
        1.00000000009,
        0.999991471404,
        -0.235207554837,
        0.00256284001174,
        0.000112618508477,
        0.000836786241228,
        0.000130213387417,
        3.03444080777e-05,
        1.44268917932e-05,
        2.80761840287e-06,
        -7.4352898275e-08,
        -1.8342077885e-06,
        -0.0135571166077,
        -1.15487532786e-06,
        -1.28104249508e-06,
        -5.68284736934e-07,
        -3.46350275922e-07,
        -1.32456770963e-07,
        -6.17273063595e-08,
        -8.12472181506e-07,
        -2.20153592813e-06,
        -7.82768985084e-07,
        -1.34681198535e-06,
        -4.06252120431e-07,
        0.999999999953,
        1.00000000006,
        0.999991465388,
        -0.329884339231,
        0.0189553650202,
        0.00137045713913,
        0.00669286305438,
        0.00122427411618,
        0.000313694852567,
        0.000125880525603,
        5.01920952536e-05,
        7.03736271975e-06,
        -5.42791307578e-06,
        -1.02465171324e-05,
        -0.00175343128983,
        -1.04801170145e-05,
        -6.21754855244e-06,
        -3.92770847038e-06,
        -2.00462743526e-06,
        -1.09893196805e-06,
        -7.3119152121e-06,
        -1.44083447475e-05,
        -8.64267803146e-06,
        -1.13389346222e-05,
        -5.83591740575e-06,
        0.99999999803,
        1.00000000002,
        0.999991466519,
        -0.280679876149,
        0.00876445804144,
        0.000623649217743,
        0.00308266147779,
        0.000555421230791,
        0.000144046000389,
        6.02393882468e-05,
        2.36018162709e-05,
        3.21940598385e-06,
        -2.80724090422e-06,
        -4.87590403925e-06,
        -5.0161295979e-06,
        -0.0156019493584,
        -2.93497005781e-06,
        -1.85029109631e-06,
        -9.37225561645e-07,
        -5.11904705389e-07,
        -3.49015739357e-06,
        -6.96588994008e-06,
        -4.086711854e-06,
        -5.40845367095e-06,
        -2.7365309231e-06,
        1.00000000005,
        0.999999999996,
        0.999991476704,
        -0.37116377224,
        0.0226371453694,
        0.0017222716625,
        0.00806705554378,
        0.00149014977717,
        0.000390388602124,
        0.000158991619945,
        6.67381564641e-05,
        9.83594884977e-06,
        -5.56347275404e-06,
        -1.290594315e-05,
        -1.31210103816e-05,
        -7.9711526573e-06,
        -0.00309933468365,
        -5.0453758164e-06,
        -2.62372925624e-06,
        -1.44998344401e-06,
        -9.25170764963e-06,
        -1.75535168596e-05,
        -1.11066772308e-05,
        -1.42102415069e-05,
        -7.62626268692e-06,
        0.999999999966,
        0.999999999962,
        0.999991482336,
        -0.452937841844,
        0.0299320260381,
        0.00229266369191,
        0.0106822092456,
        0.00198030485494,
        0.000518459876499,
        0.000209770379772,
        8.8615515805e-05,
        1.31564076895e-05,
        -7.09015331364e-06,
        -1.70422393237e-05,
        -1.73007874509e-05,
        -1.05537915364e-05,
        -6.68339918804e-06,
        -0.00144743891402,
        -3.48433814819e-06,
        -1.92777314159e-06,
        -1.22138675767e-05,
        -2.30555505663e-05,
        -1.47030366236e-05,
        -1.87499352062e-05,
        -1.01214376984e-05,
        0.999999999968,
        0.999999999994,
        0.999991482912,
        -0.458699305462,
        0.0310575733811,
        0.00250104921938,
        0.0111972756805,
        0.00211209338974,
        0.000558444784879,
        0.000222874228833,
        9.8741785538e-05,
        1.53493985874e-05,
        -5.64460372773e-06,
        -1.81305796306e-05,
        -1.82528644593e-05,
        -1.14348699587e-05,
        -7.26054123118e-06,
        -3.85308441942e-06,
        -0.000685607626345,
        -2.14809641224e-06,
        -1.30154771616e-05,
        -2.36287991794e-05,
        -1.59425729978e-05,
        -1.98398514365e-05,
        -1.11616325141e-05,
        0.999999885468,
        0.999999999663,
        0.999991477195,
        -0.500308243747,
        0.034586912311,
        0.00284869046588,
        0.0125291053699,
        0.00238324174708,
        0.000632373684814,
        0.000250344972007,
        0.00011319225746,
        1.7928661028e-05,
        -5.39073241462e-06,
        -2.03828724665e-05,
        -2.04409093232e-05,
        -1.29598946635e-05,
        -8.23874861809e-06,
        -4.40552003684e-06,
        -2.46399078711e-06,
        -0.000331534552337,
        -1.46401458653e-05,
        -2.61098053295e-05,
        -1.80724696001e-05,
        -2.22503021907e-05,
        -1.27456293195e-05,
        0.999999999998,
        1.00000000013,
        0.999991469354,
        -0.255057358615,
        0.004898935102,
        0.000357509446165,
        0.00173122449585,
        0.000314351574385,
        8.20460562242e-05,
        3.41706666113e-05,
        1.37502144192e-05,
        1.93299747395e-06,
        -1.4456754527e-06,
        -2.7664235995e-06,
        -2.83471925647e-06,
        -1.68123185019e-06,
        -1.06132356228e-06,
        -5.43038459453e-07,
        -2.97946394673e-07,
        -1.98233211867e-06,
        -0.0199948911878,
        -3.88292376206e-06,
        -2.34227205129e-06,
        -3.06028972455e-06,
        -1.58325781911e-06,
        0.999999997934,
        1.00000000018,
        0.999991463061,
        -0.265915070838,
        0.0109494792759,
        0.000727809598972,
        0.00380375351042,
        0.000670429386043,
        0.000171267754664,
        7.28151384103e-05,
        2.64527311303e-05,
        3.2749565854e-06,
        -4.24050334437e-06,
        -5.88530178702e-06,
        -6.12196310388e-06,
        -3.44998879439e-06,
        -2.16633908441e-06,
        -1.06593930713e-06,
        -5.74458822576e-07,
        -4.20184634251e-06,
        -8.81010753978e-06,
        -0.0181154560731,
        -4.7973719987e-06,
        -6.57609610604e-06,
        -3.12643842689e-06,
        0.999999998898,
        1.00000000009,
        0.999991480323,
        -0.298584691421,
        0.0119086085792,
        0.00090857059276,
        0.0042453928187,
        0.000783413698017,
        0.000205990752075,
        8.44404628706e-05,
        3.55554773409e-05,
        5.24977204927e-06,
        -2.9213819443e-06,
        -6.84785398148e-06,
        -6.96275526194e-06,
        -4.23284381089e-06,
        -2.67907083942e-06,
        -1.39453806881e-06,
        -7.70961787101e-07,
        -4.91289189242e-06,
        -9.30075557812e-06,
        -5.90056276736e-06,
        -0.00862538138574,
        -7.53826860939e-06,
        -4.05425247414e-06,
        0.999999991567,
        1.00000000005,
        0.999991465824,
        -0.307273256436,
        0.0169967293142,
        0.00121702390577,
        0.0059859046827,
        0.00108229125336,
        0.000280436037401,
        0.000116474602158,
        4.59508800823e-05,
        6.32552004356e-06,
        -5.28843110537e-06,
        -9.43585430136e-06,
        -9.69281476765e-06,
        -5.69520279382e-06,
        -3.59227751889e-06,
        -1.82461425123e-06,
        -9.97861779095e-07,
        -6.75217702128e-06,
        -1.34115358798e-05,
        -7.92872464218e-06,
        -1.04583492389e-05,
        -0.00817647881665,
        -5.3239280971e-06,
        1.00000000005,
        0.999991477393,
        -0.342962949745,
        0.0164952615839,
        0.00131833513319,
        0.00593637540508,
        0.00111389812143,
        0.000295290004362,
        0.000119232590949,
        5.25072309464e-05,
        8.10296186821e-06,
        -3.17441664576e-06,
        -9.68436112814e-06,
        -9.76848126873e-06,
        -6.0902179959e-06,
        -3.8644803685e-06,
        -2.04561117556e-06,
        -1.13912415762e-06,
        -6.95690495924e-06,
        -1.27003776521e-05,
        -8.49465009044e-06,
        -1.06067152501e-05,
        -5.93096638008e-06,
        -0.00392277893649,
        1.00000000012,
        -0.999991470297,
        1.41948852696e-05,
        0.0113461114355,
        0.0173351671776,
        0.0183032565209,
        0.00639641407249,
        0.00283115793623,
        0.0021560555731,
        0.000995202006259,
        0.000806545839047,
        0.000128094091462,
        0.000316339464257,
        5.28755969295e-05,
        2.20293216357e-05,
        9.33717552064e-06,
        3.96800977444e-06,
        0.000414729349764,
        0.000369623683106,
        0.000166985893165,
        0.00015793241772,
        6.96573377498e-05,
        0.999999999961,
        1.0,
        1.0,
        0.00094901330261,
        1.0,
        0.0,
        -0.0401548968716,
        1.00000000876,
        1.0,
        -1.0,
        0.999999604829,
        0.136733138466,
        -1.0000000002,
        -0.999999990522,
        0.999999961298,
        0.155175516957,
        -0.99999992625,
        -0.999999996084,
        0.999999966171,
        0.160091958944,
        -1.00000000002,
        -0.999999999985,
        0.999999966107,
        0.22075432733,
        -1.00000000015,
        -0.999999999992,
        0.999999968363,
        0.300141402559,
        -0.999999997591,
        -1.00000000004,
        0.999999967813,
        0.396256776588,
        -0.999999999938,
        -1.00000000012,
        0.999999965826,
        0.395031259412,
        -1.0000000001,
        -1.0,
        0.999999966404,
        0.477713033625,
        -1.0,
        -1.00000000007,
        0.99999996124,
        0.479803078303,
        -0.999999999694,
        -0.999999999845,
        0.999999924468,
        0.654186783548,
        -0.999999998965,
        -1.00000000011,
        0.999999906299,
        0.566475863275,
        -1.0,
        -1.0,
        0.999999835502,
        0.740742196011,
        -0.999999999403,
        -1.0,
        0.999998989685,
        0.719287660326,
        -0.999999997964,
        -0.999999999945,
        0.999999804755,
        0.91307600331,
        -0.999999999335,
        -1.0,
        0.99999388641,
        1.00001347193,
        -0.999999999133,
        -1.00000000023,
        0.999999914166,
        0.575894015129,
        -0.999999999413,
        -1.00000000002,
        0.999999823721,
        0.570266548157,
        -1.00000000001,
        -0.999999999927,
        0.999999757944,
        0.667025335999,
        -0.999999999447,
        -0.999999999914,
        1.0,
        0.6570704586,
        -0.999999999911,
        0.999999931251,
        0.750031438419,
        -1.0,
        -0.999999967674,
        -1.00000000006,
        0.679488936685,
        -0.0411949730698,
        -1.0,
        -0.115772811385,
        -0.999999999188,
        -1.00000000605,
        -0.999999999999,
        0.999941353411,
        0.12868183527,
        1.0,
        -0.992963637099,
        -1.04502808179,
        -0.0435073808906,
        -0.320422101993,
        -0.0426619916642,
        -0.00646286155617,
        -0.00131462835908,
        0.00141170409479,
        0.00169361909104,
        0.00192593091258,
        0.00073582983957,
        0.00118617697875,
        0.000381029308913,
        0.000203673661282,
        9.59320309214e-05,
        4.75143787504e-05,
        0.00124220234778,
        0.00146059667227,
        0.000774191134065,
        0.000852520526967,
        0.000415733055375,
        -0.999999929602,
        0.999984580693,
        -0.173767080323,
        0.999999999881,
        -0.242472530806,
        -0.0125160245106,
        -0.932038742217,
        -0.0800820119574,
        -0.0110722054202,
        -0.00166445700268,
        -0.000284682760615,
        0.000337693853492,
        0.000449964824628,
        0.000500220963121,
        0.000199111587342,
        0.000321464220936,
        0.000105841912602,
        5.68082783934e-05,
        2.75068639611e-05,
        1.38117260989e-05,
        0.000341024603349,
        0.000383574409523,
        0.000215058051909,
        0.000230137322492,
        0.000117930809806,
        -0.999999999999,
        0.999984647115,
        0.0248540528828,
        0.999999999985,
        -0.679721386519,
        -0.0309764058039,
        -0.220959372849,
        -0.92531800323,
        -0.0294929524533,
        -0.00401342120696,
        -0.000884214735929,
        0.00097452494078,
        0.00118927040062,
        0.00135384793518,
        0.000519854392372,
        0.000838181761072,
        0.000270786946787,
        0.000144928845591,
        6.84997272596e-05,
        3.40123167309e-05,
        0.000878391328055,
        0.00102623476512,
        0.000549235322122,
        0.000603407667912,
        0.000295709152556,
        -0.999999999988,
        0.999984698117,
        -0.151669683365,
        0.999999995669,
        -0.312313918367,
        -0.0147843763559,
        -0.102016306292,
        -0.0137468953234,
        -1.01095681775,
        -0.00189391292448,
        -0.000395532275961,
        0.00044497168491,
        0.000556696044744,
        0.000629567421806,
        0.000244320078974,
        0.000394021258625,
        0.000128031000341,
        6.85874012386e-05,
        3.26524623944e-05,
        1.62685159247e-05,
        0.000414360132468,
        0.000478863941427,
        0.000259776006437,
        0.000283236335125,
        0.000140637849638,
        -0.999999997731,
        0.999984703973,
        -0.261371184488,
        0.999999999973,
        -0.119555436425,
        -0.00581870779467,
        -0.0392043469315,
        -0.00531507987075,
        -0.000725402314141,
        -0.125388989991,
        -0.000148560695647,
        0.000169957670572,
        0.000216330513757,
        0.000243609974381,
        9.523532894e-05,
        0.000153589920618,
        5.01063968316e-05,
        2.6861160175e-05,
        1.28479825596e-05,
        6.41562017067e-06,
        0.000161873732737,
        0.000185752702235,
        0.000101673903085,
        0.000110311837324,
        5.52458366042e-05,
        -0.999999999883,
        0.999984699883,
        -0.317006807548,
        0.999999999672,
        -0.0473444546955,
        -0.00185039877723,
        -0.0150806276467,
        -0.00195245396899,
        -0.000285115142459,
        -6.70902637839e-05,
        -0.101040656769,
        6.82468544349e-05,
        7.63785256774e-05,
        8.8885824542e-05,
        3.28135466365e-05,
        5.29140877637e-05,
        1.67134038575e-05,
        8.90905448804e-06,
        4.09675488704e-06,
        2.00678820839e-06,
        5.47854223082e-05,
        6.65014830923e-05,
        3.38905730627e-05,
        3.82637822567e-05,
        1.78629301078e-05,
        -0.999999999947,
        0.999984703126,
        -0.36160946667,
        0.999999999622,
        0.0652896807301,
        0.00277497473356,
        0.0209736755866,
        0.00278017442628,
        0.000444434483034,
        8.76094758927e-05,
        -9.20465758147e-05,
        -0.0443295978819,
        -0.000108902784552,
        -0.000124648677368,
        -4.71690070398e-05,
        -7.61572713053e-05,
        -2.43922788274e-05,
        -1.30288932846e-05,
        -6.10801451063e-06,
        -3.01994437671e-06,
        -7.95794076644e-05,
        -9.39655751357e-05,
        -4.95397369666e-05,
        -5.48375501552e-05,
        -2.64893563939e-05,
        -1.0,
        0.99998469868,
        -0.430832455535,
        1.0,
        0.156098327461,
        0.00737275034597,
        0.0508609137625,
        0.00690943576669,
        0.00110756669166,
        0.000195044870186,
        -0.00021760994057,
        -0.000274916757259,
        -0.0599325891036,
        -0.000309389160301,
        -0.000120424095431,
        -0.000194501913803,
        -6.32671221369e-05,
        -3.388075453e-05,
        -1.61875955361e-05,
        -8.07634628043e-06,
        -0.000205068001399,
        -0.000235405825671,
        -0.000128578974215,
        -0.000139543525183,
        -6.97638699802e-05,
        -0.999999999933,
        0.999984693357,
        -0.466062701936,
        1.00000000008,
        0.216281134208,
        0.00998161518815,
        0.0702703028962,
        0.00948039632045,
        0.00148363099143,
        0.00027546518423,
        -0.000303407240194,
        -0.000376997316702,
        -0.000426231912685,
        -0.0702980594324,
        -0.00016477982576,
        -0.000266054614126,
        -8.62328766844e-05,
        -4.615607477e-05,
        -2.19488999811e-05,
        -1.09270055216e-05,
        -0.000279835097743,
        -0.000323614366201,
        -0.00017517930566,
        -0.000191094560453,
        -9.47119657366e-05,
        -1.00000000007,
        0.999984705846,
        -0.655005756323,
        0.999999999394,
        0.451342341529,
        0.0218302249756,
        0.147637353033,
        0.020119687787,
        0.0030946418119,
        0.000556630540531,
        -0.00063143895746,
        -0.000807424348966,
        -0.000906555908526,
        -0.000354762159449,
        -0.168438388889,
        -0.000572782922155,
        -0.000186882717429,
        -0.000100145196314,
        -4.799051153e-05,
        -2.39793115542e-05,
        -0.0006046167755,
        -0.000691134126037,
        -0.000379673602962,
        -0.000410845355092,
        -0.000206511243822,
        -1.0,
        0.999984660786,
        -0.557090649726,
        1.00000000007,
        0.314230646101,
        0.0152250302584,
        0.102820772201,
        0.0140131360348,
        0.00214259277674,
        0.000387296204274,
        -0.00043989631617,
        -0.000562892447914,
        -0.000631958166614,
        -0.000247385045187,
        -0.000399394923369,
        -0.242364749154,
        -0.000130337328148,
        -6.98480436797e-05,
        -3.34765940114e-05,
        -1.6728532146e-05,
        -0.00042160546113,
        -0.000481856496316,
        -0.000264780846322,
        -0.00028648419013,
        -0.000144038344315,
        -0.999999999881,
        0.999985137897,
        -0.738980510818,
        0.999999999169,
        0.526868339478,
        0.0260760929304,
        0.172890046766,
        0.0237033074861,
        0.00367875976227,
        0.000637408358685,
        -0.00073427868019,
        -0.000953520951568,
        -0.00106635698233,
        -0.000419965973134,
        -0.000678183673292,
        -0.00022202603728,
        -0.0601244743416,
        -0.0001190299989,
        -5.72809471524e-05,
        -2.86766720784e-05,
        -0.000717356943268,
        -0.000814655014918,
        -0.000451183492543,
        -0.00048602068734,
        -0.000246180508399,
        -0.999999999855,
        0.999967961334,
        -0.8665706251,
        1.00000000105,
        0.636503442014,
        0.0316342318499,
        0.209013078589,
        0.0286853072104,
        0.00445500560477,
        0.000768763278388,
        -0.000884513037538,
        -0.00115144008898,
        -0.00128878175312,
        -0.000508392902135,
        -0.00082105238427,
        -0.000269047453181,
        -0.000144276907008,
        -0.0257942900313,
        -6.94752548212e-05,
        -3.47847821684e-05,
        -0.000868579595674,
        -0.000985181453316,
        -0.000546495425388,
        -0.000588293302229,
        -0.000298305978568,
        -0.999999999968,
        1.0,
        -0.914800592,
        1.00000000019,
        0.675767473877,
        0.0344781289696,
        0.222934828541,
        0.0308065608133,
        0.00480386266375,
        0.000785276189197,
        -0.000936362636334,
        -0.00123913033252,
        -0.00138851108228,
        -0.000549879764982,
        -0.000886848640116,
        -0.000292825406636,
        -0.000156339171956,
        -7.6014319897e-05,
        -0.0114060234291,
        -3.81278705746e-05,
        -0.000944656025592,
        -0.00105969682323,
        -0.000594093732376,
        -0.00063587464126,
        -0.000324876873178,
        -0.999999999982,
        0.9996612627,
        -0.999999871987,
        0.999999996857,
        0.745641115065,
        0.0391784169164,
        0.247101655177,
        0.0344100195416,
        0.0056087993523,
        0.000880572033724,
        -0.000927982797571,
        -0.00130391582628,
        -0.00151728312923,
        -0.00060115520727,
        -0.000982169405591,
        -0.000324920055614,
        -0.000174207907843,
        -8.49342796423e-05,
        -4.23877570834e-05,
        -0.00516386687795,
        -0.00102111777579,
        -0.00113965532257,
        -0.000641796838385,
        -0.000701065531165,
        -0.000361116429625,
        -0.999999999418,
        0.999984683503,
        -0.517525916766,
        0.999999999902,
        0.261552014144,
        0.0128502844317,
        0.0857516796716,
        0.0117263045824,
        0.00179387463356,
        0.000318943207711,
        -0.00036550754037,
        -0.000471993680625,
        -0.000528692781588,
        -0.00020774281889,
        -0.000335409955923,
        -0.000109675899138,
        -5.87949626774e-05,
        -2.82464930344e-05,
        -1.41307993744e-05,
        -0.000354476522459,
        -0.203385324536,
        -0.000403626366703,
        -0.000222826200676,
        -0.000240473292105,
        -0.000121439849478,
        -1.00000000004,
        0.999984666671,
        -0.534688029347,
        0.999999999964,
        0.332663084002,
        0.0156538349886,
        0.108421313,
        0.0146680706217,
        0.00222702892283,
        0.000419198926357,
        -0.000467797576088,
        -0.000587107581956,
        -0.000662432977854,
        -0.000257241156116,
        -0.000415238482437,
        -0.000134927776705,
        -7.22583429106e-05,
        -3.44503269492e-05,
        -1.71728284021e-05,
        -0.000437206885092,
        -0.000503763885445,
        -0.223777231947,
        -0.000274038209521,
        -0.000298174962726,
        -0.000148474069567,
        -1.0000000001,
        0.999984583465,
        -0.607144190398,
        0.999999999283,
        0.369410187761,
        0.0183161714952,
        0.121255398651,
        0.0166288357018,
        0.00257160953455,
        0.000446750851594,
        -0.000514810465987,
        -0.000669358660979,
        -0.000748409692416,
        -0.000294871773053,
        -0.000476148592463,
        -0.000155913223661,
        -8.35976854197e-05,
        -4.02357630991e-05,
        -2.0145140829e-05,
        -0.000503694113883,
        -0.000571850197011,
        -0.000316821022214,
        -0.202799979123,
        -0.000341222655413,
        -0.000172902045658,
        -1.00000000012,
        0.999984576274,
        -0.619266144487,
        1.00000000015,
        0.427754051616,
        0.0205821367594,
        0.139818358145,
        0.0190313009788,
        0.00292910624644,
        0.000529538220154,
        -0.000598739743044,
        -0.000763090625432,
        -0.000857473715147,
        -0.000335090110122,
        -0.000541022099321,
        -0.000176387600874,
        -9.45095374506e-05,
        -4.52507198888e-05,
        -2.26006402598e-05,
        -0.000570852770985,
        -0.000653417261384,
        -0.00035834685486,
        -0.00038813001971,
        -0.197227476207,
        -0.000194777753511,
        0.999984410875,
        -0.696204961678,
        0.448658800529,
        0.0227190375489,
        0.147707564314,
        0.0203687590828,
        0.00317195015224,
        0.0005328886878,
        -0.00062299428735,
        -0.000821820420873,
        -0.000915565048225,
        -0.000362826636514,
        -0.000586003075041,
        -0.000192467669829,
        -0.000103244473454,
        -4.9877630257e-05,
        -2.50141278019e-05,
        -0.000621047141217,
        -0.000700909431095,
        -0.000391156482898,
        -0.000419578806337,
        -0.000214084741707,
        -0.080404054487,
        -0.99999999999,
        -0.999984686537,
        0.999999999926,
        1.42529263637e-05,
        0.0148411831272,
        0.0226010144688,
        0.0711932042892,
        0.0359945560364,
        0.0160157488471,
        0.0119990909158,
        0.00510072233898,
        0.00399935525859,
        0.000446922469826,
        0.00118645824411,
        5.73354972518e-06,
        2.88634518669e-07,
        1.73126312182e-08,
        1.11046215249e-09,
        0.00159851228014,
        0.00141942816092,
        0.000257449666386,
        0.000219362205077,
        1.50801921767e-05,
        0.124489448348,
        1.0,
        -0.000103968131004,
        1.0,
        -0.0586163555029,
        0.999999998099,
        0.99999999749,
        0.999999999805,
        0.998578997039,
        0.223084333302,
        0.0330870129406,
        0.00844037234427,
        0.00321411653589,
        0.0129738164289,
        0.0012506058686,
        -0.000439486393603,
        -0.000648680651165,
        -0.000579661493103,
        -0.000470547588296,
        -0.000421462255563,
        -0.000151092458759,
        -0.000264382763,
        -8.55775230732e-05,
        -4.62814500549e-05,
        -2.48674439986e-05,
        -1.32124851524e-05,
        -0.000306643298508,
        -0.000283574471255,
        -0.000182603767912,
        -0.000172979690713,
        -0.000103906452022,
        -0.999999990907,
        0.999999932101,
        0.998578104042,
        0.167557382197,
        0.0151775160972,
        0.00147947385387,
        0.0402891825983,
        0.00595483690826,
        0.00057385790164,
        -0.000206226812828,
        -0.000307185646821,
        -0.00027062188722,
        -0.000218931728161,
        -0.000195754304444,
        -7.01210484361e-05,
        -0.000122780146457,
        -3.97376653875e-05,
        -2.14918945389e-05,
        -1.15574992342e-05,
        -6.1434353039e-06,
        -0.000142508604778,
        -0.000131570257852,
        -8.48251043497e-05,
        -8.02715029218e-05,
        -4.82832113923e-05,
        -0.999999984209,
        0.999999999977,
        0.998578097238,
        0.217501969721,
        0.0245671107144,
        0.0024222380969,
        0.00965860876039,
        0.0161510610742,
        0.000929426132652,
        -0.000360956530253,
        -0.000553637956824,
        -0.000465654589945,
        -0.000372349111046,
        -0.000330971132588,
        -0.000118218167946,
        -0.000207469784946,
        -6.71173405685e-05,
        -3.63063241427e-05,
        -1.95804257014e-05,
        -1.04235460014e-05,
        -0.000241409933514,
        -0.000221628066082,
        -0.000143470562612,
        -0.000135293173602,
        -8.17508455566e-05,
        -0.999999988381,
        0.999999999866,
        0.998578097,
        0.217891740646,
        0.00777767840395,
        0.000759864142229,
        0.00305276712223,
        0.000294095962389,
        0.0553870182333,
        -0.000107415834592,
        -0.000161020011344,
        -0.000140442824291,
        -0.000113337625853,
        -0.000101213937181,
        -3.62338949167e-05,
        -6.3474905813e-05,
        -2.05415942907e-05,
        -1.11101993895e-05,
        -5.97821591318e-06,
        -3.1787319805e-06,
        -7.37126872348e-05,
        -6.7974983451e-05,
        -4.38614746377e-05,
        -4.14765839456e-05,
        -2.49717935425e-05,
        -1.00000000001,
        0.9999999978,
        0.998578098312,
        0.253338713926,
        -0.00663024667652,
        -0.000640158129715,
        -0.00259688986226,
        -0.000250513071033,
        8.42499794206e-05,
        0.0284932735027,
        0.000122043646166,
        0.000112271184696,
        9.17602298165e-05,
        8.24680298883e-05,
        2.96124993998e-05,
        5.17493581943e-05,
        1.67547007121e-05,
        9.06028809808e-06,
        4.8600837617e-06,
        2.58004747288e-06,
        5.99344648263e-05,
        5.56047095564e-05,
        3.57222212898e-05,
        3.39073642481e-05,
        2.03145850689e-05,
        -0.99999999861,
        0.999999999943,
        0.998578094638,
        0.314289347158,
        -0.0185160838669,
        -0.00179729551178,
        -0.00725917682675,
        -0.000699862643902,
        0.0002443960838,
        0.000359785995635,
        0.046376741862,
        0.000322819207655,
        0.000262307919075,
        0.000235058613008,
        8.42882613083e-05,
        0.000147461764931,
        4.7733254659e-05,
        2.58145311718e-05,
        1.38670168206e-05,
        7.36693922397e-06,
        0.000170995836163,
        0.000158204498748,
        0.00010183998174,
        9.64997301292e-05,
        5.79449775299e-05,
        -0.999999998748,
        1.00000000002,
        0.998578096752,
        0.30333738043,
        -0.0209574304656,
        -0.00203873098713,
        -0.00821951178724,
        -0.00079222520944,
        0.000281043175559,
        0.000416415575165,
        0.000369881700621,
        0.0273696311134,
        0.00029982146066,
        0.000268350137319,
        9.61699390331e-05,
        0.000168326884457,
        5.44822905164e-05,
        2.94655048279e-05,
        1.58375595073e-05,
        8.41636769222e-06,
        0.000195290821138,
        0.000180474676032,
        0.00011627231844,
        0.000110096542204,
        6.61709515723e-05,
        -0.999999999897,
        1.0,
        0.998578095823,
        0.347603574439,
        -0.0329440044693,
        -0.00320722365225,
        -0.0129224261754,
        -0.00124541436147,
        0.000444068692768,
        0.00065933608828,
        0.000583762477429,
        0.000472822816218,
        0.0625998071477,
        0.000423025596628,
        0.00015157370475,
        0.000265340371151,
        8.58802549196e-05,
        4.64469370863e-05,
        2.496979943e-05,
        1.32707312508e-05,
        0.000307895922722,
        0.000284429945689,
        0.000183296511558,
        0.000173520204015,
        0.000104322083775,
        -1.0,
        1.00000000014,
        0.998578085977,
        0.33901971487,
        -0.0355317775496,
        -0.00345644899507,
        -0.0139355347277,
        -0.00134317308195,
        0.000476352641787,
        0.00070572428589,
        0.000626971912791,
        0.000508239058397,
        0.000454900424206,
        0.0736045721906,
        0.000163026905085,
        0.000285344647741,
        9.23576168346e-05,
        4.99494977476e-05,
        2.68473358603e-05,
        1.42670932876e-05,
        0.000331050412644,
        0.000305940602947,
        0.000197102162778,
        0.000186635393603,
        0.000112171109487,
        -0.999999999837,
        0.99999999698,
        0.998578295729,
        0.420186947807,
        -0.0647929003341,
        -0.00629571935701,
        -0.0254064564256,
        -0.00244916764735,
        0.000861518699804,
        0.00127211041929,
        0.00113605629643,
        0.000922071275961,
        0.000825819856433,
        0.000296046413182,
        0.166854587788,
        0.000518043029186,
        0.000167683136131,
        9.0686016551e-05,
        4.87277831875e-05,
        2.58906434862e-05,
        0.000600861839518,
        0.000555618430877,
        0.000357802356321,
        0.000338927554495,
        0.000203602887323,
        -0.999999999359,
        1.0,
        0.998578541005,
        0.376673540493,
        -0.0508320186146,
        -0.00493829497714,
        -0.0199315252584,
        -0.00192141926446,
        0.000675047903049,
        0.000996260549625,
        0.000890413541598,
        0.000722834412174,
        0.000647442265015,
        0.000232110222666,
        0.000406148646719,
        0.293223630967,
        0.000131465342318,
        7.10985481169e-05,
        3.82012354634e-05,
        2.02970401646e-05,
        0.000471060153506,
        0.000435629335154,
        0.00028051476659,
        0.000265732083719,
        0.000159620952849,
        -1.0,
        0.999999997587,
        0.998590345557,
        0.459673857028,
        -0.0795582417068,
        -0.00772961164462,
        -0.0311945520125,
        -0.00300792096082,
        0.00105640410329,
        0.0015593158953,
        0.00139361648805,
        0.00113130596267,
        0.0010133399349,
        0.000363277991106,
        0.000635673800444,
        0.00020575971126,
        1.1919143769,
        0.000111282886044,
        5.97887401771e-05,
        3.17681408599e-05,
        0.000737262718147,
        0.000681812938129,
        0.000439033559445,
        0.000415894238273,
        0.000249825817975,
        -0.999999997751,
        0.999999980783,
        0.998571357691,
        0.37307735923,
        -0.0936134619616,
        -0.00909912443587,
        -0.0367019666917,
        -0.00353887544997,
        0.00124262399359,
        0.00183449675986,
        0.00163948179279,
        0.00133089178233,
        0.00119218405848,
        0.000427340729744,
        0.000747828777506,
        0.000242072822224,
        0.000130919346695,
        1.22105122316,
        7.03429250737e-05,
        3.73760346683e-05,
        0.000867309370126,
        0.000802214351249,
        0.000516509434795,
        0.000489261211564,
        0.000293890750916,
        -0.999999998873,
        0.999999973894,
        0.99898620119,
        0.537448908504,
        -0.109080408955,
        -0.0106286069439,
        -0.0427868139458,
        -0.00415604206331,
        0.00143080871411,
        0.00212190456245,
        0.0019028663027,
        0.00154394297639,
        0.00138334070011,
        0.000496244068727,
        0.000868389825575,
        0.000281504263891,
        0.000152234352333,
        8.17443062917e-05,
        1.21753017034,
        4.34575788279e-05,
        0.00100739396264,
        0.000931654986153,
        0.000599660643913,
        0.00056833095415,
        0.000341196849783,
        -0.99999999981,
        0.999999917045,
        1.0,
        0.575838913983,
        -0.124208654945,
        -0.0120961152287,
        -0.048162464982,
        -0.00477185645746,
        0.00152589653679,
        0.00233046878787,
        0.00209463646083,
        0.00172009671607,
        0.00153976299165,
        0.000554002399826,
        0.000969287695049,
        0.000313848058498,
        0.00017014553411,
        9.12500535083e-05,
        4.84969189114e-05,
        1.20936325915,
        0.0011201431349,
        0.00103691205021,
        0.000668444441943,
        0.000633764852077,
        0.000379738907127,
        -0.999999995099,
        1.0,
        0.998578155572,
        0.402224396988,
        -0.0473793105628,
        -0.00460278811369,
        -0.0185776930709,
        -0.00179091030951,
        0.000629079867717,
        0.000928350777798,
        0.000829817683632,
        0.000673662236528,
        0.000603406975053,
        0.000216325209063,
        0.000378525349885,
        0.000122524288068,
        6.62629746212e-05,
        3.56029249017e-05,
        1.89164318202e-05,
        0.000439019827751,
        0.251604615903,
        0.000406004322858,
        0.000261435809448,
        0.000247660541129,
        0.0001487639116,
        -0.999999999401,
        0.999999998932,
        0.998578293498,
        0.396550387626,
        -0.0473995588161,
        -0.00460482767969,
        -0.0185856556794,
        -0.00179165827588,
        0.000629493240338,
        0.00092904282691,
        0.000830314240054,
        0.000674040192915,
        0.000603735533889,
        0.000216440776904,
        0.000378730627991,
        0.000122590491794,
        6.62987391671e-05,
        3.56223756646e-05,
        1.89268640864e-05,
        0.000439260616556,
        0.000406220985134,
        0.267584897923,
        0.000261577996806,
        0.000247792906591,
        0.000148845303417,
        -0.999999999641,
        0.999999996892,
        0.998577960025,
        0.447259754348,
        -0.0615310346747,
        -0.00597800649576,
        -0.0241268619003,
        -0.00232585583847,
        0.000817304846975,
        0.0012063212727,
        0.00107800730253,
        0.0008750911097,
        0.000783805183723,
        0.000280995017987,
        0.00049169065695,
        0.000159154628766,
        8.60734170153e-05,
        4.6247674396e-05,
        2.45723903961e-05,
        0.000570277462092,
        0.000527377121995,
        0.000339597766942,
        0.48809502004,
        0.000321697623154,
        0.000193241818687,
        -0.999999998809,
        0.999999997955,
        0.998577351249,
        0.439760927319,
        -0.061215006249,
        -0.00594719526808,
        -0.0240027488492,
        -0.00231389466319,
        0.000813085787757,
        0.00120011410588,
        0.00107245545707,
        0.000870586712296,
        0.000779772612265,
        0.000279550213776,
        0.00048916106709,
        0.000158336025708,
        8.56304674557e-05,
        4.60094939876e-05,
        2.44459097388e-05,
        0.000567344814821,
        0.000524664975512,
        0.000337850864573,
        0.000320042333622,
        0.515636797685,
        0.000192247901741,
        -0.999999999676,
        0.998577139694,
        0.47929364336,
        -0.0763352398516,
        -0.00741672127907,
        -0.0299318229338,
        -0.00288600605049,
        0.00101360946244,
        0.00149629727131,
        0.00133716382826,
        0.00108549998734,
        0.000972290324547,
        0.000348575014358,
        0.00060993995533,
        0.000197428660547,
        0.000106775556463,
        5.73697304743e-05,
        3.04820842121e-05,
        0.000707417632984,
        0.000654206096523,
        0.000421266989718,
        0.000399060438189,
        0.000239709567457,
        1.15003776314,
        0.999999999962,
        -0.998578084918,
        -1.23062882765e-06,
        -0.000167152689571,
        -0.001701685198,
        -0.00747484470182,
        -0.00820099005226,
        -0.00462246279683,
        -0.00377377316601,
        -0.0019916210287,
        -0.00165868837734,
        -0.000302576643869,
        -0.000620270662283,
        -9.55398125595e-05,
        -1.74260187186e-05,
        -3.37000859624e-06,
        -6.86126480122e-07,
        -0.000772030495129,
        -0.00072048129801,
        -0.000370933242306,
        -0.000355941090004,
        -0.000161329188057,
        -0.999999997604,
        -1.0,
        1.0,
        -1.0,
        -0.0495009839381,
        0.0362596614382,
        0.0153151927258,
        0.0263314161972,
        0.0168578481056,
        0.0108256045302,
        0.00944174841951,
        0.00574960020368,
        0.00507285783897,
        0.0013695870954,
        0.0026218873926,
        0.000708619772943,
        0.000364466220139,
        0.000183650749474,
        9.3691362204e-05,
        0.00310161966865,
        0.00294050992805,
        0.00165032491378,
        0.00159463319228,
        0.000856969983736,
        -1.0,
        -0.0754875572465,
        -1.0,
        -0.0854592166841,
        -0.015625963727,
        -0.0421510050994,
        -0.0109626558713,
        -0.00383858688983,
        -0.00172608951832,
        -0.00131976265957,
        -0.000557699762501,
        -0.000441940180982,
        -3.75751619567e-05,
        -0.000136013694452,
        -5.75789840448e-06,
        2.38484049611e-06,
        3.7112765327e-06,
        2.87982898065e-06,
        -0.000185058197876,
        -0.00017188391814,
        -5.44682791258e-05,
        -5.28781910825e-05,
        -1.03063874069e-05,
        -0.17814014964,
        0.999999997614,
        -1.00000000201,
        0.999999916829,
        1.0,
        0.149170317301,
        -1.00000004946,
        -1.02211331382,
        -0.0597264476174,
        -0.370447634323,
        -0.0559570718491,
        -0.0109553236119,
        -0.00341753518672,
        0.000275436333926,
        0.00135272215678,
        0.00177856405651,
        0.000857911200315,
        0.00125056757806,
        0.00046887822568,
        0.000264961570911,
        0.000129635218348,
        6.66721702727e-05,
        0.00123539076789,
        0.00156088829948,
        0.000861987935256,
        0.000981586279073,
        0.000491076095282,
        0.999999999936,
        0.999999999755,
        0.999963913514,
        -0.17471069622,
        -0.281537943091,
        -0.0191865059923,
        -0.899424748756,
        -0.10649655652,
        -0.0166752845514,
        -0.00321363687597,
        -0.000944859322127,
        3.65869879743e-05,
        0.000406865284842,
        0.000523937524842,
        0.000265067758655,
        0.000387755459598,
        0.0001484762402,
        8.42297530001e-05,
        4.23716786882e-05,
        2.21846869321e-05,
        0.000390039608064,
        0.000466586027029,
        0.000273902167141,
        0.000301583847377,
        0.000159967378263,
        0.999999996771,
        0.999999998655,
        0.999963841388,
        0.0388741487026,
        -0.700883337373,
        -0.0430727796828,
        -0.261017447258,
        -0.894723240415,
        -0.0396256557972,
        -0.00704271173831,
        -0.00231500929483,
        0.000191029911686,
        0.000964353665693,
        0.00127539820869,
        0.000616943743239,
        0.000901246110101,
        0.000338273537866,
        0.000191389644689,
        9.38929874512e-05,
        4.86034780221e-05,
        0.000893707979045,
        0.00111652992595,
        0.000622230777999,
        0.000705384624606,
        0.000356534946261,
        0.99999999982,
        0.999999994575,
        0.999963837179,
        -0.146346026261,
        -0.347962635746,
        -0.0221481894972,
        -0.130266900867,
        -0.0199711390107,
        -0.991936499765,
        -0.00359656004967,
        -0.0011514507672,
        7.92715468938e-05,
        0.000486760185161,
        0.000638401010516,
        0.000313449456998,
        0.000458027549525,
        0.000173101930484,
        9.80353566492e-05,
        4.85057514818e-05,
        2.52080620802e-05,
        0.000456322502231,
        0.000562068468161,
        0.000318661738438,
        0.000357799330236,
        0.000183794420496,
        0.999999999498,
        0.99999999999,
        0.999963842622,
        -0.258076643193,
        -0.164695842969,
        -0.0107422905617,
        -0.061907773853,
        -0.00954596922803,
        -0.00168152347617,
        -0.210116402619,
        -0.000541318179303,
        3.28762064189e-05,
        0.000233220100993,
        0.000304456518364,
        0.000150965302375,
        0.000220558646905,
        8.37381685412e-05,
        4.74605426103e-05,
        2.36009794846e-05,
        1.22943327616e-05,
        0.00022030747723,
        0.000269126612129,
        0.00015417488187,
        0.000172148620934,
        8.92830176766e-05,
        0.999999999957,
        1.00000000007,
        0.999963842526,
        -0.310967077535,
        -0.0968895838655,
        -0.00578340899647,
        -0.0358983679488,
        -0.00542074010557,
        -0.00102489652608,
        -0.000325985740491,
        -0.172347007601,
        2.90670188767e-05,
        0.000131331608908,
        0.000174375541839,
        8.34315033012e-05,
        0.000121960750468,
        4.55309850709e-05,
        2.57345534183e-05,
        1.2557373012e-05,
        6.48307005569e-06,
        0.000120652091338,
        0.000151942049139,
        8.37757083802e-05,
        9.54969639818e-05,
        4.778811152e-05,
        1.0,
        0.9999999996,
        0.999963843474,
        -0.358904018922,
        0.0107403902064,
        0.000281478125452,
        0.00364595002499,
        0.000460959067281,
        7.91382421127e-05,
        3.68771057015e-05,
        -1.04390323366e-05,
        -0.0737618190411,
        -1.07217085437e-05,
        -1.66657082406e-05,
        -5.81704772244e-06,
        -8.46272408274e-06,
        -2.60179655031e-06,
        -1.4238278117e-06,
        -5.04155604924e-07,
        -2.13889817433e-07,
        -7.39956939597e-06,
        -1.3016929152e-05,
        -4.68189605005e-06,
        -6.92977841664e-06,
        -2.10892530673e-06,
        0.999999997874,
        1.0,
        0.999963843492,
        -0.42718767979,
        0.0947797989013,
        0.00600378725889,
        0.0354114842104,
        0.00544648834777,
        0.00110158232574,
        0.00032373017664,
        -2.07331343399e-05,
        -0.000132009131999,
        -0.0792108887141,
        -0.000172483524236,
        -8.46997854206e-05,
        -0.000123954576837,
        -4.6826503025e-05,
        -2.6507519578e-05,
        -1.31387810281e-05,
        -6.83210511552e-06,
        -0.000123724075259,
        -0.000151738249613,
        -8.63342044423e-05,
        -9.66660669618e-05,
        -4.98341599974e-05,
        0.999999999993,
        1.00000000009,
        0.999963841629,
        -0.465002891763,
        0.151350958346,
        0.0093081551217,
        0.0562911950784,
        0.00858716782406,
        0.00172265664385,
        0.000516829402313,
        -3.88273744327e-05,
        -0.000207837815367,
        -0.00027344698792,
        -0.0816818710642,
        -0.000132619534925,
        -0.000194040902196,
        -7.28871593758e-05,
        -4.12261280634e-05,
        -2.02922638101e-05,
        -1.05179726474e-05,
        -0.000192937608072,
        -0.000239409155095,
        -0.000134296654735,
        -0.000151560268036,
        -7.71020663851e-05,
        0.999999998918,
        0.999999999132,
        0.999963844903,
        -0.655579549382,
        0.368472759235,
        0.0240428343286,
        0.13838700894,
        0.0214167674846,
        0.00414341984895,
        0.0012409972246,
        -6.89222920732e-05,
        -0.000521105604348,
        -0.000677673270035,
        -0.000336635159135,
        -0.0431440771006,
        -0.000492401853972,
        -0.00018702661697,
        -0.000105974688318,
        -5.28169051093e-05,
        -2.75368330485e-05,
        -0.000492794275451,
        -0.000599070527113,
        -0.000344776437503,
        -0.000383748382023,
        -0.000199917572116,
        1.0,
        1.0,
        0.999963840554,
        -0.556812419586,
        0.241598376923,
        0.0157543159103,
        0.0907155310545,
        0.0140442323607,
        0.00275366011611,
        0.000816643382018,
        -4.494880328e-05,
        -0.000341499178091,
        -0.000443914857117,
        -0.000220510264199,
        -0.000322601748056,
        -0.165380435795,
        -0.000122523288252,
        -6.94211958259e-05,
        -3.46053198746e-05,
        -1.80429298795e-05,
        -0.00032292554949,
        -0.0003923827086,
        -0.000225907360555,
        -0.000251367814198,
        -0.000131001626428,
        0.999999997632,
        1.00000000018,
        0.99996391373,
        -0.739344193708,
        0.436122395991,
        0.0291164788889,
        0.164361539099,
        0.0256252313705,
        0.00510085781775,
        0.00147794636462,
        -6.67812754994e-05,
        -0.000623528737076,
        -0.000805746601558,
        -0.000404312258636,
        -0.000591666960235,
        -0.00022571274815,
        -0.11674977765,
        -0.000127965996057,
        -6.41387344891e-05,
        -3.3524280554e-05,
        -0.000594143285288,
        -0.000714976929738,
        -0.000416424837512,
        -0.000460388489911,
        -0.000242498954783,
        0.999999972274,
        1.00000000039,
        0.999963709464,
        -0.878849369577,
        0.536354762914,
        0.0359290088478,
        0.20224342502,
        0.0315635769209,
        0.00629892243447,
        0.00181825184863,
        -7.96890405205e-05,
        -0.00076819638056,
        -0.000991831778004,
        -0.000498376359647,
        -0.000729358073935,
        -0.000278409485089,
        -0.000157855228664,
        -0.0503463200523,
        -7.91818304449e-05,
        -4.14015406329e-05,
        -0.000732765466351,
        -0.000880526712297,
        -0.000513707526382,
        -0.000567402441788,
        -0.000299325775571,
        0.999999987271,
        1.00000000063,
        0.99996367517,
        -0.9155941832,
        0.568853884505,
        0.0392598891068,
        0.215563326227,
        0.0339266111076,
        0.00680095880879,
        0.00192731514551,
        -6.10995356475e-05,
        -0.000826926052105,
        -0.00106024884347,
        -0.000539529622867,
        -0.000789722199796,
        -0.000303089927566,
        -0.000171988952093,
        -8.68219903606e-05,
        -0.0226035907755,
        -4.5528852115e-05,
        -0.00079628927072,
        -0.000945988079371,
        -0.000559550990603,
        -0.000613438752235,
        -0.000327671698566,
        0.999999999999,
        1.00000000086,
        0.999963841622,
        -0.999979926828,
        0.627073919619,
        0.0438795150807,
        0.238182697188,
        0.0376337013908,
        0.00755567132577,
        0.00212388568215,
        -5.51162427545e-05,
        -0.00091761769499,
        -0.00117315128554,
        -0.000600436918057,
        -0.000878996063908,
        -0.000338156080091,
        -0.00019196860267,
        -9.71842830555e-05,
        -5.10312713355e-05,
        -0.0104599228423,
        -0.000887760447289,
        -0.00104903162643,
        -0.000624481920858,
        -0.000682304917083,
        -0.000366520913908,
        0.999999998745,
        0.999999999403,
        0.999963844945,
        -0.513316781961,
        0.192192074747,
        0.0127363202305,
        0.0723497657331,
        0.0112526767116,
        0.00221874082983,
        0.000649970466331,
        -3.15655982289e-05,
        -0.000273811229904,
        -0.000354555037361,
        -0.000177330454382,
        -0.000259464414108,
        -9.88434609191e-05,
        -5.60283158882e-05,
        -2.80313198947e-05,
        -1.46396005784e-05,
        -0.000260263205021,
        -0.177257948118,
        -0.000314229361702,
        -0.000182310378091,
        -0.00020199731861,
        -0.000106019607592,
        0.999999999999,
        0.999999999967,
        0.99996385265,
        -0.532249957352,
        0.261442598506,
        0.0164774241736,
        0.0976413523808,
        0.0149752925877,
        0.00292032918709,
        0.000884298630797,
        -6.01448973272e-05,
        -0.000363477263978,
        -0.000476216430755,
        -0.000233195384613,
        -0.000341094109116,
        -0.000128710512476,
        -7.28579619488e-05,
        -3.60375051456e-05,
        -1.87227813581e-05,
        -0.0003399732427,
        -0.000418596398751,
        -0.17542693616,
        -0.000237157908185,
        -0.000266235529205,
        -0.000136697348259,
        0.999999999999,
        0.999999999221,
        0.999963851246,
        -0.602704858469,
        0.290992590662,
        0.0194951594183,
        0.109737776295,
        0.0171195864632,
        0.00338000754527,
        0.000983757750806,
        -4.35579757618e-05,
        -0.000416827863252,
        -0.000538383508547,
        -0.00027050764623,
        -0.000395821309991,
        -0.000151093752279,
        -8.56713992174e-05,
        -4.29640535249e-05,
        -2.24626307637e-05,
        -0.000397574928197,
        -0.000478009401975,
        -0.000278741072414,
        -0.161596425013,
        -0.000307985977254,
        -0.000162397889911,
        0.999999999998,
        0.999999999624,
        0.999963854259,
        -0.616708251824,
        0.348014023872,
        0.0225401186444,
        0.130529196585,
        0.0201714551469,
        0.00395727334996,
        0.00117702770194,
        -6.7757665854e-05,
        -0.000490264520931,
        -0.000638267474286,
        -0.000316156917717,
        -0.000462524533041,
        -0.000175441176504,
        -9.93852760422e-05,
        -4.94683977745e-05,
        -2.57748241504e-05,
        -0.000462609832481,
        -0.000563541831055,
        -0.000323442216464,
        -0.000360505350458,
        -0.162777084287,
        -0.00018734261273,
        0.99999999484,
        0.999963869547,
        -0.6920979295,
        0.362025718331,
        0.0248032223447,
        0.137022604558,
        0.0215177384866,
        0.0042942533536,
        0.00122566968465,
        -4.27134699356e-05,
        -0.000524351096068,
        -0.000673527600316,
        -0.000341663873544,
        -0.000500047654141,
        -0.000191672790487,
        -0.000108742044835,
        -5.48064022083e-05,
        -2.87181082631e-05,
        -0.000503717299634,
        -0.000600232652118,
        -0.000353781963958,
        -0.000388604263572,
        -0.000206914941037,
        -0.152907257352,
        0.99999999919,
        -0.999963845442,
        1.34370512207e-05,
        0.0128394599334,
        0.0205229482311,
        0.0621895145872,
        0.051171790252,
        0.0225122335505,
        0.0171282462031,
        0.00748571971678,
        0.00593766560828,
        0.0007789232369,
        0.00197575537929,
        0.000199500473395,
        3.1273584924e-05,
        5.30299178268e-06,
        9.26242919575e-07,
        0.00265196280633,
        0.00235806855104,
        0.00102125561586,
        0.000966510552971,
        0.000359077782911,
        0.999999999187,
        0.129809120735,
        1.0,
        -5.12008551685e-05,
        1.0,
        1.0,
        -0.0656252039773,
        0.999999998494,
        1.0,
        -1.00000000001,
        1.0,
        0.245710557114,
        0.999999998533,
        0.310001176059,
        0.478349945345,
        0.0509668205487,
        0.134877038215,
        0.0280687697025,
        0.00356411665942,
        -0.00296370885318,
        -0.00333795216005,
        -0.00448258449918,
        -0.00435234386204,
        -0.00267771184077,
        -0.00370736686649,
        -0.00185676600776,
        -0.00121895224619,
        -0.00078913829357,
        -0.000501160516238,
        -0.00401134786089,
        -0.00376836657165,
        -0.00302795611209,
        -0.00288847439242,
        -0.00212476065785,
        -1.00000000021,
        0.999985050989,
        0.224754089802,
        1.00000000091,
        0.184599921181,
        0.0303056973576,
        0.795462467799,
        0.0802607608571,
        0.0167127809433,
        0.00213335831807,
        -0.00180810996749,
        -0.0020772910315,
        -0.00277340697364,
        -0.0026411453433,
        -0.00159311160684,
        -0.00221857633573,
        -0.00110380462029,
        -0.000723726244769,
        -0.000468741285547,
        -0.000297541636615,
        -0.00241281140746,
        -0.0022485310273,
        -0.00180794670707,
        -0.00171854071471,
        -0.00126602731494,
        -0.999999998486,
        0.999984847691,
        0.258683136375,
        0.999999999603,
        0.243038611111,
        0.0398064514783,
        0.105584238599,
        0.771295740807,
        0.0219957043799,
        0.00282430806189,
        -0.00245156183626,
        -0.00288098408912,
        -0.00381953508685,
        -0.00355675864308,
        -0.00209494654956,
        -0.0029380969427,
        -0.00145016723767,
        -0.000949202761061,
        -0.000615087256528,
        -0.000390272622192,
        -0.00321484874666,
        -0.00296772354479,
        -0.00238772878812,
        -0.00226023190344,
        -0.00166818499473,
        -0.999999999656,
        0.999984848013,
        0.299270455718,
        1.0,
        0.131977848623,
        0.0216774111218,
        0.0573910326585,
        0.0119491428629,
        0.912337547155,
        0.00152312991176,
        -0.00128355390688,
        -0.00146648434455,
        -0.00196129678766,
        -0.00187806723431,
        -0.00113924876155,
        -0.00158388894285,
        -0.000789499087511,
        -0.00051785710687,
        -0.000335363710655,
        -0.00021289622666,
        -0.00172007462601,
        -0.00160655706712,
        -0.00129156079191,
        -0.00122889168469,
        -0.000904900986932,
        -0.999999999013,
        0.999984847925,
        0.372352371762,
        0.999999998934,
        0.0340141273822,
        0.00558952815037,
        0.0147935761603,
        0.00307986595267,
        0.000392130627984,
        0.261171494924,
        -0.000328860771976,
        -0.000373939481322,
        -0.000500864174256,
        -0.000481848966954,
        -0.000293688138217,
        -0.000407744548048,
        -0.000203562650224,
        -0.000133567855248,
        -8.64900624925e-05,
        -5.49102606342e-05,
        -0.000442266659391,
        -0.000413854908689,
        -0.000332670993419,
        -0.000316787174525,
        -0.000233183484227,
        -0.999999999184,
        0.999984851396,
        0.467197193351,
        0.999999999132,
        -0.0476129044341,
        -0.00783688742429,
        -0.0207194869159,
        -0.00431216429944,
        -0.000546706014049,
        0.000450408862598,
        0.41115484079,
        0.000503063797223,
        0.000677641351626,
        0.000663390336646,
        0.00041143936913,
        0.000568345719603,
        0.000285362897817,
        0.000187467765518,
        0.000121348786825,
        7.70634865226e-05,
        0.000613737256325,
        0.000578267508223,
        0.000464618210072,
        0.000443749811167,
        0.000326204124948,
        -0.999999996499,
        0.99998484892,
        0.458423505267,
        0.999999996474,
        -0.0648951820715,
        -0.0106915358334,
        -0.0282493939917,
        -0.0058778388302,
        -0.000743132268495,
        0.000605080353735,
        0.000667685913459,
        0.225326931981,
        0.000902865832248,
        0.000894347886323,
        0.000561032020879,
        0.000772452571084,
        0.000389274887066,
        0.000255931032623,
        0.000165625905032,
        0.000105201319084,
        0.000831728434608,
        0.000787190437799,
        0.000632278990792,
        0.000605047862039,
        0.00044438502189,
        -0.99999999729,
        0.999984846411,
        0.538484822356,
        0.999999997268,
        -0.147595302164,
        -0.0243174730145,
        -0.0642501952394,
        -0.0133687088862,
        -0.00169022738562,
        0.00137624999237,
        0.00151861358799,
        0.00205353991744,
        0.40122259222,
        0.00203414083128,
        0.00127603327256,
        0.0017568995822,
        0.000885384800943,
        0.000582101487126,
        0.000376708579199,
        0.000239275906923,
        0.00189172594078,
        0.00179040888353,
        0.00143808640491,
        0.00137614301985,
        0.00101073315661,
        -0.999999992644,
        0.999984841983,
        0.533381556341,
        0.99999999248,
        -0.164771938163,
        -0.0271421690077,
        -0.0717226486946,
        -0.0149240438655,
        -0.00188779234056,
        0.00154036767246,
        0.00170348202954,
        0.00230188797986,
        0.00227529240532,
        0.45063771643,
        0.00142439054521,
        0.0019623142943,
        0.000988249643284,
        0.000649640227472,
        0.000420433449338,
        0.000267039886699,
        0.00211400143903,
        0.00199917785059,
        0.00160585631259,
        0.00153615982156,
        0.00112843259254,
        -0.999999923821,
        0.999984854933,
        0.690516896011,
        0.999999923783,
        -0.365564990239,
        -0.0602302654685,
        -0.159135680623,
        -0.0331119475445,
        -0.00418644158051,
        0.00340891193421,
        0.00376164137076,
        0.0050866244782,
        0.0050383840249,
        0.00316051579923,
        0.308275494887,
        0.0043515822168,
        0.00219294566184,
        0.00144176249335,
        0.000933042102705,
        0.000592644831827,
        0.00468557211465,
        0.00443455124389,
        0.00356191922945,
        0.00340846963523,
        0.00250342398966,
        -0.999999999414,
        0.999984885831,
        0.609704393907,
        0.999999999429,
        -0.26833711659,
        -0.0441576074044,
        -0.116762543163,
        -0.0243013377798,
        -0.00308238879085,
        0.00254453591485,
        0.00284789748429,
        0.00383365265304,
        0.00374560919964,
        0.00231851685477,
        0.00320450747899,
        1.00695911296,
        0.00160793800997,
        0.00105618482814,
        0.000683698697165,
        0.000434173722448,
        0.00346215238628,
        0.00325957876751,
        0.00261907703785,
        0.00250062284231,
        0.00183849036149,
        -0.999999917193,
        0.999984935678,
        0.768242527453,
        0.999999917196,
        -0.46313417575,
        -0.0761997051922,
        -0.201512490088,
        -0.0419418133635,
        -0.00532259747666,
        0.00440326363522,
        0.00493886607992,
        0.00664383943801,
        0.00647758929708,
        0.00400127604476,
        0.00553363559273,
        0.0027747547309,
        1.0072512809,
        0.0018223573341,
        0.00117971538584,
        0.00074913552632,
        0.0059817163104,
        0.00562709295548,
        0.0045216385596,
        0.00431560885162,
        0.00317339796112,
        -0.999999867873,
        0.999984718143,
        0.705303568164,
        0.999999867871,
        -0.558114105288,
        -0.091817973279,
        -0.242830759854,
        -0.0505429470803,
        -0.00641590636645,
        0.00531358984949,
        0.00596682061267,
        0.00802376711256,
        0.0078142445964,
        0.00482162361642,
        0.00667027756897,
        0.00334349748843,
        0.00219572039464,
        1.00169099475,
        0.0014214494027,
        0.000902621406237,
        0.00721242919535,
        0.00678186950786,
        0.00544973360387,
        0.00520043012418,
        0.00382435880899,
        -0.999999449848,
        0.999985057332,
        0.921754447005,
        0.999999449843,
        -0.663590619154,
        -0.109163358079,
        -0.288716495,
        -0.0600949951997,
        -0.00763035578643,
        0.00632451046802,
        0.00710827519448,
        0.00955617613412,
        0.0092986046482,
        0.00573263664042,
        0.00793251574493,
        0.0039751034579,
        0.00261034929878,
        0.00168990572917,
        0.996571681803,
        0.00107307101462,
        0.00857914406199,
        0.00806426500339,
        0.00648038413993,
        0.00618300515707,
        0.00454726064104,
        -0.999998737488,
        0.999983587799,
        0.999998572766,
        0.999998737484,
        -0.762428052751,
        -0.125415894609,
        -0.331711070377,
        -0.0690450500727,
        -0.00876811165331,
        0.00727224056214,
        0.00817877937295,
        0.01099299865,
        0.0106899318889,
        0.00658626217014,
        0.00911544237273,
        0.00456693194062,
        0.00299886980651,
        0.0019414399054,
        0.00123278054033,
        0.991817937438,
        0.00986009631904,
        0.0092659123812,
        0.00744623072017,
        0.00710372359511,
        0.00522464442664,
        -0.999999978578,
        0.999984887098,
        0.631628635744,
        0.999999978535,
        -0.244918248811,
        -0.0403041768594,
        -0.106572546581,
        -0.0221805438417,
        -0.00281335862019,
        0.00232230638014,
        0.002599011528,
        0.00349869533502,
        0.00341853796679,
        0.00211618158394,
        0.00292480562302,
        0.00146761874146,
        0.000964018809308,
        0.000624036793229,
        0.000396286340088,
        0.00315991590357,
        1.00277703104,
        0.0029750929847,
        0.00239048983833,
        0.00228239545602,
        0.00167803919425,
        -0.99999999445,
        0.999984876778,
        0.623525230687,
        0.999999994457,
        -0.245663850617,
        -0.040425757179,
        -0.106896060346,
        -0.0222477379073,
        -0.00282191612314,
        0.00232960911239,
        0.00260747526475,
        0.00350995002238,
        0.00342921682152,
        0.00212258493853,
        0.00293373561542,
        0.00147205196911,
        0.000966924277761,
        0.000625917681101,
        0.000397480044716,
        0.00316963765603,
        0.00298414458075,
        1.00342518255,
        0.0023977600249,
        0.00228930420124,
        0.00168312545217,
        -0.999999966291,
        0.999984878998,
        0.718785995003,
        0.999999966277,
        -0.341031109513,
        -0.0561180619425,
        -0.148392194285,
        -0.0308847226464,
        -0.0039179258716,
        0.00323598046331,
        0.0036236869837,
        0.00487716600148,
        0.00476267039017,
        0.00294656995217,
        0.00407316537954,
        0.00204346860435,
        0.00134221893721,
        0.000868866212283,
        0.000551757032333,
        0.00440122637853,
        0.00414286490381,
        0.00332885085024,
        1.01138367967,
        0.00317801575325,
        0.00233661114243,
        -0.999999945624,
        0.999984861468,
        0.704020619991,
        0.999999945632,
        -0.339455743568,
        -0.055857155779,
        -0.147705295384,
        -0.0307417714933,
        -0.00389999749516,
        0.00322181298925,
        0.00360863865128,
        0.00485657254583,
        0.0047415679853,
        0.00293290015077,
        0.00405451107598,
        0.00203397079775,
        0.00133596103835,
        0.000864818750473,
        0.000549184782737,
        0.00438129865044,
        0.0041237766925,
        0.00331352497397,
        0.0031632780171,
        1.01155611237,
        0.0023258071078,
        0.999984913735,
        0.794302606368,
        -0.44130437477,
        -0.0726111508925,
        -0.192017376083,
        -0.0399652694648,
        -0.00507130727316,
        0.00419371106574,
        0.00470191993584,
        0.00632590653146,
        0.00617001867682,
        0.00381276861445,
        0.00527234903721,
        0.00264407041709,
        0.00173657326422,
        0.00112417440496,
        0.000713871421348,
        0.00569870078665,
        0.00536168113678,
        0.00430832391396,
        0.00411227994529,
        0.00302379669587,
        1.01056275357,
        -0.999999999982,
        -0.999984863718,
        1.0,
        -4.14088128213e-06,
        -2.63545363356e-05,
        -0.0128136440405,
        -0.0919355683632,
        -0.0842484087775,
        -0.0491406063171,
        -0.0410865855349,
        -0.0240876356793,
        -0.0208199286094,
        -0.00564152777334,
        -0.00871441151946,
        -0.00264174348595,
        -0.0014447325757,
        -0.000788267023456,
        -0.000436318636937,
        -0.0101136680122,
        -0.00957098382588,
        -0.00575593642315,
        -0.00552279770195,
        -0.00316284918894,
        -1.0,
        0.152332153459,
        -1.0,
        -0.0513702993645,
        0.0314512092404,
        0.00561482718453,
        0.023826559779,
        0.0189867056035,
        0.0140107300061,
        0.0129047973786,
        0.00905804618791,
        0.00836509257292,
        0.00321684055049,
        0.00515855225294,
        0.00196495402619,
        0.0011993476231,
        0.000715403014736,
        0.000430850565032,
        0.00580924204592,
        0.00558652774981,
        0.0036656134646,
        0.00356765875819,
        0.00225437696658,
        -1.0,
        -0.0114567432969,
        -1.0,
        -0.0945470122072,
        -0.022547760505,
        -0.048007738635,
        -0.0161620088152,
        -0.00689769655102,
        -0.00358123148102,
        -0.00289165345745,
        -0.00143907581539,
        -0.00120341894186,
        -0.000176723157059,
        -0.000470806558531,
        -5.25672319268e-05,
        -6.32333307605e-06,
        1.02204287636e-05,
        1.26859322364e-05,
        -0.000598506362715,
        -0.000561348821791,
        -0.000232784448852,
        -0.000225476471784,
        -7.39651961037e-05,
        -0.114509011448,
        -0.999999978942,
        1.00000001299,
        -0.999999998058,
        0.999985706355,
        0.0833261825243,
        -0.999999998622,
        -0.999999999369,
        0.999999710779,
        -0.0903457586847,
        -0.999999997044,
        -0.999999996661,
        0.999999979,
        0.0259009529798,
        -0.99999999949,
        -1.00000000002,
        0.999999997971,
        -0.0704287731063,
        -0.999999999583,
        -0.99999999757,
        0.99999998965,
        -0.125900003896,
        -0.999999999727,
        -1.0,
        0.999999994348,
        -0.14847854583,
        -0.999999999909,
        -1.00000000012,
        0.9999999911,
        -0.174819896981,
        -0.999999999668,
        -0.999999999047,
        0.999999993051,
        -0.206408068669,
        -0.999999999727,
        -1.00000000003,
        0.999999987759,
        -0.226981086637,
        -0.999999999586,
        -0.999999998058,
        0.999999989923,
        -0.320049008781,
        -0.999999999119,
        -0.999999999954,
        0.99999998332,
        -0.271516584067,
        -1.00000000034,
        -1.00000000003,
        0.999999994346,
        -0.360197086785,
        -0.999999999782,
        -1.00000000003,
        0.999999994972,
        -0.439115754382,
        -0.999999999837,
        -0.999999885031,
        0.999999987391,
        -0.445997698662,
        -0.999999999231,
        -1.0,
        0.999999981706,
        -0.486819971276,
        -0.999999998553,
        -0.99999999801,
        0.999999993883,
        -0.246826185247,
        -0.999999999308,
        -0.999999998938,
        0.999999999833,
        -0.257302298697,
        -1.00000000021,
        -0.999999991754,
        0.999999988634,
        -0.289546940215,
        -0.999999999953,
        -0.999999999947,
        0.999999984166,
        -0.297888451749,
        -0.999999999391,
        0.99999998549,
        -0.333045669256,
        -1.00000000003,
        -0.999999996336,
        -0.999999998856,
        0.999600944709,
        0.000734034199174,
        -1.0,
        -0.0380309041124,
        -1.0,
        1.00000017589,
        1.0,
        1.0,
        0.240341411468,
        0.999999998739,
        0.192164392846,
        0.321518623954,
        0.0278206134974,
        0.0798304206337,
        0.0143974094464,
        0.000673528270681,
        -0.00259355841667,
        -0.00263788298104,
        -0.00293776966224,
        -0.00278855503226,
        -0.00149577096423,
        -0.00218687796255,
        -0.000992117393023,
        -0.000625536219532,
        -0.000389968280571,
        -0.000238963682786,
        -0.00240676633655,
        -0.00225153112709,
        -0.00171644279695,
        -0.00163396780038,
        -0.00114972737211,
        0.999999999894,
        0.999996185046,
        0.218380292468,
        0.999999996403,
        0.108429610528,
        0.0156679963163,
        0.919960588155,
        0.045026798958,
        0.00812780218678,
        0.000382824739468,
        -0.001514582156,
        -0.00160592840827,
        -0.00168655883226,
        -0.00157586500195,
        -0.000837502680287,
        -0.00122714986362,
        -0.000555291533637,
        -0.000349923987238,
        -0.000218111821627,
        -0.000133615438015,
        -0.00135376514098,
        -0.00126209840705,
        -0.000962267509039,
        -0.000914973150186,
        -0.000644003771487,
        1.0,
        0.999996200186,
        0.252838730681,
        0.999999999863,
        0.149114268418,
        0.0215161806751,
        0.0618999847623,
        0.639590895252,
        0.0111796763405,
        0.000528664414248,
        -0.0021266647297,
        -0.00230987002501,
        -0.00234409657297,
        -0.00216897300254,
        -0.00114597131914,
        -0.00168146874807,
        -0.000759624746907,
        -0.000478521246077,
        -0.000298239024688,
        -0.000182665982857,
        -0.00185774821382,
        -0.00172822198559,
        -0.00131770073295,
        -0.0012520589052,
        -0.000881411091187,
        0.999999999596,
        0.999996201232,
        0.293268552197,
        1.00000000007,
        0.074134594474,
        0.0107511543553,
        0.0308124326081,
        0.00555285438767,
        0.980826370206,
        0.000258521233858,
        -0.000974881512009,
        -0.00095806281449,
        -0.00111881253824,
        -0.00107474756219,
        -0.000580447832835,
        -0.000847283732639,
        -0.000385109742596,
        -0.000242909577724,
        -0.00015144472407,
        -9.28230260211e-05,
        -0.000930851569988,
        -0.000873005823606,
        -0.000665510979891,
        -0.000634030966769,
        -0.000446035313054,
        0.9999999996,
        0.999996199295,
        0.366793719091,
        0.999999999631,
        0.0072333912628,
        0.00104767333466,
        0.00300548731238,
        0.00054201001682,
        2.53545627187e-05,
        0.178076916234,
        -9.74543027865e-05,
        -9.88420364411e-05,
        -0.000110487207407,
        -0.000104975695156,
        -5.63407529e-05,
        -8.23622934199e-05,
        -3.73711912738e-05,
        -2.35635081164e-05,
        -1.46897094024e-05,
        -9.00180401696e-06,
        -9.06321541738e-05,
        -8.48010929908e-05,
        -6.46510393711e-05,
        -6.15456864403e-05,
        -4.33063314318e-05,
        0.999999999326,
        0.999996199249,
        0.462583029039,
        0.999999999312,
        -0.0477797249261,
        -0.00692208537735,
        -0.0198537056933,
        -0.00357961911471,
        -0.000167214119037,
        0.000639465444997,
        0.29238684339,
        0.000643192343532,
        0.000727385492304,
        0.000693173691607,
        0.000372661454964,
        0.000544559585703,
        0.000247203860928,
        0.000155883678898,
        9.71809476449e-05,
        5.9555200068e-05,
        0.000598971199202,
        0.000560801056911,
        0.000427529540831,
        0.000407082891695,
        0.000286421006077,
        0.999999999042,
        0.999996196823,
        0.453796755782,
        0.999999997007,
        -0.0593487301639,
        -0.00860659149684,
        -0.0246667870007,
        -0.00444525544024,
        -0.000206946209007,
        0.000780371017583,
        0.000766856171572,
        0.159029345777,
        0.000895620875314,
        0.00086037284516,
        0.000464673569136,
        0.000678284947811,
        0.000308296411454,
        0.000194459321877,
        0.000121237592141,
        7.43085277253e-05,
        0.000745180609732,
        0.000698880316639,
        0.000532768042438,
        0.000507569519728,
        0.000357068790332,
        1.0,
        0.999996204284,
        0.534168194075,
        0.999999999966,
        -0.114926814344,
        -0.0166691305772,
        -0.0477683855037,
        -0.00860805815781,
        -0.000400595884875,
        0.00150786939356,
        0.00147731095565,
        0.00173249067691,
        0.275885993806,
        0.00166596821925,
        0.000900279545361,
        0.00131396442756,
        0.000597322750102,
        0.000376776553966,
        0.000234907643752,
        0.000143981530481,
        0.00144334505582,
        0.0013539434176,
        0.00103213459674,
        0.000983381296025,
        0.000691787603633,
        0.999999999441,
        0.999996193977,
        0.529211142044,
        0.999999996818,
        -0.126520316894,
        -0.0183469915805,
        -0.0525845500952,
        -0.00947671751474,
        -0.000441281532986,
        0.00166537179697,
        0.00163880630235,
        0.00191030514341,
        0.0018342556771,
        0.309259710046,
        0.000990390732778,
        0.00144576565499,
        0.000657087343946,
        0.000414454554431,
        0.000258395178836,
        0.000158373448989,
        0.00158846376177,
        0.00148961605301,
        0.00113556649512,
        0.00108182027426,
        0.000761056021,
        0.999999999927,
        0.999996172021,
        0.687803317881,
        0.999999974377,
        -0.261416477797,
        -0.0379129170993,
        -0.108653337804,
        -0.0195807226513,
        -0.000911526048822,
        0.00343571944357,
        0.0033737973406,
        0.00394411122541,
        0.00378975832195,
        0.00204707474967,
        0.20931465396,
        0.00298802428313,
        0.00135818190913,
        0.000856686639877,
        0.000534112547302,
        0.000327368311595,
        0.00328261024282,
        0.00307878378207,
        0.00234702481232,
        0.00223604325197,
        0.00157303227109,
        0.999999999457,
        0.999996210791,
        0.606364032058,
        0.999999981236,
        -0.196331677259,
        -0.028442314315,
        -0.0815800094039,
        -0.0147089336131,
        -0.000687135260677,
        0.00262870480024,
        0.00264549941541,
        0.00298949825605,
        0.00284834325482,
        0.00153114039255,
        0.00223747048041,
        0.835092721565,
        0.00101567107394,
        0.000640465286086,
        0.000399277458921,
        0.00024468727566,
        0.00246110459516,
        0.00230417893008,
        0.00175659900028,
        0.00167256815188,
        0.00117681011433,
        1.0,
        0.99999627074,
        0.766339593241,
        0.999999917644,
        -0.327671910634,
        -0.0474620060554,
        -0.136149620962,
        -0.0245497402537,
        -0.00114747909627,
        0.00439950345152,
        0.00444349776367,
        0.00499632966951,
        0.00475435474652,
        0.00255386362109,
        0.00373262324431,
        0.00169403541893,
        0.792556705292,
        0.00106818533898,
        0.000665918796064,
        0.000408082404531,
        0.00410647071902,
        0.00384358982981,
        0.00293019369358,
        0.00278978073108,
        0.00196291898525,
        1.0,
        0.999996395689,
        0.701515491599,
        0.999999858113,
        -0.391772058474,
        -0.0567412739753,
        -0.162780382342,
        -0.0293525422194,
        -0.00137221323432,
        0.00526849508842,
        0.00533179859932,
        0.00597846712397,
        0.00568488664201,
        0.00305242777008,
        0.00446175266581,
        0.0020247117224,
        0.00127666276599,
        0.785977081862,
        0.00079587952014,
        0.000487717856239,
        0.0049091341422,
        0.00459418936741,
        0.00350242587592,
        0.0033344131498,
        0.00234616009818,
        1.00000000006,
        0.999997998136,
        0.921099200444,
        0.999999588155,
        -0.46299886125,
        -0.0670523320668,
        -0.192371491983,
        -0.0346896178048,
        -0.00162187210643,
        0.00623424637421,
        0.0063188886634,
        0.00706984088064,
        0.00671889035266,
        0.00360644002023,
        0.00527195678017,
        0.00239215715185,
        0.00150832365553,
        0.000940301030131,
        0.781309985278,
        0.000576208830347,
        0.00580108122215,
        0.0054282537921,
        0.00413830264781,
        0.00393962350477,
        0.00277202701981,
        1.0,
        0.999997178615,
        0.999987590107,
        0.999999394365,
        -0.529738062965,
        -0.0767122801576,
        -0.220101275564,
        -0.0396900854014,
        -0.00185551024166,
        0.00713954810336,
        0.0072447437518,
        0.00809275204804,
        0.0076879462688,
        0.00412558464305,
        0.00603113918546,
        0.00273645318969,
        0.00172538022602,
        0.00107560760845,
        0.000659123002839,
        0.777651697828,
        0.00663690125785,
        0.00620981175928,
        0.00473412769096,
        0.00450674607758,
        0.00317106131358,
        0.999999999602,
        0.999996170309,
        0.628546861547,
        0.999999997602,
        -0.180553377211,
        -0.0261565413101,
        -0.0750237809201,
        -0.0135268356286,
        -0.000631905111291,
        0.0024175159189,
        0.00243302711656,
        0.00274928457304,
        0.00261944332366,
        0.00140808584056,
        0.00205765303589,
        0.00093404371195,
        0.000588992115236,
        0.000367188258733,
        0.000225022216913,
        0.00226331734611,
        0.854262129111,
        0.00211899706784,
        0.00161542613577,
        0.00153814828089,
        0.00108223338334,
        0.999999999617,
        0.999996180057,
        0.620584799694,
        0.99999998214,
        -0.180956786689,
        -0.0262145793331,
        -0.0751910921755,
        -0.0135569118043,
        -0.000633289302965,
        0.00242284230535,
        0.00243834724437,
        0.00275537801676,
        0.00262527409691,
        0.00141122251157,
        0.00206223481312,
        0.00093612307163,
        0.000590303343645,
        0.000368005081267,
        0.000225522830464,
        0.00226835404402,
        0.00212372056545,
        0.844022237319,
        0.00161902114399,
        0.00154157462177,
        0.00108464122927,
        0.999999999951,
        0.999996374523,
        0.715857854127,
        0.999999954459,
        -0.245331138106,
        -0.035539748976,
        -0.101939687863,
        -0.0183802020956,
        -0.000858773887831,
        0.00328698733609,
        0.00331083973421,
        0.00373686731627,
        0.00355932928113,
        0.00191300110597,
        0.0027956008464,
        0.00126896712227,
        0.000800181444188,
        0.000498846224071,
        0.000305704105681,
        0.0030751616052,
        0.00287888963804,
        0.00219473795594,
        0.810688243448,
        0.00208970315521,
        0.00147031416433,
        0.999999999948,
        0.999996170179,
        0.701746257537,
        0.999999972158,
        -0.244174175365,
        -0.0353711036824,
        -0.101458191698,
        -0.0182933930592,
        -0.000854740636927,
        0.00327238613003,
        0.00329735126331,
        0.00371974218915,
        0.00354256479717,
        0.00190384337627,
        0.00278227036887,
        0.00126288859719,
        0.00079634451343,
        0.000496453621859,
        0.000304236723701,
        0.00306055509291,
        0.00286514132502,
        0.00218425488618,
        0.00207970491354,
        0.809896056277,
        0.00146328011406,
        0.999996181925,
        0.791120150791,
        -0.312968365497,
        -0.045333880526,
        -0.130041463423,
        -0.0234481242946,
        -0.00109587181907,
        0.00420002132421,
        0.00423928350905,
        0.00477097759699,
        0.00454095121986,
        0.00243954885028,
        0.00356543657664,
        0.00161821856882,
        0.00102038650615,
        0.000636122225704,
        0.00038982424503,
        0.00392240552141,
        0.00367148245724,
        0.00279899206667,
        0.00266489898788,
        0.00187504799611,
        0.796740651693,
        1.0,
        -0.999996205802,
        0.999999998752,
        -1.07414440809e-05,
        -0.000300919763069,
        -0.0226673885982,
        -0.103955788017,
        -0.0516991284109,
        -0.0310235352457,
        -0.0257605492265,
        -0.0148517832223,
        -0.0127716171052,
        -0.00328810118649,
        -0.00638035007331,
        -0.00173057861423,
        -0.0009121255986,
        -0.00048092759418,
        -0.000257461328168,
        -0.00764252973277,
        -0.00713140367755,
        -0.00400269315584,
        -0.00382733873689,
        -0.00209973230739,
        -1.0,
        0.219140501574,
        -1.0,
        -0.0502292279349,
        0.0337102101807,
        0.00885578163645,
        0.0253467607336,
        0.0192028646984,
        0.0137706037785,
        0.0125195072231,
        0.00854122182477,
        0.00781497563926,
        0.00280453294918,
        0.00465604076376,
        0.0016595542266,
        0.000980127920226,
        0.000566513875961,
        0.00033065005823,
        0.00529560721213,
        0.00507074196925,
        0.00323259010821,
        0.00313778768704,
        0.00192480237735,
        -1.0,
        -0.0157759055882,
        0.0,
        -1.0,
        -0.0970532686971,
        -0.0216049616507,
        -0.0480265523393,
        -0.0154449648332,
        -0.00638405808541,
        -0.00324830239798,
        -0.00259850199169,
        -0.00126753301018,
        -0.00105230008595,
        -0.000149705237773,
        -0.000403567075721,
        -4.57462394632e-05,
        -7.67051866068e-06,
        5.71961795229e-06,
        7.93827627508e-06,
        -0.000516041806372,
        -0.00048260911914,
        -0.000198095241249,
        -0.00019147682031,
        -6.38301575845e-05,
        -0.126640606449,
        0.999999999997,
        1.00000000418,
        0.999999999468,
        0.999928198415,
        0.114477145112,
        -0.0939060568645,
        -0.944921461435,
        -0.00942447979133,
        -0.0364919200948,
        -0.0079203604376,
        -0.00234087793827,
        -0.00102985273301,
        -0.000469193224681,
        -9.14667745765e-05,
        2.62543496552e-05,
        0.000111581044085,
        9.7395192923e-05,
        7.66458050853e-05,
        5.31968509747e-05,
        3.04095416425e-05,
        1.89497234723e-05,
        6.9023540653e-05,
        0.000130848339344,
        9.42192712002e-05,
        0.000119838463303,
        7.12467123919e-05,
        -0.99999999953,
        0.999999999022,
        0.999998262847,
        -0.0505007583806,
        -0.0374199951606,
        -0.00402583958711,
        -0.813931245946,
        -0.0147446370217,
        -0.00328439524961,
        -0.000997130506158,
        -0.000437387841199,
        -0.000215393841557,
        -4.20655236951e-05,
        9.53625616385e-07,
        4.46835266739e-05,
        3.86419379705e-05,
        3.16843293609e-05,
        2.23271229106e-05,
        1.31851476032e-05,
        8.13394779383e-06,
        2.5387834291e-05,
        4.90465683002e-05,
        3.79618030523e-05,
        4.76872088447e-05,
        3.01192606728e-05,
        -0.999999998238,
        0.999999998526,
        0.99999998265,
        0.0600375815669,
        -0.0699615226048,
        -0.00713983634804,
        -0.027248465247,
        -1.07261425665,
        -0.00596263966287,
        -0.00179545845243,
        -0.000802364029941,
        -0.000377648010778,
        -7.02513235726e-05,
        9.42452732424e-06,
        8.05083849522e-05,
        7.01805397298e-05,
        5.59545047799e-05,
        3.93428341119e-05,
        2.27706441033e-05,
        1.39077163767e-05,
        4.53031927737e-05,
        9.2792694136e-05,
        6.68940639708e-05,
        8.64979458102e-05,
        5.20759153463e-05,
        -1.00000000004,
        0.999999999823,
        0.999999977475,
        -0.0259811226997,
        -0.0409390904514,
        -0.00427985642706,
        -0.0160286516646,
        -0.00353583094217,
        -0.549985688664,
        -0.00106875155936,
        -0.000473256369543,
        -0.000227394891412,
        -4.33364826782e-05,
        3.44308272914e-06,
        4.78820534536e-05,
        4.15653307158e-05,
        3.35970909953e-05,
        2.36513343755e-05,
        1.38169735571e-05,
        8.47204608201e-06,
        2.69905055627e-05,
        5.40469383744e-05,
        4.0208757222e-05,
        5.12822543218e-05,
        3.15705415872e-05,
        -1.0,
        1.00000000013,
        0.999999983126,
        -0.070613845367,
        -0.0252537176772,
        -0.00268156225181,
        -0.00991851980666,
        -0.00219507931597,
        -0.000661257292261,
        -0.049151643257,
        -0.000286893838324,
        -0.000138035160007,
        -2.79422464183e-05,
        1.36545592564e-06,
        3.013477436e-05,
        2.61469440989e-05,
        2.12550082406e-05,
        1.49629454223e-05,
        8.79075345571e-06,
        5.40166300019e-06,
        1.70570677455e-05,
        3.36322992039e-05,
        2.54971229084e-05,
        3.22175608396e-05,
        2.0104859341e-05,
        -1.0,
        1.00000000008,
        0.999999984995,
        -0.0830096039491,
        -0.0191888320909,
        -0.00200488452602,
        -0.00750958990676,
        -0.00165279922811,
        -0.000496511472238,
        -0.0002165778481,
        -0.0458064507852,
        -0.000102809661128,
        -2.05093367732e-05,
        1.73141741348e-06,
        2.2659283717e-05,
        1.97197109395e-05,
        1.58795639335e-05,
        1.11694199157e-05,
        6.52158440395e-06,
        3.99702415368e-06,
        1.28129351231e-05,
        2.56605654526e-05,
        1.90365368381e-05,
        2.42781680173e-05,
        1.49247963075e-05,
        -0.999999999921,
        1.00000000028,
        0.999999984851,
        -0.108698461082,
        -0.0107554414761,
        -0.00119497423418,
        -0.00426833857876,
        -0.000960173489165,
        -0.000292122632745,
        -0.00012555484594,
        -6.29291199414e-05,
        -0.0207754851606,
        -1.30217049114e-05,
        -5.5770213441e-07,
        1.31738653694e-05,
        1.13253681845e-05,
        9.46134680774e-06,
        6.67755546723e-06,
        3.9889227579e-06,
        2.46802744049e-06,
        7.46924852501e-06,
        1.40878022211e-05,
        1.13628747612e-05,
        1.39974078355e-05,
        9.10245786154e-06,
        -1.00000000021,
        1.00000000025,
        0.999999985192,
        -0.130081659397,
        -0.00358474622375,
        -0.000414850931305,
        -0.00143655765201,
        -0.00032815481943,
        -0.000100878933023,
        -4.31657311756e-05,
        -2.25028286547e-05,
        -4.67463103211e-06,
        -0.0212406659372,
        -5.39108025454e-07,
        4.48574015763e-06,
        3.82235995314e-06,
        3.27351704136e-06,
        2.3159014808e-06,
        1.40297541125e-06,
        8.73051070114e-07,
        2.54499432279e-06,
        4.60847272257e-06,
        3.93352055056e-06,
        4.73956729395e-06,
        3.19416359986e-06,
        -1.00000000015,
        1.00000000039,
        0.999999982275,
        -0.149410965325,
        0.000712164784415,
        9.18026598137e-06,
        0.000224495560015,
        3.0417800863e-05,
        5.75878632832e-06,
        4.25760776992e-06,
        -1.1193269774e-06,
        -6.25102765664e-07,
        -1.44387470619e-06,
        -0.0203953879469,
        -4.06140990274e-07,
        -4.79399461538e-07,
        -7.67852449742e-08,
        -3.34230474713e-08,
        6.21764353869e-08,
        5.92292878401e-08,
        -2.11219398883e-07,
        -1.21248375457e-06,
        -7.36410138018e-08,
        -5.41882094509e-07,
        1.18052422817e-07,
        -0.999999999362,
        1.00000000087,
        0.99999998002,
        -0.219509255657,
        0.0192192107091,
        0.00200776387523,
        0.00752585726672,
        0.0016630727818,
        0.000505531967496,
        0.000227106756013,
        0.000110227948314,
        2.00978877563e-05,
        -1.55906044872e-06,
        -2.22520125434e-05,
        -0.0035151275883,
        -1.92731514801e-05,
        -1.56239634799e-05,
        -1.1007023033e-05,
        -6.43033583031e-06,
        -3.94368578541e-06,
        -1.25032177157e-05,
        -2.50662493552e-05,
        -1.86666523774e-05,
        -2.38269646902e-05,
        -1.46707727427e-05,
        -0.999999999604,
        0.999999999667,
        0.999999998203,
        -0.182886803131,
        0.00850567966688,
        0.000878819899975,
        0.00332113128533,
        0.000728988163483,
        0.000219276777649,
        9.67965030188e-05,
        4.57923541761e-05,
        8.82396188531e-06,
        -9.49847710333e-07,
        -9.91886002027e-06,
        -8.63867877233e-06,
        -0.0271087438533,
        -6.92402147211e-06,
        -4.86946827178e-06,
        -2.8314252333e-06,
        -1.73249743317e-06,
        -5.59518576592e-06,
        -1.13274914179e-05,
        -8.28912781308e-06,
        -1.06415403972e-05,
        -6.4772709012e-06,
        -0.999999999931,
        1.00000000022,
        0.999999982075,
        -0.24867150949,
        0.0248461655941,
        0.00267101130287,
        0.00978912078005,
        0.00218096877337,
        0.000663134742734,
        0.000291701448402,
        0.000143860925213,
        2.78390606941e-05,
        -5.54122455606e-07,
        -2.95403498197e-05,
        -2.55018409998e-05,
        -2.09595267088e-05,
        -0.00705693460889,
        -1.47780563102e-05,
        -8.7243845199e-06,
        -5.37297865108e-06,
        -1.66703158982e-05,
        -3.25045325486e-05,
        -2.51036887422e-05,
        -3.15190318931e-05,
        -1.99068015199e-05,
        -1.00000000043,
        1.00000000016,
        0.999999987726,
        -0.324990029401,
        0.0331928171109,
        0.00358115528995,
        0.0130891577373,
        0.00292114206553,
        0.000889921388615,
        0.000392270232804,
        0.000194566246837,
        3.73733822981e-05,
        -4.54888381931e-07,
        -3.94707933617e-05,
        -3.40347338686e-05,
        -2.80500167713e-05,
        -1.97843697957e-05,
        -0.00371532530103,
        -1.16957657259e-05,
        -7.20731368308e-06,
        -2.22640581297e-05,
        -4.32664938391e-05,
        -3.35879294197e-05,
        -4.20917752387e-05,
        -2.66738032138e-05,
        -0.999999999962,
        1.00000000077,
        0.999999991498,
        -0.311316034051,
        0.0362659688973,
        0.00401598611501,
        0.0143873817533,
        0.00324145680219,
        0.000993389489825,
        0.000435701069396,
        0.000221529188048,
        4.29411581023e-05,
        1.64814526056e-06,
        -4.37644707905e-05,
        -3.75368788327e-05,
        -3.14216347256e-05,
        -2.21952946612e-05,
        -1.32443438951e-05,
        -0.00195938574644,
        -8.19326795921e-06,
        -2.47060176476e-05,
        -4.6815066108e-05,
        -3.76464160522e-05,
        -4.65062764278e-05,
        -3.0164620687e-05,
        -0.99999999992,
        1.00000000144,
        0.999999979488,
        -0.340542658896,
        0.0413326905628,
        0.00462755142762,
        0.016440135158,
        0.00371948932644,
        0.00114334174414,
        0.000501068079734,
        0.000257695763143,
        4.99136622598e-05,
        2.92485279735e-06,
        -5.01472900299e-05,
        -4.29064508178e-05,
        -3.61604522712e-05,
        -2.55601155425e-05,
        -1.53107952103e-05,
        -9.48671494596e-06,
        -0.00104990431007,
        -2.83109127115e-05,
        -5.30770875783e-05,
        -4.33270057993e-05,
        -5.32091121615e-05,
        -3.48467340453e-05,
        -1.00000000032,
        1.00000000067,
        0.999999979132,
        -0.158388598432,
        0.00457053888903,
        0.000479965430013,
        0.00179099963758,
        0.000395323349715,
        0.00011926236891,
        5.23772349141e-05,
        2.51346402631e-05,
        4.90906565974e-06,
        -3.48777279871e-07,
        -5.38496433081e-06,
        -4.67579243659e-06,
        -3.78358039009e-06,
        -2.66317753287e-06,
        -1.55830438643e-06,
        -9.56005636971e-07,
        -3.04030282666e-06,
        -0.033189768281,
        -6.0615360657e-06,
        -4.53236463569e-06,
        -5.76484069257e-06,
        -3.56234532148e-06,
        -1.00000000005,
        0.999999999797,
        0.999999997793,
        -0.169624621228,
        0.0101668725075,
        0.00100209520878,
        0.00392954682398,
        0.000848403935896,
        0.000252674089286,
        0.000112864506041,
        5.10264821524e-05,
        9.5236596246e-06,
        -2.15439565787e-06,
        -1.15352821613e-05,
        -1.01399015146e-05,
        -7.89773344119e-06,
        -5.53892738643e-06,
        -3.15959022337e-06,
        -1.91746692779e-06,
        -6.49335290065e-06,
        -1.37329499951e-05,
        -0.0306449632557,
        -9.44122710397e-06,
        -1.24550710471e-05,
        -7.24612883907e-06,
        -0.999999999902,
        1.00000000005,
        0.999999992459,
        -0.188757890793,
        0.0128029295287,
        0.00137957136725,
        0.00504643111472,
        0.0011245767021,
        0.000341497762149,
        0.000149483920186,
        7.36178762511e-05,
        1.44610130258e-05,
        -2.28885215168e-07,
        -1.52872216488e-05,
        -1.32003382108e-05,
        -1.08539595871e-05,
        -7.65203993443e-06,
        -4.52113599697e-06,
        -2.78514517837e-06,
        -8.63596239556e-06,
        -1.67971289601e-05,
        -1.30074697645e-05,
        -0.0165492241984,
        -1.63074666009e-05,
        -1.03196760808e-05,
        -0.999999999662,
        1.0000000006,
        0.999999978998,
        -0.19890728068,
        0.0174661164883,
        0.00181144808849,
        0.00682593831469,
        0.00150093650508,
        0.000452400724089,
        0.000200148341847,
        9.52450671866e-05,
        1.82226562993e-05,
        -1.78785872426e-06,
        -2.03711569147e-05,
        -1.77194183884e-05,
        -1.42451246218e-05,
        -1.00220853766e-05,
        -5.83648764198e-06,
        -3.57378666193e-06,
        -1.14855311892e-05,
        -2.31719287945e-05,
        -1.70499253443e-05,
        -2.18426404512e-05,
        -0.0158131914032,
        -1.33450658059e-05,
        -1.00000000082,
        0.999999977641,
        -0.220371421046,
        0.0189201361273,
        0.00208878375352,
        0.00749964655674,
        0.00168637077342,
        0.000515195991374,
        0.000224761438006,
        0.000113367045944,
        2.23814958415e-05,
        7.18388200365e-07,
        -2.28835377961e-05,
        -1.96576544474e-05,
        -1.64051270481e-05,
        -1.15824547797e-05,
        -6.90381263291e-06,
        -4.26852633216e-06,
        -1.29336444643e-05,
        -2.45712353009e-05,
        -1.96678301098e-05,
        -2.43297406539e-05,
        -1.57362255087e-05,
        -0.00857671794538,
        0.999999998649,
        -1.00000002633,
        3.08077941039e-06,
        0.00095989805431,
        0.00740523711046,
        0.0232617319336,
        0.00834381390467,
        0.00433325243555,
        0.00341454311851,
        0.0016205288246,
        0.00132988086717,
        0.000238326767138,
        0.00054839548066,
        0.000106729497949,
        4.8609048908e-05,
        2.2350535808e-05,
        1.0377229001e-05,
        0.000704394618739,
        0.00063027231662,
        0.000302755053487,
        0.000287354227795,
        0.000136360655855,
        -0.99999999875,
        1.0,
        1.0,
        -7.30717148815e-05,
        1.0,
        -0.037046740704,
        1.00000011991,
        1.00000016513,
        0.999999990972,
        0.999998629639,
        0.10862043248,
        0.0372030052348,
        0.0108424037117,
        0.00291266918303,
        0.0127987470436,
        0.00104868608207,
        -0.000490391108811,
        -0.000636446746524,
        -0.000559546567276,
        -0.000432835949535,
        -0.000383191691904,
        -0.000125602621306,
        -0.000229335746977,
        -6.83740682864e-05,
        -3.5542091581e-05,
        -1.83483857923e-05,
        -9.38070791742e-06,
        -0.000269147657372,
        -0.000248494958518,
        -0.000153675090558,
        -0.000145552976678,
        -8.39583122874e-05,
        -1.00000000002,
        0.999999986069,
        0.999999989225,
        0.0729387438067,
        0.0162349356266,
        0.00127626693877,
        0.0551840640797,
        0.00558881715609,
        0.000457975798872,
        -0.000221940628556,
        -0.000283679804736,
        -0.000247236379584,
        -0.000191019403419,
        -0.000168902257877,
        -5.53691943838e-05,
        -0.000101127774609,
        -3.01611406316e-05,
        -1.56812464529e-05,
        -8.10235297143e-06,
        -4.14441403344e-06,
        -0.000118744502951,
        -0.000109484944981,
        -6.77958661604e-05,
        -6.41567125065e-05,
        -3.70574571033e-05,
        -0.999999966418,
        0.999999988958,
        0.999999984785,
        0.101817470386,
        0.02717351655,
        0.00216760822796,
        0.00937634177967,
        0.0210838944183,
        0.000768333552586,
        -0.000423427716489,
        -0.000513433449752,
        -0.000433638300352,
        -0.00033351108748,
        -0.000293561663473,
        -9.62653077276e-05,
        -0.000176015577643,
        -5.25658260237e-05,
        -2.73482913269e-05,
        -1.41758560235e-05,
        -7.26405654695e-06,
        -0.000207072275624,
        -0.000189977369282,
        -0.000118198613499,
        -0.000111496330714,
        -6.47233262865e-05,
        -1.00000000003,
        0.999999999619,
        0.999999982402,
        0.0922988416864,
        0.00771439072037,
        0.000607703133158,
        0.00265651915871,
        0.000217680624967,
        0.0758064627239,
        -0.000107619884108,
        -0.000136398056246,
        -0.000118301250917,
        -9.1337457466e-05,
        -8.07066646474e-05,
        -2.64581703556e-05,
        -4.83320215401e-05,
        -1.4417751512e-05,
        -7.49677496697e-06,
        -3.87538357483e-06,
        -1.98282533682e-06,
        -5.67679422866e-05,
        -5.23020710026e-05,
        -3.24098266853e-05,
        -3.06553526461e-05,
        -1.77200776171e-05,
        -1.0000000004,
        0.999999998859,
        0.999999982729,
        0.100314201763,
        -0.00913060809109,
        -0.000709109443497,
        -0.00313706878302,
        -0.000257026720061,
        0.000110975322672,
        0.0237030490046,
        0.000149224885342,
        0.000133748467573,
        0.000103738235128,
        9.20833850047e-05,
        3.01772971337e-05,
        5.50651964724e-05,
        1.64041700439e-05,
        8.52380850019e-06,
        4.39204402552e-06,
        2.24308260999e-06,
        6.455234614e-05,
        5.97725576666e-05,
        3.6862178657e-05,
        3.49793385763e-05,
        2.01179004244e-05,
        -1.0,
        0.999999998834,
        0.999999999481,
        0.121678187592,
        -0.0229778397981,
        -0.00180006084032,
        -0.00790558516572,
        -0.000647774483432,
        0.000304310826727,
        0.000394178634258,
        0.0412676613228,
        0.000346160239037,
        0.000267732032832,
        0.000236985740796,
        7.76810542918e-05,
        0.000141842457336,
        4.22906373089e-05,
        2.19840131657e-05,
        1.135030736e-05,
        5.80329359617e-06,
        0.000166477005798,
        0.000153675902849,
        9.50527557454e-05,
        9.00186828407e-05,
        5.19339971966e-05,
        -1.0,
        0.999999999847,
        0.999999999843,
        0.115453370585,
        -0.0257989330731,
        -0.00202911393717,
        -0.00888182014672,
        -0.000727768790125,
        0.000354924247962,
        0.00045242828978,
        0.000393717966998,
        0.0256662934558,
        0.000304122043575,
        0.000268852996594,
        8.81347773777e-05,
        0.000160980417767,
        4.80146204022e-05,
        2.49642931397e-05,
        1.29006847467e-05,
        6.59933806405e-06,
        0.000189040291058,
        0.000174259983082,
        0.000107928744534,
        0.000102120584767,
        5.89988506211e-05,
        -0.999999998818,
        1.0,
        0.999999993682,
        0.12883411019,
        -0.0398146895496,
        -0.00312491144649,
        -0.0137024559374,
        -0.00112276486819,
        0.000536932120154,
        0.000690179844611,
        0.000603487141079,
        0.000466471847179,
        0.0740793506514,
        0.000412652186976,
        0.000135268269011,
        0.000247030614387,
        7.36657968639e-05,
        3.82972748953e-05,
        1.97813031619e-05,
        1.01164125584e-05,
        0.000290007637922,
        0.000267530080513,
        0.000165579579817,
        0.000156743290329,
        9.04894291302e-05,
        -1.0,
        0.999999999571,
        0.999999965222,
        0.123505443633,
        -0.0428784926168,
        -0.00336085467402,
        -0.0147536966987,
        -0.00120888827712,
        0.000570947513878,
        0.000737852855955,
        0.000647134844913,
        0.000500423398307,
        0.000442875261525,
        0.0945864358326,
        0.000145170681887,
        0.000265087484226,
        7.90404753715e-05,
        4.10888257041e-05,
        2.12167834416e-05,
        1.08486935152e-05,
        0.000311150199516,
        0.000287168175083,
        0.00017765454127,
        0.00016822450336,
        9.70719830052e-05,
        -0.999999999842,
        0.999999998644,
        0.999996042217,
        0.144212869902,
        -0.0771403059402,
        -0.00603994070167,
        -0.0265381176779,
        -0.00217455274245,
        0.00101665003104,
        0.00131959076392,
        0.00116022491037,
        0.000897503656212,
        0.000794557736467,
        0.000260444282154,
        1.08289238004,
        0.000475536261698,
        0.000141775180928,
        7.36966364489e-05,
        3.80449263661e-05,
        1.94510408347e-05,
        0.000558082576308,
        0.000515273805709,
        0.000318653401187,
        0.000301812765905,
        0.000174089818895,
        -0.999999999802,
        0.99999999796,
        0.999999355179,
        0.132220751292,
        -0.0607899972656,
        -0.00475930831411,
        -0.0209127230996,
        -0.00171359666615,
        0.000800391640762,
        0.00103934137466,
        0.000913999630445,
        0.000707060584774,
        0.0006259814044,
        0.000205186166422,
        0.000374645225809,
        0.786511119512,
        0.000111695135057,
        5.80606467411e-05,
        2.99723149481e-05,
        1.53234408634e-05,
        0.000439675390732,
        0.000405955690672,
        0.000251041720022,
        0.000237778568299,
        0.000137151230282,
        -0.999999999445,
        0.999999985202,
        0.999640085582,
        0.154105074934,
        -0.0946131691331,
        -0.00741233794536,
        -0.0324819864206,
        -0.00267535745813,
        0.00124330009822,
        0.00161705125729,
        0.00142228418749,
        0.00110113773229,
        0.00097425239466,
        0.000319300063213,
        0.000582781190218,
        0.000173912300374,
        1.26004799343,
        9.02604379338e-05,
        4.66210254006e-05,
        2.38445878371e-05,
        0.00068433793759,
        0.000631960508539,
        0.000390734069891,
        0.000370078197917,
        0.000213317791665,
        -0.999999999394,
        0.999999942568,
        0.993778706362,
        0.10062271556,
        -0.115612843831,
        -0.00922443527303,
        -0.0387426061694,
        -0.00353603148531,
        0.00146054365915,
        0.00196019837076,
        0.00167193125287,
        0.00131640540668,
        0.00115305557,
        0.000386754075081,
        0.000701442202217,
        0.000208991353873,
        0.000107198152398,
        1.22872299583,
        5.57010096437e-05,
        2.88264015957e-05,
        0.000821552238716,
        0.000763899280805,
        0.000470831105577,
        0.000442004639368,
        0.000261839777564,
        -0.999999999723,
        0.999999973833,
        0.980230999815,
        0.0963879692591,
        -0.102107393501,
        -0.00510536994089,
        -0.0204214790241,
        0.0,
        0.00382903433712,
        0.00382903035045,
        0.00350994333341,
        0.00239314294504,
        0.00191451421305,
        0.000678057134752,
        0.00119657141404,
        0.000309114272536,
        0.000139599989615,
        7.47857111587e-05,
        1.21187542183,
        4.11321425861e-05,
        0.00127634282624,
        0.00143588569931,
        0.00075782856058,
        0.000757828551092,
        0.000378914275902,
        -0.999999997545,
        0.999999749143,
        0.0,
        0.999999994306,
        0.0,
        -0.0794503206023,
        0.0,
        -0.0397251574516,
        -0.0198625570911,
        0.0,
        -0.00496564439713,
        0.0,
        -0.00744846619991,
        0.0,
        0.0,
        0.0,
        -0.000543117336075,
        -0.000232764575061,
        0.0,
        1.19947002,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        -1.0,
        1.00000000008,
        0.999999625161,
        0.145804664084,
        -0.0567447879759,
        -0.00444229446411,
        -0.0195210323499,
        -0.00159953313695,
        0.000746552325378,
        0.000969752999919,
        0.000852964809526,
        0.000659859169827,
        0.000584207703234,
        0.000191493671035,
        0.000349641290884,
        0.000104239582781,
        5.4185127192e-05,
        2.79713157863e-05,
        1.43001761775e-05,
        0.000410327128903,
        0.542280793243,
        0.000378866837978,
        0.000234285364167,
        0.000221911750035,
        0.000127995640055,
        -0.999999999696,
        0.999999999454,
        0.999999583179,
        0.143430049665,
        -0.0567513140902,
        -0.00444299883175,
        -0.0195234775462,
        -0.0015997341613,
        0.000747057273521,
        0.000970165157515,
        0.000853219830163,
        0.000660039719318,
        0.00058435750993,
        0.000191543097634,
        0.00034973259644,
        0.000104267320682,
        5.41995886904e-05,
        2.79791115976e-05,
        1.430433527e-05,
        0.000410436945263,
        0.000378962100534,
        0.623059533494,
        0.000234348324149,
        0.00022196885437,
        0.000128030372157,
        -1.0,
        0.999999997745,
        0.999998519005,
        0.157943921751,
        -0.0733124152651,
        -0.00574016586898,
        -0.0252197488111,
        -0.00206674188268,
        0.00096587770467,
        0.00125386371224,
        0.00110246246243,
        0.000852853967559,
        0.000755023233924,
        0.000247492599824,
        0.000451888787581,
        0.00013472609595,
        7.00300899917e-05,
        3.61530349227e-05,
        1.84834132414e-05,
        0.000530333967225,
        0.00048964944709,
        0.000302803612226,
        0.99676343815,
        0.000286805081979,
        0.000165437773877,
        -0.999999999336,
        0.999999999989,
        0.999990558589,
        0.155343294685,
        -0.0729226585185,
        -0.00570977096392,
        -0.0250866422343,
        -0.00205541655568,
        0.000960959654002,
        0.00124739654307,
        0.00109678883258,
        0.000848420236267,
        0.000751105269051,
        0.000246198807511,
        0.000449536044211,
        0.000134022090723,
        6.96643485706e-05,
        3.59642074662e-05,
        1.83871008234e-05,
        0.000527569679353,
        0.000487093724578,
        0.000301226758466,
        0.000285303466241,
        1.00272428737,
        0.000164571197893,
        -0.999999999914,
        0.999969004875,
        0.163504618975,
        -0.0907400987349,
        -0.00710927368041,
        -0.0312124804373,
        -0.00255944782446,
        0.00119762007273,
        0.001552912154,
        0.00136572264039,
        0.00105643582831,
        0.000935079005521,
        0.000306430529486,
        0.000559602127682,
        0.000166886876058,
        8.67067789639e-05,
        4.47562595082e-05,
        2.28878553083e-05,
        0.000656705596301,
        0.000606583594729,
        0.000374907055605,
        0.000355219687518,
        0.000204858041344,
        1.26949606239,
        0.999999998132,
        -1.00000004413,
        -1.62279559076e-06,
        -0.00027681435686,
        -0.00230289213793,
        -0.0107880885162,
        -0.00738416958284,
        -0.00408529007857,
        -0.00327174057549,
        -0.0016779157564,
        -0.00138671882602,
        -0.000227254722117,
        -0.000479329958123,
        -3.50044891435e-06,
        -2.08257152876e-07,
        -1.44202058116e-08,
        -1.09536230142e-09,
        -0.000595945320648,
        -0.000554344676487,
        -0.000120212233072,
        -0.000104310240133,
        -8.6641305678e-06,
        -0.999999998971,
        -1.0,
        0.501386489802,
        -1.0,
        -0.0590043280703,
        0.0357257250974,
        0.0152177670034,
        0.025704202925,
        0.0158587454589,
        0.00990448741443,
        0.00856509902102,
        0.00506388604627,
        0.00442691286363,
        0.00110769902749,
        0.00220255383703,
        0.00055236700012,
        0.000273604659248,
        0.000132617345129,
        6.51620287972e-05,
        0.00263204778268,
        0.00249317750693,
        0.00134965541572,
        0.00130452785469,
        0.000674972303779,
        -1.0,
        -0.0397613655107,
        -1.0,
        -0.0915106467594,
        -0.0140430879719,
        -0.0398875148354,
        -0.00977784999312,
        -0.00325516868994,
        -0.0014037845294,
        -0.00105972117316,
        -0.000424817236747,
        -0.000331207362671,
        -2.13109337566e-05,
        -9.23513316122e-05,
        -6.34191086592e-07,
        3.42264832815e-06,
        3.36907808628e-06,
        2.29961422057e-06,
        -0.000129255972295,
        -0.00012009406425,
        -3.276261381e-05,
        -3.2048071365e-05,
        -3.13812861284e-06,
        -0.0967778907088,
        0.999999998641,
        -1.00000000106,
        -1.0,
        0.157090888021,
        0.0357459827456,
        0.999999998893,
        -0.999983892957,
        -1.07384451767,
        -0.0185425591239,
        -0.0937166786391,
        -0.0116649470332,
        -0.00399665322767,
        1.1394958337e-05,
        0.00360601496663,
        0.00333337957437,
        0.00350736033343,
        0.00104968762419,
        0.0018912500249,
        0.000501715202814,
        0.00024728922432,
        0.000108175219383,
        4.99413416007e-05,
        0.00206367066203,
        0.00233739151309,
        0.00114992401004,
        0.00124668982439,
        0.000564230162369,
        -0.999999968944,
        0.157089442997,
        -0.0688538853772,
        0.9999999988,
        -0.189653517274,
        -0.0045794252996,
        -0.970039591852,
        -0.0183111861872,
        -0.00239795234818,
        -0.000856202270154,
        7.75096875352e-05,
        0.000713715130344,
        0.000712288532023,
        0.000729768323354,
        0.000229342390543,
        0.000412422172138,
        0.000112971166894,
        5.5998118464e-05,
        2.53624548887e-05,
        1.18897614707e-05,
        0.000457211190552,
        0.000493901435657,
        0.000258473699932,
        0.000271014274017,
        0.000130395508049,
        -0.999999999999,
        0.157089419777,
        -0.000182284657047,
        0.999999999996,
        -0.65741688031,
        -0.0129926580333,
        -0.0621813740022,
        -0.944529906405,
        -0.00768635394611,
        -0.00258734016302,
        3.36292804786e-05,
        0.00247166453977,
        0.00230245485782,
        0.00242168998701,
        0.000729772237037,
        0.00131115727191,
        0.000350449346121,
        0.000173027754655,
        7.60121814701e-05,
        3.51199850512e-05,
        0.00143392541947,
        0.00161757284331,
        0.000801223730654,
        0.000866407921985,
        0.000394889876135,
        -0.999999999968,
        0.157089418669,
        -0.0626299614617,
        0.999999991952,
        -0.271950857804,
        -0.00566994690609,
        -0.0258583759313,
        -0.00323210197607,
        -1.02953476789,
        -0.00109937651503,
        3.57017467803e-05,
        0.0010278485267,
        0.000973221800018,
        0.00101735953967,
        0.000309771191446,
        0.00055658554319,
        0.000149707967344,
        7.39983832573e-05,
        3.27607393405e-05,
        1.51939305571e-05,
        0.00061067433408,
        0.000681758888577,
        0.000342266175607,
        0.000367342682189,
        0.000169693854565,
        -1.0,
        0.157089416442,
        -0.102772145813,
        0.999999999961,
        -0.0721449208307,
        -0.00157067372753,
        -0.00688819152265,
        -0.000872263871373,
        -0.000300779551364,
        -0.150863210534,
        1.50458185753e-05,
        0.000272272004041,
        0.00026172473224,
        0.000272002860304,
        8.35916950703e-05,
        0.00015022905528,
        4.06294526882e-05,
        2.01001157096e-05,
        8.96055603376e-06,
        4.16950497278e-06,
        0.000165332095276,
        0.000182802320551,
        9.29066080987e-05,
        9.90251009868e-05,
        4.63055550888e-05,
        -1.0,
        0.157089417707,
        -0.124598953827,
        0.99999999987,
        0.000370475171203,
        0.000248158209505,
        0.000154679683713,
        4.93471041883e-05,
        2.69350238274e-05,
        -1.77467494466e-05,
        -0.132528008254,
        -6.71142980276e-06,
        -1.8745372766e-05,
        -1.47591422575e-05,
        -6.96023956887e-06,
        -1.25385856903e-05,
        -4.08732758962e-06,
        -2.08155224601e-06,
        -1.1121227844e-06,
        -5.58514836548e-07,
        -1.52742907841e-05,
        -1.16040524839e-05,
        -9.34993404485e-06,
        -7.9281913888e-06,
        -5.39550492323e-06,
        -0.999999999171,
        0.15708941707,
        -0.13993608168,
        0.999999999492,
        0.125016208569,
        0.00230890480598,
        0.0116870290048,
        0.00148341288454,
        0.000519867340158,
        -6.85913301065e-06,
        -0.00043509215836,
        -0.0619650364839,
        -0.000404368245466,
        -0.000424616173273,
        -0.000127498907929,
        -0.000229551301129,
        -6.11622137685e-05,
        -3.01499649108e-05,
        -1.3243730505e-05,
        -6.11546819426e-06,
        -0.000251335559464,
        -0.000283151795863,
        -0.000140183715248,
        -0.000151428949463,
        -6.90291971442e-05,
        -1.0,
        0.157089420513,
        -0.166132183909,
        0.999999999718,
        0.217349312266,
        0.0044524576758,
        0.0205738746447,
        0.00263091833332,
        0.000924160793015,
        -3.68269432783e-05,
        -0.000785099965945,
        -0.000747886284339,
        -0.133715387347,
        -0.000778840391816,
        -0.000237711147108,
        -0.00042768544411,
        -0.000115125379986,
        -5.68749518109e-05,
        -2.52617217395e-05,
        -1.17303288134e-05,
        -0.000470263977829,
        -0.000522144785752,
        -0.000263620517911,
        -0.000281813914651,
        -0.000130963535088,
        -0.999999999964,
        0.157089420478,
        -0.178378846234,
        1.00000000004,
        0.282147246608,
        0.00566172609335,
        0.0266595443985,
        0.00337855526873,
        0.0011745945551,
        -3.58980395068e-05,
        -0.00102429848841,
        -0.000967729676802,
        -0.00101116143575,
        -0.210546188662,
        -0.000307108265275,
        -0.000552395244926,
        -0.000148276145789,
        -7.32248228189e-05,
        -3.23993177214e-05,
        -1.50177768731e-05,
        -0.000606313916738,
        -0.000676885067515,
        -0.000339435257904,
        -0.000364285093027,
        -0.000168145183304,
        -0.999999999404,
        0.157089472983,
        -0.248677599207,
        0.999999999878,
        0.52894832223,
        0.011179575119,
        0.0502702174691,
        0.00642936446145,
        0.00225235404523,
        -0.00010676212363,
        -0.00193833333951,
        -0.00185890539244,
        -0.00193229322573,
        -0.000592379700822,
        -0.151601801093,
        -0.00106539507462,
        -0.000287597666509,
        -0.00014221802553,
        -6.33473937469e-05,
        -2.94528101431e-05,
        -0.00117254881207,
        -0.00129724897551,
        -0.000658362516281,
        -0.00070198544112,
        -0.000327847219335,
        -0.999999999831,
        0.157089431883,
        -0.212245595856,
        1.00000000005,
        0.383138250168,
        0.00810538740796,
        0.036417265748,
        0.00465604856342,
        0.00163062599186,
        -7.73400348294e-05,
        -0.00140548060249,
        -0.00134800614157,
        -0.00140090004128,
        -0.000429570212628,
        -0.000772583142236,
        -0.327976639271,
        -0.000208587733845,
        -0.000103123025209,
        -4.59368367301e-05,
        -2.13630450292e-05,
        -0.000850370063751,
        -0.000940661018706,
        -0.000477425059882,
        -0.000509003562609,
        -0.000237750771836,
        -0.99999999698,
        0.157151161946,
        -0.280289053118,
        0.999999998194,
        0.608159175981,
        0.0133095306546,
        0.0582662800014,
        0.0077181412243,
        0.00268656458471,
        -0.000211262510313,
        -0.00213310985911,
        -0.00213775760861,
        -0.00221620738017,
        -0.000665597590164,
        -0.00116694100018,
        -0.000327282767638,
        -0.0488417164061,
        -0.000166705813746,
        -7.47724567053e-05,
        -3.32109228223e-05,
        -0.00128461574961,
        -0.00148809490117,
        -0.000756305174476,
        -0.000800433198754,
        -0.000376926787155,
        -0.999999999621,
        0.157158972235,
        -0.330033841529,
        0.999999999207,
        0.908861758032,
        0.027502817362,
        0.110011298139,
        0.0202061613497,
        0.0157159191778,
        0.00785795152005,
        0.0242882192597,
        0.00390797999102,
        0.00312638391562,
        -0.0010746942405,
        0.00390798043517,
        0.000390797990784,
        -0.000537347034782,
        -0.020130330377,
        -0.000225930004784,
        9.46464033296e-05,
        0.00508037474934,
        -0.00156319169695,
        0.0013677930237,
        0.0,
        -0.000586196856904,
        -0.999999999998,
        1.0,
        1.00000000008,
        1.0,
        0.0,
        0.999999817449,
        1.00000007215,
        1.00000094635,
        1.00000441643,
        1.00000234018,
        1.00000121787,
        0.497328581554,
        0.795726073625,
        0.130548751394,
        0.497328938942,
        0.0528411566898,
        0.0,
        0.00971344731981,
        -0.0118889974899,
        0.00796503060523,
        0.64652776838,
        0.186498161217,
        0.174065025146,
        0.167848403013,
        0.0714909842611,
        -1.0,
        0.157089418917,
        -0.377874259048,
        1.0,
        0.830679701781,
        0.0191204634844,
        0.0797213175056,
        0.0103874900011,
        0.00370221119156,
        -0.000282246686807,
        -0.0030793486728,
        -0.00303337142002,
        -0.00312153449683,
        -0.000972951186018,
        -0.00175008687497,
        -0.000477036443541,
        -0.000236226359234,
        -0.000106432331664,
        -4.97651013982e-05,
        -0.0050268486257,
        -0.00193590978754,
        -0.00210695547623,
        -0.0010918529633,
        -0.0011508003769,
        -0.000548521905516,
        -0.999999999684,
        0.157089409085,
        -0.198905651942,
        0.999999999906,
        0.326398310606,
        0.00701393434424,
        0.0310793096994,
        0.00398503676186,
        0.00139915015457,
        -7.34733157977e-05,
        -0.00120063790773,
        -0.00115693394938,
        -0.00120028614685,
        -0.000369116798666,
        -0.000663865191454,
        -0.00017954356289,
        -8.87912071124e-05,
        -3.96337295056e-05,
        -1.84495853736e-05,
        -0.000731350228072,
        -0.417071920355,
        -0.000806684218814,
        -0.000410930514869,
        -0.000437235037133,
        -0.000204960385378,
        -1.0,
        0.157089410136,
        -0.204552062852,
        0.99999999983,
        0.402774028546,
        0.00824560054292,
        0.0381483006257,
        0.00484216141264,
        0.00168431496028,
        -6.06665989858e-05,
        -0.00147210794032,
        -0.00139766287691,
        -0.00145795156829,
        -0.000444241633884,
        -0.000798946795684,
        -0.000214897263112,
        -0.000106169312607,
        -4.70793696113e-05,
        -2.1846098473e-05,
        -0.000877677415205,
        -0.000977012736891,
        -0.386781785375,
        -0.000491853629796,
        -0.000526767858205,
        -0.00024407739834,
        -0.999999999899,
        0.157089402446,
        -0.23258476374,
        1.00000000004,
        0.440076908049,
        0.0095751827466,
        0.0419602584462,
        0.00540065292132,
        0.00190352799844,
        -0.000107734574191,
        -0.00161790471113,
        -0.00156611271588,
        -0.00162245895122,
        -0.000500212403978,
        -0.00089940563867,
        -0.00024365757096,
        -0.000120551049501,
        -5.3909066073e-05,
        -2.51162669314e-05,
        -0.000991770481529,
        -0.00109103245466,
        -0.00055773356624,
        -0.163029718215,
        -0.000592449775896,
        -0.000278593136846,
        -1.00000000005,
        0.15709022924,
        -0.236334888962,
        1.00000000007,
        0.50255674817,
        0.0105656836773,
        0.0477353007349,
        0.00609819007472,
        0.00213570492922,
        -9.62546399131e-05,
        -0.00183951518869,
        -0.00176139847817,
        -0.00183183121747,
        -0.000561254192671,
        -0.00100910583816,
        -0.000272288117572,
        -0.000134636300593,
        -5.99269534761e-05,
        -2.78534127126e-05,
        -0.00111048798016,
        -0.00122960034448,
        -0.000623178851648,
        -0.000665081541766,
        -0.158340292242,
        -0.000310245667019,
        0.157073016933,
        -0.265910812123,
        0.52309893228,
        0.0117250508196,
        0.0501244110251,
        0.00652733592366,
        0.00237399033363,
        -0.00017991499394,
        -0.00186849738697,
        -0.00185646266839,
        -0.00191909631029,
        -0.000597212104458,
        -0.00106351993863,
        -0.00029093339923,
        -0.000145701561119,
        -6.52172982936e-05,
        -3.02011664406e-05,
        -0.00116122853267,
        -0.00130152797409,
        -0.000671746182079,
        -0.000709326400082,
        -0.000336186229589,
        -0.0653513498121,
        -0.999999999182,
        -0.15708942793,
        0.999999999953,
        1.61902172076e-05,
        0.0198122057941,
        0.0276395256823,
        0.0905929233476,
        0.0528505470791,
        0.0237998094454,
        0.0174882927447,
        0.00714882867783,
        0.00533845604588,
        8.73911535843e-06,
        0.000218477071987,
        1.29200178157e-08,
        5.45864080432e-11,
        3.05550259616e-13,
        2.0305856422e-15,
        0.000698759778692,
        0.000439228579359,
        9.47305738526e-06,
        7.1048090431e-06,
        7.31159976623e-08,
        0.0192528982521,
        1.0,
        -1.50506975084e-05,
        1.0,
        -0.0196507155475,
        0.999999997502,
        0.999999996048,
        0.999999999978,
        0.744096173611,
        0.00346912595628,
        0.0317280293481,
        0.0100988047627,
        0.00249930560362,
        0.0125831821073,
        0.000738893300855,
        -0.000588405693134,
        -0.000647834216552,
        -0.000556015904337,
        -0.000402748259683,
        -0.000350859391843,
        -0.000102275809324,
        -0.000197403725622,
        -5.29517089329e-05,
        -2.61759215865e-05,
        -1.2865910284e-05,
        -6.27192981742e-06,
        -0.000235366637205,
        -0.000216534117147,
        -0.000127248553767,
        -0.000120342420809,
        -6.60239801884e-05,
        -1.0,
        0.999999971123,
        0.744095763596,
        0.0020032010513,
        0.0125871482455,
        0.000996066468275,
        0.0571888663177,
        0.00499493643225,
        0.000293520446922,
        -0.000240619143918,
        -0.000262076440825,
        -0.00022321579576,
        -0.000161581299139,
        -0.000140578546605,
        -4.09943101424e-05,
        -7.9144817413e-05,
        -2.12411382355e-05,
        -1.0502541909e-05,
        -5.16747113117e-06,
        -2.52048119026e-06,
        -9.44178592624e-05,
        -8.67321064305e-05,
        -5.10478145039e-05,
        -4.82298409519e-05,
        -2.65032092595e-05,
        -1.00000000006,
        0.999999999967,
        0.744095765501,
        0.00310940805253,
        0.0224936396192,
        0.00181028047748,
        0.00894450522469,
        0.0204133000984,
        0.000526875153848,
        -0.000482916365625,
        -0.000505395432931,
        -0.00041806661661,
        -0.000301823983129,
        -0.000261273592521,
        -7.62927329637e-05,
        -0.00014745386994,
        -3.96533507325e-05,
        -1.96227989691e-05,
        -9.69311864111e-06,
        -4.73824544341e-06,
        -0.000176289517737,
        -0.000160990745158,
        -9.53228371174e-05,
        -8.97134000368e-05,
        -4.96069827402e-05,
        -0.999999994391,
        0.999999999602,
        0.744095765854,
        0.00246463048965,
        0.00492585795637,
        0.000390119721407,
        0.00195489875101,
        0.000114885569806,
        0.0780024776003,
        -9.48042558281e-05,
        -0.000103005501467,
        -8.75827621255e-05,
        -6.33888862288e-05,
        -5.51335967579e-05,
        -1.60786741505e-05,
        -3.10439552301e-05,
        -8.33258871542e-06,
        -4.12018838356e-06,
        -2.02767560948e-06,
        -9.89140836308e-07,
        -3.70392627307e-05,
        -3.40129098276e-05,
        -2.00256537452e-05,
        -1.89160456122e-05,
        -1.03983793387e-05,
        -1.00000000032,
        1.0,
        0.744095766059,
        0.00241028771263,
        -0.0105017627485,
        -0.000818047483202,
        -0.0041593682286,
        -0.000243835262688,
        0.000178810423449,
        0.0216259636806,
        0.000203254823226,
        0.000178260684484,
        0.000129366640649,
        0.00011309904627,
        3.29371322181e-05,
        6.35231097029e-05,
        1.70152660646e-05,
        8.40621996349e-06,
        4.1200947132e-06,
        2.00531230706e-06,
        7.56234322005e-05,
        6.98624884164e-05,
        4.08815317542e-05,
        3.87694091379e-05,
        2.11762432986e-05,
        -0.999999998432,
        1.0,
        0.744095759542,
        0.00283712642109,
        -0.0230483027755,
        -0.00181898098236,
        -0.00914310444742,
        -0.00053703601807,
        0.000432756105958,
        0.00047436162283,
        0.043770785264,
        0.000405866323043,
        0.000293911099147,
        0.000255904191066,
        7.4608068698e-05,
        0.000144017095371,
        3.86396174741e-05,
        1.91026282512e-05,
        9.39315751605e-06,
        4.58005193735e-06,
        0.000171752235071,
        0.00015791346326,
        9.28571692596e-05,
        8.77832823956e-05,
        4.819233045e-05,
        -0.99999999958,
        0.999999998338,
        0.744095758207,
        0.00260464535891,
        -0.0255562965915,
        -0.00202749213775,
        -0.0101444597665,
        -0.000596303271098,
        0.000498181269742,
        0.000538825121204,
        0.000456677698995,
        0.0297023858692,
        0.000330426946896,
        0.000287237923416,
        8.37793119225e-05,
        0.000161776464793,
        4.34321535291e-05,
        2.14777073216e-05,
        1.0574403134e-05,
        5.15963248955e-06,
        0.000193064501634,
        0.000177177366694,
        0.00010438333472,
        9.85583604202e-05,
        5.42151819816e-05,
        -0.999999999801,
        1.0,
        0.744095742568,
        0.00276293615272,
        -0.0384112128566,
        -0.00303292401669,
        -0.0152383675236,
        -0.000895112009835,
        0.000723867392175,
        0.000792406928749,
        0.000677359132099,
        0.000490473494551,
        0.121780833134,
        0.000426981609086,
        0.0001244902642,
        0.000240313545807,
        6.4479821328e-05,
        3.1878315768e-05,
        1.56771560042e-05,
        7.64461899519e-06,
        0.000286612868679,
        0.000263471479477,
        0.000154956566407,
        0.000146472005875,
        8.04274761325e-05,
        -0.999999999822,
        0.999999999456,
        0.744095758883,
        0.00254634916938,
        -0.0412487997874,
        -0.003252198132,
        -0.0163611656532,
        -0.000960844398849,
        0.000769206221454,
        0.000845238347387,
        0.000724446090585,
        0.000524691194922,
        0.000456974152682,
        0.174277360873,
        0.000133218622067,
        0.000257137758879,
        6.89816100499e-05,
        3.41014106871e-05,
        1.67644885984e-05,
        8.17322357957e-06,
        0.000306619478578,
        0.000282010067925,
        0.000165771283619,
        0.000156748609167,
        8.60225074477e-05,
        -0.9999999999,
        0.999999998639,
        0.744125862655,
        0.00257774416145,
        -0.072569255767,
        -0.00571404836849,
        -0.0287807126802,
        -0.00168467061447,
        0.00134905125121,
        0.00148400925563,
        0.00127394068581,
        0.000922462404245,
        0.000803476620712,
        0.000234181811152,
        1.20954446012,
        0.000452022154029,
        0.000121237139269,
        5.9928481463e-05,
        2.94515271656e-05,
        1.43569122806e-05,
        0.000538920085808,
        0.000495856908116,
        0.000291300404434,
        0.000275554739403,
        0.000151160077264,
        -0.999999999892,
        0.999999996902,
        0.744095594418,
        0.00250293539698,
        -0.0576231919919,
        -0.00453885060389,
        -0.0228533369735,
        -0.00134131342849,
        0.00106828641523,
        0.00117634628091,
        0.0010098292549,
        0.000731460520856,
        0.000637207653676,
        0.000185748398551,
        0.00035850601996,
        1.16049557212,
        9.61637869852e-05,
        4.75373880021e-05,
        2.33644632599e-05,
        1.13894557039e-05,
        0.000427444730356,
        0.000393267380872,
        0.000231092741132,
        0.000218557383097,
        0.000119902586949,
        -0.99999999998,
        0.999999977978,
        0.755218199829,
        0.00259331133953,
        -0.0847130568882,
        -0.00693811887549,
        -0.0343601997981,
        -0.00165193277938,
        0.00198232376,
        0.0020649186654,
        0.00177582905016,
        0.00123895041095,
        0.00105310774077,
        0.000320062170997,
        0.00059882599946,
        0.000154868795441,
        1.34865847346,
        7.74343993379e-05,
        3.87172027844e-05,
        1.87939762019e-05,
        0.000712396474318,
        0.000660773474611,
        0.000371685110515,
        0.000356198209424,
        0.000202620022441,
        -0.999999996802,
        0.999999976226,
        1.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0209986427764,
        0.0104993277265,
        0.0104993245077,
        0.00787449184605,
        0.00393724567971,
        0.00262483035177,
        0.000984311406119,
        0.00131241517313,
        0.0,
        0.000205064874883,
        1.31760640456,
        8.20259493427e-05,
        0.0,
        0.0013124151643,
        0.00131241515496,
        0.0,
        0.00131241522216,
        0.000574181658357,
        -0.999999998531,
        0.999999952262,
        0.0,
        0.999999977472,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.30368475183,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        -1.0,
        1.0,
        0.7440957416,
        0.00251027569022,
        -0.135955117305,
        -0.0107110622596,
        -0.0539212066979,
        -0.00316638306678,
        0.00252014125222,
        0.00277532414892,
        0.00238229329626,
        0.00172565490806,
        0.0015033196911,
        0.000438224845345,
        0.000845812553419,
        0.000226881474744,
        0.000112155326706,
        5.5125376493e-05,
        2.68724233875e-05,
        1.29379936346,
        0.00100846540019,
        0.00092779672842,
        0.000545216472032,
        0.000515639919703,
        0.000282891782256,
        -1.0,
        0.999999997357,
        0.74409716622,
        0.00298893184654,
        -0.0539227312857,
        -0.00424747415257,
        -0.0213857832794,
        -0.00125571997632,
        0.000998826073494,
        0.00110024075196,
        0.000944616935655,
        0.000684257631873,
        0.000596106735027,
        0.000173766275857,
        0.000335379547903,
        8.99613348888e-05,
        4.44710378831e-05,
        2.18570772773e-05,
        1.06546550044e-05,
        0.000399871099915,
        1.20079543955,
        0.000367899412011,
        0.000216182700259,
        0.000204465121478,
        0.000112169404278,
        -0.99999999965,
        1.0,
        0.744097958161,
        0.00292426323596,
        -0.0539090830639,
        -0.00424625061422,
        -0.0213800044562,
        -0.00125513389314,
        0.000999033543721,
        0.00110026575548,
        0.00094455585631,
        0.000684203305459,
        0.000596044755013,
        0.000173746641107,
        0.000335347070784,
        8.99525080919e-05,
        4.44663703549e-05,
        2.1855089788e-05,
        1.06537534688e-05,
        0.000399833354465,
        0.000367859550179,
        1.16425661131,
        0.000216163061598,
        0.000204440920224,
        0.000112158221349,
        -0.999999999479,
        0.999999993047,
        0.744075877907,
        0.00305096247454,
        -0.0690498305469,
        -0.005439932237,
        -0.0273891165433,
        -0.00160525387873,
        0.00128217140725,
        0.00141115050884,
        0.00121155161953,
        0.000877476333476,
        0.000764144499409,
        0.000222765350087,
        0.000429910557591,
        0.000115314134685,
        5.70161389775e-05,
        2.80182869477e-05,
        1.36577065693e-05,
        0.000512530757643,
        0.000471669997048,
        0.00027706579644,
        1.00303802052,
        0.000262159283564,
        0.000143799073381,
        -0.999999997808,
        0.999999992715,
        0.74413588039,
        0.00299702436577,
        -0.0686797523703,
        -0.00540763391188,
        -0.0272343361884,
        -0.00159286331949,
        0.00127674637947,
        0.00140468609584,
        0.00120584350971,
        0.000873063696441,
        0.000760315644024,
        0.000221680994906,
        0.000427735881313,
        0.0001147220474,
        5.67196542301e-05,
        2.78746370624e-05,
        1.35864395013e-05,
        0.000510031251929,
        0.000469316644946,
        0.000275706076641,
        0.000260779608607,
        1.00015457939,
        0.000143092294936,
        -1.00000000001,
        0.743083044341,
        0.00284127244392,
        -0.0840400979014,
        -0.00661175287423,
        -0.0342047861028,
        -0.0018512901422,
        0.00165294381972,
        0.00177415708025,
        0.00155376343237,
        0.00110195976163,
        0.000958704837004,
        0.000280310981824,
        0.000537205336586,
        0.000144287853051,
        7.16273851357e-05,
        3.51249721076e-05,
        1.69598527301e-05,
        0.000641891550369,
        0.000592303252333,
        0.000344362418356,
        0.000332654061583,
        0.000180445922722,
        1.36925540692,
        0.999999999302,
        -0.744095745788,
        -1.74667346165e-06,
        -0.000378369545626,
        -0.0025462703813,
        -0.012449643066,
        -0.00708443061106,
        -0.00378576709639,
        -0.00298768183693,
        -0.00148153450616,
        -0.00122039580369,
        -2.1434901666e-05,
        -0.00014014277413,
        -9.76054969304e-08,
        -1.81301199747e-09,
        -4.18364920299e-11,
        -1.1022860071e-12,
        -0.000335509837414,
        -0.000238680316829,
        -1.37514839684e-05,
        -1.15557884712e-05,
        -3.48249827554e-07,
        -0.999999998423,
        -1.0,
        0.0157930132889,
        -1.0,
        -0.0466850120722,
        0.0359630436632,
        0.0183844209413,
        0.0256672797658,
        0.0149206927029,
        0.00897368651446,
        0.00764528488487,
        0.004347661314,
        0.00375217792653,
        0.000851595420888,
        0.00177838829978,
        0.000405641425846,
        0.000191616974185,
        8.86282349708e-05,
        4.15880581791e-05,
        0.00215436237792,
        0.00203348354789,
        0.0010538399217,
        0.00101696222875,
        0.00050292897061,
        -1.0,
        -0.00127531548332,
        -1.0,
        -0.0746345596645,
        -0.0125875879656,
        -0.0382467371162,
        -0.00871737813034,
        -0.00274041709796,
        -0.00113167352858,
        -0.00084077401152,
        -0.000319811398639,
        -0.00024503892509,
        -1.20870173061e-05,
        -6.23417390986e-05,
        1.037541909e-06,
        2.90782305696e-06,
        2.4095543866e-06,
        1.51279997645e-06,
        -8.96539806438e-05,
        -8.31426231172e-05,
        -1.97607769233e-05,
        -1.94413193399e-05,
        -3.02628041567e-07,
        -0.00364141453752,
        0.999999997788,
        -1.00000000076,
        -0.99999999094,
        0.81090846599,
        0.0711966751677,
        0.999999995662,
        -0.999982413733,
        -1.07784692045,
        -0.0409624829181,
        -0.374206212546,
        -0.0433427715411,
        -0.0049356928524,
        -0.000397626889793,
        0.00227026992693,
        0.00207664036851,
        0.00223028800353,
        0.000714228080079,
        0.00124546715117,
        0.00035024174347,
        0.000176567185614,
        7.89294039518e-05,
        3.72028102424e-05,
        0.00134654071062,
        0.00153832200033,
        0.000774159559059,
        0.000843254584472,
        0.00038964843911,
        -0.999999985295,
        0.810897461265,
        -0.123587687816,
        0.99999999964,
        -0.205208663273,
        -0.0105468273857,
        -0.972298255651,
        -0.0788963246648,
        -0.00955917088636,
        -0.00111952679596,
        -1.82973488411e-05,
        0.000482919509127,
        0.000477064782346,
        0.000498874645877,
        0.000167262866598,
        0.000291794498256,
        8.42185175295e-05,
        4.27247955548e-05,
        1.97327587227e-05,
        9.43911049681e-06,
        0.000319617545634,
        0.00034900300953,
        0.000186247680872,
        0.000196386038551,
        9.62200596087e-05,
        -0.999999988689,
        0.810897434967,
        0.00427313144129,
        0.999999998963,
        -0.660455888815,
        -0.0286853575982,
        -0.249312958665,
        -0.949813073759,
        -0.0287216884623,
        -0.00315391505382,
        -0.000234820272089,
        0.00157044607655,
        0.00144564438556,
        0.00154997428272,
        0.000499513329051,
        0.000870548671311,
        0.000245669498685,
        0.000124193860599,
        5.57676626632e-05,
        2.63242058988e-05,
        0.000941312543232,
        0.001070866788,
        0.000542719162732,
        0.000589106874713,
        0.000274428781054,
        -0.999999999998,
        0.81089743528,
        -0.111301801476,
        0.99999999377,
        -0.281937356944,
        -0.0128610931644,
        -0.106995377105,
        -0.0124519530837,
        -1.03120304827,
        -0.00137765711104,
        -8.27228765089e-05,
        0.000673223832546,
        0.000630825416675,
        0.000672175270819,
        0.000218892385611,
        0.000381509523211,
        0.000108331776566,
        5.48251417669e-05,
        2.48058137457e-05,
        1.17524177254e-05,
        0.000413868762599,
        0.000465934587451,
        0.000239328516147,
        0.000257846628572,
        0.000121724258669,
        -0.999999998785,
        0.810897431794,
        -0.184935677978,
        1.0,
        -0.0851975854194,
        -0.00403537529334,
        -0.0324649586069,
        -0.00381355087507,
        -0.000425935104703,
        -0.10740937994,
        -2.04305483128e-05,
        0.000203599409915,
        0.000193594499395,
        0.000205227058577,
        6.73958746216e-05,
        0.000117479387138,
        3.3522327647e-05,
        1.69791154205e-05,
        7.72859521446e-06,
        3.67217992679e-06,
        0.000127784800304,
        0.000142637057607,
        7.40664829378e-05,
        7.93142282266e-05,
        3.78447854917e-05,
        -0.999999999376,
        0.81089743418,
        -0.224248093374,
        1.0,
        -0.0134290249362,
        -0.000128051319352,
        -0.00461511186368,
        -0.000440368826817,
        -4.09682978688e-05,
        -1.72974810362e-05,
        -0.0910342216457,
        2.88026179043e-05,
        1.86858004019e-05,
        2.29544530259e-05,
        5.80496671567e-06,
        1.01016847208e-05,
        2.38326087914e-06,
        1.16315091761e-06,
        3.92334899965e-07,
        1.55290868722e-07,
        9.98235454192e-06,
        1.47786171602e-05,
        5.25918943771e-06,
        7.0605366603e-06,
        2.16724666926e-06,
        -0.999999999643,
        0.810897432169,
        -0.252864109067,
        0.999999999397,
        0.108223713975,
        0.00437732308976,
        0.0403525129905,
        0.00473353890014,
        0.000560393960021,
        3.74249366348e-05,
        -0.000238873478319,
        -0.0412189293248,
        -0.000218657863207,
        -0.000234398564804,
        -7.51439964181e-05,
        -0.00013119719289,
        -3.68718380113e-05,
        -1.86088356366e-05,
        -8.34144414036e-06,
        -3.93216122443e-06,
        -0.000141909254372,
        -0.000161612124473,
        -8.16371848817e-05,
        -8.86743081099e-05,
        -4.11960086008e-05,
        -1.0,
        0.810897434906,
        -0.300473204242,
        1.0,
        0.200975573004,
        0.00896382546798,
        0.0758568752183,
        0.00901318234433,
        0.00106334903247,
        5.01583346112e-05,
        -0.000455563118279,
        -0.00042968263019,
        -0.0715610623908,
        -0.000456097366981,
        -0.00014886071813,
        -0.000259815724129,
        -7.38135290796e-05,
        -3.73322063044e-05,
        -1.69423308562e-05,
        -8.03549589061e-06,
        -0.000282467023243,
        -0.000316292563804,
        -0.000163351386067,
        -0.000175307610841,
        -8.32337040195e-05,
        -0.999999999906,
        0.810897431538,
        -0.323229438621,
        1.00000000015,
        0.2646431089,
        0.0115593691769,
        0.0996817135656,
        0.011746557038,
        0.00136760652901,
        7.58284240728e-05,
        -0.000602992434538,
        -0.000563261576737,
        -0.000600029312094,
        -0.0960146058998,
        -0.000194792762075,
        -0.000339898462228,
        -9.62755138667e-05,
        -4.86727125107e-05,
        -2.2000873816e-05,
        -1.04152804153e-05,
        -0.000368835268879,
        -0.000415415364445,
        -0.0002129986807,
        -0.000229538658653,
        -0.000108205293568,
        -1.0,
        0.810897607098,
        -0.451573765101,
        0.999999999994,
        0.507641658545,
        0.0234000700943,
        0.192493349022,
        0.0228652002363,
        0.00266100136149,
        0.000115949098961,
        -0.0011712835603,
        -0.00111337647026,
        -0.00117925069948,
        -0.000386784198674,
        -0.124139456569,
        -0.00067482616654,
        -0.000192313190302,
        -9.73399595979e-05,
        -4.43086111081e-05,
        -2.10479953419e-05,
        -0.000734404005636,
        -0.000819135670029,
        -0.000425382334211,
        -0.00045524495394,
        -0.000217287553615,
        -0.999999999356,
        0.810897454118,
        -0.385058393418,
        1.00000000035,
        0.363938422526,
        0.0167984178492,
        0.138031328028,
        0.0163919116864,
        0.00190541984528,
        8.28142166194e-05,
        -0.000840748902125,
        -0.00079931622007,
        -0.000846563211003,
        -0.000277706090462,
        -0.000484498576763,
        -0.231968488565,
        -0.000138088634415,
        -6.98964788107e-05,
        -3.18189625655e-05,
        -1.51156688207e-05,
        -0.000527305593225,
        -0.000588078831263,
        -0.000305432353915,
        -0.000326855473855,
        -0.000156027933178,
        -0.999999999938,
        0.810910858442,
        -0.508997734119,
        1.00000000053,
        0.58459723843,
        0.027693981723,
        0.222348368839,
        0.0265417915503,
        0.00309945054786,
        9.46236279928e-05,
        -0.00136547966647,
        -0.00130178902423,
        -0.00137530325334,
        -0.000452504510056,
        -0.000790097907597,
        -0.000225569699646,
        -0.0371047276161,
        -0.000114365603791,
        -5.22924350688e-05,
        -2.48923466333e-05,
        -0.000859063349265,
        -0.000956779468589,
        -0.000499224379571,
        -0.000531887393367,
        -0.000255525911638,
        -0.999999998971,
        0.811047672432,
        -0.589852844214,
        1.00000000027,
        0.701141318221,
        0.0345560548759,
        0.267131001522,
        0.0326148827201,
        0.00423867011276,
        0.000372092938136,
        -0.00142365730889,
        -0.00134276733671,
        -0.00154499106367,
        -0.000498481923192,
        -0.000893829718763,
        -0.000254802069477,
        -0.000134226062099,
        -0.0155525009157,
        -6.05407521455e-05,
        -2.77425786472e-05,
        -0.000910007779137,
        -0.0010839202756,
        -0.000521737733265,
        -0.000588471577856,
        -0.000281596755586,
        -0.999999997183,
        0.811488676049,
        -0.612444282728,
        0.999999998007,
        0.729309971378,
        0.0472041957744,
        0.283225325139,
        0.0472042398407,
        0.00413037328573,
        -0.00236020921279,
        -0.00501544391477,
        -0.000885078710526,
        -0.00073756560966,
        -0.000221269681125,
        -0.000848200348443,
        -0.000101415270701,
        -0.000119854392595,
        -8.64334417821e-05,
        -0.00661356963533,
        -3.68782696031e-05,
        -0.000368782809408,
        -0.000958835197602,
        0.0,
        -0.000553174138883,
        -0.000248928362178,
        -0.999999999994,
        1.0,
        -0.999999999661,
        1.0,
        0.760096601008,
        0.115612642589,
        0.0,
        0.0770751100827,
        0.0674408824438,
        -0.0385375190196,
        -0.0818922071885,
        0.0144515844005,
        0.00963438879772,
        0.0,
        0.00421504500284,
        0.00150537329683,
        -0.000677417744862,
        0.000282257487563,
        0.000122311576822,
        -0.00287210938264,
        0.0276988845995,
        -0.00602149121466,
        0.0048171948508,
        0.00481719476525,
        0.00210752269921,
        -0.999999999858,
        0.810897423173,
        -0.360046978374,
        1.00000000053,
        0.308287364474,
        0.0144429557573,
        0.117135574206,
        0.0139528012922,
        0.00162495721622,
        6.4200180441e-05,
        -0.000713607179054,
        -0.000682081053852,
        -0.00072108496042,
        -0.000237268900372,
        -0.000413959887051,
        -0.000118194378602,
        -5.98449944662e-05,
        -2.73009976031e-05,
        -1.29823719069e-05,
        -0.000450954084557,
        -0.294802936191,
        -0.000501400649878,
        -0.000261429312264,
        -0.000279167218952,
        -0.000133767371488,
        -0.999999999589,
        0.810897443746,
        -0.37070908015,
        1.0,
        0.383221354384,
        0.0171402669039,
        0.144813875858,
        0.017073929122,
        0.00197231924652,
        0.000103303601908,
        -0.000883134276146,
        -0.000829841747373,
        -0.000882449895629,
        -0.000287542987277,
        -0.00050162791167,
        -0.000142416151434,
        -7.20379867807e-05,
        -3.26399028572e-05,
        -1.54705005049e-05,
        -0.000544810072899,
        -0.00061170135245,
        -0.275664956518,
        -0.000314984573146,
        -0.00033869182125,
        -0.000160323060152,
        -0.999999999973,
        0.810897589368,
        -0.421387709876,
        0.999999999512,
        0.420589422736,
        0.0199297985325,
        0.1600138263,
        0.0191245854086,
        0.00223656063654,
        8.02967678633e-05,
        -0.000973297139781,
        -0.000934464161564,
        -0.000986293790132,
        -0.000325385468301,
        -0.000567765448871,
        -0.000162341452972,
        -8.22170854297e-05,
        -3.75762506258e-05,
        -1.78840362085e-05,
        -0.0006190163702,
        -0.000686383217821,
        -0.000359088543332,
        -0.153374375044,
        -0.000382729949837,
        -0.000184001621229,
        -0.999999999438,
        0.810897118064,
        -0.428589945463,
        0.999999999869,
        0.481970898693,
        0.0221028471332,
        0.182649846992,
        0.021669403297,
        0.00251942227822,
        0.000113256168015,
        -0.00111184138203,
        -0.00105462493139,
        -0.00111780867331,
        -0.000366221969386,
        -0.000638961442688,
        -0.000181969410936,
        -9.20929134129e-05,
        -4.1891762309e-05,
        -1.98911006602e-05,
        -0.00069512735512,
        -0.000776234171502,
        -0.000402498167975,
        -0.000431072559468,
        -0.153171171039,
        -0.000205475165553,
        0.810900941201,
        -0.482041986515,
        0.502154562977,
        0.0244071665911,
        0.191634868633,
        0.0230488521101,
        0.00271348725978,
        7.93717160689e-05,
        -0.00116474077037,
        -0.00112641198497,
        -0.00118634751964,
        -0.000393268078716,
        -0.000686304258935,
        -0.000196863183962,
        -9.97438782537e-05,
        -4.57511245304e-05,
        -2.18079268822e-05,
        -0.000748652480505,
        -0.000827411760095,
        -0.000435042963959,
        -0.000462360655289,
        -0.000223671136302,
        -0.049220003159,
        -0.999999997991,
        -0.810897451274,
        0.999999999928,
        1.54715467887e-05,
        0.0182040410577,
        0.026086778923,
        0.084102586633,
        0.03538144554,
        0.0160019366766,
        0.0119056632337,
        0.00493551469652,
        0.00386159104062,
        4.4513478191e-05,
        0.000346558010742,
        1.66795068056e-07,
        2.57627859081e-09,
        5.12850589468e-11,
        1.1319902795e-12,
        0.000892819742075,
        0.000608931448577,
        3.04395678501e-05,
        2.5242093309e-05,
        6.40182861104e-07,
        0.0603026948557,
        1.0,
        -4.85323562473e-05,
        1.0,
        -0.0368902245204,
        0.999999996157,
        0.99999999607,
        -0.9999999991,
        0.999825975504,
        0.248271880483,
        0.999999998105,
        0.366864338696,
        0.435167472501,
        0.0676888934916,
        0.166390855433,
        0.0390573958134,
        0.00730341776867,
        -0.00197139601918,
        -0.00279171095606,
        -0.00498803396815,
        -0.00498141180609,
        -0.00356074554831,
        -0.0046638674615,
        -0.00258662534256,
        -0.00177127626349,
        -0.00119160075103,
        -0.000785993533006,
        -0.00494833069306,
        -0.0046770948108,
        -0.00397090283571,
        -0.00379559185548,
        -0.00292401199884,
        -0.999999999225,
        0.999999996682,
        0.229058650679,
        1.00000000027,
        0.228401711546,
        0.0421663398878,
        0.73345705404,
        0.103625613453,
        0.024342893467,
        0.00457735011687,
        -0.0012241968768,
        -0.00175383526492,
        -0.00316922381037,
        -0.00312165467146,
        -0.00221173779079,
        -0.0029010607291,
        -0.00160171727773,
        -0.00109807278796,
        -0.000738527268827,
        -0.000487067467907,
        -0.00308375547865,
        -0.00288686995579,
        -0.00247207076555,
        -0.00234811982988,
        -0.00181363853535,
        -0.999999999441,
        0.999998042021,
        0.261809798716,
        0.999999998087,
        0.293745665748,
        0.0541910321588,
        0.133299767963,
        0.726440179245,
        0.0312980117926,
        0.00589702808832,
        -0.00159784010198,
        -0.00229441810766,
        -0.00414282866469,
        -0.00404946178629,
        -0.00284381597949,
        -0.00374010155727,
        -0.00205563444377,
        -0.00141024850381,
        -0.000947623396963,
        -0.000625665341437,
        -0.0039779742711,
        -0.00370375174287,
        -0.00318909232717,
        -0.00301644222057,
        -0.0023308175675,
        -0.999999999483,
        0.999998007948,
        0.303291198121,
        0.999999999548,
        0.168341750164,
        0.0310517264952,
        0.0763881832392,
        0.0179358960625,
        0.78566521349,
        0.00338052188257,
        -0.000917150764904,
        -0.00131804669772,
        -0.00238025998134,
        -0.00232353888485,
        -0.0016295866145,
        -0.00214411481253,
        -0.00117756873158,
        -0.000807963098046,
        -0.000542851634516,
        -0.000358472867291,
        -0.00228067115014,
        -0.0021216257279,
        -0.0018282768787,
        -0.00172811588298,
        -0.00133556756136,
        -0.999999998538,
        0.999997998064,
        0.376252602647,
        0.999999998572,
        0.0617808948412,
        0.0114037364389,
        0.0280412425868,
        0.00658388060074,
        0.00123942870055,
        0.295111943738,
        -0.000334700415614,
        -0.000479434979077,
        -0.00086538016063,
        -0.00084893153989,
        -0.000598357689232,
        -0.000786011558208,
        -0.000432891524581,
        -0.000296879114552,
        -0.000199554812387,
        -0.000131699582113,
        -0.000835821049378,
        -0.000780000368234,
        -0.000670180927965,
        -0.000635065444063,
        -0.000490487270545,
        -0.999999999481,
        0.999998006651,
        0.469877662677,
        0.999999999815,
        -0.027128371561,
        -0.00502168953213,
        -0.0123258073481,
        -0.00289368703248,
        -0.000542055227853,
        0.000143551413479,
        0.443138303374,
        0.00020277998248,
        0.000365232466921,
        0.000365904325191,
        0.000263294809673,
        0.000343568817773,
        0.000191401881184,
        0.000131011842655,
        8.82230290719e-05,
        5.80858416457e-05,
        0.000364887132068,
        0.000344981624862,
        0.00029285315983,
        0.000280403583772,
        0.000215989056938,
        -0.99999999874,
        0.999997999699,
        0.461093359564,
        0.999999999233,
        -0.0459780821086,
        -0.00851745664461,
        -0.0208962723268,
        -0.004905227364,
        -0.000917276001136,
        0.000241245960571,
        0.0003392002522,
        0.240965428482,
        0.000610403831142,
        0.000616110219377,
        0.000446486275741,
        0.000581289862993,
        0.000325094830001,
        0.000222377183071,
        0.000149837620625,
        9.85727170657e-05,
        0.000617090668915,
        0.000586038660973,
        0.000495424252324,
        0.000476052342401,
        0.000366348024202,
        -0.999999997648,
        0.999998014249,
        0.541017472861,
        0.999999997725,
        -0.136801462606,
        -0.0253546824338,
        -0.0621846922277,
        -0.0145973169674,
        -0.00272764383782,
        0.000715203167089,
        0.00100326185395,
        0.00180481357973,
        0.425247050485,
        0.00182789019021,
        0.001328935633,
        0.001728394909,
        0.000968333591629,
        0.000662183997418,
        0.000446303937815,
        0.000293501449535,
        0.00183449459025,
        0.00174564799506,
        0.00147302321605,
        0.00141767253745,
        0.00109054086097,
        -0.9999999959,
        0.999998016579,
        0.535942688039,
        0.999999996136,
        -0.155613809235,
        -0.0288368009281,
        -0.0707320209778,
        -0.0166037888414,
        -0.00310338480599,
        0.000814592749511,
        0.00114359296172,
        0.0020575074212,
        0.00208134560384,
        0.476794760006,
        0.00151150860011,
        0.00196654744239,
        0.00110108579371,
        0.000753041345177,
        0.000507491885256,
        0.000333782072605,
        0.00208740555541,
        0.00198493614577,
        0.00167601390736,
        0.00161214526325,
        0.0012403122181,
        -0.999999978718,
        0.999997991239,
        0.692104169507,
        0.999999978767,
        -0.376250640015,
        -0.069759348705,
        -0.171051761395,
        -0.0401523269505,
        -0.00749804627378,
        0.00196097766316,
        0.00274553775206,
        0.00493758121422,
        0.00501510329357,
        0.00365601180277,
        0.324547546719,
        0.004750878059,
        0.00266559266999,
        0.00182239041352,
        0.00122855186416,
        0.000807684759999,
        0.00504170385937,
        0.00480554381115,
        0.00404877873047,
        0.00390183206132,
        0.00300045239321,
        -0.999999986219,
        0.99999796976,
        0.61182842323,
        0.99999998619,
        -0.269324300942,
        -0.0498580798508,
        -0.122372014934,
        -0.0287276737364,
        -0.00537958979924,
        0.00142277064977,
        0.00200828606665,
        0.00361647700081,
        0.00362810364556,
        0.00261405341583,
        0.00340960608518,
        1.03304116569,
        0.00190083658144,
        0.00130093633691,
        0.000876138897781,
        0.000576759966455,
        0.00362087116277,
        0.00342620417251,
        0.00290621920729,
        0.00278451576996,
        0.00214445689579,
        -0.999999999472,
        0.999998014258,
        0.769382683348,
        0.999999999473,
        -0.482761206108,
        -0.0893543823944,
        -0.21933658638,
        -0.0514914372999,
        -0.00964573607991,
        0.0025545365933,
        0.00360929180829,
        0.0065006090764,
        0.00651179892361,
        0.00468505561538,
        0.00611369123587,
        0.00340567570181,
        1.03975394704,
        0.00233115749815,
        0.00156976787066,
        0.00103354297018,
        0.00649306710975,
        0.00613847574346,
        0.00521118975552,
        0.00498939402876,
        0.00384323226663,
        -0.999999891651,
        0.9999979066,
        0.709344344398,
        0.99999989165,
        -0.586720385359,
        -0.108583471071,
        -0.266557672124,
        -0.0625774707487,
        -0.0117249857517,
        0.00310785699822,
        0.00439383551997,
        0.00791438908901,
        0.00792055630187,
        0.00569345889701,
        0.00743173480268,
        0.00413784404097,
        0.00283255663766,
        1.03522518227,
        0.0019072585583,
        0.00125587594367,
        0.00789332816586,
        0.00745803896267,
        0.00633476535795,
        0.00606239652973,
        0.00467029422282,
        -0.999999870354,
        0.999998450303,
        0.922165410771,
        0.999999870361,
        -0.702071498796,
        -0.12991984998,
        -0.318954555957,
        -0.0748785219064,
        -0.0140321595978,
        0.00372228395449,
        0.00526490783622,
        0.00948410467537,
        0.00948420417157,
        0.00681242191269,
        0.00889443201968,
        0.00495022775072,
        0.00338891382056,
        0.00228172252368,
        1.03083046875,
        0.00150257883321,
        0.00944732572427,
        0.00892216181904,
        0.00758165620919,
        0.00725298324064,
        0.00558801016602,
        -0.999999242348,
        0.999996110636,
        0.999999030164,
        0.999999242351,
        -0.810163760097,
        -0.149912982072,
        -0.368053761893,
        -0.0864058445667,
        -0.0161951914004,
        0.00429773973617,
        0.00608148189074,
        0.0109561289068,
        0.0109498123214,
        0.0078607633712,
        0.0102649912913,
        0.005711323936,
        0.00391013401153,
        0.00263254412578,
        0.00173370959307,
        1.02631989867,
        0.0109034721432,
        0.0102937057948,
        0.0087500219837,
        0.00836838573323,
        0.00644786753468,
        -0.99999999799,
        0.999997987204,
        0.6332243967,
        0.999999998014,
        -0.24365212562,
        -0.0451056051452,
        -0.110707438293,
        -0.0259893686401,
        -0.00486682883783,
        0.00128720950195,
        0.00181694922793,
        0.00327192845898,
        0.00328236221628,
        0.00236488693309,
        0.00308463320202,
        0.00171964343187,
        0.00117692988983,
        0.000792623499479,
        0.000521783389341,
        0.00327577027577,
        1.02669162957,
        0.00309960693138,
        0.00262922681215,
        0.00251909145583,
        0.00194005089463,
        -0.999999976208,
        0.999997990128,
        0.625015584174,
        0.999999976207,
        -0.244655598204,
        -0.0452900095014,
        -0.111162314003,
        -0.0260959759248,
        -0.00488685916399,
        0.00129258309567,
        0.00182467671501,
        0.00328583080543,
        0.00329606145066,
        0.00237457018282,
        0.00309733970905,
        0.00172665218786,
        0.00118173434077,
        0.00079585261435,
        0.000523913193237,
        0.00328927404377,
        0.00311225124985,
        1.02850722987,
        0.00264005441539,
        0.00252937747218,
        0.00194798431083,
        -0.999999971012,
        0.999998088455,
        0.72041389704,
        0.999999971056,
        -0.348989036864,
        -0.0646043320787,
        -0.158567745815,
        -0.0372249919139,
        -0.00697122872336,
        0.0018441980617,
        0.00260354129591,
        0.00468853327369,
        0.00470238267624,
        0.00338722965498,
        0.00441843900292,
        0.00246292041684,
        0.0016856667095,
        0.00113521909016,
        0.000747333447261,
        0.00469228738344,
        0.00443931884974,
        0.00376612900651,
        1.04070657127,
        0.00360796101545,
        0.00277871639871,
        -0.99999998819,
        0.999998056635,
        0.705042376411,
        0.999999988161,
        -0.347438862195,
        -0.0643152082838,
        -0.157861487633,
        -0.0370591464965,
        -0.00694045882976,
        0.00183631504014,
        0.00259277235884,
        0.00466922807635,
        0.00468218164843,
        0.00337208778011,
        0.00439893380945,
        0.00245181179561,
        0.00167809004681,
        0.00113009898115,
        0.000743976941283,
        0.00467161595152,
        0.00441929458194,
        0.00374950700192,
        0.00359173703651,
        1.0409075141,
        0.00276627412111,
        0.999997824793,
        0.796453939463,
        -0.458814860565,
        -0.0849262749526,
        -0.20846052759,
        -0.048938181083,
        -0.00916673755245,
        0.00242695389587,
        0.00342826613707,
        0.00617438161099,
        0.00618704032304,
        0.00445282481079,
        0.0058100531556,
        0.00323709873822,
        0.00221570495532,
        0.00149206517554,
        0.00098234816559,
        0.00617047406639,
        0.0058346466569,
        0.0049523646767,
        0.004742329736,
        0.00365277611942,
        1.04256924073,
        -0.999999996156,
        -0.999998090618,
        0.99999999699,
        -4.52791545964e-07,
        -2.9881543661e-07,
        -0.00280598475652,
        -0.0581954486325,
        -0.101492104979,
        -0.0581486930444,
        -0.0488127809186,
        -0.0290801015276,
        -0.0253003779239,
        -0.00726656270018,
        -0.0105734367662,
        -0.00344610921185,
        -0.00195310796731,
        -0.00110268142674,
        -0.000631246242022,
        -0.0121447177417,
        -0.0115376514565,
        -0.00716437245939,
        -0.00688907169797,
        -0.00407788127691,
        -1.0,
        0.139018496216,
        -1.0,
        -0.0522005919191,
        0.028830197934,
        0.00234293374942,
        0.022045694198,
        0.0185722670604,
        0.0141312423522,
        0.013194431758,
        0.00954235786384,
        0.00889713558306,
        0.0036763007466,
        0.00568745895374,
        0.00232077510056,
        0.00146552088122,
        0.000903128839075,
        0.000561805028348,
        0.00633948714783,
        0.0061235194429,
        0.00414016955611,
        0.00404064074178,
        0.00263303387551,
        -1.0,
        -0.0107765206203,
        -1.0,
        -0.0912997366821,
        -0.0233952315314,
        -0.0476292912621,
        -0.0168313489307,
        -0.00743684396422,
        -0.0039501317039,
        -0.00322148949551,
        -0.00164129194608,
        -0.00138377686942,
        -0.000213309327008,
        -0.000555926287931,
        -6.3216599287e-05,
        -5.63838672776e-06,
        1.5781083951e-05,
        1.90885336542e-05,
        -0.000701332977182,
        -0.000659686332503,
        -0.000278884166381,
        -0.000270601482399,
        -8.90706077274e-05,
        -0.10273063009,
        -0.999999986632,
        1.00000001489,
        0.99999999945,
        0.999479409217,
        0.128216503501,
        -0.0975006003435,
        -1.19859422478,
        -0.0112931946574,
        -0.0394594429457,
        -0.0092908575147,
        -0.00301686005276,
        -0.00142166315568,
        -0.000747541375875,
        -0.000186884931916,
        -7.34190550087e-05,
        0.000124311823449,
        7.84248985522e-05,
        9.38596007646e-05,
        6.84132232969e-05,
        4.2445398481e-05,
        2.63849774614e-05,
        3.6709525686e-05,
        0.000118471660693,
        8.92708983468e-05,
        0.00013265489836,
        8.09278273013e-05,
        -0.999999999463,
        0.999999995882,
        0.999999937063,
        -0.0354131909986,
        -0.0407232450216,
        -0.00489080139169,
        -0.661133434288,
        -0.0165883701283,
        -0.00393761218836,
        -0.00125772516365,
        -0.000561267423312,
        -0.000288762136141,
        -6.400955576e-05,
        -9.48128585637e-06,
        6.12658850313e-05,
        4.95811463133e-05,
        4.5276535016e-05,
        3.31852292695e-05,
        2.04805695867e-05,
        1.30454209598e-05,
        2.89839054067e-05,
        6.32760511033e-05,
        5.11479697193e-05,
        6.37033378172e-05,
        4.25862036369e-05,
        -0.999999996956,
        0.999999999446,
        0.999912517979,
        0.0742222485744,
        -0.072645579969,
        -0.00837403815056,
        -0.029321061844,
        -1.22397093474,
        -0.00686565897685,
        -0.00217587623968,
        -0.000989178927763,
        -0.000494386816101,
        -0.000109070954903,
        -9.91793703805e-06,
        0.000103732332396,
        8.2767254363e-05,
        7.61459027092e-05,
        5.57666030401e-05,
        3.37809364566e-05,
        2.14417903882e-05,
        4.9299281765e-05,
        0.000108305040297,
        8.50627843807e-05,
        0.000110029770374,
        7.04584687748e-05,
        -0.99999999978,
        1.0,
        0.999912199421,
        -0.00924649345378,
        -0.043645658946,
        -0.00512857905213,
        -0.0176964371422,
        -0.00417461315555,
        -0.499666452249,
        -0.00133009095071,
        -0.000603391827051,
        -0.000307065009017,
        -6.78472844271e-05,
        -8.38433717551e-06,
        6.28862440366e-05,
        4.9923350947e-05,
        4.65527740317e-05,
        3.41391510115e-05,
        2.08392163581e-05,
        1.32702735393e-05,
        2.9883563088e-05,
        6.43193903892e-05,
        5.20211394244e-05,
        6.65053677227e-05,
        4.34128988935e-05,
        -1.00000000008,
        1.0,
        0.999912302877,
        -0.0507285441458,
        -0.0276007107873,
        -0.00328476640783,
        -0.0112217233886,
        -0.00265595417406,
        -0.000845667463903,
        -0.0564234839457,
        -0.000380172862344,
        -0.000195136459825,
        -4.40958637595e-05,
        -6.26956368945e-06,
        4.03386905002e-05,
        3.19778398558e-05,
        2.99973625126e-05,
        2.20023071172e-05,
        1.34922962591e-05,
        8.60669984382e-06,
        1.92059252892e-05,
        4.08029825097e-05,
        3.35708744427e-05,
        4.25894860857e-05,
        2.81206198711e-05,
        -1.00000000009,
        1.00000000005,
        0.999912453981,
        -0.0599033268909,
        -0.0207891567893,
        -0.00244486694893,
        -0.00842750977178,
        -0.0019844421801,
        -0.000628986502355,
        -0.000282460284887,
        -0.0522650014344,
        -0.000143265634927,
        -3.25561292064e-05,
        -3.96611503574e-06,
        3.02873690338e-05,
        2.40994536812e-05,
        2.23992152096e-05,
        1.64124787383e-05,
        1.00163528557e-05,
        6.37624980721e-06,
        1.44308690277e-05,
        3.10623122899e-05,
        2.5072456909e-05,
        3.20399498793e-05,
        2.08991349207e-05,
        -1.00000000016,
        0.999999999859,
        0.999912543765,
        -0.085684737858,
        -0.0123457599324,
        -0.00152252653077,
        -0.00506148497832,
        -0.00121242781117,
        -0.000387988978325,
        -0.000172262313464,
        -9.09495047752e-05,
        -0.024052835449,
        -2.1109933892e-05,
        -4.12434828109e-06,
        1.85234032329e-05,
        1.45769403991e-05,
        1.39726491124e-05,
        1.02651842125e-05,
        6.37786620657e-06,
        4.08969425391e-06,
        8.83630088372e-06,
        1.8061759608e-05,
        1.56682980376e-05,
        1.94551212096e-05,
        1.32831020455e-05,
        -1.00000000022,
        0.999999999875,
        0.99991253708,
        -0.103866444057,
        -0.00466453480545,
        -0.000589066331213,
        -0.00192360701985,
        -0.000464944886743,
        -0.000149639502366,
        -6.6185456634e-05,
        -3.5659343109e-05,
        -8.31352687888e-06,
        -0.0252447325465,
        -1.90365303491e-06,
        7.09023535234e-06,
        5.54660492714e-06,
        5.40121529007e-06,
        3.97372330924e-06,
        2.49000860875e-06,
        1.60217502289e-06,
        3.38261194925e-06,
        6.73248668934e-06,
        6.06001351152e-06,
        7.41980983253e-06,
        5.17996879746e-06,
        -0.99999999976,
        1.00000000057,
        0.999912549799,
        -0.123008731426,
        -0.000368745528941,
        -0.000107853400512,
        -0.000201427900259,
        -6.63569056003e-05,
        -2.45186010692e-05,
        -9.28330854474e-06,
        -8.02273648237e-06,
        -2.18953150991e-06,
        -1.67791727835e-06,
        -0.0244684671832,
        1.01871336472e-06,
        6.61837855794e-07,
        1.00273094903e-06,
        7.59782512433e-07,
        5.66622719577e-07,
        3.87731224852e-07,
        4.93868174713e-07,
        1.92345959814e-07,
        1.14646787386e-06,
        9.49953999836e-07,
        1.158863636e-06,
        -1.00000000089,
        1.00000000039,
        0.999912905367,
        -0.186451042826,
        0.0190534522954,
        0.0022386389406,
        0.00772613495054,
        0.00182379578094,
        0.000582088371585,
        0.000264909813002,
        0.000134976865267,
        2.95559667678e-05,
        3.67631249906e-06,
        -2.73560689446e-05,
        -0.00452837800054,
        -2.16999425119e-05,
        -2.02586717238e-05,
        -1.48608566668e-05,
        -9.07251733965e-06,
        -5.77807055113e-06,
        -1.29882531595e-05,
        -2.79498617702e-05,
        -2.26258306276e-05,
        -2.89270698842e-05,
        -1.88902427699e-05,
        -1.00000000109,
        1.00000000007,
        0.999912484638,
        -0.15317271794,
        0.00802754551814,
        0.000933366405057,
        0.00324598933134,
        0.000761722741915,
        0.000241276699131,
        0.000108990219784,
        5.48007894306e-05,
        1.22804160765e-05,
        1.27512941297e-06,
        -1.15758552445e-05,
        -9.22824780312e-06,
        -0.0277778838493,
        -8.52343310207e-06,
        -6.24299741568e-06,
        -3.79364736159e-06,
        -2.41087550917e-06,
        -5.50956220453e-06,
        -1.19991810193e-05,
        -9.53158589103e-06,
        -1.22652158157e-05,
        -7.91532180699e-06,
        -0.999999999476,
        0.999999999571,
        0.999912517413,
        -0.212075640846,
        0.0253480611889,
        0.00305253593659,
        0.0103357808231,
        0.00245808056086,
        0.000785651072035,
        0.000353100019934,
        0.000183294533425,
        4.13267265184e-05,
        6.65951369184e-06,
        -3.72154671039e-05,
        -2.94030572355e-05,
        -2.78198318719e-05,
        -0.00805189087878,
        -2.04233165736e-05,
        -1.25814656786e-05,
        -8.04117891342e-06,
        -1.77122393547e-05,
        -3.71456112968e-05,
        -3.11340600388e-05,
        -3.92183194281e-05,
        -2.61990999895e-05,
        -1.00000000029,
        1.00000000004,
        0.999912519925,
        -0.288408901748,
        0.0341275252455,
        0.0041216311897,
        0.0139260518214,
        0.00331648196804,
        0.00106153803781,
        0.000477505401482,
        0.000248658019579,
        5.58813365448e-05,
        9.26737653245e-06,
        -5.0110673655e-05,
        -3.95494966638e-05,
        -3.75109368608e-05,
        -2.75460821735e-05,
        -0.00437331376102,
        -1.69885579822e-05,
        -1.08632952907e-05,
        -2.384082867e-05,
        -4.98390535999e-05,
        -4.1972970783e-05,
        -5.27816444843e-05,
        -3.53631420722e-05,
        -0.999999999848,
        1.00000000008,
        0.999912519524,
        -0.267719571417,
        0.0378326016937,
        0.00466776846317,
        0.0155179923813,
        0.00372501048752,
        0.00119805169726,
        0.000536647434108,
        0.000284676544279,
        6.43643517169e-05,
        1.26973459937e-05,
        -5.62403803632e-05,
        -4.41600259929e-05,
        -4.24730684173e-05,
        -3.12288959227e-05,
        -1.9411258903e-05,
        -0.00237525220451,
        -1.24520367687e-05,
        -2.67646790543e-05,
        -5.46601183808e-05,
        -4.75544590853e-05,
        -5.90471069193e-05,
        -4.03670670961e-05,
        -1.0000000011,
        0.999999999685,
        0.999912515884,
        -0.293419013261,
        0.0434306432098,
        0.00540547991889,
        0.017852847266,
        0.00430008289913,
        0.00138625645332,
        0.000620317760878,
        0.000331645826749,
        7.50046991547e-05,
        1.57275998745e-05,
        -6.48415408914e-05,
        -5.07979875376e-05,
        -4.91477944329e-05,
        -3.61570112194e-05,
        -2.25452312926e-05,
        -1.4481068758e-05,
        -0.00130998910452,
        -3.08551487822e-05,
        -6.24075937969e-05,
        -5.50344162033e-05,
        -6.7986170972e-05,
        -4.68607020312e-05,
        -0.999999999715,
        0.999999999954,
        0.999912472889,
        -0.128443548222,
        0.00401900708443,
        0.000473940128632,
        0.00163043018634,
        0.000384515383292,
        0.000122118312875,
        5.4949617371e-05,
        2.79668578707e-05,
        6.31600029603e-06,
        8.02243307782e-07,
        -5.84871632638e-06,
        -4.6478446612e-06,
        -4.33201660218e-06,
        -3.17537620233e-06,
        -1.94023285613e-06,
        -1.23583027307e-06,
        -2.78520105396e-06,
        -0.0328657355005,
        -5.97574468472e-06,
        -4.84748962869e-06,
        -6.18397292149e-06,
        -4.04624245183e-06,
        -1.00000000066,
        1.00000000009,
        0.999912479198,
        -0.140082118781,
        0.00955465000819,
        0.00106331544184,
        0.00382514301413,
        0.000883647759228,
        0.000277333510647,
        0.000126597383134,
        6.11844761404e-05,
        1.34278985237e-05,
        3.35487920494e-07,
        -1.34205471027e-05,
        -1.08080497282e-05,
        -9.69708062041e-06,
        -7.08409319983e-06,
        -4.22811346187e-06,
        -2.6667739425e-06,
        -6.38059268633e-06,
        -1.45417364643e-05,
        -0.0307844415272,
        -1.08256860016e-05,
        -1.4314195369e-05,
        -8.83896142443e-06,
        -0.999999999081,
        1.00000000033,
        0.999912492459,
        -0.15498238319,
        0.0127396178372,
        0.00153795586098,
        0.00519722108174,
        0.00123652815741,
        0.000394938372695,
        0.000177003008414,
        9.19845588194e-05,
        2.08933349219e-05,
        3.43482498909e-06,
        -1.87768291135e-05,
        -1.48346989801e-05,
        -1.40469897386e-05,
        -1.03116423218e-05,
        -6.35759766556e-06,
        -4.06448363155e-06,
        -8.94268960385e-06,
        -1.87065294427e-05,
        -1.57279004049e-05,
        -0.017887213356,
        -1.97817501041e-05,
        -1.32424110384e-05,
        -1.00000000017,
        0.999999999199,
        0.999912492952,
        -0.16592198177,
        0.0173391442725,
        0.00202430254508,
        0.00701827699101,
        0.00164993185662,
        0.000523508733911,
        0.000236625018353,
        0.000119480913309,
        2.67079652894e-05,
        2.97116335683e-06,
        -2.50237920608e-05,
        -1.99217582945e-05,
        -1.84611210253e-05,
        -1.35269637692e-05,
        -8.23396005838e-06,
        -5.23658949837e-06,
        -1.19061990215e-05,
        -2.58147089689e-05,
        -2.06427143893e-05,
        -2.64956257658e-05,
        -0.0171397999156,
        -1.7172562023e-05,
        -0.999999999827,
        0.99991251523,
        -0.182937442918,
        0.0194095427784,
        0.00239047787906,
        0.00795681383118,
        0.0019074266866,
        0.000612166770484,
        0.000273450455121,
        0.000144616855932,
        3.29850661705e-05,
        6.40795539422e-06,
        -2.89207434365e-05,
        -2.27354702087e-05,
        -2.18176269141e-05,
        -1.60355367519e-05,
        -9.95999512174e-06,
        -6.38679960912e-06,
        -1.3774878763e-05,
        -2.81903248565e-05,
        -2.44401034829e-05,
        -3.03756678357e-05,
        -2.07250588972e-05,
        -0.00968689608766,
        0.999999998742,
        -0.999910149631,
        9.68732086557e-07,
        6.94749219931e-05,
        0.00318350102283,
        0.0157471871607,
        0.0103911952815,
        0.00506850839197,
        0.00395037686304,
        0.0018998501313,
        0.00157289880389,
        0.000299654646665,
        0.00055095140292,
        0.000118855940702,
        5.59023342233e-05,
        2.64346826222e-05,
        1.26404585877e-05,
        0.000684185130668,
        0.000620889842254,
        0.000319499382605,
        0.000304174930869,
        0.000149998132387,
        -0.999999998384,
        1.0,
        1.0,
        -0.000442173854119,
        1.0,
        -0.0367407819622,
        1.00000013019,
        1.00000005368,
        -0.999999997929,
        0.999955794161,
        0.247415866307,
        0.999999999059,
        0.366004296958,
        0.464267560008,
        0.0644426697023,
        0.163179411945,
        0.0364609892021,
        0.00597319904986,
        -0.00261916990603,
        -0.00326925878665,
        -0.00510409933308,
        -0.00503106842612,
        -0.00337346645386,
        -0.00451758784738,
        -0.00240099639679,
        -0.00161459968725,
        -0.0010692602188,
        -0.000692991305251,
        -0.00484027686721,
        -0.00455527140878,
        -0.00378035753589,
        -0.00361135797893,
        -0.00272692047839,
        -0.999999996367,
        0.999995957604,
        0.227571465551,
        0.999999999376,
        0.22408007046,
        0.0394063098592,
        0.726341386928,
        0.0998940695856,
        0.0223190626235,
        0.00365320499595,
        -0.00164025934806,
        -0.00206802916596,
        -0.00323515591535,
        -0.00314159307702,
        -0.00206834752587,
        -0.00278654789874,
        -0.00147166202857,
        -0.000988059536261,
        -0.000654437665278,
        -0.000424226727302,
        -0.00299823012475,
        -0.00280250487563,
        -0.00232655015368,
        -0.00221429426146,
        -0.00167465173242,
        -0.999999998713,
        0.999999770732,
        0.260989936232,
        0.999999999718,
        0.2908736879,
        0.0510811598808,
        0.129597576703,
        0.755906538005,
        0.0289571529548,
        0.00475339337076,
        -0.00216798109419,
        -0.00276592267522,
        -0.00434501868712,
        -0.00415598080568,
        -0.00268638950482,
        -0.00364041274021,
        -0.00191051926974,
        -0.00128051884987,
        -0.000848633988504,
        -0.000550039706429,
        -0.00393580174108,
        -0.00365071891611,
        -0.00303280304082,
        -0.00287597506816,
        -0.00217832938936,
        -1.0,
        0.999999756671,
        0.301977094396,
        0.99999999997,
        0.162992081191,
        0.0286430652611,
        0.0726422343834,
        0.0162306580867,
        0.798757001779,
        0.00265992998667,
        -0.0012042814373,
        -0.00152741955175,
        -0.00239420789957,
        -0.00230724594199,
        -0.00150501012886,
        -0.00203361292281,
        -0.00107059339544,
        -0.000718160302928,
        -0.000475796864912,
        -0.000308411282744,
        -0.00219331254877,
        -0.00204223960797,
        -0.00169601376276,
        -0.00161119834576,
        -0.00121942414845,
        -0.999999999409,
        0.999999774796,
        0.374940538968,
        0.999999999163,
        0.0528477753684,
        0.0092913536627,
        0.0235579168038,
        0.00526351397271,
        0.00086164863134,
        0.291660119918,
        -0.000388171535302,
        -0.000490348875366,
        -0.000767458239566,
        -0.000743392594946,
        -0.000487908227249,
        -0.000657995915377,
        -0.000347130127008,
        -0.000232987350444,
        -0.000154327201172,
        -0.000100040225143,
        -0.000708508177677,
        -0.000661413943999,
        -0.000549160975898,
        -0.000522328060665,
        -0.00039511681864,
        -0.999999999171,
        0.999999756197,
        0.469090415842,
        0.99999999926,
        -0.0390755650836,
        -0.00688352941317,
        -0.0174338734384,
        -0.00389483249814,
        -0.000634476858982,
        0.000279484156261,
        0.456076795228,
        0.000346579856319,
        0.000538611770173,
        0.000534306598141,
        0.000360515913455,
        0.000482015600249,
        0.000256673637616,
        0.00017270029061,
        0.00011428904775,
        7.41036433516e-05,
        0.00051522156042,
        0.000486573875683,
        0.000403590714211,
        0.000385932188988,
        0.000291272182315,
        -0.999999998064,
        0.999999769993,
        0.460283701238,
        0.999999998323,
        -0.0585635339184,
        -0.0103230975791,
        -0.0261365357188,
        -0.00583853110026,
        -0.000949213437901,
        0.000414283737718,
        0.000509842995762,
        0.250161931623,
        0.000789886752975,
        0.000791517833179,
        0.000540104746653,
        0.00071962797075,
        0.000384636351865,
        0.000259052675796,
        0.000171370428617,
        0.000111124402987,
        0.000766905748285,
        0.000727688189198,
        0.000603321963082,
        0.000578174442756,
        0.000435951224138,
        -0.999999998858,
        0.999999759936,
        0.540243311753,
        0.999999998822,
        -0.152113364071,
        -0.0268174732009,
        -0.0678914727233,
        -0.0151661469239,
        -0.00246502788159,
        0.00107450798585,
        0.00132084082952,
        0.00204550485605,
        0.44759963763,
        0.00205264200253,
        0.00140287403721,
        0.00186826662859,
        0.000999101372044,
        0.000672987372129,
        0.000445178711636,
        0.000288678400595,
        0.00199018058769,
        0.00188962939164,
        0.00156660902907,
        0.00150175095633,
        0.0011322055101,
        -0.999999988709,
        0.999999750122,
        0.535127572843,
        0.999999988784,
        -0.171515149479,
        -0.0302337420425,
        -0.0765461966308,
        -0.0170995910082,
        -0.00278020556119,
        0.00121379527313,
        0.00149407936673,
        0.00231500395611,
        0.00231902630818,
        0.502879888143,
        0.00158187102792,
        0.00210789323428,
        0.00112652556508,
        0.000758692370377,
        0.000501903085964,
        0.000325456376166,
        0.00224659250891,
        0.00213137642244,
        0.0017671493599,
        0.00169336943128,
        0.00127686878338,
        -0.999999922745,
        0.99999970659,
        0.691617025597,
        0.999999922774,
        -0.398629496531,
        -0.0702856824044,
        -0.177925225504,
        -0.0397462960384,
        -0.00645857405173,
        0.00281204634462,
        0.00345322883889,
        0.00534572058584,
        0.00537133372329,
        0.00367628935171,
        0.345459031307,
        0.00489372556042,
        0.00261827987571,
        0.00176387250867,
        0.00116674269187,
        0.000756589375098,
        0.0052110956603,
        0.00495074386172,
        0.00410424501591,
        0.00393539053531,
        0.00296664518769,
        -0.999999997795,
        0.99999970442,
        0.611145318296,
        0.999999997749,
        -0.288684329514,
        -0.0508489750993,
        -0.1287937745,
        -0.0287727604602,
        -0.00468760029717,
        0.00206595657024,
        0.00256325510257,
        0.00398407312039,
        0.00394992985452,
        0.00266335294542,
        0.00356168077676,
        1.0332880003,
        0.00189616442195,
        0.00127573872377,
        0.000844268217677,
        0.000547409271235,
        0.00380770651584,
        0.00359502958143,
        0.0029819418949,
        0.00285112518651,
        0.00215190490402,
        -0.999999974605,
        0.99999971529,
        0.769037658035,
        0.999999974622,
        -0.50878324184,
        -0.0896049582612,
        -0.226974702071,
        -0.0507072362245,
        -0.0082643466211,
        0.00364874700931,
        0.00453372674321,
        0.00705093243673,
        0.00697706004194,
        0.00469423696149,
        0.00628181233595,
        0.00334187265217,
        1.04141143716,
        0.00224798034001,
        0.00148779241819,
        0.00096464247966,
        0.00671962395363,
        0.0063385159179,
        0.00525798582329,
        0.00502521276098,
        0.00379349864535,
        -0.999999895144,
        0.999999966895,
        0.707552537891,
        0.999999895141,
        -0.616051480804,
        -0.108487989399,
        -0.274818955319,
        -0.0613962561502,
        -0.0100085946189,
        0.00442323543322,
        0.00550049771043,
        0.00855722224495,
        0.00845856296667,
        0.00568413536954,
        0.00760934966224,
        0.0040464687274,
        0.00272165223365,
        1.03733388445,
        0.00180135924026,
        0.0011679385063,
        0.00814229463991,
        0.00767661522723,
        0.00636825312435,
        0.00608490936919,
        0.0045939207098,
        -0.999999570055,
        1.00000000028,
        0.922034810992,
        0.999999570062,
        -0.735123809944,
        -0.129449669854,
        -0.327927079817,
        -0.0732616219378,
        -0.0119445612071,
        0.00528328164202,
        0.00657417795997,
        0.0102298275561,
        0.0101034827675,
        0.00678306805833,
        0.00908315302745,
        0.00482868349151,
        0.0032474870867,
        0.00214945424266,
        1.03316387756,
        0.00139362013734,
        0.00972178281434,
        0.00916212832276,
        0.00760086130596,
        0.0072613142909,
        0.00548251592853,
        -0.999999204367,
        0.99999894441,
        0.999999122964,
        0.999999204367,
        -0.846714345986,
        -0.149092669149,
        -0.377696909931,
        -0.0843804925369,
        -0.0137589487426,
        0.0060896674874,
        0.00758110264323,
        0.0117987700632,
        0.0116458218237,
        0.00781297353143,
        0.010464657933,
        0.00556171880394,
        0.00374025304586,
        0.00247566662884,
        0.00160510944662,
        1.02887007165,
        0.0112024731347,
        0.0105544561474,
        0.00875609027398,
        0.0083638114134,
        0.00631529468424,
        -0.999999984774,
        0.999999755649,
        0.632822492361,
        0.999999984842,
        -0.262219268471,
        -0.0461877755115,
        -0.116987017023,
        -0.0261351522597,
        -0.00425786807299,
        0.00187647150432,
        0.00232807771721,
        0.00361850835248,
        0.00358764745173,
        0.00241919239146,
        0.00323512051315,
        0.00172233914221,
        0.00115879399662,
        0.00076687489543,
        0.000497228967937,
        0.00345854413167,
        1.02671147285,
        0.00326543303492,
        0.00270855143615,
        0.00258975124805,
        0.00195462582513,
        -0.999999993605,
        0.999999704237,
        0.624650040493,
        0.999999993514,
        -0.263168192535,
        -0.0463535812853,
        -0.117409136006,
        -0.0262292985554,
        -0.00427327113796,
        0.00188345925719,
        0.00233698603188,
        0.00363243514904,
        0.00360106142963,
        0.00242791966031,
        0.00324691704198,
        0.00172854247168,
        0.00116295506714,
        0.000769630761397,
        0.000499014753795,
        0.003471270745,
        0.00327728558439,
        1.0285571052,
        0.00271838362441,
        0.00259909507697,
        0.00196169119538,
        -0.999999960473,
        0.999999762108,
        0.719965857226,
        0.999999960465,
        -0.370823060761,
        -0.0653153764168,
        -0.165437191709,
        -0.0369592003904,
        -0.00602187601026,
        0.00265506646067,
        0.00329526562699,
        0.00512254911509,
        0.00507641299381,
        0.0034212198334,
        0.00457587212945,
        0.00243569620815,
        0.00163866562327,
        0.00108446673372,
        0.000703147634602,
        0.00489260044012,
        0.00461836168963,
        0.00383083444848,
        1.0416838675,
        0.00366242619476,
        0.00276435684382,
        -0.999999994034,
        0.999999735416,
        0.704841923506,
        0.999999994064,
        -0.369147099987,
        -0.0650183791733,
        -0.164687794587,
        -0.0367916358589,
        -0.00599477690747,
        0.00264363204876,
        0.00328160607872,
        0.00510158848775,
        0.00505462478513,
        0.00340574750337,
        0.00455550535107,
        0.00242466426986,
        0.00163121026109,
        0.00107954068401,
        0.000699951589021,
        0.0048711153673,
        0.00459764836135,
        0.00381367715442,
        0.00364586509808,
        1.04183107075,
        0.00275190445149,
        0.999999646058,
        0.795731813445,
        -0.484097734654,
        -0.0852606241227,
        -0.215965340762,
        -0.0482477174562,
        -0.00786288203979,
        0.00347036815716,
        0.00431076635663,
        0.00670338967833,
        0.00663566785499,
        0.00446647458182,
        0.00597622013161,
        0.00317975599471,
        0.00213900946937,
        0.00141565364641,
        0.000917873523189,
        0.00639199542689,
        0.00603055346629,
        0.00500244803219,
        0.00478137889515,
        0.00360930918069,
        1.04413255014,
        -0.999999999674,
        -0.999999584757,
        0.999999999788,
        -1.46035330528e-06,
        -2.72605830846e-06,
        -0.00631632619781,
        -0.0756753212892,
        -0.101509674334,
        -0.0580728519871,
        -0.0485345370069,
        -0.0285865545123,
        -0.0247775574458,
        -0.00690201725701,
        -0.00961545871917,
        -0.00303223720192,
        -0.00169302326131,
        -0.000942362378626,
        -0.000531993733669,
        -0.0111048014535,
        -0.0105290344565,
        -0.00643429698555,
        -0.00618173858735,
        -0.00360523123075,
        -1.0,
        0.135338743347,
        -1.0,
        -0.0518396914795,
        0.0299703650255,
        0.00373968467025,
        0.0228196267725,
        0.0187594644271,
        0.01408419112,
        0.0130732686507,
        0.00933301955175,
        0.00866612943706,
        0.00347147020672,
        0.00545436550485,
        0.00216056000207,
        0.00134450809641,
        0.000816994425394,
        0.000501184451101,
        0.0061066174089,
        0.00588738146689,
        0.0039293253832,
        0.00383035133486,
        0.0024630616444,
        -1.0,
        -0.0103740569878,
        -1.0,
        -0.0926235538745,
        -0.0230067479845,
        -0.0477565253997,
        -0.0165231630399,
        -0.00719005753959,
        -0.00378030375816,
        -0.0030695290102,
        -0.0015472648889,
        -0.00129973345572,
        -0.000195541385976,
        -0.000515579312057,
        -5.76940118462e-05,
        -5.60085100193e-06,
        1.33788342005e-05,
        1.61794099781e-05,
        -0.000652802009509,
        -0.000613286408826,
        -0.000256640741555,
        -0.000248851264552,
        -8.14009510883e-05,
        -0.10780654573,
        -0.999999970086,
        1.00000001204,
        0.999999999714,
        0.998325678379,
        0.125539229276,
        -0.0973642214869,
        -1.30128585417,
        -0.0121705267345,
        -0.0407969801197,
        -0.0100278295243,
        -0.00342831835229,
        -0.00158559775261,
        -0.000942787979608,
        -0.000417826198352,
        -0.000203556291162,
        0.000104456515141,
        2.67837206845e-05,
        9.10646554358e-05,
        6.82984928685e-05,
        5.08890733555e-05,
        2.84577052041e-05,
        -3.21404648165e-05,
        9.10646518232e-05,
        5.35674418604e-05,
        0.000120526748648,
        6.96376757202e-05,
        -0.999999998599,
        0.999999996835,
        0.999663217379,
        -0.029474631057,
        -0.0420709130354,
        -0.00549635250494,
        -0.650485052978,
        -0.0177189060818,
        -0.00445311524117,
        -0.00150800603657,
        -0.000706590015238,
        -0.000444323018615,
        -0.000152638503803,
        -7.88974196033e-05,
        4.89655403082e-05,
        2.03955806892e-05,
        4.38702145952e-05,
        3.12911795393e-05,
        2.06616194703e-05,
        1.45405115554e-05,
        -8.15803091399e-06,
        3.80756119724e-05,
        3.1955822765e-05,
        4.76037359275e-05,
        2.75387549777e-05,
        -0.999999999458,
        0.999999999604,
        1.00000020778,
        0.0835175307307,
        -0.0745519217104,
        -0.00923041200919,
        -0.0307346367729,
        -1.26858655086,
        -0.00749307624657,
        -0.00245003302852,
        -0.00112932134631,
        -0.000586409923981,
        -0.000142548098017,
        -2.93316185124e-05,
        0.000119484110713,
        8.90701499985e-05,
        9.0786075889e-05,
        6.82053299448e-05,
        4.24379499681e-05,
        2.75588722158e-05,
        4.88929671497e-05,
        0.000116291298432,
        9.67577721954e-05,
        0.000125694060032,
        8.35244262931e-05,
        -0.999999998807,
        0.999999999899,
        1.00000116365,
        0.0013041922275,
        -0.0454660145834,
        -0.00572286804187,
        -0.0188215697082,
        -0.00462086705043,
        -0.499399970696,
        -0.00151966334094,
        -0.000700762915565,
        -0.000369429446001,
        -8.93409758203e-05,
        -2.04798009098e-05,
        7.32783170677e-05,
        5.43287009455e-05,
        5.61173060917e-05,
        4.22213508872e-05,
        2.64504282263e-05,
        1.72264980455e-05,
        2.99609721022e-05,
        6.98584661705e-05,
        5.98184945076e-05,
        7.68663935298e-05,
        5.19986948515e-05,
        -0.999999999703,
        0.99999999963,
        1.00000126641,
        -0.0383453519322,
        -0.0292402657581,
        -0.0037201661647,
        -0.0121347003637,
        -0.00298928113029,
        -0.000984011655822,
        -0.0644122662393,
        -0.000451602712589,
        -0.000240081927051,
        -5.86061459132e-05,
        -1.42340562535e-05,
        4.75593704461e-05,
        3.51832291125e-05,
        3.65781558358e-05,
        2.75305921238e-05,
        1.73164683131e-05,
        1.12955093661e-05,
        1.9456997225e-05,
        4.48274552225e-05,
        3.90240893405e-05,
        4.98071840202e-05,
        3.40430713884e-05,
        -0.999999999578,
        0.999999999941,
        0.999999791435,
        -0.0455123532187,
        -0.021977480522,
        -0.00276981592816,
        -0.00909728355951,
        -0.00223027319161,
        -0.000730341349836,
        -0.000333925129168,
        -0.056956463386,
        -0.000175836702198,
        -4.34207724308e-05,
        -9.94219570226e-06,
        3.57666863287e-05,
        2.65700054866e-05,
        2.73711657324e-05,
        2.05765558501e-05,
        1.28899520748e-05,
        8.3931221867e-06,
        1.46567526134e-05,
        3.41744673657e-05,
        2.92187599057e-05,
        3.75258158537e-05,
        2.53740424618e-05,
        -1.0000000001,
        0.999999999884,
        0.999999469429,
        -0.0714671344088,
        -0.013505792219,
        -0.00177268418173,
        -0.00564621131386,
        -0.00140483535987,
        -0.000463787074077,
        -0.00021002798413,
        -0.000114328003919,
        -0.0261986603423,
        -2.86537129077e-05,
        -8.06513213623e-06,
        2.2547958924e-05,
        1.65697454744e-05,
        1.75600069916e-05,
        1.32309690579e-05,
        8.41798298271e-06,
        5.51547117951e-06,
        9.23864901077e-06,
        2.05359091262e-05,
        1.87788866695e-05,
        2.35009827859e-05,
        1.65494748827e-05,
        -1.00000000024,
        1.0000000002,
        0.999999181729,
        -0.0876948643714,
        -0.00549710816078,
        -0.000733874686884,
        -0.00230792909267,
        -0.000577886918154,
        -0.00019148081851,
        -8.64178379867e-05,
        -4.76883242029e-05,
        -1.20053178458e-05,
        -0.0276411845282,
        -3.62541798573e-06,
        9.27140701105e-06,
        6.78136230818e-06,
        7.27303130761e-06,
        5.48538336151e-06,
        3.51176779506e-06,
        2.30660494483e-06,
        3.79791365686e-06,
        8.26995372008e-06,
        7.78269174353e-06,
        9.63645435077e-06,
        6.8998128237e-06,
        -1.0000000004,
        0.999999999214,
        0.999999080366,
        -0.106822169547,
        -0.00119085649701,
        -0.000214614357324,
        -0.000543986263038,
        -0.000152366099598,
        -5.34383924665e-05,
        -2.26720504914e-05,
        -1.53756179543e-05,
        -4.14930951485e-06,
        -2.32564815475e-06,
        -0.0268900160187,
        2.44733275226e-06,
        1.65130633505e-06,
        2.15220688844e-06,
        1.64602031894e-06,
        1.1496771921e-06,
        7.79955550729e-07,
        1.00057330874e-06,
        1.41387718663e-06,
        2.32660606528e-06,
        2.42475063254e-06,
        2.24225228871e-06,
        -1.00000000005,
        0.999999998925,
        0.999997012647,
        -0.166458210489,
        0.0188059927265,
        0.00236779463488,
        0.00778571613535,
        0.00191174979364,
        0.00062882479008,
        0.000290011744505,
        0.000152933066307,
        3.69678591557e-05,
        8.48890520463e-06,
        -3.03085678038e-05,
        -0.00515651702385,
        -2.24680461295e-05,
        -2.32141186634e-05,
        -1.74664234045e-05,
        -1.0943505166e-05,
        -7.12761460063e-06,
        -1.23916533064e-05,
        -2.8882432043e-05,
        -2.47448412573e-05,
        -3.17908566135e-05,
        -2.15129917571e-05,
        -0.999999999972,
        1.00000000051,
        0.999999347059,
        -0.135028849674,
        0.00757955932837,
        0.000943868090837,
        0.00312863887497,
        0.00076388775852,
        0.000249702497315,
        0.00011462923225,
        5.97736862479e-05,
        1.46501309062e-05,
        3.11671715132e-06,
        -1.22282309094e-05,
        -9.1101106647e-06,
        -0.0284729629576,
        -9.3105259536e-06,
        -6.99537408366e-06,
        -4.36169320002e-06,
        -2.8347180842e-06,
        -5.00966594707e-06,
        -1.183970879e-05,
        -9.93188669185e-06,
        -1.28540272777e-05,
        -8.58799648173e-06,
        -1.00000000018,
        0.999999999712,
        0.999999157263,
        -0.189867643582,
        0.0255080856734,
        0.00328483391553,
        0.0106159846853,
        0.00262541087914,
        0.000865313461965,
        0.000395146572108,
        0.000212051209734,
        5.22678925577e-05,
        1.34900393984e-05,
        -4.19025303337e-05,
        -3.09174023868e-05,
        -3.23854556634e-05,
        -0.00860154513079,
        -2.43859091951e-05,
        -1.5407675703e-05,
        -1.00681076267e-05,
        -1.71522148567e-05,
        -3.89770846151e-05,
        -3.45821123502e-05,
        -4.38013610757e-05,
        -3.02895579634e-05,
        -0.999999999736,
        1.00000000015,
        0.999999144401,
        -0.266639697072,
        0.0345540915592,
        0.00446116004052,
        0.0143906141827,
        0.00356321062295,
        0.00117578466562,
        0.000537237421189,
        0.000289013870237,
        7.10862561085e-05,
        1.85918537716e-05,
        -5.67807046611e-05,
        -4.18531677965e-05,
        -4.393981766e-05,
        -3.30952138965e-05,
        -0.00476393776758,
        -2.09320452508e-05,
        -1.36840004913e-05,
        -2.32344475578e-05,
        -5.26305742041e-05,
        -4.69155743536e-05,
        -5.93258272663e-05,
        -4.11380350601e-05,
        -1.00000000012,
        0.999999998908,
        0.999999131279,
        -0.241368978021,
        0.0386281241405,
        0.00508347698671,
        0.0161640803556,
        0.00403112023623,
        0.00133585969619,
        0.000608114422466,
        0.000332354883204,
        8.21157371892e-05,
        2.34215820439e-05,
        -6.41925021156e-05,
        -4.70671067817e-05,
        -5.00857870855e-05,
        -3.77672691089e-05,
        -2.40581240664e-05,
        -0.00263632901692,
        -1.57726862234e-05,
        -2.62597901923e-05,
        -5.81436897424e-05,
        -5.3514335924e-05,
        -6.68603582947e-05,
        -4.72469738059e-05,
        -0.999999999631,
        1.00000000029,
        0.999999126668,
        -0.264971436138,
        0.0445397209795,
        0.00590669867454,
        0.0186743840286,
        0.00467120715801,
        0.00155108131821,
        0.000705417700293,
        0.000388033969959,
        9.59031546935e-05,
        2.82393140339e-05,
        -7.4309717847e-05,
        -5.43619400594e-05,
        -5.81739089308e-05,
        -4.38883990803e-05,
        -2.80364483577e-05,
        -1.84018404804e-05,
        -0.00148081827595,
        -3.03906145519e-05,
        -6.66650814192e-05,
        -6.21665155992e-05,
        -7.72992488625e-05,
        -5.50387527361e-05,
        -1.00000000075,
        0.999999999373,
        0.999999440966,
        -0.110023077233,
        0.00351816148422,
        0.000443763415835,
        0.00145664651691,
        0.000357290243098,
        0.000117077963344,
        5.35680775472e-05,
        2.82336356054e-05,
        6.95812365072e-06,
        1.6022348087e-06,
        -5.72316014005e-06,
        -4.24985482183e-06,
        -4.38186087583e-06,
        -3.2946348247e-06,
        -2.0646875491e-06,
        -1.34463996272e-06,
        -2.34490115592e-06,
        -0.0334763788538,
        -5.46103644114e-06,
        -4.67732079765e-06,
        -6.00352901079e-06,
        -4.06372438361e-06,
        -0.999999999591,
        1.00000000088,
        0.999999397923,
        -0.121972383825,
        0.00903559779708,
        0.00107801732533,
        0.00369234511043,
        0.000887608309142,
        0.00028754959655,
        0.000133322885839,
        6.69843702118e-05,
        1.61520182808e-05,
        2.41252799997e-06,
        -1.4203798644e-05,
        -1.07041402948e-05,
        -1.06096437429e-05,
        -7.95078605477e-06,
        -4.86980096015e-06,
        -3.14169089149e-06,
        -5.82073966465e-06,
        -1.44310088029e-05,
        -0.0314288695336,
        -1.12959140457e-05,
        -1.50355421925e-05,
        -9.60375719485e-06,
        -1.00000000087,
        0.999999999831,
        0.999999244831,
        -0.134251798928,
        0.0125440365258,
        0.00161943196809,
        0.0052233797666,
        0.00129247131181,
        0.0004258122439,
        0.000194013196638,
        0.000104278306498,
        2.58338283617e-05,
        6.74329617627e-06,
        -2.06756241253e-05,
        -1.52521881558e-05,
        -1.59931745327e-05,
        -1.20422659003e-05,
        -7.61520531279e-06,
        -4.97765822773e-06,
        -8.46700264111e-06,
        -1.91885996453e-05,
        -1.7084717148e-05,
        -0.0186106404656,
        -2.16054637405e-05,
        -1.49735412654e-05,
        -1.00000000062,
        1.00000000017,
        0.999999245031,
        -0.145755895041,
        0.0171204498878,
        0.0021421015233,
        0.00707524794244,
        0.00173093337678,
        0.000566730567756,
        0.000260185088346,
        0.000136259554436,
        3.33583435791e-05,
        7.32647134595e-06,
        -2.76665152171e-05,
        -2.05789510741e-05,
        -2.1113057574e-05,
        -1.58693527414e-05,
        -9.91428399821e-06,
        -6.44881377974e-06,
        -1.13309251846e-05,
        -2.66282866429e-05,
        -2.25222849759e-05,
        -2.90580062688e-05,
        -0.0178520347854,
        -1.95138505069e-05,
        -0.999999999617,
        0.999999173667,
        -0.160017936173,
        0.0195488957884,
        0.00256959273274,
        0.00817689804862,
        0.00203717278511,
        0.000674009540589,
        0.000306162001,
        0.000167033915096,
        4.15163022517e-05,
        1.17724215546e-05,
        -3.25484648057e-05,
        -2.38883927588e-05,
        -2.53749529492e-05,
        -1.91277804833e-05,
        -1.21780160332e-05,
        -7.98184258815e-06,
        -1.33243185408e-05,
        -2.95526710618e-05,
        -2.71223845136e-05,
        -3.39112281155e-05,
        -2.39269055427e-05,
        -0.0102835903956,
        0.99999999962,
        -1.00000975546,
        3.17484348124e-07,
        7.18872332275e-06,
        0.00148417271579,
        0.0117077199711,
        0.0113624920154,
        0.00561330698347,
        0.0043781462861,
        0.00212461008995,
        0.00176682195892,
        0.000347472571079,
        0.00057746132974,
        0.000128948482214,
        6.17193966174e-05,
        2.96559058631e-05,
        1.44238006785e-05,
        0.000712876611121,
        0.000647963709127,
        0.000338831683415,
        0.000322872934029,
        0.000161727277585,
        -1.0000000003,
        1.0,
        1.0,
        -0.0006631745871,
        1.0,
        -0.0365710690983,
        1.00000004442,
        1.00000005209,
        0.999999999997,
        0.96361110479,
        0.135449137946,
        0.0197027341993,
        0.00647162386629,
        0.000730686972079,
        0.0019535319564,
        0.000110656027314,
        -0.000388340101682,
        -0.000462745233426,
        -0.00050078522833,
        -0.000386854261066,
        -0.000334783025112,
        -9.29813582754e-05,
        -0.000183663129568,
        -4.7138972489e-05,
        -2.28140421481e-05,
        -1.09839853582e-05,
        -5.24770502719e-06,
        -0.000220427910202,
        -0.000202486816006,
        -0.000116501539942,
        -0.000110110296599,
        -5.91579984417e-05,
        -0.999999999336,
        0.999999999942,
        0.963610172629,
        0.0719014495624,
        0.00749766561527,
        0.000279471966012,
        0.0381603871437,
        0.000743846971396,
        4.21696553866e-05,
        -0.000151990232606,
        -0.000179454756716,
        -0.000192840691252,
        -0.000148904076295,
        -0.000128698593007,
        -3.576084789e-05,
        -7.06554207353e-05,
        -1.81448044759e-05,
        -8.78362858611e-06,
        -4.23331767764e-06,
        -2.02371231893e-06,
        -8.48452994928e-05,
        -7.78217560389e-05,
        -4.48462851666e-05,
        -4.23434236783e-05,
        -2.27869489853e-05,
        -0.999999927269,
        0.999999994605,
        0.96361017542,
        0.118613805714,
        0.0138609923227,
        0.000526499172543,
        0.00137800147218,
        0.0132445699566,
        7.82860360633e-05,
        -0.000313574714582,
        -0.000357581345205,
        -0.000373938604607,
        -0.000288196795984,
        -0.000247854671466,
        -6.89847537143e-05,
        -0.000136438732629,
        -3.51150059568e-05,
        -1.70134891646e-05,
        -8.23324228072e-06,
        -3.94468868989e-06,
        -0.000164192922614,
        -0.000149711067121,
        -8.68078462271e-05,
        -8.16416325961e-05,
        -4.42174520774e-05,
        -0.999999990542,
        0.999999999674,
        0.963610176426,
        0.0868973894005,
        0.00263486507695,
        9.82324743912e-05,
        0.000261409379797,
        1.48191974054e-05,
        0.0521986033621,
        -5.35208186666e-05,
        -6.31477125624e-05,
        -6.78245128434e-05,
        -5.23690362501e-05,
        -4.52588570188e-05,
        -1.25761126059e-05,
        -2.48480827995e-05,
        -6.38137951825e-06,
        -3.08917573575e-06,
        -1.4889506668e-06,
        -7.11812057722e-07,
        -2.98395235172e-05,
        -2.73665829702e-05,
        -1.57721538558e-05,
        -1.48909025961e-05,
        -8.01435919862e-06,
        -1.00000000007,
        0.999999998759,
        0.963610175839,
        0.0788160820324,
        -0.00721583675379,
        -0.000264022073904,
        -0.000714428211561,
        -4.0409701986e-05,
        0.000130424585538,
        0.0207744220482,
        0.000160118954689,
        0.000177087248636,
        0.000136997203346,
        0.000119009515801,
        3.30113964195e-05,
        6.51553283227e-05,
        1.6694666965e-05,
        8.07437681501e-06,
        3.87513296629e-06,
        1.84817252054e-06,
        7.80686935522e-05,
        7.20406023367e-05,
        4.12534868475e-05,
        3.91081337129e-05,
        2.09078911192e-05,
        -0.99999999945,
        0.999999999046,
        0.963610177418,
        0.0906414290997,
        -0.0152131693666,
        -0.000565620521177,
        -0.00150886297677,
        -8.55071731403e-05,
        0.000304093766883,
        0.000360687397169,
        0.0453016907905,
        0.000388959899695,
        0.000300404902682,
        0.000259805728479,
        7.21745182952e-05,
        0.000142582552007,
        3.66057072445e-05,
        1.77182788978e-05,
        8.53494270251e-06,
        4.07890032011e-06,
        0.00017117089571,
        0.00015712020797,
        9.0471647458e-05,
        8.54654445952e-05,
        4.5955027027e-05,
        -0.999999999338,
        1.00000000019,
        0.963610182393,
        0.0809060652177,
        -0.0168171218505,
        -0.000628829839022,
        -0.0016689847919,
        -9.46432142129e-05,
        0.000347892931272,
        0.000408022023185,
        0.000436252457442,
        0.0322337219244,
        0.000336734813684,
        0.000290777959436,
        8.08204728443e-05,
        0.000159713539626,
        4.10316374336e-05,
        1.9865950115e-05,
        9.58163141957e-06,
        4.58232527982e-06,
        0.000191864568966,
        0.000175792527995,
        0.000101417007684,
        9.56885511197e-05,
        5.15543898703e-05,
        -0.999999999875,
        0.999999999817,
        0.963610232707,
        0.0818922723055,
        -0.0250345157523,
        -0.000930055414458,
        -0.00248274347902,
        -0.00014068302503,
        0.000498121815349,
        0.000591721186601,
        0.000638836446337,
        0.000493428591047,
        0.155506856773,
        0.000426830143512,
        0.000118565854296,
        0.000234219961325,
        6.01266310145e-05,
        2.91020625886e-05,
        1.4016155878e-05,
        6.69777914612e-06,
        0.000281157069701,
        0.000258141228055,
        0.000148602723882,
        0.000140402748619,
        7.54748866282e-05,
        -0.999999999426,
        0.999999999526,
        0.963610227192,
        0.0725056911199,
        -0.0268452938949,
        -0.000995970666999,
        -0.00266193048915,
        -0.000150812165154,
        0.000529760623622,
        0.000631035485598,
        0.000682690189552,
        0.000527373294158,
        0.000456361030818,
        0.226200750284,
        0.000126753322867,
        0.000250374589424,
        6.42631869531e-05,
        3.11022158889e-05,
        1.49748993792e-05,
        7.15473077161e-06,
        0.000300501118183,
        0.000276022999827,
        0.000158823794304,
        0.000150103570644,
        8.06512403391e-05,
        -0.999999999742,
        0.999999995953,
        0.963916514501,
        0.0610479611745,
        -0.0468623964928,
        -0.00174592668376,
        -0.00465183927055,
        -0.000266839150406,
        0.000920551870241,
        0.0010996486839,
        0.00118864676442,
        0.000918887540317,
        0.000795225304662,
        0.000220994529524,
        1.01160108459,
        0.000436515965091,
        0.000112050436577,
        5.42778967702e-05,
        2.60942529681e-05,
        1.24739309967e-05,
        0.000523641644441,
        0.000481188232072,
        0.000276834840603,
        0.000262116671757,
        0.000140525297299,
        -0.999999998085,
        0.999999999268,
        0.963619435919,
        0.0646018872729,
        -0.0372853472906,
        -0.00138356447064,
        -0.00369741700142,
        -0.000209685493698,
        0.000734220567634,
        0.000875251976671,
        0.00094732855993,
        0.000731847621787,
        0.000633343681491,
        0.000175929001378,
        0.000347486762347,
        1.03777120537,
        8.91924605787e-05,
        4.3166640414e-05,
        2.0780803561e-05,
        9.92958724898e-06,
        0.000417047269557,
        0.000383114976212,
        0.000220422018543,
        0.00020834158723,
        0.000111933144328,
        -0.999999998889,
        0.999999979401,
        0.849511078896,
        -1.00000001428,
        -0.0834261002357,
        0.0,
        -0.00865554890939,
        0.00123650703812,
        0.0054097211202,
        0.00432777552547,
        0.00472120918853,
        0.00344371938645,
        0.00258278946048,
        0.000860929840491,
        0.00139901096742,
        0.000349752743217,
        1.33660121402,
        0.000201780433208,
        8.40751801707e-05,
        5.21266139723e-05,
        0.00193709215119,
        0.00161424341217,
        0.00112997043626,
        0.0010761623052,
        0.000457368977514,
        -0.999999980772,
        0.999999976805,
        0.0,
        0.0,
        0.0,
        -0.504960652149,
        0.0,
        0.0,
        0.0,
        -0.288548812184,
        -0.183622020786,
        -0.301357316381,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.30742654246,
        -0.00470870853654,
        0.0,
        -0.100452450033,
        0.0,
        0.0,
        0.0,
        0.0,
        -0.999999999816,
        1.0,
        0.963610239581,
        0.0406356518821,
        -0.0772773068631,
        -0.00286637251641,
        -0.0076625081339,
        -0.000434118596529,
        0.00152233503437,
        0.00181444511892,
        0.00196381404345,
        0.0015170839696,
        0.0013129051624,
        0.000364648211635,
        0.000720274950812,
        0.000184865744939,
        8.94704158008e-05,
        4.30750157778e-05,
        1.0,
        2.05797056486e-05,
        0.000864449312156,
        0.000794104604314,
        0.000456886295476,
        0.000431826585611,
        0.000231999686136,
        -1.0,
        1.0,
        0.963610151941,
        0.0344078352223,
        -0.0873199836354,
        -0.00323889606209,
        -0.00865830718416,
        -0.000490536961287,
        0.00172017479636,
        0.00205024960547,
        0.00221902856026,
        0.00171424261453,
        0.00148352894414,
        0.000412037681752,
        0.000813881362305,
        0.000208890843512,
        0.000101097967498,
        4.8673040598e-05,
        2.32542447104e-05,
        1.0,
        0.000976792691177,
        0.000897305652004,
        0.000516263043427,
        0.000487946527628,
        0.00026215031812,
        -1.0,
        0.999999999079,
        0.963608566793,
        0.0845882825821,
        -0.0349233054762,
        -0.00129541424807,
        -0.00346288948333,
        -0.000196228782923,
        0.00068780508532,
        0.000819852896563,
        0.000887387340734,
        0.000685522729671,
        0.000593256527008,
        0.000164778038718,
        0.000325477179375,
        8.35372241175e-05,
        4.04296974741e-05,
        1.94645017414e-05,
        9.29950016402e-06,
        0.000390620373886,
        1.04322382269,
        0.00035883923016,
        0.000206455706537,
        0.000195136163166,
        0.000104836209689,
        -0.999999999951,
        0.99999999707,
        0.963613580718,
        0.0822646145851,
        -0.0349089060141,
        -0.00129488446998,
        -0.00346148662633,
        -0.000196168629191,
        0.000687615424121,
        0.000819591588495,
        0.00088705806349,
        0.000685284447081,
        0.000593062583359,
        0.000164717266709,
        0.000325358033681,
        8.35061445837e-05,
        4.04167113804e-05,
        1.94573570391e-05,
        9.29649979816e-06,
        0.000390481333422,
        0.000358722036465,
        1.03934916029,
        0.000206387766368,
        0.000195065313912,
        0.000104800773152,
        -0.999999999022,
        0.999999993039,
        0.963685815899,
        0.0806780378922,
        -0.0446437690422,
        -0.00166555690556,
        -0.00442562713414,
        -0.000255310083938,
        0.000873751374319,
        0.00104465177251,
        0.0011295965042,
        0.000872504530718,
        0.000755224167094,
        0.000210434565729,
        0.000414821002896,
        0.00010659780702,
        5.16307418803e-05,
        2.48210435064e-05,
        1.18722719476e-05,
        0.000497785136781,
        0.00045728931453,
        0.000263552637943,
        1.15402234992,
        0.000249155585637,
        0.000133649895516,
        -0.999999997692,
        0.999999999324,
        0.963517564979,
        0.079455339663,
        -0.0443018001648,
        -0.0016549701455,
        -0.00439447671539,
        -0.000253527482708,
        0.000869736465439,
        0.00103825255601,
        0.00112294563083,
        0.00086859361701,
        0.000750913338111,
        0.000209355604655,
        0.000412582004367,
        0.000105991186921,
        5.13757345395e-05,
        2.46809252468e-05,
        1.18014213961e-05,
        0.000495238459213,
        0.000455136157305,
        0.000262066600314,
        0.000247881944231,
        1.13071652458,
        0.000133090943373,
        -0.99999999874,
        1.0,
        -0.0686929685224,
        -0.0654698928803,
        -0.00235127532646,
        -0.00548630884717,
        -0.000447861946021,
        0.0017634572056,
        0.001791448237,
        0.00185252013954,
        0.0014032267925,
        0.00124731268693,
        0.000360551327225,
        0.000643145605941,
        0.000170531033719,
        9.01378333483e-05,
        4.08056410382e-05,
        1.88802220357e-05,
        0.000818548964209,
        0.000740591903818,
        0.000409274479851,
        0.00045799763468,
        0.000224126504798,
        1.36266749886,
        0.999999998778,
        -0.963610179044,
        -1.1887321261e-06,
        -0.000284594065005,
        -0.00175768159167,
        -0.0087461302269,
        -0.00692123043253,
        -0.00368950182375,
        -0.00289035701001,
        -0.00141459362624,
        -0.00112155946282,
        -2.69080650206e-06,
        -5.68987383027e-05,
        -4.95952492388e-09,
        -2.52503142986e-11,
        -1.63838436038e-13,
        -1.30404735955e-15,
        -0.000167399240337,
        -0.000110316648109,
        -2.76290101856e-06,
        -2.09497692089e-06,
        -2.61421990216e-08,
        -1.00000000006,
        -1.0,
        0.620668198221,
        -1.0,
        -0.0280901483472,
        0.0113195816923,
        0.00310910361872,
        0.004602388719,
        0.00913138349891,
        0.00617718757725,
        0.00665547861674,
        0.00407332214973,
        0.00349613012811,
        0.000761277437504,
        0.00162326046942,
        0.000355637306301,
        0.000164641690476,
        7.46527930022e-05,
        3.43516850744e-05,
        0.00197789858348,
        0.00186408000895,
        0.000948336687616,
        0.000914512302534,
        0.000443672363632,
        -1.0,
        -0.0505290135154,
        -1.0,
        -0.04583231732,
        -0.00378698716123,
        -0.00592094890588,
        -0.00149555581701,
        -0.00160625327667,
        -0.000744432089289,
        -0.00069915898508,
        -0.000284630600471,
        -0.000216478473732,
        -9.45889450406e-06,
        -5.29593992211e-05,
        1.36557934393e-06,
        2.64756088975e-06,
        2.07319173037e-06,
        1.2609253246e-06,
        -7.7062184739e-05,
        -7.14082389237e-05,
        -1.59429932997e-05,
        -1.57316154606e-05,
        3.55969523859e-07,
        -0.153264240299,
        0.999999998403,
        -1.00000000258,
        -1.0,
        0.999989678551,
        0.0773698736194,
        1.0,
        -0.489405381404,
        -0.997873881436,
        -0.124672152048,
        -0.45138463676,
        -0.152453116989,
        -0.00983683734968,
        0.00258850844542,
        0.0115134850229,
        0.00855935340162,
        0.00875607146342,
        0.00229423838868,
        0.00438429130766,
        0.00104433222827,
        0.000489627196617,
        0.000204514468216,
        9.00605350729e-05,
        0.00488892585509,
        0.0054738504605,
        0.00256245540353,
        0.00276681412874,
        0.0011918214067,
        -0.99999992881,
        0.999997864969,
        -0.187177951693,
        0.999999919893,
        -0.0778758804644,
        -0.0285673915056,
        -0.851245405873,
        -0.0744069382797,
        -0.0268858171523,
        -0.00189553545856,
        0.000724094806283,
        0.00192888551141,
        0.00156078145936,
        0.00154387376918,
        0.000428695369058,
        0.000818470702324,
        0.000201956374251,
        9.54202141598e-05,
        4.14943862179e-05,
        1.86464715211e-05,
        0.000928077374627,
        0.000983293613692,
        0.000494885266653,
        0.000513391454495,
        0.000237976448474,
        -0.999999991115,
        0.999997933949,
        -0.0134283512443,
        0.999999999996,
        -0.333813884242,
        -0.0884458264955,
        -0.309415689652,
        -0.848868276847,
        -0.104188057319,
        -0.00660882223211,
        0.00181552805485,
        0.00777445268796,
        0.00584906332595,
        0.00597150340192,
        0.00157638642563,
        0.00300700808425,
        0.000719203786578,
        0.000337979618473,
        0.000141456890903,
        6.24082569926e-05,
        0.00335516998175,
        0.00374494320521,
        0.00176242869144,
        0.00189808896197,
        0.000823127509166,
        -0.999999999999,
        0.999997948027,
        -0.173091417562,
        1.0,
        -0.12731378657,
        -0.0361193080855,
        -0.118605004905,
        -0.0406246175222,
        -0.989501576954,
        -0.00266192402275,
        0.000789260399263,
        0.00301227638645,
        0.00229850081625,
        0.00232960248703,
        0.000621172833439,
        0.00118568342413,
        0.000285402318056,
        0.000134258071793,
        5.66886826287e-05,
        2.51163853107e-05,
        0.00132804008481,
        0.0014649331516,
        0.000699670438979,
        0.000747091145462,
        0.000328897465522,
        -1.0,
        0.9999979295,
        -0.277552437915,
        0.999999998316,
        -0.0230494605373,
        -0.00714504765616,
        -0.0216316410843,
        -0.00757444797543,
        -0.000516469401822,
        -0.31146505219,
        0.000167932252115,
        0.000556451390759,
        0.000432923567437,
        0.000434636836743,
        0.000117469517581,
        0.000224386162669,
        5.44676640414e-05,
        2.56573599899e-05,
        1.09553826297e-05,
        4.87986687386e-06,
        0.000252564299629,
        0.000274355433533,
        0.000133585489885,
        0.00014107034033,
        6.33199072575e-05,
        -0.999999999589,
        0.999997941902,
        -0.336079591026,
        0.99999999932,
        0.0125469683991,
        0.0057091135688,
        0.0123086935127,
        0.00468672178863,
        0.000353663653209,
        -0.000159296172136,
        -0.368295178182,
        -0.000325866895845,
        -0.000278519568841,
        -0.000269449935134,
        -7.75430255095e-05,
        -0.000148097400791,
        -3.72982306034e-05,
        -1.76845711394e-05,
        -7.87136654053e-06,
        -3.57534444027e-06,
        -0.000169714000061,
        -0.000173593205619,
        -9.13850163447e-05,
        -9.25220009282e-05,
        -4.47512346582e-05,
        -0.999999999992,
        0.999997960133,
        -0.374576932001,
        1.0,
        0.0733558593302,
        0.0197434960724,
        0.0679034253811,
        0.0232808217377,
        0.0015572012854,
        -0.000429885440324,
        -0.00174430466247,
        -0.196705868206,
        -0.0013152865279,
        -0.00133536540357,
        -0.000352991307576,
        -0.00067482793231,
        -0.000161542098683,
        -7.58893254085e-05,
        -3.19031236839e-05,
        -1.41005575564e-05,
        -0.00075515740167,
        -0.000836623566508,
        -0.000396609636269,
        -0.000424893250946,
        -0.000185689411812,
        -1.00000000001,
        0.999997931239,
        -0.443987225965,
        0.999999999997,
        0.120335697234,
        0.0352106222413,
        0.112317126539,
        0.0389035883294,
        0.00261531184804,
        -0.000799263272152,
        -0.0028793823343,
        -0.00221147417805,
        -0.69095684305,
        -0.00223177695551,
        -0.000597756031139,
        -0.00114187187233,
        -0.000275626066955,
        -0.000129702716879,
        -5.50153308237e-05,
        -2.44263318779e-05,
        -0.00128183729517,
        -0.0014047014532,
        -0.000676115457655,
        -0.000718539809043,
        -0.000318837560104,
        -0.999999998965,
        0.999997925532,
        -0.475164060207,
        1.0,
        0.154667158392,
        0.0438470095083,
        0.143988845147,
        0.0494958184106,
        0.00328231869045,
        -0.000969296754559,
        -0.00367549455819,
        -0.00280365049933,
        -0.00283878854329,
        -0.911172703964,
        -0.000756699810495,
        -0.00144515910553,
        -0.00034778506039,
        -0.000163578264486,
        -6.91077309142e-05,
        -3.06244275064e-05,
        -0.00161950001887,
        -0.00178433779126,
        -0.00085300950128,
        -0.000910086107043,
        -0.000401061182903,
        -0.999999999993,
        0.999999999998,
        -0.659437547799,
        1.00000000034,
        0.276592635187,
        0.0826379782873,
        0.25876155672,
        0.0901187139358,
        0.00589335678242,
        -0.00189196730307,
        -0.00657840086975,
        -0.00521379219591,
        -0.0051403980365,
        -0.00139009609284,
        -0.249018657347,
        -0.00266262304773,
        -0.000639491737248,
        -0.000302586313234,
        -0.000128378301431,
        -5.7255234986e-05,
        -0.00297115518015,
        -0.00326066037859,
        -0.00157188537133,
        -0.00166634835554,
        -0.000744827656034,
        -0.999999999868,
        0.999997765489,
        -0.563968023983,
        0.999999999695,
        0.203997224402,
        0.0611657900933,
        0.190852854599,
        0.0664036315602,
        0.00449555000285,
        -0.0013958879686,
        -0.00488773165474,
        -0.0037779392671,
        -0.00380456410764,
        -0.00102304975592,
        -0.00195388407987,
        -0.699368173705,
        -0.000472728170008,
        -0.000222584913484,
        -9.46609546374e-05,
        -4.20839820311e-05,
        -0.00219541152459,
        -0.00239749835942,
        -0.00115936512072,
        -0.00122928800023,
        -0.000547989676173,
        -0.999999982409,
        0.705412825951,
        -0.276507728294,
        1.0,
        -1.00000009514,
        0.0,
        -1.00000011148,
        0.0,
        0.264530509456,
        0.264530506556,
        0.19839785496,
        -0.0991988509879,
        0.077154709554,
        0.0103332186782,
        0.0275552510802,
        0.0130887471595,
        -0.103880340445,
        0.00516661041728,
        0.00202358899213,
        0.000818046587733,
        0.0716436699329,
        0.0330663015717,
        0.0151553878407,
        0.0,
        0.00551104988417,
        -1.0,
        0.999997933691,
        -0.851033543711,
        1.0,
        0.372235649158,
        0.116238538726,
        0.349599172283,
        0.122562924748,
        0.00836690858999,
        -0.00274142397653,
        -0.00899459386759,
        -0.0070095645713,
        -0.00703301678899,
        -0.00190303608362,
        -0.00363498255824,
        -0.000882993411423,
        -0.000415997459946,
        -0.0493189025623,
        -0.000177767877095,
        -7.92144781798e-05,
        -0.00409272897672,
        -0.00444118399685,
        -0.00216549393904,
        -0.00228509863176,
        -0.00102710144801,
        -1.0,
        0.999997933691,
        -0.91611341067,
        1.0,
        0.39052364273,
        0.127103758984,
        0.368285380637,
        0.130177367907,
        0.00898227696668,
        -0.00306783789483,
        -0.0095013144019,
        -0.00747505577435,
        -0.00747140971311,
        -0.00203498285999,
        -0.00388693925933,
        -0.000948003101017,
        -0.000446949434406,
        -0.000191896579182,
        -0.0196158459024,
        -8.5705066481e-05,
        -0.00438491389864,
        -0.00472793209026,
        -0.00232466248473,
        -0.00244184295006,
        -0.00110663711527,
        -1.0,
        0.999997930849,
        -1.0,
        1.0,
        0.423944023493,
        0.140790459113,
        0.400625847757,
        0.142185932361,
        0.00986231931677,
        -0.00343487709387,
        -0.0103497448833,
        -0.00818076797415,
        -0.00816141185744,
        -0.00223009027911,
        -0.00425956576677,
        -0.00104091982256,
        -0.000490929144559,
        -0.000211259205962,
        -9.44559935476e-05,
        -0.0079578688923,
        -0.00480982513817,
        -0.00516990348309,
        -0.002552369327,
        -0.0026750533233,
        -0.00121718778945,
        -0.999999999535,
        0.999997858708,
        -0.530669325515,
        0.99999999997,
        0.175412876305,
        0.0534636666502,
        0.164363352961,
        0.0573506695188,
        0.00389108254516,
        -0.00124029374633,
        -0.00422044384162,
        -0.0032715622831,
        -0.00328947775854,
        -0.000886777255767,
        -0.00169382301773,
        -0.000410472795142,
        -0.000193304463937,
        -8.23731662879e-05,
        -3.66556644041e-05,
        -0.00190486439898,
        -0.877158833247,
        -0.00207489561421,
        -0.00100674775841,
        -0.0010652096397,
        -0.000476457859779,
        -0.999999998978,
        0.999996976882,
        -0.544551954897,
        0.999999999072,
        0.214258919207,
        0.0617922562063,
        0.19973367385,
        0.0689664790732,
        0.00461359068835,
        -0.00138588306018,
        -0.005111620614,
        -0.00391410792655,
        -0.00395550755424,
        -0.00105716444637,
        -0.00201921936423,
        -0.00048669941552,
        -0.000228997538368,
        -9.69620755982e-05,
        -4.30120175318e-05,
        -0.00226523531477,
        -0.00248795932214,
        -0.840251560048,
        -0.00119400095337,
        -0.0012710061113,
        -0.000562325900198,
        -0.999999999998,
        0.999942080393,
        -0.619099005897,
        1.00000000018,
        0.231365293748,
        0.0720467230658,
        0.217186649496,
        0.0762811760545,
        0.0053677020572,
        -0.0014313888312,
        -0.00541989788531,
        -0.00430535290163,
        -0.00430908107913,
        -0.00117139179237,
        -0.0022356154515,
        -0.00054049898374,
        -0.000255979864406,
        -0.000109162738128,
        -4.87643067862e-05,
        -0.00250866040853,
        -0.00272299658797,
        -0.00133027971223,
        -0.367805640062,
        -0.0014052973822,
        -0.00063415436176,
        -1.0,
        0.999960160799,
        -0.627991602553,
        1.00000000005,
        0.2632096184,
        0.078348547699,
        0.245977090278,
        0.0856754470158,
        0.0056409157639,
        -0.00174847716799,
        -0.00624456454432,
        -0.00492279899385,
        -0.00490198427362,
        -0.0013256694429,
        -0.00252774830825,
        -0.000606080700968,
        -0.000287185016097,
        -0.000122553679978,
        -5.4152097959e-05,
        -0.00283607358846,
        -0.00309756514811,
        -0.00149804536643,
        -0.0015884616079,
        -0.352724660862,
        -0.000710319384152,
        0.987181764667,
        -0.65078882791,
        0.206735191287,
        0.134615934926,
        0.228998974439,
        0.089743973451,
        0.03365400567,
        0.0673080350546,
        0.0252405020676,
        0.0,
        0.0,
        0.00131460924638,
        0.0,
        0.00166517199398,
        0.000219101536121,
        8.76406137714e-05,
        3.56039991108e-05,
        0.00455731223784,
        0.0,
        0.0,
        0.00175281234693,
        -0.000701124797577,
        -0.137606512896,
        -0.999999999999,
        -0.999997866098,
        0.999999997063,
        1.89149512629e-05,
        0.0238073062056,
        0.0309035853934,
        0.108771258267,
        0.124722894069,
        0.0487701920689,
        0.0348759915408,
        0.0135110332606,
        0.00895364307497,
        3.36114817387e-07,
        5.33918710047e-05,
        5.43858842531e-11,
        2.25588394278e-14,
        1.21864607394e-17,
        7.5179534814e-21,
        0.00026576590949,
        0.000145141210436,
        6.8760186535e-07,
        4.61619550046e-07,
        6.13343482155e-10,
        0.0163526545687,
        1.0,
        -1.0945853897e-05,
        1.0,
        -0.0490519328571,
        0.999999999686,
        0.999999999911,
        0.999999999306,
        0.99882384269,
        0.215935212627,
        0.00122342881701,
        0.00637987755724,
        0.000887241880326,
        0.00076051682332,
        8.69913974106e-05,
        -0.000246314674135,
        -0.000333667840702,
        -0.000538143918265,
        -0.000370193022858,
        -0.000318165773687,
        -8.4045829986e-05,
        -0.000170057749521,
        -4.16889353769e-05,
        -1.97357058112e-05,
        -9.29969102738e-06,
        -4.35093890326e-06,
        -0.000205502316619,
        -0.000188469819099,
        -0.000106087808811,
        -0.000100197222363,
        -5.26751099244e-05,
        -0.999999913763,
        0.999999931556,
        0.998824077291,
        0.102888441917,
        0.000446861211435,
        0.000326385320469,
        0.0383056608308,
        0.000277995478218,
        3.17044368446e-05,
        -9.16832095915e-05,
        -0.000123838378127,
        -0.000199042475705,
        -0.00013690516939,
        -0.000117572410557,
        -3.10697107744e-05,
        -6.28751949256e-05,
        -1.54206296362e-05,
        -7.30156000759e-06,
        -3.44307501121e-06,
        -1.61153997282e-06,
        -7.60056958576e-05,
        -6.96367780035e-05,
        -3.92408500216e-05,
        -3.70369294461e-05,
        -1.94930481164e-05,
        -1.00000000085,
        0.999999990748,
        0.998824080148,
        0.182960720255,
        0.000881806552237,
        0.000661500977358,
        0.000550084034554,
        0.0131291738346,
        6.18868882827e-05,
        -0.0001948821077,
        -0.000260247588268,
        -0.000412820845633,
        -0.000283778976545,
        -0.000242969236457,
        -6.42960174175e-05,
        -0.000130186020961,
        -3.1983210217e-05,
        -1.51541988911e-05,
        -7.16592615726e-06,
        -3.35930299184e-06,
        -0.00015758013978,
        -0.000143826839449,
        -8.13843098667e-05,
        -7.66178710525e-05,
        -4.04981417135e-05,
        -0.999999859694,
        0.999999999922,
        0.998824079601,
        0.12102080891,
        0.000135701706635,
        9.90590984982e-05,
        8.44152602352e-05,
        9.62847266232e-06,
        0.0552423765154,
        -2.78080656465e-05,
        -3.75669595871e-05,
        -6.03944325192e-05,
        -4.15403638598e-05,
        -3.56762032872e-05,
        -9.42749269705e-06,
        -1.90780985008e-05,
        -4.67889333955e-06,
        -2.21539518024e-06,
        -1.04462485644e-06,
        -4.88925697087e-07,
        -2.30617372293e-05,
        -2.11307155313e-05,
        -1.19063992764e-05,
        -1.12382035646e-05,
        -5.91435780269e-06,
        -1.0,
        0.999999999425,
        0.998824079273,
        0.0975690593651,
        -0.000483439508736,
        -0.000341193450428,
        -0.000299707473653,
        -3.47316168348e-05,
        8.98618133346e-05,
        0.020499893012,
        0.000123348314868,
        0.000201919707254,
        0.000138986877816,
        0.000119853609365,
        3.16114349601e-05,
        6.39243750202e-05,
        1.56413352629e-05,
        7.39908717086e-06,
        3.47569439866e-06,
        1.62327139669e-06,
        7.71354885226e-05,
        7.10404747034e-05,
        3.98050419314e-05,
        3.7700777935e-05,
        1.97260386584e-05,
        -0.999999999998,
        1.00000000003,
        0.998824077315,
        0.107959936607,
        -0.0010062686944,
        -0.000729205085413,
        -0.000625494358053,
        -7.15892121501e-05,
        0.00020203790603,
        0.000273819347759,
        0.0456637223705,
        0.000441845735253,
        0.000303954218627,
        0.00026126655643,
        6.90125888565e-05,
        0.000139637436392,
        3.42295828063e-05,
        1.62041404732e-05,
        7.63474891249e-06,
        3.57178140684e-06,
        0.000168732730354,
        0.000154769330658,
        8.71053455751e-05,
        8.22757332086e-05,
        4.3247329833e-05,
        -1.00000000008,
        1.00000000004,
        0.998824082028,
        0.0911985696123,
        -0.00111573716189,
        -0.000812701431355,
        -0.000693900714218,
        -7.92175222792e-05,
        0.0002273401623,
        0.000307387520257,
        0.000494685679484,
        0.0332773410075,
        0.000340264329904,
        0.000292299260887,
        7.72313593407e-05,
        0.000156284161757,
        3.83233222108e-05,
        1.81446095812e-05,
        8.55383672203e-06,
        4.00303648304e-06,
        0.000188897958863,
        0.000173133102532,
        9.75220083595e-05,
        9.2067550227e-05,
        4.84361268564e-05,
        -1.0,
        0.999999999083,
        0.998824101296,
        0.0832723994487,
        -0.00163495380333,
        -0.00118636814109,
        -0.00101642113512,
        -0.000116257000843,
        0.00032952157839,
        0.00044632285036,
        0.000719702444064,
        0.000495082426848,
        0.184184594383,
        0.000425484983529,
        0.000112398404247,
        0.000227429247201,
        5.57551103155e-05,
        2.63951975485e-05,
        1.24381941056e-05,
        5.81947535155e-06,
        0.000274836052254,
        0.00025204186907,
        0.000141881980672,
        0.000133997261808,
        7.04500322998e-05,
        -0.999999999555,
        0.99999999888,
        0.998823916398,
        0.0665763095268,
        -0.0017489395903,
        -0.00126804663434,
        -0.00108719217976,
        -0.000124399257781,
        0.000351687845809,
        0.000476519577716,
        0.000768718125157,
        0.000528808341429,
        0.000454513567046,
        0.340660748875,
        0.000120061188079,
        0.000242930439978,
        5.95520324914e-05,
        2.81921098237e-05,
        1.3283768659e-05,
        6.21479141273e-06,
        0.000293556215633,
        0.000269241716041,
        0.000151544425074,
        0.000143134079468,
        7.52435836949e-05,
        -1.0,
        0.999999996904,
        1.0,
        0.0322976929952,
        -0.00300413543105,
        -0.00224668207032,
        -0.00188721324785,
        -0.000247593594418,
        0.000570311443434,
        0.000795629341023,
        0.00130038095176,
        0.000892798866467,
        0.000764216437665,
        0.000202578005906,
        1.08604068192,
        0.000408795136639,
        0.00010098574074,
        4.78393409123e-05,
        2.24412752337e-05,
        1.05193476454e-05,
        0.000493708068842,
        0.000453677696013,
        0.00025777141004,
        0.000240788828232,
        0.000126762880686,
        -0.99999999961,
        0.99999999991,
        0.998823122227,
        0.0426012427859,
        -0.00241650070629,
        -0.00175322308161,
        -0.00150272539447,
        -0.000172324287168,
        0.000485996418778,
        0.000658628200105,
        0.00106265885794,
        0.000730904646167,
        0.00062815875107,
        0.000165954058406,
        0.00033580813818,
        1.07075000168,
        8.23213826826e-05,
        3.89717095035e-05,
        1.83625389121e-05,
        8.59199666072e-06,
        0.000405740052589,
        0.000372176796006,
        0.000209469364582,
        0.000197826933395,
        0.00010401419582,
        -0.999999999826,
        0.999999999959,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        1.16968674544,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        -1.0,
        0.999999999996,
        0.998824055881,
        -0.119749465493,
        -0.00428900769017,
        -0.00311077063536,
        -0.0026662754756,
        -0.000305040701611,
        0.000863220423727,
        0.0011694665012,
        0.00188626465568,
        0.0012975761799,
        0.00111523141838,
        0.000294598190301,
        0.000596089355683,
        0.000146128814612,
        6.91783574905e-05,
        1.0,
        3.2597125001e-05,
        1.52507888271e-05,
        0.000720323711826,
        0.000660630195025,
        0.000371859087986,
        0.000351211293399,
        0.000184636717036,
        -1.0,
        1.0,
        0.998824055881,
        -0.0399301682499,
        -0.00497395511845,
        -0.00360760417977,
        -0.00309208143514,
        -0.000353758474497,
        0.00100107871892,
        0.0013562360549,
        0.00218750741965,
        0.00150480463645,
        0.00129333832533,
        0.000341647043989,
        0.000691287849186,
        0.000169466427003,
        8.02265351374e-05,
        3.78030912129e-05,
        1.0,
        1.76864373841e-05,
        0.000835363089813,
        0.000766135754838,
        0.000431246982569,
        0.000407301514738,
        0.000214124301623,
        -1.0,
        1.0,
        0.998824055881,
        -0.0610401115351,
        -0.00561633781333,
        -0.00407355049497,
        -0.00349142521602,
        -0.00039944776839,
        0.00113036922931,
        0.00153139722546,
        0.00247002756026,
        0.00169915344017,
        0.001460375465,
        0.000385771614653,
        0.000780569275882,
        0.00019135346363,
        9.05880093161e-05,
        4.26854718238e-05,
        1.99706952364e-05,
        1.0,
        0.000943252276286,
        0.000865083737978,
        0.000486943607658,
        0.000459905469263,
        0.000241779041232,
        -1.0,
        0.999999999435,
        0.998818283044,
        0.076478407277,
        -0.00226503918912,
        -0.00164311068912,
        -0.00140819919691,
        -0.000161201440617,
        0.000455743541006,
        0.000617468999316,
        0.000996016308406,
        0.000685171666623,
        0.000588894266706,
        0.000155557052174,
        0.000314758623888,
        7.71637015925e-05,
        3.65302567161e-05,
        1.72127865014e-05,
        8.05326755608e-06,
        0.000380352254622,
        1.07509866553,
        0.000348848079037,
        0.000196364408044,
        0.000185456771154,
        9.7498427128e-05,
        -0.999999999603,
        0.999999999352,
        0.998823506626,
        0.0731445647568,
        -0.00226417217872,
        -0.00164308937929,
        -0.00140757333221,
        -0.000161373127951,
        0.000455322304658,
        0.000616999764845,
        0.000995304929392,
        0.000684658456176,
        0.000588510495011,
        0.000155475440597,
        0.000314566047371,
        7.71142330742e-05,
        3.65091629473e-05,
        1.7203631355e-05,
        8.04861507404e-06,
        0.000380116542472,
        0.000348623481494,
        1.06976513839,
        0.000196249322487,
        0.000185338088795,
        9.74385254541e-05,
        -0.999999999391,
        0.999999995782,
        0.999089975898,
        0.0585592244346,
        -0.00285859439439,
        -0.00208702664759,
        -0.00178345931401,
        -0.00020779922188,
        0.000578918694649,
        0.000782218798409,
        0.00127480874199,
        0.000870056970413,
        0.000752137169979,
        0.00019679860203,
        0.000399971246438,
        9.83992998425e-05,
        4.65106001097e-05,
        2.18609777452e-05,
        1.02706756795e-05,
        0.000482037057551,
        0.000445386312571,
        0.000251376348769,
        1.03224515369,
        0.000236237997913,
        0.000124293851521,
        -0.999999999765,
        0.999999997716,
        0.998898896606,
        0.0610230152284,
        -0.00286918223815,
        -0.00213852075997,
        -0.0017821007559,
        -0.000216397956329,
        0.000562047052615,
        0.000764427305488,
        0.00125727785036,
        0.000857643107705,
        0.000740895884049,
        0.000195327095996,
        0.000397389608108,
        9.76635472389e-05,
        4.59551892145e-05,
        2.16094628572e-05,
        1.01732860276e-05,
        0.00047821461172,
        0.000438924683499,
        0.000248649146141,
        0.000232371890378,
        1.0307085187,
        0.000122921360673,
        -0.999999999941,
        0.678956354482,
        0.184109910894,
        0.0,
        -0.0175638866104,
        0.0,
        -0.0037636898513,
        -0.00405320402277,
        -0.00277324500081,
        0.0,
        -0.0026552955965,
        0.0,
        -0.000663823897754,
        -0.000663823907908,
        -0.00033191194842,
        0.0,
        -5.1861242372e-05,
        0.0,
        -0.00132764780187,
        -0.00099573585734,
        0.0,
        0.0,
        0.0,
        1.17424249032,
        0.999999999931,
        -0.998824084767,
        -1.36348665212e-06,
        -0.000357869235257,
        -0.00208752858682,
        -0.0104185742358,
        -0.0066107285858,
        -0.00351710422382,
        -0.00281515929472,
        -0.00122750745173,
        -0.000831070949349,
        -3.41827109254e-07,
        -1.75383486129e-05,
        -2.12539750293e-10,
        -3.17617278774e-13,
        -5.77334998902e-16,
        -1.2646711015e-18,
        -6.51370569796e-05,
        -3.93810318869e-05,
        -5.02349213739e-07,
        -3.5999085096e-07,
        -1.59803281359e-09,
        -1.00000000002,
        -1.0,
        1.0,
        -1.0,
        -0.00167728959881,
        0.0149101401834,
        0.00132443213357,
        0.00453340102962,
        0.00539041613902,
        0.00429413447782,
        0.00691402905774,
        0.00380147290614,
        0.00324392361924,
        0.000676420305866,
        0.0014741304657,
        0.00030965457792,
        0.000140364835549,
        6.23427258121e-05,
        2.81083880206e-05,
        0.00180718683415,
        0.00170034049352,
        0.000848541124922,
        0.000817622665136,
        0.00038883952134,
        -1.0,
        -0.0819843261143,
        -1.0,
        -0.0028175948972,
        -0.00476400249166,
        -0.00230325299197,
        -0.00140835511432,
        -0.000907793418969,
        -0.000494445601532,
        -0.000693739309859,
        -0.000252348246584,
        -0.000190458491277,
        -7.3069559189e-06,
        -4.47498284783e-05,
        1.54359800699e-06,
        2.36628987581e-06,
        1.76000422804e-06,
        1.03769070367e-06,
        -6.59182673641e-05,
        -6.10258954697e-05,
        -1.27487261917e-05,
        -1.26210482405e-05,
        8.02580731527e-07,
        -0.268990962378,
        0.999999997286,
        -1.0000000007,
        -1.0,
        0.988995312697,
        0.0330239332245,
        0.99999998496,
        -0.0606432066559,
        -0.999937284103,
        -0.0210989881671,
        -0.0357580951166,
        -0.0104390272942,
        -0.00182313631373,
        0.000279764079649,
        0.00402955334366,
        0.00318078624583,
        0.00329423848412,
        0.000923944611849,
        0.00171185380451,
        0.000431098573786,
        0.000207416475349,
        8.86447810913e-05,
        3.99810372387e-05,
        0.00188859263674,
        0.00212744261279,
        0.00102209622618,
        0.00110521447647,
        0.000488036466644,
        -0.999999999996,
        0.989032150046,
        -0.0709627976343,
        0.99999999987,
        -0.0106516107235,
        -0.00502726056828,
        -0.857343111289,
        -0.00648263096532,
        -0.00201081957667,
        -0.000371371737479,
        0.000104472354799,
        0.000738146034297,
        0.000630869019429,
        0.000634757949769,
        0.000187361162142,
        0.000347146495445,
        9.02467981257e-05,
        4.36863103036e-05,
        1.93878137386e-05,
        8.90033918727e-06,
        0.000389228863426,
        0.000416487790595,
        0.000213653125569,
        0.000222876771816,
        0.000105222565438,
        -0.9999999945,
        0.98903208943,
        -0.002697834124,
        0.999999999998,
        -0.0414982572272,
        -0.0150033366317,
        -0.0245954947312,
        -0.855712033158,
        -0.0071905103967,
        -0.00121794414681,
        0.000201704733521,
        0.00274479539872,
        0.00218809559492,
        0.0022668399207,
        0.000639095317324,
        0.00118292300676,
        0.00029911972819,
        0.000144114491221,
        6.1797325926e-05,
        2.79064252956e-05,
        0.00130666822975,
        0.00146672509513,
        0.000707832653489,
        0.000764094349409,
        0.000339604521828,
        -0.999999999991,
        0.989032083923,
        -0.0651166138653,
        0.999999994422,
        -0.0164309378042,
        -0.00630092059953,
        -0.00978521977498,
        -0.00290133878592,
        -0.988209186979,
        -0.000506223698207,
        9.67577105451e-05,
        0.00109981534686,
        0.000889604709016,
        0.000915150067557,
        0.000260543287268,
        0.000482548710136,
        0.000122762388885,
        5.92029650699e-05,
        2.55991840285e-05,
        1.16063396059e-05,
        0.000535006064003,
        0.000593718871615,
        0.00029063043288,
        0.000311151256535,
        0.000140299013884,
        -0.9999999986,
        0.989032079381,
        -0.105605041658,
        0.999999997307,
        -0.00364666476113,
        -0.00148797593557,
        -0.0021834514385,
        -0.000657691576068,
        -0.000118716947559,
        -0.127165316133,
        2.58861365522e-05,
        0.000247419135902,
        0.00020336268327,
        0.000207579718597,
        5.97231581658e-05,
        0.00011069578896,
        2.83423091617e-05,
        1.36815420491e-05,
        5.96817711925e-06,
        2.71715922184e-06,
        0.000123225933318,
        0.000135057119236,
        6.71356256707e-05,
        7.12377431701e-05,
        3.26198875196e-05,
        -0.999999999525,
        0.989032077828,
        -0.127958329236,
        0.999999999055,
        0.000819051903074,
        0.000627246015439,
        0.0005348839018,
        0.000187688827185,
        3.92240308498e-05,
        -1.87840926467e-05,
        -0.138886870066,
        -6.29320636716e-05,
        -6.2236709867e-05,
        -5.93397228578e-05,
        -1.91142219914e-05,
        -3.54348995503e-05,
        -9.66005043267e-06,
        -4.71363226186e-06,
        -2.20346320807e-06,
        -1.03556088792e-06,
        -4.07205669236e-05,
        -4.00712798961e-05,
        -2.28614353628e-05,
        -2.2535239137e-05,
        -1.17303543232e-05,
        -0.999999999959,
        0.9890320827,
        -0.143110062808,
        0.99999999934,
        0.00823090530803,
        0.00298386582356,
        0.00486605276913,
        0.00143785515918,
        0.000259115733933,
        -4.30815417135e-05,
        -0.00055215157135,
        -0.0709210454764,
        -0.000440313028981,
        -0.000453955639109,
        -0.000127961764272,
        -0.000237390115014,
        -6.00164847051e-05,
        -2.88992437256e-05,
        -1.24329025657e-05,
        -5.62128913612e-06,
        -0.000262889615801,
        -0.000293285359834,
        -0.000142310112903,
        -0.00015296333447,
        -6.8384427005e-05,
        -0.999999999525,
        0.989032071366,
        -0.169841217176,
        0.999999999525,
        0.014041334103,
        0.00552789943908,
        0.00837584262428,
        0.0025058633068,
        0.000449933856373,
        -9.10710335007e-05,
        -0.000948206254759,
        -0.000772159805257,
        -0.199848551869,
        -0.00079087843413,
        -0.00022613887144,
        -0.000419183783243,
        -0.00010692149458,
        -5.15774300504e-05,
        -2.24011317858e-05,
        -1.0176911364e-05,
        -0.000465810261798,
        -0.000513525693767,
        -0.000253307225667,
        -0.00026991803827,
        -0.00012265752798,
        -1.00000000005,
        0.989032069067,
        -0.182071231087,
        1.00000000041,
        0.0182086427195,
        0.0069662226257,
        0.0108348530725,
        0.00321869567616,
        0.000569472170201,
        -0.000108178424601,
        -0.00122228415075,
        -0.000988070073962,
        -0.00101558514477,
        -0.259571603079,
        -0.000288986490977,
        -0.000535513233365,
        -0.000136189475183,
        -6.56658427545e-05,
        -2.84049067355e-05,
        -1.2879709467e-05,
        -0.000593991313697,
        -0.000658549899141,
        -0.000322572783438,
        -0.000345124683503,
        -0.000155731316436,
        -0.99999999903,
        0.989011486207,
        -0.253240460259,
        0.999999998812,
        0.0332667220366,
        0.0133789836898,
        0.0198848297721,
        0.00597690651825,
        0.00107799916662,
        -0.000225810215929,
        -0.00225404240929,
        -0.00184151981019,
        -0.00188737405785,
        -0.000540943553609,
        -0.109648550504,
        -0.00100284144232,
        -0.000256353214525,
        -0.000123700561323,
        -5.38828458904e-05,
        -2.44864212601e-05,
        -0.00111520716427,
        -0.00122441182606,
        -0.000607200474605,
        -0.00064546413868,
        -0.000294785278389,
        -0.999999999994,
        0.989032322207,
        -0.216340994136,
        0.999999999033,
        0.0243226305651,
        0.00978326904535,
        0.014540496321,
        0.0043688719341,
        0.000787856290019,
        -0.000166795723565,
        -0.00164746851717,
        -0.00134924725658,
        -0.00137892212969,
        -0.000395725433424,
        -0.000733589431132,
        -0.308118967805,
        -0.000187529261051,
        -9.05016819314e-05,
        -3.94121566578e-05,
        -1.79287762658e-05,
        -0.000816091757887,
        -0.000896372156045,
        -0.000444240101385,
        -0.000472147512389,
        -0.000215577027339,
        -0.999999999977,
        1.0,
        -0.2929739362,
        0.999999979736,
        0.0462786766237,
        0.0273691034195,
        0.0276179192436,
        0.0138622750392,
        0.00373215377897,
        0.00373215316572,
        0.00188074707335,
        0.000940373516811,
        -0.00216285867114,
        -0.000211584018649,
        -0.000235093359564,
        -0.000176320008667,
        -0.0467981438596,
        0.0,
        -2.93866686804e-05,
        0.0,
        -0.000376149371444,
        -0.000329130701358,
        0.000282112055912,
        -0.000282112023206,
        -0.00023509334309,
        -0.99999999999,
        0.0,
        -0.999999999473,
        0.999999999813,
        0.999999873709,
        0.999999531466,
        0.999999761937,
        0.999999366191,
        1.00000364884,
        1.0000023373,
        -0.503930643871,
        0.251965583342,
        -0.20157230153,
        0.0629913978233,
        0.226769116175,
        0.070865367185,
        0.0291335359435,
        -0.0223513408374,
        -0.00393695939562,
        0.00157478460869,
        0.0881879401039,
        0.264563971082,
        0.151179423113,
        0.069290534938,
        0.0,
        -1.0,
        0.989032076569,
        -0.352096254345,
        1.0,
        0.0474177204213,
        0.0205528373217,
        0.0285716599012,
        0.0087186271957,
        0.00159914264029,
        -0.000390659942153,
        -0.00324902806855,
        -0.00271370526024,
        -0.00275231277739,
        -0.000800250998703,
        -0.00148339707168,
        -0.000382214798653,
        -0.000184708150141,
        -8.11872707852e-05,
        -0.00908164970151,
        -3.70969338536e-05,
        -0.00165668940771,
        -0.00179664600953,
        -0.000905346730815,
        -0.000953448542225,
        -0.0004424693611,
        -1.0,
        0.989032076569,
        -0.384390163021,
        1.0,
        0.0515866077328,
        0.0227703581484,
        0.0311459723478,
        0.00954099822958,
        0.00175713221317,
        -0.000443114268579,
        -0.00354491369591,
        -0.00297541307148,
        -0.00301205278496,
        -0.000878576444927,
        -0.00162858888597,
        -0.000420423978963,
        -0.000203241506957,
        -8.95317021907e-05,
        -4.09530343663e-05,
        -0.00376292063049,
        -0.00182056718589,
        -0.00196821305919,
        -0.000995820415015,
        -0.00104640937026,
        -0.000487526075065,
        -1.00000000002,
        0.989032173743,
        -0.203169546616,
        1.00000000007,
        0.0208357782146,
        0.00851446439059,
        0.0124764560447,
        0.0037605487744,
        0.000680416491607,
        -0.000148638006065,
        -0.00141459216768,
        -0.00116309684189,
        -0.00118688597064,
        -0.000341564533478,
        -0.000633131869962,
        -0.000162133047237,
        -7.82641781674e-05,
        -3.41504644121e-05,
        -1.55497652932e-05,
        -0.000704904861134,
        -0.397443637422,
        -0.000772239651231,
        -0.000384055144582,
        -0.000407406490732,
        -0.000186645417792,
        -0.999999999965,
        0.989032056023,
        -0.2087028171,
        0.999999999191,
        0.0255562339739,
        0.00992196689728,
        0.0152238285688,
        0.00454161784419,
        0.000812433437353,
        -0.000159497049238,
        -0.00172230067215,
        -0.00139744713601,
        -0.00143340749504,
        -0.000408871554198,
        -0.000757889701713,
        -0.000193027126462,
        -9.30900172551e-05,
        -4.03600178309e-05,
        -1.83192091426e-05,
        -0.000841590970452,
        -0.000930009185242,
        -0.365924015795,
        -0.00045730786437,
        -0.000488148559211,
        -0.000221141405444,
        -0.99999999968,
        0.989043819172,
        -0.237343186939,
        0.999999997808,
        0.0277623928401,
        0.0115223419919,
        0.016651091252,
        0.00504021894491,
        0.000920534442097,
        -0.000199058323993,
        -0.00188650647574,
        -0.00155628950331,
        -0.0015873480279,
        -0.000457673445698,
        -0.000848494529121,
        -0.000217746743973,
        -0.000105073083819,
        -4.59803207298e-05,
        -2.09436668454e-05,
        -0.000945035720411,
        -0.00103298715448,
        -0.000515544733633,
        -0.138974365718,
        -0.000545948496864,
        -0.000251149567655,
        -0.999999999977,
        0.989014538347,
        -0.240945191796,
        0.999999999196,
        0.0316178484011,
        0.0126371510825,
        0.0188847416586,
        0.00567067085836,
        0.00102542438081,
        -0.000211631124269,
        -0.00214282861205,
        -0.00174915590741,
        -0.00178956075322,
        -0.000512573563426,
        -0.000951154793905,
        -0.000242702145583,
        -0.000117025210142,
        -5.09764486076e-05,
        -2.31613233517e-05,
        -0.00105654174123,
        -0.00116067933514,
        -0.000574779939981,
        -0.000611737095795,
        -0.132703772135,
        -0.000279070913331,
        0.987688344928,
        -0.271102042082,
        0.0330903876933,
        0.0143967526359,
        0.0198010746831,
        0.00645480888097,
        0.0013800447931,
        0.000454634522868,
        -0.00267875847062,
        -0.00187513092611,
        -0.00200906877126,
        -0.000535751667925,
        -0.000971049922023,
        -0.000235437749596,
        -0.000133414718187,
        -5.93826297039e-05,
        -2.45901644892e-05,
        -0.00113010120663,
        -0.00102964778168,
        -0.000606906197208,
        -0.000627833989491,
        -0.000307638654992,
        -0.0629496261381,
        -0.999999998609,
        -0.98903207115,
        0.999999997236,
        1.80561815323e-05,
        0.0217723631878,
        0.0291802952849,
        0.100805126478,
        0.0481608804473,
        0.0191901374211,
        0.0138244846707,
        0.00539184070182,
        0.00399647136654,
        1.10172554436e-06,
        6.91488450599e-05,
        5.10068154955e-10,
        6.33755439906e-13,
        9.91645735253e-16,
        1.80930070954e-18,
        0.000281266158753,
        0.000161567393112,
        1.67999121712e-06,
        1.192078507e-06,
        4.0838377676e-09,
        0.0187630693849,
        1.0,
        -1.38347566606e-05,
        1.0,
        -0.0194073673818,
        0.999999998755,
        0.999999998334,
        -0.999999998919,
        0.999625118262,
        0.249191301344,
        1.00000000009,
        0.492154639084,
        0.930491989674,
        0.207651688249,
        0.490154483763,
        0.121935083767,
        0.0271637301263,
        -0.0030181931651,
        -0.00573456364666,
        -0.0138836507005,
        -0.0141854095462,
        -0.0113936077235,
        -0.0143363371255,
        -0.00852634289373,
        -0.0060363543017,
        -0.00421601454639,
        -0.00283425653164,
        -0.0148644747939,
        -0.0143363422192,
        -0.0125254707864,
        -0.0119972479782,
        -0.00950724533202,
        -0.99999999042,
        0.999489067916,
        0.221775135201,
        0.999999997625,
        0.31642209159,
        0.128943532906,
        0.892536054685,
        0.306541319109,
        0.0733636503726,
        0.0157455498855,
        -0.00338279773412,
        -0.00553246605739,
        -0.00899841143422,
        -0.00827788822184,
        -0.00670025055249,
        -0.0086004973875,
        -0.00492128145791,
        -0.00357381362615,
        -0.00247533189909,
        -0.00165139485249,
        -0.00808821549943,
        -0.00868244274229,
        -0.00800368373598,
        -0.00709664658729,
        -0.00545588648941,
        -0.999999997337,
        1.00001163601,
        0.259765732907,
        1.0,
        0.396345026847,
        0.171415742615,
        0.397091253171,
        0.907229516547,
        0.103196383252,
        0.0227495957472,
        -0.00152178273536,
        -0.00403392110807,
        -0.0102078163364,
        -0.00937572132313,
        -0.00777058969693,
        -0.00987351006936,
        -0.0058628542807,
        -0.00425069563019,
        -0.00292270767876,
        -0.00199143840286,
        -0.00951434971645,
        -0.00988375747923,
        -0.00943476222267,
        -0.0082837129088,
        -0.00646876011359,
        -0.99999999882,
        1.00002533972,
        0.302920159153,
        0.999999998822,
        0.236336640131,
        0.102427051939,
        0.237018423701,
        0.0619250249757,
        0.9468703795,
        0.013565949612,
        -0.000909917554327,
        -0.00240286106979,
        -0.00599112185608,
        -0.00531861967375,
        -0.00443000969367,
        -0.00566466843077,
        -0.00334345786185,
        -0.00244307884966,
        -0.00167462247047,
        -0.00114288634475,
        -0.00530803562032,
        -0.0056900067424,
        -0.00554578939243,
        -0.00474142727103,
        -0.00367331213613,
        -0.999999998907,
        0.99997191028,
        0.375887894695,
        1.0,
        0.103380597445,
        0.0441858761664,
        0.102987890722,
        0.0259417674394,
        0.00593356974721,
        0.97965190149,
        -0.000391139061396,
        -0.00105852482065,
        -0.00290349769088,
        -0.00312934660231,
        -0.00253855031541,
        -0.00313566329648,
        -0.00191272118879,
        -0.00133903938524,
        -0.000933772710876,
        -0.000631590284709,
        -0.00339988303391,
        -0.00308992872418,
        -0.00266265176318,
        -0.00265872084956,
        -0.00214980985395,
        -0.999999993648,
        0.999990918723,
        0.468737042796,
        0.999999993644,
        -0.0108589828525,
        -0.00466337148367,
        -0.0108418469283,
        -0.00276359614923,
        -0.000623452732745,
        4.1366057984e-05,
        0.984107022143,
        0.000110927229815,
        0.000295654956992,
        0.000302087174969,
        0.000246756473583,
        0.000307597555953,
        0.000186010779845,
        0.000131710804267,
        9.14283115646e-05,
        6.19898676381e-05,
        0.000321396464653,
        0.000304663934062,
        0.00027190199108,
        0.00025991298954,
        0.000207842729458,
        -0.999999998011,
        1.00000120442,
        0.460311820398,
        0.999999998011,
        -0.0344608788559,
        -0.0147892195329,
        -0.0343978651982,
        -0.00876122091439,
        -0.00197779642811,
        0.000131017052894,
        0.000352112121109,
        0.68688609462,
        0.000939565375256,
        0.000962973921818,
        0.00078619365176,
        0.000979528536129,
        0.000592611087926,
        0.000419338576315,
        0.000291153475537,
        0.000197376220551,
        0.00102566610796,
        0.000969969007164,
        0.000863863628472,
        0.000827852326368,
        0.000662371426917,
        -0.999999997229,
        1.00001107993,
        0.540507636343,
        0.99999999723,
        -0.148850751523,
        -0.0637241346114,
        -0.148401406695,
        -0.037544601109,
        -0.00854342815276,
        0.000564311081313,
        0.00152283723525,
        0.00413293149297,
        0.971638034014,
        0.00437059620811,
        0.00355396414767,
        0.00440410201207,
        0.00267821492662,
        0.00188247925309,
        0.00131059897172,
        0.000887220737377,
        0.00471380584605,
        0.00434781866623,
        0.0037939053685,
        0.00372969068327,
        0.00300396095217,
        -0.999999997805,
        1.00001071512,
        0.535657601622,
        0.999999997817,
        -0.172668669258,
        -0.0738958350496,
        -0.17211995328,
        -0.0435074447846,
        -0.00991034813626,
        0.000654305864388,
        0.00176680202421,
        0.00480515315136,
        0.0051009868052,
        0.963605416457,
        0.00414585236778,
        0.00513422039,
        0.00312415127781,
        0.002194137737,
        0.00152808003096,
        0.00103426877056,
        0.00550974178932,
        0.00506674298162,
        0.00441008582983,
        0.00434907555816,
        0.00350560549418,
        -0.999999903751,
        1.0000088586,
        0.691926159994,
        0.999999903755,
        -0.451861585377,
        -0.19271462836,
        -0.449680946559,
        -0.112617115807,
        -0.0259345148086,
        0.00170496518652,
        0.00463162667394,
        0.0128810235073,
        0.01421798998,
        0.0114997812433,
        0.756704618764,
        0.0141480462362,
        0.00866310360738,
        0.00603464422277,
        0.00421677978773,
        0.00284916928171,
        0.0155856036062,
        0.0139099125502,
        0.0117973693706,
        0.0120142292754,
        0.00976181955244,
        -0.999999977295,
        1.00000239029,
        0.611537444659,
        0.9999999773,
        -0.317060005731,
        -0.135896318923,
        -0.316281400159,
        -0.0802667129589,
        -0.0181983400164,
        0.00120381442854,
        0.00324177209003,
        0.00873088905706,
        0.00910374308455,
        0.00741612463314,
        0.00921234607045,
        0.969195782174,
        0.00558933289172,
        0.00394048150355,
        0.00274006421189,
        0.00185608330566,
        0.00976426160608,
        0.00910701421558,
        0.00802058289633,
        0.00779454621965,
        0.00625940064361,
        -0.999999861832,
        1.00000125831,
        0.769065159014,
        0.999999861834,
        -0.587332365332,
        -0.251714599081,
        -0.585861758032,
        -0.148639784612,
        -0.0337113420846,
        0.00222989478552,
        0.00600559133553,
        0.0161859196762,
        0.0168992199698,
        0.0137642060946,
        0.0170941171013,
        0.0103736016715,
        0.939966914252,
        0.0073113487392,
        0.00508461716202,
        0.00344405277407,
        0.018134841833,
        0.0168964800051,
        0.0148680914741,
        0.0144645085269,
        0.0116189281561,
        -0.99999946251,
        1.00000050655,
        0.71277621528,
        0.999999462513,
        -0.718794429855,
        -0.308158144006,
        -0.717108952019,
        -0.182098342286,
        -0.0412572773003,
        0.0027299894234,
        0.00734840071244,
        0.0197623115217,
        0.0205499732735,
        0.0167464560441,
        0.0208123268198,
        0.01262165124,
        0.00890348816489,
        0.932769831777,
        0.00618968638669,
        0.00419332897035,
        0.0220170121019,
        0.0205797646549,
        0.0181571388375,
        0.0176061377326,
        0.0141304963058,
        -0.999999551501,
        1.00000074127,
        0.922111131684,
        0.999999551503,
        -0.864604191149,
        -0.370776805193,
        -0.862694562097,
        -0.219230570051,
        -0.0496275758662,
        0.00328474580885,
        0.00883738945289,
        0.0237239770958,
        0.0245848688914,
        0.0200435823545,
        0.0249247264307,
        0.0151071318768,
        0.0106646203144,
        0.00741182527667,
        0.923894084772,
        0.00502207699109,
        0.0263038754278,
        0.024654340278,
        0.0218010430328,
        0.0210803210565,
        0.0169066169948,
        -0.999999102057,
        0.999998951433,
        1.00002768912,
        0.99999910206,
        -1.00124079005,
        -0.429464635758,
        -0.999128324903,
        -0.254047659454,
        -0.057470439413,
        0.00380504177445,
        0.0102328510373,
        0.0274308068499,
        0.0283499167398,
        0.0231212981962,
        0.02876515317,
        0.0174272059847,
        0.0123095326907,
        0.00855301785467,
        0.00579602447243,
        0.913082344992,
        0.0302993685603,
        0.0284605511775,
        0.0252108964507,
        0.0243241424317,
        0.0194971363636,
        -0.999999963911,
        1.00000258493,
        0.632118985465,
        0.999999963918,
        -0.284606752019,
        -0.121997285574,
        -0.283919782359,
        -0.0720701008982,
        -0.0163356711494,
        0.00108073946782,
        0.00290983977386,
        0.007832564645,
        0.00815860698096,
        0.00664708653687,
        0.00825851576009,
        0.00500977506669,
        0.00353267396461,
        0.00245627024244,
        0.00166392374186,
        0.00874695987769,
        0.969875848226,
        0.00816489844473,
        0.00719572973863,
        0.00698704378732,
        0.00560971863074,
        -0.999999967923,
        1.00000259051,
        0.623893154237,
        0.999999967927,
        -0.286078397337,
        -0.122624702232,
        -0.285385035409,
        -0.0724404196347,
        -0.0164198209268,
        0.00108625318418,
        0.00292492586241,
        0.00787326852617,
        0.00820158796837,
        0.00668200126138,
        0.0083018010801,
        0.00503607580608,
        0.00355116664516,
        0.00246913723595,
        0.00167263325093,
        0.00879322094318,
        0.00820767076445,
        0.969328828498,
        0.00723305210118,
        0.00702369800042,
        0.00563920566007,
        -0.99999992918,
        1.00000227722,
        0.719648579674,
        0.999999929182,
        -0.41799743356,
        -0.179097222771,
        -0.416900769325,
        -0.105702055874,
        -0.0239919897505,
        0.00158639273695,
        0.00427457924641,
        0.0115397595539,
        0.0120850987857,
        0.00983932506412,
        0.0122133087995,
        0.00741537576367,
        0.00522297912718,
        0.00363323811078,
        0.00246062678655,
        0.0129844223471,
        0.0120685378416,
        0.0105985738448,
        0.95216560965,
        0.0103365639321,
        0.00830838538882,
        -0.999999982572,
        1.00000205864,
        0.70388242151,
        0.999999982578,
        -0.416147766657,
        -0.178334932893,
        -0.415090647792,
        -0.105293622129,
        -0.0238855599235,
        0.00157969666272,
        0.00425531158027,
        0.0114737405534,
        0.0119893395201,
        0.00976410068526,
        0.012124556463,
        0.00735880610576,
        0.00518558773041,
        0.00360652957005,
        0.0024427809667,
        0.0128701416955,
        0.0119834419381,
        0.0105391021457,
        0.0102599774854,
        0.95862698536,
        0.00824295949132,
        1.00000114176,
        0.796504934595,
        -0.556913047408,
        -0.238675437529,
        -0.555515609805,
        -0.14093502826,
        -0.0319655490916,
        0.00211436475748,
        0.00569449654054,
        0.0153493984945,
        0.0160289487419,
        0.013055078931,
        0.0162128947225,
        0.00983916280445,
        0.00693438374628,
        0.00482254355306,
        0.00326651179883,
        0.0172023025286,
        0.016025128423,
        0.0140995816369,
        0.0137190242895,
        0.0110205729953,
        0.950991546377,
        -0.999999998793,
        -0.999927308544,
        0.999999998794,
        -8.97333012888e-08,
        -3.65839092556e-09,
        -0.000620118554466,
        -0.0314772393126,
        -0.252262076931,
        -0.124643935775,
        -0.137567870267,
        -0.0762138936239,
        -0.0610686173101,
        -0.0240291042504,
        -0.0132087874888,
        -0.00459845491647,
        -0.00266867102503,
        -0.00154188498197,
        -0.000902839012791,
        -0.0149458174457,
        -0.0142688175396,
        -0.00922217950606,
        -0.00887663174313,
        -0.00540033072801,
        -1.0,
        0.0497724415076,
        -1.0,
        -0.0249641288951,
        0.0265847156616,
        -0.00034419408453,
        0.0205239108607,
        0.0181573269558,
        0.0141775180438,
        0.0133841338885,
        0.00991956414032,
        0.00931986716963,
        0.0040800655714,
        0.00613345850182,
        0.00264444681774,
        0.00171570139818,
        0.00108515283433,
        0.000692631582283,
        0.00678083451622,
        0.00657238731391,
        0.00455236722961,
        0.00445229081896,
        0.00297394935235,
        -1.0,
        -0.00388917383148,
        -1.0,
        -0.0416897097749,
        -0.0241178106472,
        -0.0473409815165,
        -0.0174186257368,
        -0.00792976629013,
        -0.00430050763023,
        -0.00353725905769,
        -0.00184357732725,
        -0.00156610101526,
        -0.000257300713274,
        -0.000648485910391,
        -7.96275969144e-05,
        -8.77385119066e-06,
        1.91206095893e-05,
        2.43874468404e-05,
        -0.000811095769438,
        -0.000764626500207,
        -0.000332824684164,
        -0.00032321819653,
        -0.000110606882292,
        -0.092965449123,
        -0.999999998751,
        0.999999996788,
        0.999999999737,
        0.996597270442,
        0.158246855335,
        -0.105888486064,
        -1.28854948812,
        -0.0150527746959,
        -0.0461964476202,
        -0.0124574694054,
        -0.00545014366851,
        -0.0025953071388,
        -0.00155718434685,
        -0.000940798468508,
        -0.000486619785349,
        4.86619780593e-05,
        -0.000275751197379,
        -2.43309884903e-05,
        1.62206593123e-05,
        8.11032964206e-06,
        1.01379121046e-05,
        -0.000308192515477,
        -0.000291971844355,
        -0.000178427246796,
        -5.67723061874e-05,
        3.24413186484e-05,
        -0.99999999821,
        0.999999998522,
        0.999664546131,
        -0.0246664220461,
        -0.0428749941339,
        -0.00535626904808,
        -0.696446772541,
        -0.0181391890302,
        -0.00474235496245,
        -0.00185818016249,
        -0.000567853167925,
        -0.000517710550711,
        -0.000438452624207,
        -0.000257001895548,
        7.71193162528e-05,
        0.0,
        0.0,
        4.98224896313e-05,
        1.28503138326e-05,
        2.33045929106e-05,
        -0.000166902591807,
        -0.000153949736336,
        8.99504050991e-05,
        0.0,
        0.0,
        -0.999999998355,
        0.999999999509,
        0.999999990876,
        0.0912499344129,
        -0.076528364545,
        -0.0100038751492,
        -0.0320632662984,
        -1.26937721775,
        -0.00806345156993,
        -0.00270190471746,
        -0.00126107542689,
        -0.000674607975844,
        -0.000176325662452,
        -5.05136253362e-05,
        0.000132511602213,
        9.24629820854e-05,
        0.000103570917743,
        7.94189516298e-05,
        5.04378695976e-05,
        3.33302634977e-05,
        4.59461413533e-05,
        0.000120992268879,
        0.000105860461911,
        0.000138441156039,
        9.47149035401e-05,
        -0.999999999618,
        0.999999999841,
        1.00000106054,
        0.00984371392305,
        -0.0472363619505,
        -0.00626813161031,
        -0.0198680693228,
        -0.00502949218288,
        -0.545145512525,
        -0.00169424617468,
        -0.000792798556904,
        -0.000429493488881,
        -0.000111238778255,
        -3.39230640305e-05,
        8.20364611447e-05,
        5.68998210522e-05,
        6.46509938226e-05,
        4.96256318199e-05,
        3.17278837317e-05,
        2.10154547811e-05,
        2.8381240379e-05,
        7.33773339277e-05,
        6.6016431368e-05,
        8.55108162831e-05,
        5.94890870476e-05,
        -1.00000000005,
        1.00000000048,
        1.00000124953,
        -0.028427425802,
        -0.0306977704799,
        -0.00411269247685,
        -0.0129415520941,
        -0.00328665627394,
        -0.00110861442315,
        -0.0727117912317,
        -0.000516747019532,
        -0.000282101986258,
        -7.35549790458e-05,
        -2.32182018236e-05,
        5.37286664078e-05,
        3.71748648788e-05,
        4.25085547112e-05,
        3.26462545277e-05,
        2.09474200097e-05,
        1.38961109418e-05,
        1.85888798994e-05,
        4.74961842794e-05,
        4.34486517064e-05,
        5.59094415027e-05,
        3.9277713066e-05,
        -0.999999998886,
        0.999999999913,
        0.999999819684,
        -0.0339485851367,
        -0.0230797085314,
        -0.00306657069892,
        -0.00970726636656,
        -0.00245455032785,
        -0.000824252432952,
        -0.00038207934795,
        -0.0612893092272,
        -0.000207091586447,
        -5.47951673498e-05,
        -1.67336128233e-05,
        4.04967631046e-05,
        2.81477385884e-05,
        3.18699275319e-05,
        2.44602356851e-05,
        1.56320350625e-05,
        1.03575521512e-05,
        1.40453283851e-05,
        3.62646041772e-05,
        3.26306125087e-05,
        4.21957717159e-05,
        2.93556953494e-05,
        -1.00000000005,
        0.999999999772,
        0.999999530286,
        -0.0601259040135,
        -0.0145626528242,
        -0.00200554096463,
        -0.00617999770685,
        -0.00158335680758,
        -0.000535516938334,
        -0.000246153935612,
        -0.00013726914641,
        -0.0280767844056,
        -3.67366442413e-05,
        -1.26953158277e-05,
        2.61406137853e-05,
        1.79705862379e-05,
        2.0903145474e-05,
        1.60764195306e-05,
        1.04178740055e-05,
        6.94113544363e-06,
        9.05082956213e-06,
        2.233397493e-05,
        2.14427686856e-05,
        2.70663983773e-05,
        1.95468827425e-05,
        -1.00000000006,
        0.999999999603,
        0.999999252592,
        -0.0747728180127,
        -0.0062927552486,
        -0.000877828228811,
        -0.00267925456784,
        -0.000689762309701,
        -0.000233927260252,
        -0.00010726273498,
        -6.04165083914e-05,
        -1.62118169616e-05,
        -0.0295807844936,
        -5.82324487657e-06,
        1.13829106269e-05,
        7.79331961795e-06,
        9.15453002447e-06,
        7.04581564629e-06,
        4.58795166509e-06,
        3.06256119844e-06,
        3.93818514728e-06,
        9.55708387811e-06,
        9.39567394676e-06,
        1.17596386224e-05,
        8.6049374375e-06,
        -0.999999999991,
        0.999999999951,
        0.99999916384,
        -0.0939422143946,
        -0.00196398458382,
        -0.000325975324672,
        -0.000876811161582,
        -0.000240921736289,
        -8.4527421063e-05,
        -3.73975495605e-05,
        -2.38280880926e-05,
        -6.64001682459e-06,
        -3.38632083425e-06,
        -0.0288338484191,
        3.97317313797e-06,
        2.57620594417e-06,
        3.43358301496e-06,
        2.6660160581e-06,
        1.8365197742e-06,
        1.25210301635e-06,
        1.36238409349e-06,
        2.56621619183e-06,
        3.54960312322e-06,
        3.98303569383e-06,
        3.43121690869e-06,
        -1.00000000034,
        1.00000000003,
        0.999997312303,
        -0.150457974438,
        0.0186207984054,
        0.00247101420386,
        0.00783212517907,
        0.00198264984779,
        0.000667860153126,
        0.000312482064835,
        0.000169288546434,
        4.38555916967e-05,
        1.33748181056e-05,
        -3.23432980779e-05,
        -0.0056519999574,
        -2.24337596016e-05,
        -2.54890057486e-05,
        -1.9565130845e-05,
        -1.25089038561e-05,
        -8.28551059956e-06,
        -1.1190286617e-05,
        -2.89297824925e-05,
        -2.60280831564e-05,
        -3.37130817275e-05,
        -2.34544052108e-05,
        -1.0000000001,
        0.999999999823,
        0.999999414797,
        -0.120619948601,
        0.00712649803414,
        0.000934454457229,
        0.00298782787182,
        0.000751995519162,
        0.000251955639257,
        0.000117292047313,
        6.28956331994e-05,
        1.6527210281e-05,
        4.79080177989e-06,
        -1.23865243465e-05,
        -8.64168549491e-06,
        -0.0327055528556,
        -9.6922800466e-06,
        -7.43316264241e-06,
        -4.72573495328e-06,
        -3.12440115687e-06,
        -4.29747271412e-06,
        -1.12770466462e-05,
        -9.91340526649e-06,
        -1.29367287859e-05,
        -8.87582224129e-06,
        -1.00000000027,
        1.00000000026,
        0.99999923261,
        -0.172245677713,
        0.0256181643334,
        0.00347239039925,
        0.0108300863167,
        0.00276061661955,
        0.000932212232143,
        0.000431918445081,
        0.00023794966765,
        6.27413528412e-05,
        2.06046999868e-05,
        -4.53245338178e-05,
        -3.12767503162e-05,
        -3.60207370267e-05,
        -0.0101617208341,
        -2.76803435665e-05,
        -1.78358471916e-05,
        -1.18539606827e-05,
        -1.56872981455e-05,
        -3.95029662014e-05,
        -3.68746943363e-05,
        -4.70653536769e-05,
        -3.34529142487e-05,
        -0.99999999987,
        0.99999999988,
        0.999999219902,
        -0.249559747692,
        0.034931660991,
        0.00474610588443,
        0.0147769740832,
        0.00377088983586,
        0.00127460839358,
        0.000591014911203,
        0.000326253574079,
        8.58191168239e-05,
        2.8416231806e-05,
        -6.18107101406e-05,
        -4.26067862482e-05,
        -4.91881458098e-05,
        -3.78051310875e-05,
        -0.00570803651467,
        -2.43850669911e-05,
        -1.62121312089e-05,
        -2.13824224177e-05,
        -5.36826806395e-05,
        -5.03409520119e-05,
        -6.41595942402e-05,
        -4.57230592526e-05,
        -0.99999999976,
        1.00000000037,
        0.999999215156,
        -0.220462663354,
        0.039313053531,
        0.00543697672466,
        0.016705560803,
        0.00429167231213,
        0.00145628257531,
        0.000673089179879,
        0.000376828646605,
        9.94237621041e-05,
        3.48145625187e-05,
        -7.02868798983e-05,
        -4.81762894696e-05,
        -5.63811681607e-05,
        -4.33777182107e-05,
        -2.81693907551e-05,
        -0.00320300013486,
        -1.8777319578e-05,
        -2.42884155321e-05,
        -5.96132281841e-05,
        -5.77389231672e-05,
        -7.27343249681e-05,
        -5.27872892338e-05,
        -1.00000000045,
        0.999999999251,
        0.999999213214,
        -0.242394430566,
        0.045509690252,
        0.00633849969492,
        0.0193742302857,
        0.00499107925966,
        0.00169660267866,
        0.000783643094334,
        0.000441187756818,
        0.000116367549917,
        4.16013131589e-05,
        -8.1651195874e-05,
        -5.58310917635e-05,
        -6.57108854967e-05,
        -5.05764178334e-05,
        -3.29324410406e-05,
        -2.19741987986e-05,
        -0.00182404280343,
        -2.81981286491e-05,
        -6.85831824487e-05,
        -6.72975051025e-05,
        -8.43926080174e-05,
        -6.16914742054e-05,
        -1.00000000069,
        1.00000000041,
        0.999999482099,
        -0.0953176952642,
        0.00300637681227,
        0.00039877499422,
        0.00126398592416,
        0.000319456443795,
        0.000107271845574,
        4.97875734891e-05,
        2.6949293049e-05,
        7.11201250744e-06,
        2.15826718463e-06,
        -5.26495581463e-06,
        -3.66084890894e-06,
        -4.14113251022e-06,
        -3.17799281125e-06,
        -2.0298078506e-06,
        -1.34451768678e-06,
        -1.82630470737e-06,
        -0.0382933948089,
        -4.72418933243e-06,
        -4.23857716763e-06,
        -5.48782073801e-06,
        -3.81148074083e-06,
        -1.00000000021,
        0.999999999126,
        0.999999438579,
        -0.107550415024,
        0.00853928906255,
        0.00107272033054,
        0.00354347046086,
        0.000877927803443,
        0.000291520092623,
        0.000137034457988,
        7.08837676565e-05,
        1.83760086128e-05,
        4.32667114858e-06,
        -1.44599327639e-05,
        -1.02226314135e-05,
        -1.10919172798e-05,
        -8.48409466658e-06,
        -5.29675693674e-06,
        -3.476033416e-06,
        -5.02776564667e-06,
        -1.38847403416e-05,
        -0.0360016028842,
        -1.13195993726e-05,
        -1.52159015765e-05,
        -9.96113502129e-06,
        -1.0000000001,
        0.999999999381,
        0.999999309332,
        -0.117701117214,
        0.0123057356353,
        0.00167223304885,
        0.00520520248287,
        0.001327636987,
        0.000448242294157,
        0.000207180570783,
        0.000114352801101,
        3.03044010491e-05,
        1.00341653319e-05,
        -2.18481799914e-05,
        -1.50720835107e-05,
        -1.73765150703e-05,
        -1.3354491255e-05,
        -8.61200279543e-06,
        -5.72609799283e-06,
        -7.56474244807e-06,
        -1.89882207722e-05,
        -1.78003121868e-05,
        -0.0215883297722,
        -2.26764015502e-05,
        -1.61568989753e-05,
        -0.999999999819,
        1.00000000038,
        0.999999311433,
        -0.129699258674,
        0.0168942213367,
        0.00222770797483,
        0.0070930780274,
        0.00178932538818,
        0.000600497154718,
        0.000279552748883,
        0.000150607356757,
        3.95273093739e-05,
        1.17306069768e-05,
        -2.94267601164e-05,
        -2.04872970533e-05,
        -2.30913280106e-05,
        -1.77156114893e-05,
        -1.12900694009e-05,
        -7.47120412168e-06,
        -1.02025393489e-05,
        -2.65887709631e-05,
        -2.36162243614e-05,
        -3.07035852504e-05,
        -0.0207254906571,
        -2.11969034983e-05,
        -1.00000000059,
        0.999999237103,
        -0.14172689852,
        0.0196026618689,
        0.00270884513115,
        0.00832726397818,
        0.00213760237672,
        0.000724483230055,
        0.000334004828075,
        0.000186823131053,
        4.96039503243e-05,
        1.73260029747e-05,
        -3.51280819509e-05,
        -2.41004363648e-05,
        -2.81530501762e-05,
        -2.16578924595e-05,
        -1.40571511194e-05,
        -9.36978892426e-06,
        -1.21481564479e-05,
        -2.98468921104e-05,
        -2.88530228223e-05,
        -3.63538812715e-05,
        -2.63554901668e-05,
        -0.0120949119337,
        0.999999999698,
        -1.00000862911,
        1.09598487124e-07,
        8.96126560736e-07,
        0.000725252867534,
        0.0100168224585,
        0.0123229509538,
        0.00614175236553,
        0.0048015231779,
        0.00234397071851,
        0.00195459507075,
        0.00039300812323,
        0.00068488996147,
        0.000157155680384,
        7.62433646185e-05,
        3.70887441647e-05,
        1.82757164303e-05,
        0.000841573964664,
        0.000765959004349,
        0.000405645138333,
        0.000386808874336,
        0.000196140621932,
        -1.00000000017,
        1.0,
        1.0,
        -0.000828122526465,
        1.0,
        -0.036688906816,
        1.00000004673,
        1.00000005808,
        -0.999999999265,
        0.999540732831,
        0.246547743071,
        0.99999999748,
        0.354102472598,
        0.458520074383,
        0.0683926909404,
        0.163431898885,
        0.0401177955665,
        0.00829002560188,
        -0.0012953173003,
        -0.0022575531486,
        -0.00471863949353,
        -0.00477414576845,
        -0.00363150622124,
        -0.00463537177132,
        -0.0026924037951,
        -0.00187358018309,
        -0.00127449716407,
        -0.000854676667935,
        -0.00490368528419,
        -0.00462611364676,
        -0.00402473309931,
        -0.00383968111789,
        -0.00302086202948,
        -0.999999998304,
        0.999913334064,
        0.228958061906,
        0.99999999537,
        0.223096368245,
        0.0429168344359,
        0.855705835873,
        0.102536960048,
        0.0251265574202,
        0.00526512336882,
        -0.000807382014906,
        -0.00142085271075,
        -0.0027929095966,
        -0.00273254913161,
        -0.00221045589901,
        -0.00277345634473,
        -0.00164007884632,
        -0.00114314242999,
        -0.000782769061706,
        -0.000526435570996,
        -0.00291016589034,
        -0.0026900515913,
        -0.00252608040752,
        -0.0023220396355,
        -0.00188767569104,
        -0.99999999882,
        1.00000065845,
        0.261435451047,
        0.99999999977,
        0.285658946377,
        0.0550596621912,
        0.131694409721,
        0.772066969502,
        0.0323297020777,
        0.00675991411647,
        -0.00101798398188,
        -0.00178582077012,
        -0.00349731717976,
        -0.00338965908258,
        -0.00279548515535,
        -0.0035013486247,
        -0.00207937662512,
        -0.00145295276204,
        -0.000994096967238,
        -0.000670092444022,
        -0.0036621996764,
        -0.00336663169555,
        -0.0032273378177,
        -0.0029460424029,
        -0.00240747664649,
        -0.999999999948,
        1.00000126526,
        0.303598412194,
        0.999999999992,
        0.16770548108,
        0.0323263991046,
        0.0773170797397,
        0.0189807306412,
        0.919546007201,
        0.0039686799334,
        -0.000597626374797,
        -0.0010483049695,
        -0.00205403872655,
        -0.00199126594512,
        -0.00164157211942,
        -0.00205626366343,
        -0.00122103074764,
        -0.000853175774331,
        -0.000583735936411,
        -0.000393463290851,
        -0.00215081925609,
        -0.00197751810364,
        -0.00189475729313,
        -0.00173002145614,
        -0.00141347748621,
        -0.999999998914,
        0.999999832113,
        0.376698301677,
        1.0,
        0.0682751950469,
        0.0132176845557,
        0.0315381165596,
        0.0077341329009,
        0.00160975731489,
        0.326212789323,
        -0.00023842281555,
        -0.000417535128584,
        -0.000901621834605,
        -0.000913131623569,
        -0.000697283348343,
        -0.000889788875675,
        -0.00051615163296,
        -0.000359477764662,
        -0.000245876919484,
        -0.000164320556299,
        -0.000938151396749,
        -0.000887709280089,
        -0.000769994422697,
        -0.00073825553414,
        -0.000578871488966,
        -0.999999999265,
        1.00000034282,
        0.469833166665,
        0.999999999312,
        -0.0163074433169,
        -0.00315721352954,
        -0.00753287182586,
        -0.00184744344535,
        -0.000384573982174,
        5.69943804336e-05,
        0.438548318128,
        9.9778951957e-05,
        0.000215010137041,
        0.000217548202173,
        0.000166405753462,
        0.000212258825392,
        0.0001231942105,
        8.58061464094e-05,
        5.86912671005e-05,
        3.92314646826e-05,
        0.000223758001405,
        0.000211589054274,
        0.000183947495023,
        0.000176164541416,
        0.000138265939081,
        -0.999999998449,
        1.00000124132,
        0.461118504975,
        0.999999998568,
        -0.0339578756818,
        -0.00657729706876,
        -0.0156897065544,
        -0.0038469511841,
        -0.000800190482403,
        0.000118236071868,
        0.000207052901098,
        0.234298075573,
        0.00045275752319,
        0.000460950804052,
        0.00034870775228,
        0.000446021617786,
        0.000257963414284,
        0.000179585070512,
        0.000122827573139,
        8.19952339075e-05,
        0.000470728102729,
        0.000446986437252,
        0.000382847633332,
        0.000369415616506,
        0.000288115013045,
        -0.999999999502,
        1.00000096293,
        0.541051727924,
        0.999999999579,
        -0.119611536852,
        -0.0232001203851,
        -0.0552990315847,
        -0.0135546156496,
        -0.00281549347939,
        0.00041388470979,
        0.000724298992416,
        0.00163029671909,
        0.385429166603,
        0.00167923774045,
        0.00124404970646,
        0.00159967739494,
        0.000919024417356,
        0.000639183002111,
        0.000437134069302,
        0.000291079586457,
        0.00169206144363,
        0.00161931013356,
        0.00134785708532,
        0.00131968152356,
        0.00101676859844,
        -0.999999993334,
        1.00000081007,
        0.53607022015,
        0.999999993393,
        -0.137376592021,
        -0.0266441283521,
        -0.0635104535448,
        -0.0155674869726,
        -0.00323376749248,
        0.000475466412512,
        0.000832105424817,
        0.00187080559785,
        0.00192610265843,
        0.427419516139,
        0.00142809052801,
        0.00183594921556,
        0.00105503874989,
        0.000733808535993,
        0.000501849213327,
        0.00033420519305,
        0.00194180885523,
        0.00185776157446,
        0.00154806244244,
        0.00151483272125,
        0.00116768386125,
        -0.999999966402,
        1.00000054958,
        0.692118949547,
        0.999999966404,
        -0.346355029737,
        -0.0672737946979,
        -0.16022813104,
        -0.0392609509048,
        -0.00814302553408,
        0.00119051006505,
        0.00208220281969,
        0.00482787165596,
        0.00503025611529,
        0.00364980839541,
        0.276758903587,
        0.00471835581275,
        0.00269239483925,
        0.00187074295577,
        0.00127927602608,
        0.000849646635402,
        0.00500202467887,
        0.00482427337888,
        0.0039006992349,
        0.00387693816384,
        0.00294985108078,
        -0.999999983501,
        1.00000015739,
        0.61195536648,
        0.999999983525,
        -0.244778578531,
        -0.0474218183967,
        -0.113106222313,
        -0.027732295636,
        -0.00576771196333,
        0.000851873575106,
        0.00149141861112,
        0.00327149702107,
        0.0033348812174,
        0.00251717556221,
        0.00322147383872,
        0.975574087247,
        0.00186186535287,
        0.00129603527386,
        0.000886423640871,
        0.000591586773566,
        0.00340074960442,
        0.00323190808538,
        0.00275973647959,
        0.00266703017602,
        0.00207739980655,
        -0.999999972256,
        1.00000006523,
        0.76938079957,
        0.999999972248,
        -0.446224697377,
        -0.0864396309776,
        -0.206179384132,
        -0.0505544245248,
        -0.0105155409967,
        0.00155384833927,
        0.00272044257825,
        0.00595230063847,
        0.0060612762508,
        0.00458364636034,
        0.00586337930151,
        0.00339078589835,
        0.958763770421,
        0.00236051045734,
        0.00161448778078,
        0.00107772732522,
        0.00618844999149,
        0.0058770710169,
        0.00503122498631,
        0.0048559529967,
        0.00378647766645,
        -0.999999877878,
        0.99999993595,
        0.711075635115,
        0.999999877876,
        -0.544215377004,
        -0.105403760174,
        -0.25143701245,
        -0.0616541298313,
        -0.0128267099754,
        0.00189661869831,
        0.00332077236346,
        0.00723850796919,
        0.00735956757517,
        0.00558096343795,
        0.00713415381311,
        0.00412932028175,
        0.00287500318954,
        0.950112616103,
        0.00196640365604,
        0.00131307566729,
        0.00752745008892,
        0.00714126063112,
        0.00613652742534,
        0.00591146994929,
        0.00461690820581,
        -0.999999740344,
        0.999999322492,
        0.922177906408,
        0.999999740344,
        -0.652870072649,
        -0.126429242785,
        -0.301616910216,
        -0.0739616871318,
        -0.0153898251537,
        0.00227705109163,
        0.003987074329,
        0.00866065883756,
        0.0087927735089,
        0.00668505137471,
        0.00853998125667,
        0.00494707321156,
        0.00344475116492,
        0.00235612564454,
        0.943052485006,
        0.00157379290016,
        0.00900832688495,
        0.00853789390841,
        0.00736228266632,
        0.00707980060388,
        0.00553752891683,
        -0.999999405119,
        0.999998518221,
        0.999997086777,
        0.999999405125,
        -0.75464954184,
        -0.146119237345,
        -0.348615389954,
        -0.0854892237966,
        -0.0177908350957,
        0.00263386173464,
        0.00461177449864,
        0.00998869662403,
        0.010128871554,
        0.0077174154251,
        0.00985349653991,
        0.00571183183881,
        0.00397767607044,
        0.00272065090938,
        0.00181774740757,
        0.937304699564,
        0.0103915103267,
        0.00984094907201,
        0.00851050496095,
        0.00817201354994,
        0.00639964282433,
        -0.999999983576,
        1.0000001723,
        0.632985217996,
        0.999999983563,
        -0.220537084064,
        -0.0427235217553,
        -0.101902751297,
        -0.0249856696336,
        -0.00519675315367,
        0.000767700350469,
        0.00134406060814,
        0.00294508416549,
        0.00300081131849,
        0.00226682071927,
        0.00290049038377,
        0.00167677523676,
        0.001167236826,
        0.000798335267703,
        0.000532848385683,
        0.0030616458706,
        0.976283036651,
        0.00290877301024,
        0.00248649194762,
        0.0024016502583,
        0.00187154891688,
        -0.999999988397,
        1.00000020013,
        0.624758827913,
        0.999999988456,
        -0.221577242546,
        -0.0429246413314,
        -0.102383234209,
        -0.0251032479059,
        -0.00522112907534,
        0.000771239489896,
        0.00135031433214,
        0.0029595002844,
        0.00301582258858,
        0.00227772599332,
        0.00291458127158,
        0.00168481756683,
        0.00117282531477,
        0.000802155349233,
        0.000535385516544,
        0.00307657782762,
        0.00292317418207,
        0.974661504555,
        0.00249815676489,
        0.00241323338717,
        0.00188036508881,
        -0.999999989889,
        1.00000014933,
        0.720294917175,
        0.999999989904,
        -0.319996290571,
        -0.0619975541297,
        -0.147866183709,
        -0.0362546946753,
        -0.00753983183787,
        0.00111341922015,
        0.00194924540461,
        0.00428010346033,
        0.00436482590579,
        0.00329215939222,
        0.00421407726607,
        0.00243497484548,
        0.00169491825488,
        0.00115923807589,
        0.000773591187611,
        0.00444894265059,
        0.00422921820272,
        0.00360773811655,
        0.970350722033,
        0.00348831308362,
        0.00271596225905,
        -0.999999959836,
        1.0000001587,
        0.704712148883,
        0.999999959843,
        -0.318595199378,
        -0.0617229968329,
        -0.147215647002,
        -0.0360953678084,
        -0.00750697306112,
        0.00110872240346,
        0.00194107966925,
        0.00425869491155,
        0.00434155734925,
        0.00327654489922,
        0.00419346125518,
        0.00242351815339,
        0.00168698812605,
        0.00115381564965,
        0.000770026650087,
        0.00442689519358,
        0.00420733680153,
        0.00359195485505,
        0.00347163769271,
        0.971109414831,
        0.00270389735275,
        1.00000003267,
        0.796699696936,
        -0.423620142371,
        -0.0820655219462,
        -0.195740041978,
        -0.0479941896843,
        -0.00998247124793,
        0.00147478675942,
        0.00258194179936,
        0.00565563515037,
        0.00576182228179,
        0.00435362119125,
        0.0055702806343,
        0.00322044726406,
        0.00224184578368,
        0.00153332162644,
        0.00102344526324,
        0.00587961829981,
        0.00558549265051,
        0.00477627921556,
        0.00461249654267,
        0.00359494065046,
        0.964019242638,
        -0.99999999576,
        -0.999992109291,
        0.999999995824,
        -1.31655276406e-07,
        -3.15925464473e-08,
        -0.00116158098527,
        -0.0410943656215,
        -0.0883407579323,
        -0.0541282312734,
        -0.0466580962053,
        -0.0291213920931,
        -0.0255691854321,
        -0.00774003410914,
        -0.012036596563,
        -0.00405514550633,
        -0.0023230793611,
        -0.00132553281035,
        -0.000766738526849,
        -0.0137055514271,
        -0.0130573795964,
        -0.00828586574839,
        -0.00797180682,
        -0.00478416927628,
        -1.0,
        0.145534882655,
        -1.0,
        -0.0526692667114,
        0.0276883944104,
        0.000962441133053,
        0.021270611152,
        0.0183664736588,
        0.0141595136714,
        0.0132960837515,
        0.0097371606455,
        0.009114633548,
        0.00387970746231,
        0.00591408457334,
        0.00248260750771,
        0.00158971869982,
        0.000992857766311,
        0.000625864012899,
        0.00656431959503,
        0.00635211678936,
        0.00434829202649,
        0.00424847310439,
        0.00280383541947,
        -1.0,
        -0.0113554860303,
        -1.0,
        -0.0899949041227,
        -0.0237636217215,
        -0.0474830722572,
        -0.0171284338502,
        -0.00768312980003,
        -0.00412339229369,
        -0.00337733651995,
        -0.00173996394719,
        -0.00147246848724,
        -0.000233729212758,
        -0.000600073463654,
        -7.03689108872e-05,
        -6.56263110476e-06,
        1.78009405152e-05,
        2.19033410437e-05,
        -0.000753946048699,
        -0.000709997797178,
        -0.000304105731383,
        -0.000295228020566,
        -9.86366285432e-05,
        -0.0978726958287,
        -0.999999993987,
        1.00000000449,
        0.999999999817,
        0.993263709402,
        0.237430933032,
        -0.115623714773,
        -1.13259321215,
        -0.0144529638535,
        -0.0433588922949,
        -0.0112411942978,
        -0.0020073560134,
        0.0,
        -0.00100367796802,
        0.0,
        0.000501839162861,
        0.000501839077315,
        0.000301103417313,
        0.00010036779674,
        0.000276011487432,
        0.000156824702491,
        0.000106640794688,
        0.00035128733166,
        0.000301103437858,
        0.000602206786526,
        0.000175643648942,
        0.000351287296051,
        -0.999999999761,
        0.999999998725,
        0.999992600123,
        -0.0293889308921,
        -0.0405799009466,
        -0.00608612319978,
        -0.695924211592,
        -0.0121605280665,
        -0.00405680791915,
        0.0,
        -0.00307309720396,
        -0.00255218040587,
        -0.00164871786049,
        -0.00139027823554,
        -0.000374282509603,
        -0.000738919936334,
        -0.000219074855616,
        -0.000125441114586,
        -7.8450428736e-05,
        0.0,
        -0.000857322739318,
        -0.00151313056984,
        0.000507236839507,
        -0.000496849662233,
        0.000317318856512,
        -0.999999998865,
        0.999999998791,
        0.966424897736,
        0.0993489476759,
        -0.0789762677636,
        -0.0108696618879,
        -0.0336082346225,
        -1.13946858591,
        -0.00870804709776,
        -0.00299103256392,
        -0.00140875780235,
        -0.000774504241601,
        -0.000216255838071,
        -7.74133086625e-05,
        0.000145918047364,
        9.40979863454e-05,
        0.000117338052057,
        9.19285325279e-05,
        5.95161047644e-05,
        3.99892859075e-05,
        4.01222781057e-05,
        0.000124005332339,
        0.000114485389913,
        0.000151389377883,
        0.00010642098563,
        -1.0,
        0.999999999968,
        0.966403069738,
        0.018562651189,
        -0.0494763399369,
        -0.00690334073448,
        -0.0211420992522,
        -0.00550903533269,
        -0.543557075954,
        -0.00190488357584,
        -0.000888689711171,
        -0.000491560508945,
        -0.000134870352215,
        -5.07431679468e-05,
        9.17625768471e-05,
        5.8665604887e-05,
        7.43510441451e-05,
        5.83079490713e-05,
        3.79702777509e-05,
        2.55404047317e-05,
        2.50411484594e-05,
        7.64560385991e-05,
        7.20544624169e-05,
        9.49776750571e-05,
        6.74788293045e-05,
        -0.999999999944,
        0.999999998909,
        0.966422384731,
        -0.0183956298276,
        -0.0320748361944,
        -0.00451963282474,
        -0.0137298454495,
        -0.00358852243533,
        -0.0012383695272,
        -0.0803188288393,
        -0.000580548704659,
        -0.000325208363022,
        -9.14855851296e-05,
        -3.49119834042e-05,
        6.01357717348e-05,
        3.84499247961e-05,
        4.89138046473e-05,
        3.83572234693e-05,
        2.5075622888e-05,
        1.69137352205e-05,
        1.64980198439e-05,
        4.939329095e-05,
        4.778210162e-05,
        6.21165025904e-05,
        4.48251213727e-05,
        -1.0,
        0.999999999766,
        0.966454158834,
        -0.0222228874761,
        -0.0241662837699,
        -0.00337975322162,
        -0.0103185106238,
        -0.00268801796061,
        -0.000923177778714,
        -0.000435932388804,
        -0.0673437735284,
        -0.0002437424822,
        -6.96645613071e-05,
        -2.58203757534e-05,
        4.52717893544e-05,
        2.91125488569e-05,
        3.66705175843e-05,
        2.87286421875e-05,
        1.87227344806e-05,
        1.26256943301e-05,
        1.24997406432e-05,
        3.76001726593e-05,
        3.60182454429e-05,
        4.68228983801e-05,
        3.36356352335e-05,
        -0.9999999999,
        0.999999999824,
        0.966476526695,
        -0.0487105009836,
        -0.0156191163009,
        -0.00225449780363,
        -0.00672225754183,
        -0.00177175157905,
        -0.000612046067861,
        -0.000287385798764,
        -0.000164793151065,
        -0.0302071408739,
        -4.76333814114e-05,
        -1.91071811729e-05,
        2.98586010563e-05,
        1.89878401815e-05,
        2.45439857253e-05,
        1.92623253903e-05,
        1.27121467532e-05,
        8.61669635126e-06,
        8.21550481555e-06,
        2.36589921675e-05,
        2.41795588559e-05,
        3.06961739579e-05,
        2.28476316304e-05,
        -1.0,
        1.00000000001,
        0.966467175736,
        -0.0617383656781,
        -0.00718640924085,
        -0.00104695282418,
        -0.00310148653445,
        -0.000820522415966,
        -0.000284517623268,
        -0.000132897553836,
        -7.65581167161e-05,
        -2.19461133592e-05,
        -0.0310397559936,
        -9.03197665385e-06,
        1.37872460271e-05,
        8.72301944669e-06,
        1.13853996589e-05,
        8.94275734367e-06,
        5.92276208616e-06,
        4.01779652256e-06,
        3.77703908552e-06,
        1.07707820868e-05,
        1.11835523315e-05,
        1.41504570975e-05,
        1.06130538789e-05,
        -1.00000000004,
        0.999999999559,
        0.966467910574,
        -0.0809920995788,
        -0.00283586792577,
        -0.000462132165541,
        -0.00126182118719,
        -0.000348224512961,
        -0.00012351685607,
        -5.63195843136e-05,
        -3.51067019119e-05,
        -1.0258298843e-05,
        -5.17196308605e-06,
        -0.0302296091935,
        5.83924031281e-06,
        3.54136350987e-06,
        5.06703037077e-06,
        4.0041951344e-06,
        2.75718116115e-06,
        1.89763548953e-06,
        1.57327969157e-06,
        3.78756270521e-06,
        5.00018257585e-06,
        5.86813683902e-06,
        4.92736217232e-06,
        -1.00000000003,
        0.999999999898,
        0.96653051537,
        -0.1340968638,
        0.0186186696389,
        0.00259486036864,
        0.00795681190169,
        0.00207308349647,
        0.000718076293915,
        0.000333931390359,
        0.000184080717583,
        4.99109272918e-05,
        1.88358633862e-05,
        -3.44250842219e-05,
        -0.00594018852613,
        -2.19768732839e-05,
        -2.78890711448e-05,
        -2.18767419122e-05,
        -1.42402229388e-05,
        -9.57101861809e-06,
        -9.36384364046e-06,
        -2.87293563186e-05,
        -2.69248375182e-05,
        -3.56424940845e-05,
        -2.52302505914e-05,
        -1.0000000004,
        0.999999999712,
        0.966452027228,
        -0.106150787674,
        0.00659967782068,
        0.000908411975802,
        0.00280823576421,
        0.000727563982187,
        0.000249777846913,
        0.000117758703655,
        6.47907791615e-05,
        1.81587001675e-05,
        6.489823506e-06,
        -1.22017572568e-05,
        -7.87189406168e-06,
        -0.037247650141,
        -9.81464345135e-06,
        -7.68508148236e-06,
        -4.97607212298e-06,
        -3.34402717426e-06,
        -3.36067222566e-06,
        -1.03691283434e-05,
        -9.58060635231e-06,
        -1.26595537048e-05,
        -8.90402475284e-06,
        -1.00000000014,
        1.00000000013,
        0.966457089979,
        -0.15451734313,
        0.0258162088797,
        0.00367380016057,
        0.0110824830092,
        0.00290810427059,
        0.00100743655135,
        0.000469799849248,
        0.000264565262585,
        7.38243886186e-05,
        2.90237154348e-05,
        -4.85926392964e-05,
        -3.09052865551e-05,
        -3.97208252734e-05,
        -0.0117729940962,
        -3.11754580279e-05,
        -2.04609759907e-05,
        -1.38144905697e-05,
        -1.32723255057e-05,
        -3.93317174031e-05,
        -3.86917114247e-05,
        -5.01041202843e-05,
        -3.64668641457e-05,
        -1.00000000039,
        1.00000000024,
        0.966457498165,
        -0.232432681939,
        0.0355309687371,
        0.00506709894976,
        0.0152643194687,
        0.00400931025701,
        0.00139100053344,
        0.000647180928649,
        0.000364558113406,
        0.000101151261488,
        4.00845548255e-05,
        -6.6883719906e-05,
        -4.24645613282e-05,
        -5.47370767976e-05,
        -4.29737656695e-05,
        -0.0067033139459,
        -2.82283283732e-05,
        -1.90585661539e-05,
        -1.82310225012e-05,
        -5.39636576745e-05,
        -5.3219781178e-05,
        -6.89398313517e-05,
        -5.02261060061e-05,
        -1.00000000098,
        0.999999999547,
        0.966457656598,
        -0.199385736246,
        0.0402876791383,
        0.00584176280791,
        0.017384055425,
        0.00459496396821,
        0.00160051312626,
        0.000741301957567,
        0.000422535300225,
        0.000117244894983,
        4.83482905318e-05,
        -7.65748631305e-05,
        -4.82973687339e-05,
        -6.31563445281e-05,
        -4.96358629533e-05,
        -3.28140444653e-05,
        -0.00381392830086,
        -2.2205424354e-05,
        -2.08046627522e-05,
        -6.02527565374e-05,
        -6.13927549358e-05,
        -7.86844425359e-05,
        -5.83115112642e-05,
        -1.00000000053,
        0.999999999312,
        0.966457737196,
        -0.219605495654,
        0.0468841766484,
        0.00684262157402,
        0.0202674775836,
        0.00537079140753,
        0.00187458038878,
        0.000865956180706,
        0.000495511522165,
        0.000137073011615,
        5.74302634192e-05,
        -8.93994099465e-05,
        -5.62155084534e-05,
        -7.39618858773e-05,
        -5.81565211589e-05,
        -3.85413081487e-05,
        -2.60994706528e-05,
        -0.00220325179404,
        -2.42380769625e-05,
        -6.9647591436e-05,
        -7.18214013313e-05,
        -9.17537567393e-05,
        -6.84006922375e-05,
        -1.00000000071,
        0.999999999764,
        0.966449856068,
        -0.0804926019247,
        0.00240174669814,
        0.000333705601698,
        0.0010243028301,
        0.000266291334728,
        9.15640347326e-05,
        4.31088785186e-05,
        2.39061487984e-05,
        6.72966528453e-06,
        2.46812762013e-06,
        -4.46765835145e-06,
        -2.87331000782e-06,
        -3.60967969483e-06,
        -2.82785840082e-06,
        -1.83810620018e-06,
        -1.23731521098e-06,
        -1.22943185638e-06,
        -0.0435840470771,
        -3.74570208466e-06,
        -3.52754386765e-06,
        -4.62721358579e-06,
        -3.29029245443e-06,
        -1.00000000025,
        0.99999999989,
        0.966450885626,
        -0.0930387069811,
        0.00798160410524,
        0.00105163347384,
        0.00336004541599,
        0.000856544854467,
        0.000291392361018,
        0.000138684369303,
        7.36380419478e-05,
        2.04018784052e-05,
        6.31223295186e-06,
        -1.43694713149e-05,
        -9.41900545836e-06,
        -1.13161616528e-05,
        -8.83667701701e-06,
        -5.61432064122e-06,
        -3.74413355963e-06,
        -3.98149935196e-06,
        -1.29788702907e-05,
        -0.0409613242434,
        -1.10164159996e-05,
        -1.5032600094e-05,
        -1.0054769624e-05,
        -1.00000000021,
        1.00000000019,
        0.96645512033,
        -0.101002782635,
        0.0120221708142,
        0.00171563800479,
        0.00516312651028,
        0.00135592735081,
        0.000469274489383,
        0.000219289896347,
        0.000123999242542,
        3.49266574227e-05,
        1.37707759486e-05,
        -2.27089637221e-05,
        -1.44481832592e-05,
        -1.85821594955e-05,
        -1.4583315794e-05,
        -9.581724894e-06,
        -6.47508711222e-06,
        -6.21588662727e-06,
        -1.83047197698e-05,
        -1.81549772143e-05,
        -0.0246651236344,
        -2.34014099483e-05,
        -1.71161438747e-05,
        -1.00000000035,
        0.999999999386,
        0.966455330423,
        -0.113530057643,
        0.0166657496631,
        0.00230949511452,
        0.00710488400313,
        0.0018456811572,
        0.000635200733545,
        0.000298478807674,
        0.000164888436386,
        4.60192223618e-05,
        1.68251925905e-05,
        -3.09002352394e-05,
        -1.98651826411e-05,
        -2.49403935226e-05,
        -1.95403893756e-05,
        -1.268827519e-05,
        -8.53364139855e-06,
        -8.48680885747e-06,
        -2.60046761639e-05,
        -2.43079320654e-05,
        -3.20199459313e-05,
        -0.0237068016197,
        -2.26646438239e-05,
        -1.00000000008,
        0.966457350038,
        -0.123264837781,
        0.0196733761,
        0.00285189993131,
        0.00848483547624,
        0.00224170359846,
        0.000779051529285,
        0.000362241699838,
        0.000207031390181,
        5.81984397784e-05,
        2.38592890013e-05,
        -3.748487397e-05,
        -2.36919728935e-05,
        -3.09008634601e-05,
        -2.42766933282e-05,
        -1.60477076552e-05,
        -1.08669513193e-05,
        -1.02222857781e-05,
        -2.95071924309e-05,
        -3.01631031403e-05,
        -3.85150950509e-05,
        -2.86161242736e-05,
        -0.0139578876945,
        0.99999999975,
        -0.965972690099,
        3.70757979487e-08,
        1.10403929731e-07,
        0.000339788576207,
        0.00856044456576,
        0.01280070842,
        0.00653583718451,
        0.00526971363367,
        0.00264390593401,
        0.00220704564571,
        0.000449945800621,
        0.000844816666565,
        0.000198684229763,
        9.75341660747e-05,
        4.79627095969e-05,
        2.39092628223e-05,
        0.00103386682304,
        0.000942057171715,
        0.000504747049321,
        0.000481501592168,
        0.000246824708498,
        -0.999999999829,
        1.0,
        1.0,
        -0.00098325892238,
        1.0,
        -0.0373426319557,
        1.00000005567,
        -1.0000000087,
        0.999999913822,
        0.994013057902,
        0.0113963949577,
        0.00353735694311,
        0.00234484354527,
        0.00196338845387,
        0.00342590870473,
        0.000363424512637,
        -0.000671621100322,
        -0.000632438248459,
        -0.000527447719175,
        -0.000353312426647,
        -0.000301483592767,
        -7.5598371077e-05,
        -0.000156855228322,
        -3.66572274627e-05,
        -1.69581903474e-05,
        -7.81414661738e-06,
        -3.57696325741e-06,
        -0.000190910266436,
        -0.000174774843628,
        -9.61702711258e-05,
        -9.07575468506e-05,
        -4.66503780051e-05,
        0.999999999127,
        0.994012371927,
        0.00471234312138,
        0.00122421473699,
        0.000684393190077,
        0.0146407933646,
        0.0011864634236,
        0.000125383685277,
        -0.000235994040718,
        -0.000221916002457,
        -0.000184619197798,
        -0.000123670083028,
        -0.000105463585236,
        -2.64555245057e-05,
        -5.48960691287e-05,
        -1.28345219463e-05,
        -5.93842875862e-06,
        -2.73801211125e-06,
        -1.25375765907e-06,
        -6.68316505032e-05,
        -6.11351405393e-05,
        -3.36706061112e-05,
        -3.1757850895e-05,
        -1.63390469752e-05,
        0.999999860805,
        0.994012371491,
        0.00934136131864,
        0.00253134209092,
        0.00145501290684,
        0.00245952413724,
        0.00488452406355,
        0.000255328103209,
        -0.000519222326708,
        -0.000485348179486,
        -0.0003998876951,
        -0.000267841429717,
        -0.00022785672112,
        -5.7238926126e-05,
        -0.00011881459931,
        -2.78226929811e-05,
        -1.28815270033e-05,
        -5.9532855319e-06,
        -2.72975135932e-06,
        -0.000144797631441,
        -0.000132043043607,
        -7.29804750165e-05,
        -6.8690288423e-05,
        -3.54684073602e-05,
        1.0,
        0.994012371772,
        0.00535068408394,
        0.00030079142986,
        0.000167848322255,
        0.000291464753134,
        3.08309288928e-05,
        0.0214115037118,
        -5.7761441182e-05,
        -5.43348425837e-05,
        -4.52310812911e-05,
        -3.02986503509e-05,
        -2.58421234442e-05,
        -6.48185794871e-06,
        -1.34497846987e-05,
        -3.14418441455e-06,
        -1.45472883794e-06,
        -6.70624783138e-07,
        -3.07057771225e-07,
        -1.63729717601e-05,
        -1.4980404732e-05,
        -8.24866776769e-06,
        -7.78113171077e-06,
        -4.00236424918e-06,
        1.0,
        0.994012371274,
        0.00345839151888,
        -0.00153490772412,
        -0.000825541990857,
        -0.00148239117294,
        -0.000160215586972,
        0.000270963390709,
        0.0201405827627,
        0.000257064776975,
        0.000216975793482,
        0.000145358496152,
        0.000124402089657,
        3.11397647326e-05,
        6.45831832174e-05,
        1.50635681898e-05,
        6.96320881113e-06,
        3.19923839388e-06,
        1.46199805554e-06,
        7.85047842921e-05,
        7.21443870214e-05,
        3.95272598373e-05,
        3.73976328215e-05,
        1.91377367637e-05,
        0.999999999897,
        0.994012376595,
        0.00345516542103,
        -0.00310082393816,
        -0.00171905441685,
        -0.00300285998824,
        -0.000318831852744,
        0.000586952751478,
        0.000552897698564,
        0.0484333949845,
        0.00046132972838,
        0.000309030368435,
        0.000263728117594,
        6.61265082633e-05,
        0.000137200725781,
        3.20613544915e-05,
        1.48316543279e-05,
        6.83346996413e-06,
        3.12781292222e-06,
        0.000166978851067,
        0.000152890574993,
        8.41150695564e-05,
        7.93871952112e-05,
        4.07988323543e-05,
        1.0,
        0.994012379391,
        0.0024856242731,
        -0.0034283935216,
        -0.00191030548566,
        -0.00332159639289,
        -0.00035157436766,
        0.000656482660973,
        0.000617686196229,
        0.000514435161043,
        0.0371033226784,
        0.000344597596454,
        0.00029394616161,
        7.37232196649e-05,
        0.000152972740604,
        3.57578241217e-05,
        1.6543654811e-05,
        7.62568323918e-06,
        3.49132822916e-06,
        0.000186210708421,
        0.000170398954975,
        9.38103627278e-05,
        8.85022539626e-05,
        4.55145617602e-05,
        0.999999999501,
        0.994012445513,
        0.00149790742572,
        -0.00497599557678,
        -0.0027633301034,
        -0.00481953278651,
        -0.000511183984261,
        0.0009455739137,
        0.00089036743539,
        0.000742444186002,
        0.000497337496153,
        0.269847498866,
        0.000424364099621,
        0.000106413623337,
        0.000220794442411,
        5.16009854308e-05,
        2.38717054579e-05,
        1.10002069328e-05,
        5.03545597604e-06,
        0.000268733637366,
        0.000246011000928,
        0.000135377279174,
        0.000127750936797,
        6.5669263906e-05,
        1.0,
        0.994012653139,
        0.000509141643572,
        -0.00531683716298,
        -0.00295062050898,
        -0.00514931672482,
        -0.00054636310896,
        0.00100885295873,
        0.000950084717848,
        0.000792431865489,
        0.000530822252708,
        0.000452962625359,
        0.489288383804,
        0.000113580692203,
        0.000235663439402,
        5.5073888008e-05,
        2.54779167716e-05,
        1.17396807182e-05,
        5.3737711326e-06,
        0.000286823835334,
        0.000262591779392,
        0.000144488986902,
        0.000136356331075,
        7.00864097423e-05,
        0.999999995235,
        0.987653557832,
        -0.00289358315782,
        -0.00921282158608,
        -0.00505466559616,
        -0.0098270096467,
        -0.000722095115386,
        0.00198576139321,
        0.00180523764795,
        0.00153445201544,
        0.00103801165686,
        0.000834922425305,
        0.000197447871294,
        1.10564223321,
        0.000451309417026,
        0.000105775644403,
        4.865679641e-05,
        2.22128852737e-05,
        1.01368325523e-05,
        0.000552854034955,
        0.000507723095015,
        0.000298992486566,
        0.000253861547753,
        0.000129751457276,
        1.00000000005,
        0.994063218023,
        -0.00142164533952,
        -0.00731006106788,
        -0.00406744284623,
        -0.00708838776858,
        -0.000763777129538,
        0.00138066828291,
        0.00130383937477,
        0.00108691002652,
        0.000728184526979,
        0.00062155557359,
        0.000155883187549,
        0.00032355911682,
        1.07796069704,
        7.56289355243e-05,
        3.49854429499e-05,
        1.61091018043e-05,
        7.37598199731e-06,
        0.000393750661429,
        0.000360561554774,
        0.000198534783918,
        0.000187165751054,
        9.61779668038e-05,
        1.0,
        0.994012392287,
        -0.00469324783994,
        -0.011054070463,
        -0.00613589605998,
        -0.0107060679009,
        -0.00113589635479,
        0.00209826787181,
        0.0019759868419,
        0.00164799036717,
        0.00110393436666,
        0.000941997857923,
        0.000236209985015,
        0.000490101470748,
        0.00011453647758,
        1.17065144588,
        5.29863916889e-05,
        2.44153954121e-05,
        1.1176110737e-05,
        0.000596501892608,
        0.000546095235945,
        0.000300491809357,
        0.000283574397429,
        0.000145759648867,
        1.0,
        0.994012392287,
        -0.0118930230725,
        -0.0128933547803,
        -0.00715687775711,
        -0.0124874533238,
        -0.00132490356464,
        0.00244740014922,
        0.00230477450459,
        0.00192220142026,
        0.00128761969976,
        0.00109873816712,
        0.000275513400983,
        0.000571650282667,
        0.000133594416086,
        6.18029030732e-05,
        1.0,
        2.84779231447e-05,
        1.303572892e-05,
        0.00069575494255,
        0.000636960861834,
        0.000350491202728,
        0.000330758831436,
        0.000170012892653,
        1.0,
        0.994012392287,
        -0.0085302720368,
        -0.014937189473,
        -0.00829149371384,
        -0.0144669776465,
        -0.00153493857624,
        0.00283537069379,
        0.00267014090838,
        0.00222691660667,
        0.00149173979835,
        0.00127291537311,
        0.000319189352727,
        0.000662271459895,
        0.000154772622496,
        7.16002923618e-05,
        3.29924389891e-05,
        1.0,
        1.51022435876e-05,
        0.000806049960111,
        0.000737935090811,
        0.000406053179969,
        0.000383192544362,
        0.000196964439203,
        1.0,
        0.994012392287,
        -0.0103307319163,
        -0.0168540678387,
        -0.00935559683057,
        -0.016323529597,
        -0.00173192375996,
        0.00319923749714,
        0.00301280582704,
        0.00251270027007,
        0.0016831775973,
        0.00143627092442,
        0.000360151608536,
        0.000747262229872,
        0.000174634981476,
        8.07889463661e-05,
        3.72264538639e-05,
        1.7040358187e-05,
        1.0,
        0.000909492237078,
        0.000832635760339,
        0.000458162963124,
        0.000432368485459,
        0.000222241391956,
        0.999999999103,
        0.994032161007,
        0.000461073821546,
        -0.00685618971423,
        -0.00380782737715,
        -0.00664358735095,
        -0.000706865710879,
        0.00129994633811,
        0.00122472599935,
        0.00102153038719,
        0.000684330938611,
        0.000583953021576,
        0.000146452911682,
        0.000303889338234,
        7.10198716538e-05,
        3.28539354272e-05,
        1.51386055865e-05,
        6.92883459638e-06,
        0.000369804061587,
        1.03885544221,
        0.00033857012211,
        0.000186326680558,
        0.000175818134389,
        9.03648406246e-05,
        0.999999999926,
        0.99404235068,
        0.00029755771468,
        -0.00685356570243,
        -0.00380352771079,
        -0.00663896985854,
        -0.000704545724901,
        0.00129982311328,
        0.00122442694566,
        0.00102149960357,
        0.000684184574466,
        0.000583862631634,
        0.000146400512898,
        0.000303822427479,
        7.10160743066e-05,
        3.28419294415e-05,
        1.51309089738e-05,
        6.92793304093e-06,
        0.000369743167594,
        0.000338483522503,
        1.08073207189,
        0.00018628657157,
        0.000175758512992,
        9.03522128284e-05,
        0.999999998033,
        0.995465384686,
        0.000616472921016,
        -0.00972041355606,
        -0.00597782626015,
        -0.0099696548943,
        -0.00175818426567,
        0.00131863813734,
        0.00140654734295,
        0.00118677432899,
        0.000813160187509,
        0.000703273676597,
        0.000171697674977,
        0.000373614139744,
        9.06563718371e-05,
        4.08640464322e-05,
        1.897259294e-05,
        8.24148835078e-06,
        0.000450534697755,
        0.000412074419764,
        0.000230761674177,
        1.06695143203,
        0.000217025860661,
        0.000108512930161,
        0.999999999512,
        1.0,
        -0.00183720066379,
        -0.00965619615164,
        -0.00419176530316,
        -0.00950763927082,
        -0.000523970673953,
        0.00183389726871,
        0.00176840094251,
        0.00150641562605,
        0.000949696810144,
        0.000835078229919,
        0.000190348714693,
        0.000433913197416,
        0.000102338018112,
        4.55404181143e-05,
        2.02117586023e-05,
        9.78607296255e-06,
        0.000507596571031,
        0.000466661364529,
        0.000278359408865,
        0.000245611244138,
        1.06494202892,
        0.000118712101353,
        0.0,
        0.999999997932,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        -0.0499100949661,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        0.0,
        -0.00623876157143,
        1.17229642192,
        0.999999999982,
        -0.994012374874,
        -5.41703356348e-07,
        -0.000157564831478,
        -0.000844516978699,
        -0.00425815133459,
        -0.00642180063562,
        -0.00334540102295,
        -0.00268964555635,
        -0.0011861572888,
        -0.000694843630605,
        -3.92184359443e-08,
        -5.03347160133e-06,
        -8.51329759353e-12,
        -4.29115161715e-15,
        -2.69282536795e-18,
        -1.9989575149e-21,
        -2.28690353861e-05,
        -1.31453078319e-05,
        -7.36560291026e-08,
        -5.01863340787e-08,
        -8.87526336395e-11,
        -1.0,
        0.0535195133394,
        -1.0,
        -0.00461353160985,
        0.0360688577883,
        0.00654847747171,
        0.0255182498418,
        0.0136994294817,
        0.00783741660501,
        0.00654207870514,
        0.00353264606157,
        0.00299607263104,
        0.000597036259864,
        0.00133123985316,
        0.000267590722845,
        0.000118651375715,
        5.15764883239e-05,
        2.27649092739e-05,
        0.00164254064076,
        0.00154256276611,
        0.000754521081674,
        0.000726358888917,
        0.000338350600885,
        -1.0,
        -0.00441291763382,
        -1.0,
        -0.00806469178605,
        -0.0109977324756,
        -0.0103775020403,
        -0.00757324006363,
        -0.00220814778814,
        -0.000862197670513,
        -0.000626999397004,
        -0.000222832932902,
        -0.000166845402166,
        -5.56887166151e-06,
        -3.76105797526e-05,
        1.60326426943e-06,
        2.07714079207e-06,
        1.47335880223e-06,
        8.42503407384e-07,
        -5.61093806743e-05,
        -5.18892675254e-05,
        -1.01044234579e-05,
        -1.00394177719e-05,
        1.07650821155e-06,
        -0.0155671139201,
        0.999999999451,
        0.999999928815,
        -1.0,
        0.999999974738,
        0.999999980644,
        0.999995945589,
        -0.994874991433,
        -0.00435422833148,
        -0.993393540089,
        -1.00002455179,
        -0.995611997518,
        -1.00000770005,
        -1.00883215354,
        -1.00002586945,
        -1.00910849791,
        -1.00043563798,
        -1.01254111213,
        -1.0009050659,
        -0.991283431415,
        -1.0011491686,
        -0.992556656261,
        -1.00147755869,
        -0.988411648103,
        -1.00160317954,
        -0.988595608139,
        -1.002202602,
        -0.990485216845,
        -1.00187834911,
        -0.991166203372,
        -1.00242496948,
        -0.990763935577,
        -1.00267208074,
        -0.995741990906,
        -1.00291175485,
        -1.0,
        -1.00312327256,
        -0.993186545447,
        -1.00177765956,
        -0.987991113424,
        -1.001844869,
        -0.993008479702,
        -1.00206350329,
        -0.988830499697,
        -1.00211400842,
        -0.995888929081,
        -1.0023067548,
        -1.0,
        0.999999974738,
        1.0,
        0.999998807932,
        -0.896958372018,
        -0.000911054992981,
        -0.998690573634,
        -0.420390918841,
        -0.999523330414,
        -1.00000906536,
        -0.998984377017,
        -1.00000362166,
        -0.994794893751,
        -1.0000650455,
        -0.994795204253,
        -1.00017894761,
        -0.993880720839,
        -1.00025167337,
        -0.993732143735,
        -1.00036763497,
        -0.993733535488,
        -1.00042195138,
        -0.993547422057,
        -1.00076785476,
        -0.993599702424,
        -0.453643968084,
        -0.993552957926,
        -0.471049073737,
        -0.99352941811,
        -0.473250426034,
        -0.993712934095,
        -0.474258221165,
        -1.0,
        -0.474759422538,
        -0.993666956519,
        -0.446717090426,
        -0.993648757511,
        -0.449828104227,
        -0.993625660915,
        -0.46348685682,
        -0.993557731704,
        -0.464064207934,
        -0.993714156489,
        -0.469914001254,
        1.0,
        0.999999559144,
        -0.851336990363,
        -9.71572687329e-05,
        -0.857755040898,
        -0.0482968534246,
        -0.850320454037,
        -0.246373548693,
        -0.836386749885,
        -1.00127571079,
        -0.920662341599,
        -1.00025614991,
        -0.999594624088,
        -0.999965656989,
        -0.994537802754,
        -1.00006137499,
        -0.977010617029,
        -1.00063642896,
        -0.96831880596,
        -1.0009594692,
        -0.950302714202,
        -1.0043510992,
        -0.959330015012,
        -1.00239343537,
        -0.949188969592,
        -1.00634416686,
        -0.945965823324,
        -1.00851498933,
        -0.947928894694,
        -1.01021084604,
        -0.947823283715,
        -1.01201306096,
        -0.964499946837,
        -1.00177082221,
        -0.955585104158,
        -1.00223512241,
        -0.956879012946,
        -1.00359811636,
        -0.950889592086,
        -1.0039201321,
        -0.954305011778,
        -1.00540073842,
        0.999999997211,
        -0.905108968772,
        -9.02433977751e-06,
        -0.934939477203,
        -0.00634260358689,
        -0.985571959346,
        -0.0255559730507,
        -0.992573249289,
        -0.373051867684,
        -1.1248318788,
        -0.999956342745,
        -0.80066716073,
        -0.997589938146,
        -0.83787484404,
        -0.993637319521,
        -0.847334995308,
        -0.985403054535,
        -0.856727307732,
        -0.982711530826,
        -0.852964686783,
        -0.987673285242,
        -0.852443317811,
        -1.00814410477,
        -0.845655538171,
        -0.633627595188,
        -0.841913328499,
        -0.260440358622,
        -0.832936314559,
        -0.11179540391,
        -0.826916568628,
        -0.0492941870852,
        -0.848824436177,
        -1.00728865576,
        -0.859624185568,
        -1.00595907635,
        -0.847607902059,
        -0.99838512758,
        -0.854474158912,
        -0.996308327377,
        -0.841851836311,
        -0.84248924744,
        0.999999999032,
        -1.03044709999,
        -1.18601932524e-05,
        -0.950428359939,
        -0.0111771680366,
        -1.01542517508,
        -0.035326290261,
        -1.02139433018,
        -0.578055965196,
        -0.955373664039,
        -1.00002268005,
        -0.830326078633,
        -0.995662744904,
        -0.877440751683,
        -0.99133323822,
        -0.880278737897,
        -0.981489020626,
        -0.889511141157,
        -0.978027113228,
        -0.880813977108,
        -0.905285618114,
        -0.88164032993,
        -1.02039975493,
        -0.871805655891,
        -0.0327084586249,
        -0.867178348691,
        -0.0045785532311,
        -0.856686004863,
        -0.00072386402281,
        -0.849685034712,
        -0.000122550201515,
        -0.878211414258,
        -1.02091906278,
        -0.889842753515,
        -1.02033480104,
        -0.875106565059,
        -0.421969030703,
        -0.882661722922,
        -0.375296990667,
        -0.867668677016,
        -0.060952628031,
        0.999999998507,
        -0.873201125084,
        -1.14924633809e-05,
        -0.97722856706,
        -0.0151044570083,
        -1.05091649397,
        -0.0363317797463,
        -1.06146622618,
        -0.664077655435,
        -0.846202738903,
        -1.00014952745,
        -0.844386740363,
        -0.99495123244,
        -0.90096810227,
        -0.990821032553,
        -0.90290352068,
        -0.980934115976,
        -0.91485623409,
        -0.975810566738,
        -0.906062141243,
        -0.0955659129335,
        -0.905538232035,
        -0.376361888325,
        -0.895721053499,
        -0.00123160862376,
        -0.891421264893,
        -5.58908721371e-05,
        -0.878957193215,
        -3.07141927185e-06,
        -0.87123567419,
        -1.88295447747e-07,
        -0.900700673442,
        -0.70866500536,
        -0.915860686757,
        -0.549491349849,
        -0.89788675868,
        -0.0625357984947,
        -0.907956583738,
        -0.0521064061487,
        -0.88951799914,
        -0.00326570000195,
        0.999999998691,
        -0.531904112559,
        -7.51749921453e-06,
        -0.303809206017,
        -0.0113639141566,
        -0.163855140334,
        -0.0243853910142,
        -0.189675874378,
        -0.467531929345,
        -0.5288341525,
        -1.00019140549,
        -0.610944045226,
        -0.99412204837,
        -0.823555655244,
        -0.990049596276,
        -0.901198479239,
        -0.979740180458,
        -0.91219563322,
        -0.980954982266,
        -0.903005334947,
        -0.0182875484289,
        -0.902790153786,
        -0.15407505596,
        -0.893002562868,
        -7.1613022344e-05,
        -0.888822691485,
        -9.05048499309e-07,
        -0.876872388802,
        -1.42522302827e-08,
        -0.869475240425,
        -2.68293301308e-10,
        -0.898342061097,
        -0.354579221424,
        -0.912556730433,
        -0.254655836285,
        -0.895298653366,
        -0.0149620604605,
        -0.904803575428,
        -0.0119848668129,
        -0.887141446279,
        -0.000277234909268,
        0.999999998786,
        -0.02936083623,
        -8.14219591215e-06,
        -0.369228713199,
        -0.014179210462,
        -0.058220108471,
        -0.0271023949239,
        -0.170567745423,
        -0.544654633375,
        -0.346775142931,
        -1.00040954935,
        -0.464048745592,
        -0.997396670601,
        -0.890301358192,
        -0.993843514057,
        -0.886607454712,
        -0.996572962239,
        -0.887454159237,
        -1.0103555022,
        -0.880979820092,
        -0.00262881990575,
        -0.883216680229,
        -0.0547286305056,
        -0.877467636748,
        -3.55512323975e-06,
        -0.874924302446,
        -1.32989477277e-08,
        -0.871438482499,
        -5.94949192141e-11,
        -0.86870064203,
        -3.14535215327e-13,
        -0.88325556175,
        -0.157641715592,
        -0.885209745662,
        -0.103867133142,
        -0.880493051768,
        -0.00285508805652,
        -0.881790575804,
        -0.00216250635327,
        -0.877270267681,
        -1.99047796158e-05,
        0.999999999628,
        -0.0840488327863,
        -3.07346266206e-06,
        -0.906185163525,
        -0.00621186841703,
        -0.267247537998,
        -0.01051263044,
        -0.971262271648,
        -0.221840443082,
        -0.92744139653,
        -1.0004774507,
        -0.912577747811,
        -0.998258563746,
        -0.910673999747,
        -0.994982176175,
        -0.907658934112,
        -0.995438065852,
        -0.907555320029,
        -0.906027500381,
        -0.902765857329,
        -0.000336192165754,
        -0.904691360157,
        -0.017331486558,
        -0.900560015672,
        -1.63966844629e-07,
        -0.898711894165,
        -2.09447880109e-10,
        -0.896693170276,
        -3.30458298738e-13,
        -0.894940891781,
        -6.04675396484e-16,
        -0.905073029477,
        -0.0605564963312,
        -0.905624859125,
        -0.0380329873526,
        -0.902961090799,
        -0.000491741992976,
        -0.903319428993,
        -0.00035227532407,
        -0.900862189227,
        -1.2884221974e-06,
        0.999999957437,
        -0.852071036419,
        -1.93692706456e-05,
        -0.854646532591,
        -0.00209384894794,
        -0.849491461852,
        -0.0756068197968,
        -0.838661726603,
        -0.951665397312,
        -0.81314280701,
        -1.00300364466,
        -0.778300697103,
        -0.999166028809,
        -0.838165279581,
        -0.999430639761,
        -1.01994910218,
        -1.00123260846,
        -0.975020475286,
        -1.00180050098,
        -0.908628212517,
        -1.00327308927,
        -0.933468953669,
        -1.00234828017,
        -0.903969240506,
        -1.00421242405,
        -0.895860882581,
        -1.00522777138,
        -0.897435294418,
        -1.00596322367,
        -0.895297737882,
        -1.00671048793,
        -0.950704852008,
        -1.00206357649,
        -0.923671512679,
        -1.00243371656,
        -0.924174304257,
        -1.00284531992,
        -0.909982411564,
        -1.00308228745,
        -0.915351335066,
        -1.0036713944,
        0.999999956838,
        -0.88044462439,
        -5.04126062988e-06,
        -0.89089049804,
        -0.000125548121257,
        -0.879339716256,
        -0.0300650242954,
        -0.869455239468,
        -0.507795784119,
        -0.842225527486,
        -0.99720043978,
        -0.803250873864,
        -0.99916094912,
        -0.824916257208,
        -0.999466334238,
        -1.02016445472,
        -1.00012886992,
        -1.01526060995,
        -1.00026864058,
        -0.960216919959,
        -0.999552566155,
        -0.985276583592,
        -0.822547985602,
        -0.955975311679,
        -0.85953628315,
        -0.947632454062,
        -0.861032978731,
        -0.950687113125,
        -0.861520031032,
        -0.949154523414,
        -0.861864342049,
        -0.999872198071,
        -0.803325334321,
        -0.975116725999,
        -0.812108028484,
        -0.976662040031,
        -0.847278187591,
        -0.961090187659,
        -0.84847765817,
        -0.968351878549,
        -0.858322812249,
        0.999999984817,
        -0.900346732399,
        -1.61508923352e-06,
        -0.912571876342,
        -1.15216003378e-05,
        -0.899737663369,
        -0.0134842792021,
        -0.891240825017,
        -0.359016801644,
        -0.866599562149,
        -0.995858280091,
        -0.826641529439,
        -0.999063900388,
        -0.844420209158,
        -0.999483648154,
        -1.00520617307,
        -1.00001619961,
        -1.01300948488,
        -1.00005685509,
        -0.971311392574,
        -0.998620253892,
        -0.993067607386,
        -0.754615593617,
        -0.968457165976,
        -0.790310809729,
        -0.960069938016,
        -0.791677503416,
        -0.963456582199,
        -0.792057295516,
        -0.962669953781,
        -0.792230845497,
        -1.00476296468,
        -0.736291913534,
        -0.984197998375,
        -0.744615819223,
        -0.985952345606,
        -0.778439076809,
        -0.971719738547,
        -0.779538647903,
        -0.978116032704,
        -0.789197321046,
        0.999999983112,
        -0.894004843632,
        -5.46064755813e-07,
        -0.903129493347,
        -1.29874323024e-06,
        -0.893076324776,
        -0.00633875663685,
        -0.885998595208,
        -0.291500613896,
        -0.869398405404,
        -0.995010875562,
        -0.821374223467,
        -0.998956994734,
        -0.843642460835,
        -0.999396936259,
        -0.997569079896,
        -0.999988926902,
        -1.00296366906,
        -1.00004365962,
        -0.96286775832,
        -0.998657970319,
        -0.98342570829,
        -0.800592914723,
        -0.949884686912,
        -0.841130836083,
        -0.951142763623,
        -0.842924047934,
        -0.948066358982,
        -0.843475234837,
        -0.953577579724,
        -0.843737388102,
        -0.988437351928,
        -0.780287456393,
        -0.959651300333,
        -0.789521652599,
        -0.98173749672,
        -0.827343362459,
        -0.955734657808,
        -0.828600045126,
        -0.963650883596,
        -0.839772712484,
        0.999999973997,
        -0.835092136604,
        -1.80800125451e-07,
        -0.841735571046,
        -1.44415233176e-07,
        -0.827711737267,
        -0.00284418976341,
        -0.827037367889,
        -0.235781192325,
        -0.795689224088,
        -1.00031703219,
        -0.882977949583,
        -1.00114890815,
        -0.945429851186,
        -0.999687316155,
        -1.04049795106,
        -0.999854335957,
        -0.91129957655,
        -1.00023039283,
        -0.829535183521,
        -1.00130782079,
        -0.837774757422,
        -0.872161080973,
        -0.834611167989,
        -0.922248763179,
        -0.832783036157,
        -0.925555671865,
        -0.837986968872,
        -0.926890076727,
        -0.863369857407,
        -0.927792129266,
        -0.860649580584,
        -0.84845948516,
        -0.739962059117,
        -0.859426067368,
        -0.961447710196,
        -0.904120062528,
        -0.819092630728,
        -0.905896051146,
        -0.936180430146,
        -0.920017881703,
        0.999999982522,
        -0.402383569232,
        -4.40271916706e-08,
        -0.957790066902,
        -5.76629573954e-09,
        -0.905307829867,
        -0.000533529539635,
        -1.12978805203,
        -0.0632211939417,
        -0.787051614913,
        -0.983220775494,
        -1.32385105674,
        -0.778202817523,
        -1.38377056596,
        -1.00078567388,
        -1.11009294282,
        -0.87592869697,
        -1.0024864135,
        -0.794694012867,
        -0.524026297125,
        -1.00214717005,
        -0.719692573465,
        -0.312583751997,
        -0.493626734492,
        -0.333447202593,
        -0.525423680299,
        -0.335122520173,
        -0.508845785726,
        -0.335850652766,
        -0.520232071193,
        -0.336329262695,
        -0.704241088223,
        -0.303269197353,
        -0.701851077988,
        -0.30762363482,
        -0.841074110687,
        -0.325554997334,
        -0.564275726641,
        -0.326304163828,
        -0.52490302258,
        -0.332333358274,
        -1.0,
        0.999999999112,
        -1.18549645438e-06,
        -1.55128782863e-07,
        -0.0143564914653,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        -1.0,
        1.0,
        1.0,
        1.00000000094,
        -1.0,
        0.999999974738,
    ];

    (n, rowind, colst, nz)
}
