#[derive(Debug)]
#[derive(serde_derive::Deserialize,serde_derive::Serialize)]
struct Serialized {
	#[serde(rename="s")]
	#[serde(skip_serializing_if="Option::is_none")]
	last_s: Option<(u64, u32)>,

	#[serde(rename="l")]
	#[serde(skip_serializing_if="Option::is_none")]
	legacy_last: Option<std::time::SystemTime>,
}

impl serde::Serialize for crate::Tracker {
	fn serialize<S: serde::Serializer>(&self, serializer: S)
	-> Result<<S as serde::Serializer>::Ok, <S as serde::Serializer>::Error> {
		Serialized{
			last_s: self.last.map(|t| {
				let duration = t.duration_since(std::time::UNIX_EPOCH).unwrap();
				(duration.as_secs(), duration.subsec_nanos())
			}),

			legacy_last: None,
		}.serialize(serializer)
	}
}

impl<'de> serde::Deserialize<'de> for crate::Tracker {
	fn deserialize<D: serde::Deserializer<'de>>(deserializer: D) -> Result<Self, <D as serde::Deserializer<'de>>::Error> {
		let Serialized{
			last_s,
			legacy_last,
		} = serde::Deserialize::deserialize(deserializer)?;

		Ok(crate::Tracker {
			last: last_s.map(|(s, ns)| std::time::UNIX_EPOCH + std::time::Duration::new(s, ns))
				.or(legacy_last),
		})
	}
}

#[test]
fn test_serde_serialize() -> Result<(), Box<dyn std::error::Error>> {
	let mut now = std::time::UNIX_EPOCH;
	let cfg = crate::Config::new(std::time::Duration::from_secs(1), 10);
	let mut t = crate::Tracker::default();

	assert_eq!(
		serde_json::to_string(&t)?,
		r#"{}"#);

	now += std::time::Duration::from_secs(20);
	t.acquire_at(&cfg, 4, now)?;
	dbg!(t.capacity_at(&cfg, now));

	assert_eq!(
		serde_json::to_string(&t)?,
		r#"{"s":[14,0]}"#);

	Ok(())
}

#[test]
fn test_serde_deserialize() -> Result<(), Box<dyn std::error::Error>> {
	assert_eq!(
		serde_json::from_str::<crate::Tracker>(r#"{}"#)?.last,
		None);

	assert_eq!(
		serde_json::from_str::<crate::Tracker>(r#"{"s":[14,0],"l":{"secs_since_epoch":14,"nanos_since_epoch":0}}"#)?.last,
		Some(std::time::UNIX_EPOCH + std::time::Duration::from_secs(14)));

	assert_eq!(
		serde_json::from_str::<crate::Tracker>(r#"{"s":[14,0]}"#)?.last,
		Some(std::time::UNIX_EPOCH + std::time::Duration::from_secs(14)));

	Ok(())
}

#[test]
fn test_serde_old() -> Result<(), Box<dyn std::error::Error>> {
	assert_eq!(
		serde_json::from_str::<crate::Tracker>(r#"{"l":null}"#)?.last,
		None);

	assert_eq!(
		serde_json::from_str::<crate::Tracker>(r#"{"l":{"secs_since_epoch":14,"nanos_since_epoch":0}}"#)?.last,
		Some(std::time::UNIX_EPOCH + std::time::Duration::from_secs(14)));

	Ok(())
}
