/*
    Fork of gobble (rewrite of Devour in Rust), by Sebastien MacDougall-Landry
    gobble upstream: <https://github.com/EmperorPenguin18/gobble>
    Custom patches applied for personal use
*/

#![deny(
    missing_debug_implementations,
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]
#![deny(clippy::all)]

extern crate anyhow;
extern crate xcb;

use anyhow::{anyhow, Error};
use std::{
    env,
    process::{self, Child, Command, Output},
};
use xcb::Connection;

// TODO: Change function names to be more descriptive
fn issue(args: &[String]) -> Result<Child, Error> {
    let child: Child = Command::new(&args[1]).args(&args[2..]).spawn()?;

    // TODO: check if the command is returning success before continuing
    /*
        if !child.wait().expect("failed to wait for child").success() {
            return Err(anyhow!("command is not returning success"));
        }

        check_exit_code(args)?;
    */

    Ok(child)
}

// TODO: Merge `is_command_running` and `check_command` into one function
fn is_command_executable(command: &str) -> bool {
    let output: Output = Command::new("which")
        .arg(command)
        .output()
        .expect("failed to execute `which` command");

    output.status.success()
}

fn check_command(command: &str) -> Result<(), Error> {
    if is_command_executable(command) {
        Ok(())
    } else {
        Err(anyhow!(
            "command '{}' is not available on this system",
            command
        ))
    }
}

fn main() -> Result<(), Error> {
    let args: Vec<String> = env::args().collect();

    if args.len() < 2 {
        return Err(anyhow!("no command specified"));
    }

    let conn_scrn_num: (Connection, i32) = Connection::connect(None)?;
    let conn: Connection = conn_scrn_num.0;
    let win: u32 = xcb::get_input_focus(&conn).get_reply()?.focus();

    check_command(&args[1])?;
    let mut child: Child = issue(&args)?;

    xcb::unmap_window_checked(&conn, win).request_check()?;
    conn.flush();

    let exit_code: i32 = child.wait()?.code().unwrap_or(-1);

    xcb::map_window_checked(&conn, win).request_check()?;
    conn.flush();

    process::exit(exit_code);
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_command_available() {
        assert!(is_command_executable("which"));
    }

    #[test]
    fn test_command_not_available() {
        assert!(!is_command_executable("not_a_command"));
    }

    #[test]
    fn test_check_command() {
        assert!(check_command("which").is_ok());
    }

    #[test]
    fn test_check_command_not_available() {
        assert!(check_command("not_a_command").is_err());
    }
}
