//! Module that stores the actual code that queries the system.
//!
//!

// Std imports
use std::{cmp, fmt, process};

/// Enum to store the different answer types that a query can have
/// and then off this we can build formatters and things like that.
enum Answer {
    Percentage(u8),
    Text(String),
}

impl fmt::Display for Answer {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            // For percentage we use the cmp::min function to cap it at
            // 100%.
            Self::Percentage(pc) => write!(f, "{}%", cmp::min(100, *pc)),
            Self::Text(txt) => write!(f, "{}", txt),
        }
    }
}

/// The main functions that every Query needs.
/// This is a trait to help build a prototype for generic
/// Queries in the future.
trait Query {
    fn ask(&mut self) -> Answer;
    fn been_asked(&self) -> bool;
}

/// External command query template.
struct CommandQueryTemplate {
    path: String,
    args: Vec<String>,
    asked: bool,
}

/// impl query for command
impl Query for CommandQueryTemplate {
    fn ask(&mut self) -> Answer {
        let ans = process::Command::new(&self.path)
            .args(&self.args)
            .output()
            .expect("failed");
        self.asked = true;
        Answer::Text(String::from_utf8_lossy(&ans.stdout).to_string())
    }

    fn been_asked(&self) -> bool {
        self.asked
    }
}

mod tests {
    #[test]
    /// Test to confirm that the display of percentages is
    /// working correctly.
    fn answer_percentage_display() {
        use super::Answer;
        let under_100 = Answer::Percentage(5);
        let over_100 = Answer::Percentage(105);

        assert_eq!("5%".to_string(), under_100.to_string());
        assert_eq!("100%".to_string(), over_100.to_string());
    }

    #[test]
    /// Test to see if the basic functionality of CommandQueryTemplate
    /// is working.
    fn command_template_basic() {
        use super::{CommandQueryTemplate, Query};
        let mut query = CommandQueryTemplate {
            path: "/usr/bin/echo".to_string(),
            args: vec!["hello".to_string()],
            asked: false,
        };
        let ans = query.ask();
        assert_eq!("hello\n".to_string(), ans.to_string());
    }
}
