//! NIST P-256 tests and test vectors

use crate::{ecdsa_tests, signature::TestVector};
use core::convert::TryFrom;
use ring_compat::signature::{
    ecdsa::p256::{Signature, SigningKey, VerifyingKey},
    Signer, Verifier,
};

ecdsa_tests!(SigningKey, VerifyingKey, TEST_VECTORS);

/// ECDSA test vectors for the NIST P-256 elliptic curve (SHA-256)
///
/// Sourced from NIST's CAVP web site (FIPS 186-4 ECDSA Test Vectors):
///
/// <https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/Digital-Signatures>
const TEST_VECTORS: &[TestVector] = &[
    TestVector {
        sk: b"\x51\x9b\x42\x3d\x71\x5f\x8b\x58\x1f\x4f\xa8\xee\x59\xf4\x77\x1a\x5b\x44\xc8\x13\x0b\x4e\x3e\xac\xca\x54\xa5\x6d\xda\x72\xb4\x64",
        pk: b"\x1c\xcb\xe9\x1c\x07\x5f\xc7\xf4\xf0\x33\xbf\xa2\x48\xdb\x8f\xcc\xd3\x56\x5d\xe9\x4b\xbf\xb1\x2f\x3c\x59\xff\x46\xc2\x71\xbf\x83\xce\x40\x14\xc6\x88\x11\xf9\xa2\x1a\x1f\xdb\x2c\x0e\x61\x13\xe0\x6d\xb7\xca\x93\xb7\x40\x4e\x78\xdc\x7c\xcd\x5c\xa8\x9a\x4c\xa9",
        msg: b"\x59\x05\x23\x88\x77\xc7\x74\x21\xf7\x3e\x43\xee\x3d\xa6\xf2\xd9\xe2\xcc\xad\x5f\xc9\x42\xdc\xec\x0c\xbd\x25\x48\x29\x35\xfa\xaf\x41\x69\x83\xfe\x16\x5b\x1a\x04\x5e\xe2\xbc\xd2\xe6\xdc\xa3\xbd\xf4\x6c\x43\x10\xa7\x46\x1f\x9a\x37\x96\x0c\xa6\x72\xd3\xfe\xb5\x47\x3e\x25\x36\x05\xfb\x1d\xdf\xd2\x80\x65\xb5\x3c\xb5\x85\x8a\x8a\xd2\x81\x75\xbf\x9b\xd3\x86\xa5\xe4\x71\xea\x7a\x65\xc1\x7c\xc9\x34\xa9\xd7\x91\xe9\x14\x91\xeb\x37\x54\xd0\x37\x99\x79\x0f\xe2\xd3\x08\xd1\x61\x46\xd5\xc9\xb0\xd0\xde\xbd\x97\xd7\x9c\xe8",
        sig: b"\xf3\xac\x80\x61\xb5\x14\x79\x5b\x88\x43\xe3\xd6\x62\x95\x27\xed\x2a\xfd\x6b\x1f\x6a\x55\x5a\x7a\xca\xbb\x5e\x6f\x79\xc8\xc2\xac\x8b\xf7\x78\x19\xca\x05\xa6\xb2\x78\x6c\x76\x26\x2b\xf7\x37\x1c\xef\x97\xb2\x18\xe9\x6f\x17\x5a\x3c\xcd\xda\x2a\xcc\x05\x89\x03",
    },
    TestVector {
        sk: b"\x0f\x56\xdb\x78\xca\x46\x0b\x05\x5c\x50\x00\x64\x82\x4b\xed\x99\x9a\x25\xaa\xf4\x8e\xbb\x51\x9a\xc2\x01\x53\x7b\x85\x47\x98\x13",
        pk: b"\xe2\x66\xdd\xfd\xc1\x26\x68\xdb\x30\xd4\xca\x3e\x8f\x77\x49\x43\x2c\x41\x60\x44\xf2\xd2\xb8\xc1\x0b\xf3\xd4\x01\x2a\xef\xfa\x8a\xbf\xa8\x64\x04\xa2\xe9\xff\xe6\x7d\x47\xc5\x87\xef\x7a\x97\xa7\xf4\x56\xb8\x63\xb4\xd0\x2c\xfc\x69\x28\x97\x3a\xb5\xb1\xcb\x39",
        msg: b"\xc3\x5e\x2f\x09\x25\x53\xc5\x57\x72\x92\x6b\xdb\xe8\x7c\x97\x96\x82\x7d\x17\x02\x4d\xbb\x92\x33\xa5\x45\x36\x6e\x2e\x59\x87\xdd\x34\x4d\xeb\x72\xdf\x98\x71\x44\xb8\xc6\xc4\x3b\xc4\x1b\x65\x4b\x94\xcc\x85\x6e\x16\xb9\x6d\x7a\x82\x1c\x8e\xc0\x39\xb5\x03\xe3\xd8\x67\x28\xc4\x94\xa9\x67\xd8\x30\x11\xa0\xe0\x90\xb5\xd5\x4c\xd4\x7f\x4e\x36\x6c\x09\x12\xbc\x80\x8f\xbb\x2e\xa9\x6e\xfa\xc8\x8f\xb3\xeb\xec\x93\x42\x73\x8e\x22\x5f\x7c\x7c\x2b\x01\x1c\xe3\x75\xb5\x66\x21\xa2\x06\x42\xb4\xd3\x6e\x06\x0d\xb4\x52\x4a\xf1",
        sig: b"\x97\x6d\x3a\x4e\x9d\x23\x32\x6d\xc0\xba\xa9\xfa\x56\x0b\x7c\x4e\x53\xf4\x28\x64\xf5\x08\x48\x3a\x64\x73\xb6\xa1\x10\x79\xb2\xdb\x1b\x76\x6e\x9c\xeb\x71\xba\x6c\x01\xdc\xd4\x6e\x0a\xf4\x62\xcd\x4c\xfa\x65\x2a\xe5\x01\x7d\x45\x55\xb8\xee\xef\xe3\x6e\x19\x32",
    },
    TestVector {
        sk: b"\xe2\x83\x87\x12\x39\x83\x7e\x13\xb9\x5f\x78\x9e\x6e\x1a\xf6\x3b\xf6\x1c\x91\x8c\x99\x2e\x62\xbc\xa0\x40\xd6\x4c\xad\x1f\xc2\xef",
        pk: b"\x74\xcc\xd8\xa6\x2f\xba\x0e\x66\x7c\x50\x92\x9a\x53\xf7\x8c\x21\xb8\xff\x0c\x3c\x73\x7b\x0b\x40\xb1\x75\x0b\x23\x02\xb0\xbd\xe8\x29\x07\x4e\x21\xf3\xa0\xef\x88\xb9\xef\xdf\x10\xd0\x6a\xa4\xc2\x95\xcc\x16\x71\xf7\x58\xca\x0e\x4c\xd1\x08\x80\x3d\x0f\x26\x14",
        msg: b"\x3c\x05\x4e\x33\x3a\x94\x25\x9c\x36\xaf\x09\xab\x5b\x4f\xf9\xbe\xb3\x49\x2f\x8d\x5b\x42\x82\xd1\x68\x01\xda\xcc\xb2\x9f\x70\xfe\x61\xa0\xb3\x7f\xfe\xf5\xc0\x4c\xd1\xb7\x0e\x85\xb1\xf5\x49\xa1\xc4\xdc\x67\x29\x85\xe5\x0f\x43\xea\x03\x7e\xfa\x99\x64\xf0\x96\xb5\xf6\x2f\x7f\xfd\xf8\xd6\xbf\xb2\xcc\x85\x95\x58\xf5\xa3\x93\xcb\x94\x9d\xbd\x48\xf2\x69\x34\x3b\x52\x63\xdc\xdb\x9c\x55\x6e\xca\x07\x4f\x2e\x98\xe6\xd9\x4c\x2c\x29\xa6\x77\xaf\xaf\x80\x6e\xdf\x79\xb1\x5a\x3f\xcd\x46\xe7\x06\x7b\x76\x69\xf8\x31\x88\xee",
        sig: b"\x35\xfb\x60\xf5\xca\x0f\x3c\xa0\x85\x42\xfb\x3c\xc6\x41\xc8\x26\x3a\x2c\xab\x7a\x90\xee\x6a\x5e\x15\x83\xfa\xc2\xbb\x6f\x6b\xd1\xee\x59\xd8\x1b\xc9\xdb\x10\x55\xcc\x0e\xd9\x7b\x15\x9d\x87\x84\xaf\x04\xe9\x85\x11\xd0\xa9\xa4\x07\xb9\x9b\xb2\x92\x57\x2e\x96",
    },
    TestVector {
        sk: b"\xa3\xd2\xd3\xb7\x59\x6f\x65\x92\xce\x98\xb4\xbf\xe1\x0d\x41\x83\x7f\x10\x02\x7a\x90\xd7\xbb\x75\x34\x94\x90\x01\x8c\xf7\x2d\x07",
        pk: b"\x32\x2f\x80\x37\x1b\xf6\xe0\x44\xbc\x49\x39\x1d\x97\xc1\x71\x4a\xb8\x7f\x99\x0b\x94\x9b\xc1\x78\xcb\x7c\x43\xb7\xc2\x2d\x89\xe1\x3c\x15\xd5\x4a\x5c\xc6\xb9\xf0\x9d\xe8\x45\x7e\x87\x3e\xb3\xde\xb1\xfc\xeb\x54\xb0\xb2\x95\xda\x60\x50\x29\x4f\xae\x7f\xd9\x99",
        msg: b"\x09\x89\x12\x24\x10\xd5\x22\xaf\x64\xce\xb0\x7d\xa2\xc8\x65\x21\x90\x46\xb4\xc3\xd9\xd9\x9b\x01\x27\x8c\x07\xff\x63\xea\xf1\x03\x9c\xb7\x87\xae\x9e\x2d\xd4\x64\x36\xcc\x04\x15\xf2\x80\xc5\x62\xbe\xbb\x83\xa2\x3e\x63\x9e\x47\x6a\x02\xec\x8c\xff\x7e\xa0\x6c\xd1\x2c\x86\xdc\xc3\xad\xef\xbf\x1a\x9e\x9a\x9b\x66\x46\xc7\x59\x9e\xc6\x31\xb0\xda\x9a\x60\xde\xbe\xb9\xb3\xe1\x93\x24\x97\x7f\x3b\x4f\x36\x89\x2c\x8a\x38\x67\x1c\x8e\x1c\xc8\xe5\x0f\xcd\x50\xf9\xe5\x1d\xea\xf9\x82\x72\xf9\x26\x6f\xc7\x02\xe4\xe5\x7c\x30",
        sig: b"\xd7\xc5\x62\x37\x0a\xf6\x17\xb5\x81\xc8\x4a\x24\x68\xcc\x8b\xd5\x0b\xb1\xcb\xf3\x22\xde\x41\xb7\x88\x7c\xe0\x7c\x0e\x58\x84\xca\xb4\x6d\x9f\x2d\x8c\x4b\xf8\x35\x46\xff\x17\x8f\x1d\x78\x93\x7c\x00\x8d\x64\xe8\xec\xc5\xcb\xb8\x25\xcb\x21\xd9\x4d\x67\x0d\x89",
    },
    TestVector {
        sk: b"\x53\xa0\xe8\xa8\xfe\x93\xdb\x01\xe7\xae\x94\xe1\xa9\x88\x2a\x10\x2e\xbd\x07\x9b\x3a\x53\x58\x27\xd5\x83\x62\x6c\x27\x2d\x28\x0d",
        pk: b"\x1b\xce\xc4\x57\x0e\x1e\xc2\x43\x65\x96\xb8\xde\xd5\x8f\x60\xc3\xb1\xeb\xc6\xa4\x03\xbc\x55\x43\x04\x0b\xa8\x29\x63\x05\x72\x44\x8a\xf6\x2a\x4c\x68\x3f\x09\x6b\x28\x55\x83\x20\x73\x7b\xf8\x3b\x99\x59\xa4\x6a\xd2\x52\x10\x04\xef\x74\xcf\x85\xe6\x74\x94\xe1",
        msg: b"\xdc\x66\xe3\x9f\x9b\xbf\xd9\x86\x53\x18\x53\x1f\xfe\x92\x07\xf9\x34\xfa\x61\x5a\x5b\x28\x57\x08\xa5\xe9\xc4\x6b\x77\x75\x15\x0e\x81\x8d\x7f\x24\xd2\xa1\x23\xdf\x36\x72\xff\xf2\x09\x4e\x3f\xd3\xdf\x6f\xbe\x25\x9e\x39\x89\xdd\x5e\xdf\xcc\xcb\xe7\xd4\x5e\x26\xa7\x75\xa5\xc4\x32\x9a\x08\x4f\x05\x7c\x42\xc1\x3f\x32\x48\xe3\xfd\x6f\x0c\x76\x67\x8f\x89\x0f\x51\x3c\x32\x29\x2d\xd3\x06\xea\xa8\x4a\x59\xab\xe3\x4b\x16\xcb\x5e\x38\xd0\xe8\x85\x52\x5d\x10\x33\x6c\xa4\x43\xe1\x68\x2a\xa0\x4a\x7a\xf8\x32\xb0\xee\xe4\xe7",
        sig: b"\x18\xca\xaf\x7b\x66\x35\x07\xa8\xbc\xd9\x92\xb8\x36\xde\xc9\xdc\x57\x03\xc0\x80\xaf\x5e\x51\xdf\xa3\xa9\xa7\xc3\x87\x18\x26\x04\x77\xc6\x89\x28\xac\x3b\x88\xd9\x85\xfb\x43\xfb\x61\x5f\xb7\xff\x45\xc1\x8b\xa5\xc8\x1a\xf7\x96\xc6\x13\xdf\xa9\x83\x52\xd2\x9c",
    },
    TestVector {
        sk: b"\x4a\xf1\x07\xe8\xe2\x19\x4c\x83\x0f\xfb\x71\x2a\x65\x51\x1b\xc9\x18\x6a\x13\x30\x07\x85\x5b\x49\xab\x4b\x38\x33\xae\xfc\x4a\x1d",
        pk: b"\xa3\x2e\x50\xbe\x3d\xae\x2c\x8b\xa3\xf5\xe4\xbd\xae\x14\xcf\x76\x45\x42\x0d\x42\x5e\xad\x94\x03\x6c\x22\xdd\x6c\x4f\xc5\x9e\x00\xd6\x23\xbf\x64\x11\x60\xc2\x89\xd6\x74\x2c\x62\x57\xae\x6b\xa5\x74\x44\x6d\xd1\xd0\xe7\x4d\xb3\xaa\xa8\x09\x00\xb7\x8d\x4a\xe9",
        msg: b"\x60\x09\x74\xe7\xd8\xc5\x50\x8e\x2c\x1a\xab\x07\x83\xad\x0d\x7c\x44\x94\xab\x2b\x4d\xa2\x65\xc2\xfe\x49\x64\x21\xc4\xdf\x23\x8b\x0b\xe2\x5f\x25\x65\x91\x57\xc8\xa2\x25\xfb\x03\x95\x36\x07\xf7\xdf\x99\x6a\xcf\xd4\x02\xf1\x47\xe3\x7a\xee\x2f\x16\x93\xe3\xbf\x1c\x35\xea\xb3\xae\x36\x0a\x2b\xd9\x1d\x04\x62\x2e\xa4\x7f\x83\xd8\x63\xd2\xdf\xec\xb6\x18\xe8\xb8\xbd\xc3\x9e\x17\xd1\x5d\x67\x2e\xee\x03\xbb\x4c\xe2\xcc\x5c\xf6\xb2\x17\xe5\xfa\xf3\xf3\x36\xfd\xd8\x7d\x97\x2d\x3a\x8b\x8a\x59\x3b\xa8\x59\x55\xcc\x9d\x71",
        sig: b"\x85\x24\xc5\x02\x4e\x2d\x9a\x73\xbd\xe8\xc7\x2d\x91\x29\xf5\x78\x73\xbb\xad\x0e\xd0\x52\x15\xa3\x72\xa8\x4f\xdb\xc7\x8f\x2e\x68\xd1\x8c\x2c\xaf\x3b\x10\x72\xf8\x70\x64\xec\x5e\x89\x53\xf5\x13\x01\xca\xda\x03\x46\x9c\x64\x02\x44\x76\x03\x28\xeb\x5a\x05\xcb",
    },
    TestVector {
        sk: b"\x78\xdf\xaa\x09\xf1\x07\x68\x50\xb3\xe2\x06\xe4\x77\x49\x4c\xdd\xcf\xb8\x22\xaa\xa0\x12\x84\x75\x05\x35\x92\xc4\x8e\xba\xf4\xab",
        pk: b"\x8b\xcf\xe2\xa7\x21\xca\x6d\x75\x39\x68\xf5\x64\xec\x43\x15\xbe\x48\x57\xe2\x8b\xef\x19\x08\xf6\x1a\x36\x6b\x1f\x03\xc9\x74\x79\x0f\x67\x57\x6a\x30\xb8\xe2\x0d\x42\x32\xd8\x53\x0b\x52\xfb\x4c\x89\xcb\xc5\x89\xed\xe2\x91\xe4\x99\xdd\xd1\x5f\xe8\x70\xab\x96",
        msg: b"\xdf\xa6\xcb\x9b\x39\xad\xda\x6c\x74\xcc\x8b\x2a\x8b\x53\xa1\x2c\x49\x9a\xb9\xde\xe0\x1b\x41\x23\x64\x2b\x4f\x11\xaf\x33\x6a\x91\xa5\xc9\xce\x05\x20\xeb\x23\x95\xa6\x19\x0e\xcb\xf6\x16\x9c\x4c\xba\x81\x94\x1d\xe8\xe7\x6c\x9c\x90\x8e\xb8\x43\xb9\x8c\xe9\x5e\x0d\xa2\x9c\x5d\x43\x88\x04\x02\x64\xe0\x5e\x07\x03\x0a\x57\x7c\xc5\xd1\x76\x38\x71\x54\xea\xba\xe2\xaf\x52\xa8\x3e\x85\xc6\x1c\x7c\x61\xda\x93\x0c\x9b\x19\xe4\x5d\x7e\x34\xc8\x51\x6d\xc3\xc2\x38\xfd\xdd\x6e\x45\x0a\x77\x45\x5d\x53\x4c\x48\xa1\x52\x01\x0b",
        sig: b"\xc5\xa1\x86\xd7\x2d\xf4\x52\x01\x54\x80\xf7\xf3\x38\x97\x0b\xfe\x82\x50\x87\xf0\x5c\x00\x88\xd9\x53\x05\xf8\x7a\xac\xc9\xb2\x54\x84\xa5\x8f\x9e\x9d\x9e\x73\x53\x44\xb3\x16\xb1\xaa\x1a\xb5\x18\x56\x65\xb8\x51\x47\xdc\x82\xd9\x2e\x96\x9d\x7b\xee\x31\xca\x30",
    },
    TestVector {
        sk: b"\x80\xe6\x92\xe3\xeb\x9f\xcd\x8c\x7d\x44\xe7\xde\x9f\x7a\x59\x52\x68\x64\x07\xf9\x00\x25\xa1\xd8\x7e\x52\xc7\x09\x6a\x62\x61\x8a",
        pk: b"\xa8\x8b\xc8\x43\x02\x79\xc8\xc0\x40\x0a\x77\xd7\x51\xf2\x6c\x0a\xbc\x93\xe5\xde\x4a\xd9\xa4\x16\x63\x57\x95\x2f\xe0\x41\xe7\x67\x2d\x36\x5a\x1e\xef\x25\xea\xd5\x79\xcc\x9a\x06\x9b\x6a\xbc\x1b\x16\xb8\x1c\x35\xf1\x87\x85\xce\x26\xa1\x0b\xa6\xd1\x38\x11\x85",
        msg: b"\x51\xd2\x54\x7c\xbf\xf9\x24\x31\x17\x4a\xa7\xfc\x73\x02\x13\x95\x19\xd9\x80\x71\xc7\x55\xff\x1c\x92\xe4\x69\x4b\x58\x58\x7e\xa5\x60\xf7\x2f\x32\xfc\x6d\xd4\xde\xe7\xd2\x2b\xb7\x38\x73\x81\xd0\x25\x6e\x28\x62\xd0\x64\x4c\xdf\x2c\x27\x7c\x5d\x74\x0f\xa0\x89\x83\x0e\xb5\x2b\xf7\x9d\x1e\x75\xb8\x59\x6e\xcf\x0e\xa5\x8a\x0b\x9d\xf6\x1e\x0c\x97\x54\xbf\xcd\x62\xef\xab\x6e\xa1\xbd\x21\x6b\xf1\x81\xc5\x59\x3d\xa7\x9f\x10\x13\x5a\x9b\xc6\xe1\x64\xf1\x85\x4b\xc8\x85\x97\x34\x34\x1a\xad\x23\x7b\xa2\x9a\x81\xa3\xfc\x8b",
        sig: b"\x9d\x0c\x6a\xfb\x6d\xf3\xbc\xed\x45\x5b\x45\x9c\xc2\x13\x87\xe1\x49\x29\x39\x26\x64\xbb\x87\x41\xa3\x69\x3a\x17\x95\xca\x69\x02\xd7\xf9\xdd\xd1\x91\xf1\xf4\x12\x86\x94\x29\x20\x9e\xe3\x81\x4c\x75\xc7\x2f\xa4\x6a\x9c\xcc\xf8\x04\xa2\xf5\xcc\x0b\x7e\x73\x9f",
    },
    TestVector {
        sk: b"\x5e\x66\x6c\x0d\xb0\x21\x4c\x3b\x62\x7a\x8e\x48\x54\x1c\xc8\x4a\x8b\x6f\xd1\x5f\x30\x0d\xa4\xdf\xf5\xd1\x8a\xec\x6c\x55\xb8\x81",
        pk: b"\x1b\xc4\x87\x57\x0f\x04\x0d\xc9\x41\x96\xc9\xbe\xfe\x8a\xb2\xb6\xde\x77\x20\x8b\x1f\x38\xbd\xaa\xe2\x8f\x96\x45\xc4\xd2\xbc\x3a\xec\x81\x60\x2a\xbd\x83\x45\xe7\x18\x67\xc8\x21\x03\x13\x73\x78\x65\xb8\xaa\x18\x68\x51\xe1\xb4\x8e\xac\xa1\x40\x32\x0f\x5d\x8f",
        msg: b"\x55\x8c\x2a\xc1\x30\x26\x40\x2b\xad\x4a\x0a\x83\xeb\xc9\x46\x8e\x50\xf7\xff\xab\x06\xd6\xf9\x81\xe5\xdb\x1d\x08\x20\x98\x06\x5b\xcf\xf6\xf2\x1a\x7a\x74\x55\x8b\x1e\x86\x12\x91\x4b\x8b\x5a\x0a\xa2\x8e\xd5\xb5\x74\xc3\x6a\xc4\xea\x58\x68\x43\x2a\x62\xbb\x8e\xf0\x69\x5d\x27\xc1\xe3\xce\xaf\x75\xc7\xb2\x51\xc6\x5d\xdb\x26\x86\x96\xf0\x7c\x16\xd2\x76\x79\x73\xd8\x5b\xeb\x44\x3f\x21\x1e\x64\x45\xe7\xfe\x5d\x46\xf0\xdc\xe7\x0d\x58\xa4\xcd\x9f\xe7\x06\x88\xc0\x35\x68\x8e\xa8\xc6\xba\xec\x65\xa5\xfc\x7e\x2c\x93\xe8",
        sig: b"\x2f\x9e\x2b\x4e\x9f\x74\x7c\x65\x7f\x70\x5b\xff\xd1\x24\xee\x17\x8b\xbc\x53\x91\xc8\x6d\x05\x67\x17\xb1\x40\xc1\x53\x57\x0f\xd9\xf5\x41\x3b\xfd\x85\x94\x9d\xa8\xd8\x3d\xe8\x3a\xb0\xd1\x9b\x29\x86\x61\x3e\x22\x4d\x19\x01\xd7\x69\x19\xde\x23\xcc\xd0\x31\x99",
    },
    TestVector {
        sk: b"\xf7\x3f\x45\x52\x71\xc8\x77\xc4\xd5\x33\x46\x27\xe3\x7c\x27\x8f\x68\xd1\x43\x01\x4b\x0a\x05\xaa\x62\xf3\x08\xb2\x10\x1c\x53\x08",
        pk: b"\xb8\x18\x8b\xd6\x87\x01\xfc\x39\x6d\xab\x53\x12\x5d\x4d\x28\xea\x33\xa9\x1d\xaf\x6d\x21\x48\x5f\x47\x70\xf6\xea\x8c\x56\x5d\xde\x42\x3f\x05\x88\x10\xf2\x77\xf8\xfe\x07\x6f\x6d\xb5\x6e\x92\x85\xa1\xbf\x2c\x2a\x1d\xae\x14\x50\x95\xed\xd9\xc0\x49\x70\xbc\x4a",
        msg: b"\x4d\x55\xc9\x9e\xf6\xbd\x54\x62\x16\x62\xc3\xd1\x10\xc3\xcb\x62\x7c\x03\xd6\x31\x13\x93\xb2\x64\xab\x97\xb9\x0a\x4b\x15\x21\x4a\x55\x93\xba\x25\x10\xa5\x3d\x63\xfb\x34\xbe\x25\x1f\xac\xb6\x97\xc9\x73\xe1\x1b\x66\x5c\xb7\x92\x0f\x16\x84\xb0\x03\x1b\x4d\xd3\x70\xcb\x92\x7c\xa7\x16\x8b\x0b\xf8\xad\x28\x5e\x05\xe9\xe3\x1e\x34\xbc\x24\x02\x47\x39\xfd\xc1\x0b\x78\x58\x6f\x29\xef\xf9\x44\x12\x03\x4e\x3b\x60\x6e\xd8\x50\xec\x2c\x19\x00\xe8\xe6\x81\x51\xfc\x4a\xee\x5a\xde\xbb\x06\x6e\xb6\xda\x4e\xaa\x56\x81\x37\x8e",
        sig: b"\x1c\xc6\x28\x53\x3d\x00\x04\xb2\xb2\x0e\x7f\x4b\xaa\xd0\xb8\xbb\x5e\x06\x73\xdb\x15\x9b\xbc\xcf\x92\x49\x1a\xef\x61\xfc\x96\x20\x88\x0e\x0b\xbf\x82\xa8\xcf\x81\x8e\xd4\x6b\xa0\x3c\xf0\xfc\x6c\x89\x8e\x36\xfc\xa3\x6c\xc7\xfd\xb1\xd2\xdb\x75\x03\x63\x44\x30",
    },
    TestVector {
        sk: b"\xb2\x0d\x70\x5d\x9b\xd7\xc2\xb8\xdc\x60\x39\x3a\x53\x57\xf6\x32\x99\x0e\x59\x9a\x09\x75\x57\x3a\xc6\x7f\xd8\x9b\x49\x18\x79\x06",
        pk: b"\x51\xf9\x9d\x2d\x52\xd4\xa6\xe7\x34\x48\x4a\x01\x8b\x7c\xa2\xf8\x95\xc2\x92\x9b\x67\x54\xa3\xa0\x32\x24\xd0\x7a\xe6\x11\x66\xce\x47\x37\xda\x96\x3c\x6e\xf7\x24\x7f\xb8\x8d\x19\xf9\xb0\xc6\x67\xca\xc7\xfe\x12\x83\x7f\xda\xb8\x8c\x66\xf1\x0d\x3c\x14\xca\xd1",
        msg: b"\xf8\x24\x8a\xd4\x7d\x97\xc1\x8c\x98\x4f\x1f\x5c\x10\x95\x0d\xc1\x40\x47\x13\xc5\x6b\x6e\xa3\x97\xe0\x1e\x6d\xd9\x25\xe9\x03\xb4\xfa\xdf\xe2\xc9\xe8\x77\x16\x9e\x71\xce\x3c\x7f\xe5\xce\x70\xee\x42\x55\xd9\xcd\xc2\x6f\x69\x43\xbf\x48\x68\x78\x74\xde\x64\xf6\xcf\x30\xa0\x12\x51\x2e\x78\x7b\x88\x05\x9b\xbf\x56\x11\x62\xbd\xcc\x23\xa3\x74\x2c\x83\x5a\xc1\x44\xcc\x14\x16\x7b\x1b\xd6\x72\x7e\x94\x05\x40\xa9\xc9\x9f\x3c\xbb\x41\xfb\x1d\xcb\x00\xd7\x6d\xda\x04\x99\x58\x47\xc6\x57\xf4\xc1\x9d\x30\x3e\xb0\x9e\xb4\x8a",
        sig: b"\x98\x86\xae\x46\xc1\x41\x5c\x3b\xc9\x59\xe8\x2b\x76\x0a\xd7\x60\xaa\xb6\x68\x85\xa8\x4e\x62\x0a\xa3\x39\xfd\xf1\x02\x46\x5c\x42\x2b\xf3\xa8\x0b\xc0\x4f\xaa\x35\xeb\xec\xc0\xf4\x86\x4a\xc0\x2d\x34\x9f\x6f\x12\x6e\x0f\x98\x85\x01\xb8\xd3\x07\x54\x09\xa2\x6c",
    },
    TestVector {
        sk: b"\xd4\x23\x4b\xeb\xfb\xc8\x21\x05\x03\x41\xa3\x7e\x12\x40\xef\xe5\xe3\x37\x63\xcb\xbb\x2e\xf7\x6a\x1c\x79\xe2\x47\x24\xe5\xa5\xe7",
        pk: b"\x8f\xb2\x87\xf0\x20\x2a\xd5\x7a\xe8\x41\xae\xa3\x5f\x29\xb2\xe1\xd5\x3e\x19\x6d\x0d\xdd\x9a\xec\x24\x81\x3d\x64\xc0\x92\x2f\xb7\x1f\x6d\xaf\xf1\xaa\x2d\xd2\xd6\xd3\x74\x16\x23\xee\xcb\x5e\x7b\x61\x29\x97\xa1\x03\x9a\xab\x2e\x5c\xf2\xde\x96\x9c\xfe\xa5\x73",
        msg: b"\x3b\x6e\xe2\x42\x59\x40\xb3\xd2\x40\xd3\x5b\x97\xb6\xdc\xd6\x1e\xd3\x42\x3d\x8e\x71\xa0\xad\xa3\x5d\x47\xb3\x22\xd1\x7b\x35\xea\x04\x72\xf3\x5e\xdd\x1d\x25\x2f\x87\xb8\xb6\x5e\xf4\xb7\x16\x66\x9f\xc9\xac\x28\xb0\x0d\x34\xa9\xd6\x6a\xd1\x18\xc9\xd9\x4e\x7f\x46\xd0\xb4\xf6\xc2\xb2\xd3\x39\xfd\x6b\xcd\x35\x12\x41\xa3\x87\xcc\x82\x60\x90\x57\x04\x8c\x12\xc4\xec\x3d\x85\xc6\x61\x97\x5c\x45\xb3\x00\xcb\x96\x93\x0d\x89\x37\x0a\x32\x7c\x98\xb6\x7d\xef\xaa\x89\x49\x7a\xa8\xef\x99\x4c\x77\xf1\x13\x0f\x75\x2f\x94\xa4",
        sig: b"\x49\x0e\xfd\x10\x6b\xe1\x1f\xc3\x65\xc7\x46\x7e\xb8\x9b\x8d\x39\xe1\x5d\x65\x17\x53\x56\x77\x5d\xea\xb2\x11\x16\x3c\x25\x04\xcb\x64\x43\x00\xfc\x0d\xa4\xd4\x0f\xb8\xc6\xea\xd5\x10\xd1\x4f\x0b\xd4\xe1\x32\x1a\x46\x9e\x9c\x0a\x58\x14\x64\xc7\x18\x6b\x7a\xa7",
    },
    TestVector {
        sk: b"\xb5\x8f\x52\x11\xdf\xf4\x40\x62\x6b\xb5\x6d\x0a\xd4\x83\x19\x3d\x60\x6c\xf2\x1f\x36\xd9\x83\x05\x43\x32\x72\x92\xf4\xd2\x5d\x8c",
        pk: b"\x68\x22\x9b\x48\xc2\xfe\x19\xd3\xdb\x03\x4e\x4c\x15\x07\x7e\xb7\x47\x1a\x66\x03\x1f\x28\xa9\x80\x82\x18\x73\x91\x52\x98\xba\x76\x30\x3e\x8e\xe3\x74\x2a\x89\x3f\x78\xb8\x10\x99\x1d\xa6\x97\x08\x3d\xd8\xf1\x11\x28\xc4\x76\x51\xc2\x7a\x56\x74\x0a\x80\xc2\x4c",
        msg: b"\xc5\x20\x4b\x81\xec\x0a\x4d\xf5\xb7\xe9\xfd\xa3\xdc\x24\x5f\x98\x08\x2a\xe7\xf4\xef\xe8\x19\x98\xdc\xaa\x28\x6b\xd4\x50\x7c\xa8\x40\xa5\x3d\x21\xb0\x1e\x90\x4f\x55\xe3\x8f\x78\xc3\x75\x7d\x5a\x5a\x4a\x44\xb1\xd5\xd4\xe4\x80\xbe\x3a\xfb\x5b\x39\x4a\x5d\x28\x40\xaf\x42\xb1\xb4\x08\x3d\x40\xaf\xbf\xe2\x2d\x70\x2f\x37\x0d\x32\xdb\xfd\x39\x2e\x12\x8e\xa4\x72\x4d\x66\xa3\x70\x1d\xa4\x1a\xe2\xf0\x3b\xb4\xd9\x1b\xb9\x46\xc7\x96\x94\x04\xcb\x54\x4f\x71\xeb\x7a\x49\xeb\x4c\x4e\xc5\x57\x99\xbd\xa1\xeb\x54\x51\x43\xa7",
        sig: b"\xe6\x7a\x97\x17\xcc\xf9\x68\x41\x48\x9d\x65\x41\xf4\xf6\xad\xb1\x2d\x17\xb5\x9a\x6b\xef\x84\x7b\x61\x83\xb8\xfc\xf1\x6a\x32\xeb\x9a\xe6\xba\x6d\x63\x77\x06\x84\x9a\x6a\x9f\xc3\x88\xcf\x02\x32\xd8\x5c\x26\xea\x0d\x1f\xe7\x43\x7a\xdb\x48\xde\x58\x36\x43\x33",
    },
    TestVector {
        sk: b"\x54\xc0\x66\x71\x1c\xdb\x06\x1e\xda\x07\xe5\x27\x5f\x7e\x95\xa9\x96\x2c\x67\x64\xb8\x4f\x6f\x1f\x3a\xb5\xa5\x88\xe0\xa2\xaf\xb1",
        pk: b"\x0a\x7d\xbb\x8b\xf5\x0c\xb6\x05\xeb\x22\x68\xb0\x81\xf2\x6d\x6b\x08\xe0\x12\xf9\x52\xc4\xb7\x0a\x5a\x1e\x6e\x7d\x46\xaf\x98\xbb\xf2\x6d\xd7\xd7\x99\x93\x00\x62\x48\x08\x49\x96\x2c\xcf\x50\x04\xed\xcf\xd3\x07\xc0\x44\xf4\xe8\xf6\x67\xc9\xba\xa8\x34\xee\xae",
        msg: b"\x72\xe8\x1f\xe2\x21\xfb\x40\x21\x48\xd8\xb7\xab\x03\x54\x9f\x11\x80\xbc\xc0\x3d\x41\xca\x59\xd7\x65\x38\x01\xf0\xba\x85\x3a\xdd\x1f\x6d\x29\xed\xd7\xf9\xab\xc6\x21\xb2\xd5\x48\xf8\xdb\xf8\x97\x9b\xd1\x66\x08\xd2\xd8\xfc\x32\x60\xb4\xeb\xc0\xdd\x42\x48\x24\x81\xd5\x48\xc7\x07\x57\x11\xb5\x75\x96\x49\xc4\x1f\x43\x9f\xad\x69\x95\x49\x56\xc9\x32\x68\x41\xea\x64\x92\x95\x68\x29\xf9\xe0\xdc\x78\x9f\x73\x63\x3b\x40\xf6\xac\x77\xbc\xae\x6d\xfc\x79\x30\xcf\xe8\x9e\x52\x6d\x16\x84\x36\x5c\x5b\x0b\xe2\x43\x7f\xdb\x01",
        sig: b"\xb5\x3c\xe4\xda\x1a\xa7\xc0\xdc\x77\xa1\x89\x6a\xb7\x16\xb9\x21\x49\x9a\xed\x78\xdf\x72\x5b\x15\x04\xab\xa1\x59\x7b\xa0\xc6\x4b\xd7\xc2\x46\xdc\x7a\xd0\xe6\x77\x00\xc3\x73\xed\xcf\xdd\x1c\x0a\x04\x95\xfc\x95\x45\x49\xad\x57\x9d\xf6\xed\x14\x38\x84\x08\x51",
    },
    TestVector {
        sk: b"\x34\xfa\x46\x82\xbf\x6c\xb5\xb1\x67\x83\xad\xcd\x18\xf0\xe6\x87\x9b\x92\x18\x5f\x76\xd7\xc9\x20\x40\x9f\x90\x4f\x52\x2d\xb4\xb1",
        pk: b"\x10\x5d\x22\xd9\xc6\x26\x52\x0f\xac\xa1\x3e\x7c\xed\x38\x2d\xcb\xe9\x34\x98\x31\x5f\x00\xcc\x0a\xc3\x9c\x48\x21\xd0\xd7\x37\x37\x6c\x47\xf3\xcb\xbf\xa9\x7d\xfc\xeb\xe1\x62\x70\xb8\xc7\xd5\xd3\xa5\x90\x0b\x88\x8c\x42\x52\x0d\x75\x1e\x8f\xaf\x3b\x40\x1e\xf4",
        msg: b"\x21\x18\x8c\x3e\xdd\x5d\xe0\x88\xda\xcc\x10\x76\xb9\xe1\xbc\xec\xd7\x9d\xe1\x00\x3c\x24\x14\xc3\x86\x61\x73\x05\x4d\xc8\x2d\xde\x85\x16\x9b\xaa\x77\x99\x3a\xdb\x20\xc2\x69\xf6\x0a\x52\x26\x11\x18\x28\x57\x8b\xcc\x7c\x29\xe6\xe8\xd2\xda\xe8\x18\x06\x15\x2c\x8b\xa0\xc6\xad\xa1\x98\x6a\x19\x83\xeb\xee\xc1\x47\x3a\x73\xa0\x47\x95\xb6\x31\x9d\x48\x66\x2d\x40\x88\x1c\x17\x23\xa7\x06\xf5\x16\xfe\x75\x30\x0f\x92\x40\x8a\xa1\xdc\x6a\xe4\x28\x8d\x20\x46\xf2\x3c\x1a\xa2\xe5\x4b\x7f\xb6\x44\x8a\x0d\xa9\x22\xbd\x7f\x34",
        sig: b"\x54\x2c\x40\xa1\x81\x40\xa6\x26\x6d\x6f\x02\x86\xe2\x4e\x9a\x7b\xad\x76\x50\xe7\x2e\xf0\xe2\x13\x1e\x62\x9c\x07\x6d\x96\x26\x63\x4f\x7f\x65\x30\x5e\x24\xa6\xbb\xb5\xcf\xf7\x14\xba\x8f\x5a\x2c\xee\x5b\xdc\x89\xba\x8d\x75\xdc\xbf\x21\x96\x6c\xe3\x8e\xb6\x6f",
    },
];
