//! Virtual key codes corresponded to physical scan codes.

#![allow(missing_docs)]

use std::ffi::{CStr, CString};

use crate::{bind, EnumInt};

use super::scan_code::ScanCode;

#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum KeyCode {
    Unknown,
    Return,
    Escape,
    Backspace,
    Tab,
    Space,
    Exclaim,
    Quotedbl,
    Hash,
    Percent,
    Dollar,
    Ampersand,
    Quote,
    Leftparen,
    Rightparen,
    Asterisk,
    Plus,
    Comma,
    Minus,
    Period,
    Slash,
    Num0,
    Num1,
    Num2,
    Num3,
    Num4,
    Num5,
    Num6,
    Num7,
    Num8,
    Num9,
    Colon,
    Semicolon,
    Less,
    Equals,
    Greater,
    Question,
    At,
    LeftBracket,
    Backslash,
    RightBracket,
    Caret,
    Underscore,
    Backquote,
    A,
    B,
    C,
    D,
    E,
    F,
    G,
    H,
    I,
    J,
    K,
    L,
    M,
    N,
    O,
    P,
    Q,
    R,
    S,
    T,
    U,
    V,
    W,
    X,
    Y,
    Z,
    CapsLock,
    F1,
    F2,
    F3,
    F4,
    F5,
    F6,
    F7,
    F8,
    F9,
    F10,
    F11,
    F12,
    PrintScreen,
    ScrollLock,
    Pause,
    Insert,
    Home,
    PageUp,
    Delete,
    End,
    PageDown,
    Right,
    Left,
    Down,
    Up,
    NumLockClear,
    KeypadDivide,
    KeypadMultiply,
    KeypadMinus,
    KeypadPlus,
    KeypadEnter,
    Keypad1,
    Keypad2,
    Keypad3,
    Keypad4,
    Keypad5,
    Keypad6,
    Keypad7,
    Keypad8,
    Keypad9,
    Keypad0,
    KeypadPeriod,
    Application,
    Power,
    KeypadEquals,
    F13,
    F14,
    F15,
    F16,
    F17,
    F18,
    F19,
    F20,
    F21,
    F22,
    F23,
    F24,
    Execute,
    Help,
    Menu,
    Select,
    Stop,
    Again,
    Undo,
    Cut,
    Copy,
    Paste,
    Find,
    Mute,
    VolumeUp,
    VolumeDown,
    KeypadComma,
    KeypadEqualsAs400,
    Alterase,
    SysReq,
    Cancel,
    Clear,
    Prior,
    Return2,
    Separator,
    Out,
    Oper,
    ClearAgain,
    CrSel,
    ExSel,
    Keypad00,
    Keypad000,
    ThousandsSeparator,
    DecimalSeparator,
    CurrencyUnit,
    CurrencySubunit,
    KeypadLeftParen,
    KeypadRightParen,
    KeypadLeftBrace,
    KeypadRightBrace,
    KeypadTab,
    KeypadBackspace,
    KeypadA,
    KeypadB,
    KeypadC,
    KeypadD,
    KeypadE,
    KeypadF,
    KeypadXor,
    KeypadPower,
    KeypadPercent,
    KeypadLess,
    KeypadGreater,
    KeypadAmpersand,
    KeypadDblAmpersand,
    KeypadVerticalBar,
    KeypadDblverticalBar,
    KeypadColon,
    KeypadHash,
    KeypadSpace,
    KeypadAt,
    KeypadExclam,
    KeypadMemStore,
    KeypadMemRecall,
    KeypadMemClear,
    KeypadMemAdd,
    KeypadMemSubtract,
    KeypadMemMultiply,
    KeypadMemDivide,
    KeypadPlusMinus,
    KeypadClear,
    KeypadClearEntry,
    KeypadBinary,
    KeypadOctal,
    KeypadDecimal,
    KeypadHexadecimal,
    LCtrl,
    LShift,
    LAlt,
    LGui,
    RCtrl,
    RShift,
    RAlt,
    RGui,
    Mode,
    AudioNext,
    AudioPrev,
    AudioStop,
    AudioPlay,
    AudioMute,
    MediaSelect,
    Www,
    Mail,
    Calculator,
    Computer,
    ApplicationSearch,
    ApplicationHome,
    ApplicationBack,
    ApplicationForward,
    ApplicationStop,
    ApplicationRefresh,
    ApplicationBookmarks,
    BrightnessDown,
    BrightnessUp,
    DisplaySwitch,
    KeyboardIllumToggle,
    KeyboardIllumDown,
    KeyboardIllumUp,
    Eject,
    Sleep,
    App1,
    App2,
    AudioRewind,
    AudioFastForward,
}

impl KeyCode {
    pub fn from_name(name: &str) -> Self {
        let c_str = CString::new(name).expect("name must be a valid string");
        unsafe { bind::SDL_GetKeyFromName(c_str.as_ptr()) }.into()
    }

    pub fn name(&self) -> Option<String> {
        let name: String = unsafe { CStr::from_ptr(bind::SDL_GetKeyName(self.as_raw())) }
            .to_str()
            .unwrap()
            .into();
        (!name.is_empty()).then(|| name)
    }

    pub(crate) fn as_raw(&self) -> bind::SDL_Keycode {
        use KeyCode::*;
        (match self {
            Unknown => bind::SDLK_UNKNOWN,
            Return => bind::SDLK_RETURN,
            Escape => bind::SDLK_ESCAPE,
            Backspace => bind::SDLK_BACKSPACE,
            Tab => bind::SDLK_TAB,
            Space => bind::SDLK_SPACE,
            Exclaim => bind::SDLK_EXCLAIM,
            Quotedbl => bind::SDLK_QUOTEDBL,
            Hash => bind::SDLK_HASH,
            Percent => bind::SDLK_PERCENT,
            Dollar => bind::SDLK_DOLLAR,
            Ampersand => bind::SDLK_AMPERSAND,
            Quote => bind::SDLK_QUOTE,
            Leftparen => bind::SDLK_LEFTPAREN,
            Rightparen => bind::SDLK_RIGHTPAREN,
            Asterisk => bind::SDLK_ASTERISK,
            Plus => bind::SDLK_PLUS,
            Comma => bind::SDLK_COMMA,
            Minus => bind::SDLK_MINUS,
            Period => bind::SDLK_PERIOD,
            Slash => bind::SDLK_SLASH,
            Num0 => bind::SDLK_0,
            Num1 => bind::SDLK_1,
            Num2 => bind::SDLK_2,
            Num3 => bind::SDLK_3,
            Num4 => bind::SDLK_4,
            Num5 => bind::SDLK_5,
            Num6 => bind::SDLK_6,
            Num7 => bind::SDLK_7,
            Num8 => bind::SDLK_8,
            Num9 => bind::SDLK_9,
            Colon => bind::SDLK_COLON,
            Semicolon => bind::SDLK_SEMICOLON,
            Less => bind::SDLK_LESS,
            Equals => bind::SDLK_EQUALS,
            Greater => bind::SDLK_GREATER,
            Question => bind::SDLK_QUESTION,
            At => bind::SDLK_AT,
            LeftBracket => bind::SDLK_LEFTBRACKET,
            Backslash => bind::SDLK_BACKSLASH,
            RightBracket => bind::SDLK_RIGHTBRACKET,
            Caret => bind::SDLK_CARET,
            Underscore => bind::SDLK_UNDERSCORE,
            Backquote => bind::SDLK_BACKQUOTE,
            A => bind::SDLK_a,
            B => bind::SDLK_b,
            C => bind::SDLK_c,
            D => bind::SDLK_d,
            E => bind::SDLK_e,
            F => bind::SDLK_f,
            G => bind::SDLK_g,
            H => bind::SDLK_h,
            I => bind::SDLK_i,
            J => bind::SDLK_j,
            K => bind::SDLK_k,
            L => bind::SDLK_l,
            M => bind::SDLK_m,
            N => bind::SDLK_n,
            O => bind::SDLK_o,
            P => bind::SDLK_p,
            Q => bind::SDLK_q,
            R => bind::SDLK_r,
            S => bind::SDLK_s,
            T => bind::SDLK_t,
            U => bind::SDLK_u,
            V => bind::SDLK_v,
            W => bind::SDLK_w,
            X => bind::SDLK_x,
            Y => bind::SDLK_y,
            Z => bind::SDLK_z,
            CapsLock => bind::SDLK_CAPSLOCK,
            F1 => bind::SDLK_F1,
            F2 => bind::SDLK_F2,
            F3 => bind::SDLK_F3,
            F4 => bind::SDLK_F4,
            F5 => bind::SDLK_F5,
            F6 => bind::SDLK_F6,
            F7 => bind::SDLK_F7,
            F8 => bind::SDLK_F8,
            F9 => bind::SDLK_F9,
            F10 => bind::SDLK_F10,
            F11 => bind::SDLK_F11,
            F12 => bind::SDLK_F12,
            PrintScreen => bind::SDLK_PRINTSCREEN,
            ScrollLock => bind::SDLK_SCROLLLOCK,
            Pause => bind::SDLK_PAUSE,
            Insert => bind::SDLK_INSERT,
            Home => bind::SDLK_HOME,
            PageUp => bind::SDLK_PAGEUP,
            Delete => bind::SDLK_DELETE,
            End => bind::SDLK_END,
            PageDown => bind::SDLK_PAGEDOWN,
            Right => bind::SDLK_RIGHT,
            Left => bind::SDLK_LEFT,
            Down => bind::SDLK_DOWN,
            Up => bind::SDLK_UP,
            NumLockClear => bind::SDLK_NUMLOCKCLEAR,
            KeypadDivide => bind::SDLK_KP_DIVIDE,
            KeypadMultiply => bind::SDLK_KP_MULTIPLY,
            KeypadMinus => bind::SDLK_KP_MINUS,
            KeypadPlus => bind::SDLK_KP_PLUS,
            KeypadEnter => bind::SDLK_KP_ENTER,
            Keypad1 => bind::SDLK_KP_1,
            Keypad2 => bind::SDLK_KP_2,
            Keypad3 => bind::SDLK_KP_3,
            Keypad4 => bind::SDLK_KP_4,
            Keypad5 => bind::SDLK_KP_5,
            Keypad6 => bind::SDLK_KP_6,
            Keypad7 => bind::SDLK_KP_7,
            Keypad8 => bind::SDLK_KP_8,
            Keypad9 => bind::SDLK_KP_9,
            Keypad0 => bind::SDLK_KP_0,
            KeypadPeriod => bind::SDLK_KP_PERIOD,
            Application => bind::SDLK_APPLICATION,
            Power => bind::SDLK_POWER,
            KeypadEquals => bind::SDLK_KP_EQUALS,
            F13 => bind::SDLK_F13,
            F14 => bind::SDLK_F14,
            F15 => bind::SDLK_F15,
            F16 => bind::SDLK_F16,
            F17 => bind::SDLK_F17,
            F18 => bind::SDLK_F18,
            F19 => bind::SDLK_F19,
            F20 => bind::SDLK_F20,
            F21 => bind::SDLK_F21,
            F22 => bind::SDLK_F22,
            F23 => bind::SDLK_F23,
            F24 => bind::SDLK_F24,
            Execute => bind::SDLK_EXECUTE,
            Help => bind::SDLK_HELP,
            Menu => bind::SDLK_MENU,
            Select => bind::SDLK_SELECT,
            Stop => bind::SDLK_STOP,
            Again => bind::SDLK_AGAIN,
            Undo => bind::SDLK_UNDO,
            Cut => bind::SDLK_CUT,
            Copy => bind::SDLK_COPY,
            Paste => bind::SDLK_PASTE,
            Find => bind::SDLK_FIND,
            Mute => bind::SDLK_MUTE,
            VolumeUp => bind::SDLK_VOLUMEUP,
            VolumeDown => bind::SDLK_VOLUMEDOWN,
            KeypadComma => bind::SDLK_KP_COMMA,
            KeypadEqualsAs400 => bind::SDLK_KP_EQUALSAS400,
            Alterase => bind::SDLK_ALTERASE,
            SysReq => bind::SDLK_SYSREQ,
            Cancel => bind::SDLK_CANCEL,
            Clear => bind::SDLK_CLEAR,
            Prior => bind::SDLK_PRIOR,
            Return2 => bind::SDLK_RETURN2,
            Separator => bind::SDLK_SEPARATOR,
            Out => bind::SDLK_OUT,
            Oper => bind::SDLK_OPER,
            ClearAgain => bind::SDLK_CLEARAGAIN,
            CrSel => bind::SDLK_CRSEL,
            ExSel => bind::SDLK_EXSEL,
            Keypad00 => bind::SDLK_KP_00,
            Keypad000 => bind::SDLK_KP_000,
            ThousandsSeparator => bind::SDLK_THOUSANDSSEPARATOR,
            DecimalSeparator => bind::SDLK_DECIMALSEPARATOR,
            CurrencyUnit => bind::SDLK_CURRENCYUNIT,
            CurrencySubunit => bind::SDLK_CURRENCYSUBUNIT,
            KeypadLeftParen => bind::SDLK_KP_LEFTPAREN,
            KeypadRightParen => bind::SDLK_KP_RIGHTPAREN,
            KeypadLeftBrace => bind::SDLK_KP_LEFTBRACE,
            KeypadRightBrace => bind::SDLK_KP_RIGHTBRACE,
            KeypadTab => bind::SDLK_KP_TAB,
            KeypadBackspace => bind::SDLK_KP_BACKSPACE,
            KeypadA => bind::SDLK_KP_A,
            KeypadB => bind::SDLK_KP_B,
            KeypadC => bind::SDLK_KP_C,
            KeypadD => bind::SDLK_KP_D,
            KeypadE => bind::SDLK_KP_E,
            KeypadF => bind::SDLK_KP_F,
            KeypadXor => bind::SDLK_KP_XOR,
            KeypadPower => bind::SDLK_KP_POWER,
            KeypadPercent => bind::SDLK_KP_PERCENT,
            KeypadLess => bind::SDLK_KP_LESS,
            KeypadGreater => bind::SDLK_KP_GREATER,
            KeypadAmpersand => bind::SDLK_KP_AMPERSAND,
            KeypadDblAmpersand => bind::SDLK_KP_DBLAMPERSAND,
            KeypadVerticalBar => bind::SDLK_KP_VERTICALBAR,
            KeypadDblverticalBar => bind::SDLK_KP_DBLVERTICALBAR,
            KeypadColon => bind::SDLK_KP_COLON,
            KeypadHash => bind::SDLK_KP_HASH,
            KeypadSpace => bind::SDLK_KP_SPACE,
            KeypadAt => bind::SDLK_KP_AT,
            KeypadExclam => bind::SDLK_KP_EXCLAM,
            KeypadMemStore => bind::SDLK_KP_MEMSTORE,
            KeypadMemRecall => bind::SDLK_KP_MEMRECALL,
            KeypadMemClear => bind::SDLK_KP_MEMCLEAR,
            KeypadMemAdd => bind::SDLK_KP_MEMADD,
            KeypadMemSubtract => bind::SDLK_KP_MEMSUBTRACT,
            KeypadMemMultiply => bind::SDLK_KP_MEMMULTIPLY,
            KeypadMemDivide => bind::SDLK_KP_MEMDIVIDE,
            KeypadPlusMinus => bind::SDLK_KP_PLUSMINUS,
            KeypadClear => bind::SDLK_KP_CLEAR,
            KeypadClearEntry => bind::SDLK_KP_CLEARENTRY,
            KeypadBinary => bind::SDLK_KP_BINARY,
            KeypadOctal => bind::SDLK_KP_OCTAL,
            KeypadDecimal => bind::SDLK_KP_DECIMAL,
            KeypadHexadecimal => bind::SDLK_KP_HEXADECIMAL,
            LCtrl => bind::SDLK_LCTRL,
            LShift => bind::SDLK_LSHIFT,
            LAlt => bind::SDLK_LALT,
            LGui => bind::SDLK_LGUI,
            RCtrl => bind::SDLK_RCTRL,
            RShift => bind::SDLK_RSHIFT,
            RAlt => bind::SDLK_RALT,
            RGui => bind::SDLK_RGUI,
            Mode => bind::SDLK_MODE,
            AudioNext => bind::SDLK_AUDIONEXT,
            AudioPrev => bind::SDLK_AUDIOPREV,
            AudioStop => bind::SDLK_AUDIOSTOP,
            AudioPlay => bind::SDLK_AUDIOPLAY,
            AudioMute => bind::SDLK_AUDIOMUTE,
            MediaSelect => bind::SDLK_MEDIASELECT,
            Www => bind::SDLK_WWW,
            Mail => bind::SDLK_MAIL,
            Calculator => bind::SDLK_CALCULATOR,
            Computer => bind::SDLK_COMPUTER,
            ApplicationSearch => bind::SDLK_AC_SEARCH,
            ApplicationHome => bind::SDLK_AC_HOME,
            ApplicationBack => bind::SDLK_AC_BACK,
            ApplicationForward => bind::SDLK_AC_FORWARD,
            ApplicationStop => bind::SDLK_AC_STOP,
            ApplicationRefresh => bind::SDLK_AC_REFRESH,
            ApplicationBookmarks => bind::SDLK_AC_BOOKMARKS,
            BrightnessDown => bind::SDLK_BRIGHTNESSDOWN,
            BrightnessUp => bind::SDLK_BRIGHTNESSUP,
            DisplaySwitch => bind::SDLK_DISPLAYSWITCH,
            KeyboardIllumToggle => bind::SDLK_KBDILLUMTOGGLE,
            KeyboardIllumDown => bind::SDLK_KBDILLUMDOWN,
            KeyboardIllumUp => bind::SDLK_KBDILLUMUP,
            Eject => bind::SDLK_EJECT,
            Sleep => bind::SDLK_SLEEP,
            App1 => bind::SDLK_APP1,
            App2 => bind::SDLK_APP2,
            AudioRewind => bind::SDLK_AUDIOREWIND,
            AudioFastForward => bind::SDLK_AUDIOFASTFORWARD,
        }) as bind::SDL_Keycode
    }
}

impl From<ScanCode> for KeyCode {
    fn from(scan_code: ScanCode) -> Self {
        let raw = scan_code.as_raw();
        unsafe { bind::SDL_GetKeyFromScancode(raw) }.into()
    }
}

impl From<bind::SDL_Keycode> for KeyCode {
    fn from(code: bind::SDL_Keycode) -> Self {
        use KeyCode::*;
        match code as EnumInt {
            bind::SDLK_RETURN => Return,
            bind::SDLK_ESCAPE => Escape,
            bind::SDLK_BACKSPACE => Backspace,
            bind::SDLK_TAB => Tab,
            bind::SDLK_SPACE => Space,
            bind::SDLK_EXCLAIM => Exclaim,
            bind::SDLK_QUOTEDBL => Quotedbl,
            bind::SDLK_HASH => Hash,
            bind::SDLK_PERCENT => Percent,
            bind::SDLK_DOLLAR => Dollar,
            bind::SDLK_AMPERSAND => Ampersand,
            bind::SDLK_QUOTE => Quote,
            bind::SDLK_LEFTPAREN => Leftparen,
            bind::SDLK_RIGHTPAREN => Rightparen,
            bind::SDLK_ASTERISK => Asterisk,
            bind::SDLK_PLUS => Plus,
            bind::SDLK_COMMA => Comma,
            bind::SDLK_MINUS => Minus,
            bind::SDLK_PERIOD => Period,
            bind::SDLK_SLASH => Slash,
            bind::SDLK_0 => Num0,
            bind::SDLK_1 => Num1,
            bind::SDLK_2 => Num2,
            bind::SDLK_3 => Num3,
            bind::SDLK_4 => Num4,
            bind::SDLK_5 => Num5,
            bind::SDLK_6 => Num6,
            bind::SDLK_7 => Num7,
            bind::SDLK_8 => Num8,
            bind::SDLK_9 => Num9,
            bind::SDLK_COLON => Colon,
            bind::SDLK_SEMICOLON => Semicolon,
            bind::SDLK_LESS => Less,
            bind::SDLK_EQUALS => Equals,
            bind::SDLK_GREATER => Greater,
            bind::SDLK_QUESTION => Question,
            bind::SDLK_AT => At,
            bind::SDLK_LEFTBRACKET => LeftBracket,
            bind::SDLK_BACKSLASH => Backslash,
            bind::SDLK_RIGHTBRACKET => RightBracket,
            bind::SDLK_CARET => Caret,
            bind::SDLK_UNDERSCORE => Underscore,
            bind::SDLK_BACKQUOTE => Backquote,
            bind::SDLK_a => A,
            bind::SDLK_b => B,
            bind::SDLK_c => C,
            bind::SDLK_d => D,
            bind::SDLK_e => E,
            bind::SDLK_f => F,
            bind::SDLK_g => G,
            bind::SDLK_h => H,
            bind::SDLK_i => I,
            bind::SDLK_j => J,
            bind::SDLK_k => K,
            bind::SDLK_l => L,
            bind::SDLK_m => M,
            bind::SDLK_n => N,
            bind::SDLK_o => O,
            bind::SDLK_p => P,
            bind::SDLK_q => Q,
            bind::SDLK_r => R,
            bind::SDLK_s => S,
            bind::SDLK_t => T,
            bind::SDLK_u => U,
            bind::SDLK_v => V,
            bind::SDLK_w => W,
            bind::SDLK_x => X,
            bind::SDLK_y => Y,
            bind::SDLK_z => Z,
            bind::SDLK_CAPSLOCK => CapsLock,
            bind::SDLK_F1 => F1,
            bind::SDLK_F2 => F2,
            bind::SDLK_F3 => F3,
            bind::SDLK_F4 => F4,
            bind::SDLK_F5 => F5,
            bind::SDLK_F6 => F6,
            bind::SDLK_F7 => F7,
            bind::SDLK_F8 => F8,
            bind::SDLK_F9 => F9,
            bind::SDLK_F10 => F10,
            bind::SDLK_F11 => F11,
            bind::SDLK_F12 => F12,
            bind::SDLK_PRINTSCREEN => PrintScreen,
            bind::SDLK_SCROLLLOCK => ScrollLock,
            bind::SDLK_PAUSE => Pause,
            bind::SDLK_INSERT => Insert,
            bind::SDLK_HOME => Home,
            bind::SDLK_PAGEUP => PageUp,
            bind::SDLK_DELETE => Delete,
            bind::SDLK_END => End,
            bind::SDLK_PAGEDOWN => PageDown,
            bind::SDLK_RIGHT => Right,
            bind::SDLK_LEFT => Left,
            bind::SDLK_DOWN => Down,
            bind::SDLK_UP => Up,
            bind::SDLK_NUMLOCKCLEAR => NumLockClear,
            bind::SDLK_KP_DIVIDE => KeypadDivide,
            bind::SDLK_KP_MULTIPLY => KeypadMultiply,
            bind::SDLK_KP_MINUS => KeypadMinus,
            bind::SDLK_KP_PLUS => KeypadPlus,
            bind::SDLK_KP_ENTER => KeypadEnter,
            bind::SDLK_KP_1 => Keypad1,
            bind::SDLK_KP_2 => Keypad2,
            bind::SDLK_KP_3 => Keypad3,
            bind::SDLK_KP_4 => Keypad4,
            bind::SDLK_KP_5 => Keypad5,
            bind::SDLK_KP_6 => Keypad6,
            bind::SDLK_KP_7 => Keypad7,
            bind::SDLK_KP_8 => Keypad8,
            bind::SDLK_KP_9 => Keypad9,
            bind::SDLK_KP_0 => Keypad0,
            bind::SDLK_KP_PERIOD => KeypadPeriod,
            bind::SDLK_APPLICATION => Application,
            bind::SDLK_POWER => Power,
            bind::SDLK_KP_EQUALS => KeypadEquals,
            bind::SDLK_F13 => F13,
            bind::SDLK_F14 => F14,
            bind::SDLK_F15 => F15,
            bind::SDLK_F16 => F16,
            bind::SDLK_F17 => F17,
            bind::SDLK_F18 => F18,
            bind::SDLK_F19 => F19,
            bind::SDLK_F20 => F20,
            bind::SDLK_F21 => F21,
            bind::SDLK_F22 => F22,
            bind::SDLK_F23 => F23,
            bind::SDLK_F24 => F24,
            bind::SDLK_EXECUTE => Execute,
            bind::SDLK_HELP => Help,
            bind::SDLK_MENU => Menu,
            bind::SDLK_SELECT => Select,
            bind::SDLK_STOP => Stop,
            bind::SDLK_AGAIN => Again,
            bind::SDLK_UNDO => Undo,
            bind::SDLK_CUT => Cut,
            bind::SDLK_COPY => Copy,
            bind::SDLK_PASTE => Paste,
            bind::SDLK_FIND => Find,
            bind::SDLK_MUTE => Mute,
            bind::SDLK_VOLUMEUP => VolumeUp,
            bind::SDLK_VOLUMEDOWN => VolumeDown,
            bind::SDLK_KP_COMMA => KeypadComma,
            bind::SDLK_KP_EQUALSAS400 => KeypadEqualsAs400,
            bind::SDLK_ALTERASE => Alterase,
            bind::SDLK_SYSREQ => SysReq,
            bind::SDLK_CANCEL => Cancel,
            bind::SDLK_CLEAR => Clear,
            bind::SDLK_PRIOR => Prior,
            bind::SDLK_RETURN2 => Return2,
            bind::SDLK_SEPARATOR => Separator,
            bind::SDLK_OUT => Out,
            bind::SDLK_OPER => Oper,
            bind::SDLK_CLEARAGAIN => ClearAgain,
            bind::SDLK_CRSEL => CrSel,
            bind::SDLK_EXSEL => ExSel,
            bind::SDLK_KP_00 => Keypad00,
            bind::SDLK_KP_000 => Keypad000,
            bind::SDLK_THOUSANDSSEPARATOR => ThousandsSeparator,
            bind::SDLK_DECIMALSEPARATOR => DecimalSeparator,
            bind::SDLK_CURRENCYUNIT => CurrencyUnit,
            bind::SDLK_CURRENCYSUBUNIT => CurrencySubunit,
            bind::SDLK_KP_LEFTPAREN => KeypadLeftParen,
            bind::SDLK_KP_RIGHTPAREN => KeypadRightParen,
            bind::SDLK_KP_LEFTBRACE => KeypadLeftBrace,
            bind::SDLK_KP_RIGHTBRACE => KeypadRightBrace,
            bind::SDLK_KP_TAB => KeypadTab,
            bind::SDLK_KP_BACKSPACE => KeypadBackspace,
            bind::SDLK_KP_A => KeypadA,
            bind::SDLK_KP_B => KeypadB,
            bind::SDLK_KP_C => KeypadC,
            bind::SDLK_KP_D => KeypadD,
            bind::SDLK_KP_E => KeypadE,
            bind::SDLK_KP_F => KeypadF,
            bind::SDLK_KP_XOR => KeypadXor,
            bind::SDLK_KP_POWER => KeypadPower,
            bind::SDLK_KP_PERCENT => KeypadPercent,
            bind::SDLK_KP_LESS => KeypadLess,
            bind::SDLK_KP_GREATER => KeypadGreater,
            bind::SDLK_KP_AMPERSAND => KeypadAmpersand,
            bind::SDLK_KP_DBLAMPERSAND => KeypadDblAmpersand,
            bind::SDLK_KP_VERTICALBAR => KeypadVerticalBar,
            bind::SDLK_KP_DBLVERTICALBAR => KeypadDblverticalBar,
            bind::SDLK_KP_COLON => KeypadColon,
            bind::SDLK_KP_HASH => KeypadHash,
            bind::SDLK_KP_SPACE => KeypadSpace,
            bind::SDLK_KP_AT => KeypadAt,
            bind::SDLK_KP_EXCLAM => KeypadExclam,
            bind::SDLK_KP_MEMSTORE => KeypadMemStore,
            bind::SDLK_KP_MEMRECALL => KeypadMemRecall,
            bind::SDLK_KP_MEMCLEAR => KeypadMemClear,
            bind::SDLK_KP_MEMADD => KeypadMemAdd,
            bind::SDLK_KP_MEMSUBTRACT => KeypadMemSubtract,
            bind::SDLK_KP_MEMMULTIPLY => KeypadMemMultiply,
            bind::SDLK_KP_MEMDIVIDE => KeypadMemDivide,
            bind::SDLK_KP_PLUSMINUS => KeypadPlusMinus,
            bind::SDLK_KP_CLEAR => KeypadClear,
            bind::SDLK_KP_CLEARENTRY => KeypadClearEntry,
            bind::SDLK_KP_BINARY => KeypadBinary,
            bind::SDLK_KP_OCTAL => KeypadOctal,
            bind::SDLK_KP_DECIMAL => KeypadDecimal,
            bind::SDLK_KP_HEXADECIMAL => KeypadHexadecimal,
            bind::SDLK_LCTRL => LCtrl,
            bind::SDLK_LSHIFT => LShift,
            bind::SDLK_LALT => LAlt,
            bind::SDLK_LGUI => LGui,
            bind::SDLK_RCTRL => RCtrl,
            bind::SDLK_RSHIFT => RShift,
            bind::SDLK_RALT => RAlt,
            bind::SDLK_RGUI => RGui,
            bind::SDLK_MODE => Mode,
            bind::SDLK_AUDIONEXT => AudioNext,
            bind::SDLK_AUDIOPREV => AudioPrev,
            bind::SDLK_AUDIOSTOP => AudioStop,
            bind::SDLK_AUDIOPLAY => AudioPlay,
            bind::SDLK_AUDIOMUTE => AudioMute,
            bind::SDLK_MEDIASELECT => MediaSelect,
            bind::SDLK_WWW => Www,
            bind::SDLK_MAIL => Mail,
            bind::SDLK_CALCULATOR => Calculator,
            bind::SDLK_COMPUTER => Computer,
            bind::SDLK_AC_SEARCH => ApplicationSearch,
            bind::SDLK_AC_HOME => ApplicationHome,
            bind::SDLK_AC_BACK => ApplicationBack,
            bind::SDLK_AC_FORWARD => ApplicationForward,
            bind::SDLK_AC_STOP => ApplicationStop,
            bind::SDLK_AC_REFRESH => ApplicationRefresh,
            bind::SDLK_AC_BOOKMARKS => ApplicationBookmarks,
            bind::SDLK_BRIGHTNESSDOWN => BrightnessDown,
            bind::SDLK_BRIGHTNESSUP => BrightnessUp,
            bind::SDLK_DISPLAYSWITCH => DisplaySwitch,
            bind::SDLK_KBDILLUMTOGGLE => KeyboardIllumToggle,
            bind::SDLK_KBDILLUMDOWN => KeyboardIllumDown,
            bind::SDLK_KBDILLUMUP => KeyboardIllumUp,
            bind::SDLK_EJECT => Eject,
            bind::SDLK_SLEEP => Sleep,
            bind::SDLK_APP1 => App1,
            bind::SDLK_APP2 => App2,
            bind::SDLK_AUDIOREWIND => AudioRewind,
            bind::SDLK_AUDIOFASTFORWARD => AudioFastForward,
            _ => Unknown,
        }
    }
}
