use crate::data::Point;
use crate::data::Polygon;

use std::iter::DoubleEndedIterator;
use std::iter::FromFn;
use std::iter::FromIterator;

type Index = usize; // XXX: Use u32?

// Features:
//   Stable indices. Adding/Removing points do not invalidate indices.
//   Binary search after sorting.
//
// remove vertex. O(1)
// insert vertex. O(1)
// set_next O(n)
//
// Convert polygon with holes to simple polygon + mapping of new points to old points.
// &'a Polygon -> PolygonIndexed<'a>
// PolygonIndexed with holes -> PolygonIndex without holes + [(Index,Index)] corridors.
// PolygonIndexed without holes -> [(Index,Index,Index)]

// Index -> Point<T,2>
// Index -> [Neighbours]

#[derive(Clone, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Node<T> {
  value: T,
  prev: Index,
  next: Index,
}

#[derive(Clone, Debug, Eq, PartialEq, Ord, PartialOrd)]
pub struct Neighbours {
  prev: Index,
  next: Index,
}

#[derive(Clone, Debug)]
pub struct CircularList<T> {
  nodes: Vec<Node<T>>,
}

impl<T> CircularList<T> {
  pub fn new() -> CircularList<T> {
    CircularList { nodes: Vec::new() }
  }

  pub fn is_empty(&self) -> bool {
    self.nodes.is_empty()
  }

  pub fn len(&self) -> usize {
    self.nodes.len()
  }

  /// Note: Invalidates all indices.
  pub fn yoink(&mut self, idx: Index) -> (T, Option<Neighbours>) {
    assert!(!self.is_empty());

    // Remove node and remove all pointers to it.
    let mut node = self.nodes.swap_remove(idx);
    let neighbours;
    if self.nodes.len() == 0 {
      neighbours = None;
    } else {
      if node.next >= self.nodes.len() - 1 {
        node.next = idx;
      }
      if node.prev >= self.nodes.len() - 1 {
        node.prev = idx;
      }
      self.nodes[node.prev].next = node.next;
      self.nodes[node.next].prev = node.prev;
      neighbours = Some(Neighbours {
        next: node.next,
        prev: node.prev,
      });

      // The last node in the vec has moved and we need to update the pointers to it.
      let Node {
        value: _,
        next,
        prev,
      } = self.nodes[idx];
      self.nodes[prev].next = idx;
      self.nodes[next].prev = idx;
    }

    (node.value, neighbours)
  }
}

impl<T> FromIterator<T> for CircularList<T> {
  fn from_iter<I>(iter: I) -> CircularList<T>
  where
    I: IntoIterator<Item = T>,
  {
    let mut nodes: Vec<Node<T>> = iter
      .into_iter()
      .enumerate()
      .map(|(nth, v)| Node {
        value: v,
        next: nth + 1,
        prev: nth.saturating_sub(1),
      })
      .collect();
    let last = nodes.len() - 1;
    nodes[0].prev = last as Index;
    nodes[last].next = 0;
    CircularList { nodes }
  }
}

impl<T> IntoIterator for CircularList<T> {
  type Item = T;
  type IntoIter = IntoIter<T>;
  fn into_iter(self) -> Self::IntoIter {
    let head = if self.is_empty() { None } else { Some(0) };
    IntoIter {
      list: self,
      at: head,
    }
  }
}

pub struct IntoIter<T> {
  list: CircularList<T>,
  at: Option<Index>,
}

impl<T> Iterator for IntoIter<T> {
  type Item = T;

  fn next(&mut self) -> Option<T> {
    let (val, neighbours) = self.list.yoink(self.at?);
    self.at = neighbours.map(|n| n.next);
    Some(val)
  }

  fn size_hint(&self) -> (usize, Option<usize>) {
    (self.list.len(), Some(self.list.len()))
  }

  fn count(self) -> usize {
    self.list.len()
  }

  // XXX: 'last' could be efficiently implemented.
}

impl<T> DoubleEndedIterator for IntoIter<T> {
  fn next_back(&mut self) -> Option<T> {
    let prev = self.list.nodes[self.at?].prev;
    let (val, neighbours) = self.list.yoink(prev);
    self.at = neighbours.map(|n| n.next);
    Some(val)
  }
}

#[cfg(test)]
mod tests {
  use super::*;

  #[test]
  fn basic_1() {
    let mut lst = CircularList::from_iter(0..10).into_iter();

    assert_eq!(Some(0), lst.next());
    assert_eq!(Some(9), lst.next_back());
    assert_eq!(Some(8), lst.next_back());
    assert_eq!(Some(1), lst.next());
    assert_eq!(Some(2), lst.next());
  }

  #[test]
  fn yoink_empty() {
    let mut lst = CircularList::from_iter(0..1);
    assert_eq!((0, None), lst.yoink(0));
  }

  #[test]
  fn yoink_full() {
    let mut lst = CircularList::from_iter(0..10);
    assert_eq!((0, Some(Neighbours { prev: 0, next: 1 })), lst.yoink(0));
  }
}
