//! A JSON symbol table format (.json).
//!
//! The JSON file contains a single array. Each element is a symbol, and contains the properties
//! "type" (which can be "function" or "data"), "name", "address" (integer),
//! optional "length" (integer), and optional "description".
//!
//! # Example
//! Note: The following example shows pretty-printed JSON for readability. However, the actual
//! output generated by this module will be minified JSON.
//! ```json
//! [
//!     {
//!         "type": "function",
//!         "name": "main",
//!         "address": 33554432,
//!         "description": "the main function"
//!     },
//!     {
//!         "type": "function",
//!         "name": "function1",
//!         "address": 37748736
//!     },
//!     {
//!         "type": "data",
//!         "name": "SOME_DATA",
//!         "address": 50331647,
//!         "length": 4
//!     }
//! ]
//! ```

use std::error::Error;
use std::io::Write;

use serde::Serialize;

use super::symgen_yml::{Generate, SymGen, Uint};

/// Generator for the .json format.
pub struct JsonFormatter {}

#[derive(Debug, Serialize)]
#[serde(rename_all = "lowercase")]
enum SymbolType {
    Function,
    Data,
}

#[derive(Debug, Serialize)]
struct Entry<'a> {
    #[serde(rename(serialize = "type"))]
    stype: SymbolType,
    name: &'a str,
    address: Uint,
    #[serde(skip_serializing_if = "Option::is_none")]
    length: Option<Uint>,
    #[serde(skip_serializing_if = "Option::is_none")]
    description: Option<&'a str>,
}

impl Generate for JsonFormatter {
    fn generate<W: Write>(
        &self,
        mut writer: W,
        symgen: &SymGen,
        version: &str,
    ) -> Result<(), Box<dyn Error>> {
        let mut needs_comma = false;
        writer.write_all(b"[")?;
        for f in symgen.functions_realized(version) {
            if needs_comma {
                writer.write_all(b",")?;
            }
            serde_json::to_writer(
                &mut writer,
                &Entry {
                    stype: SymbolType::Function,
                    name: f.name,
                    address: f.address,
                    length: f.length,
                    description: f.description,
                },
            )?;
            needs_comma = true;
        }
        for d in symgen.data_realized(version) {
            if needs_comma {
                writer.write_all(b",")?;
            }
            serde_json::to_writer(
                &mut writer,
                &Entry {
                    stype: SymbolType::Data,
                    name: d.name,
                    address: d.address,
                    length: d.length,
                    description: d.description,
                },
            )?;
            needs_comma = true;
        }
        writer.write_all(b"]")?;
        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    fn get_test_symgen() -> SymGen {
        SymGen::read(
            r"
            main:
              versions:
                - v1
                - v2
              address:
                v1: 0x2000000
                v2: 0x2000000
              length:
                v1: 0x100000
                v2: 0x100000
              description: foo
              functions:
                - name: fn1
                  address:
                    v1: 0x2000000
                    v2: 0x2002000
                  length:
                    v1: 0x1000
                    v2: 0x1000
                  description: bar
                - name: fn2
                  address:
                    v1:
                      - 0x2001000
                      - 0x2002000
                    v2: 0x2003000
              data:
                - name: SOME_DATA
                  address:
                    v1: 0x2003000
                    v2: 0x2004000
                  length:
                    v1: 0x1000
                    v2: 0x2000
                  description: baz
        "
            .as_bytes(),
        )
        .expect("Read failed")
    }

    #[test]
    fn test_generate() {
        let symgen = get_test_symgen();
        let f = JsonFormatter {};
        assert_eq!(
            f.generate_str(&symgen, "v1").expect("generate failed"),
            r#"[
                {
                    "type": "function",
                    "name": "fn1",
                    "address": 33554432,
                    "length": 4096,
                    "description": "bar"
                },
                {
                    "type": "function",
                    "name": "fn2",
                    "address": 33558528
                },
                {
                    "type": "function",
                    "name": "fn2",
                    "address": 33562624
                },
                {
                    "type": "data",
                    "name": "SOME_DATA",
                    "address": 33566720,
                    "length": 4096,
                    "description": "baz"
                }
            ]"#
            .replace(" ", "")
            .replace("\n", "")
        );
        assert_eq!(
            f.generate_str(&symgen, "v2").expect("generate failed"),
            r#"[
                {
                    "type": "function",
                    "name": "fn1",
                    "address": 33562624,
                    "length": 4096,
                    "description": "bar"
                },
                {
                    "type": "function",
                    "name": "fn2",
                    "address": 33566720
                },
                {
                    "type": "data",
                    "name": "SOME_DATA",
                    "address": 33570816,
                    "length": 8192,
                    "description": "baz"
                }
            ]"#
            .replace(" ", "")
            .replace("\n", "")
        );
    }
}
