/// ## Rust Rest API Stack with User Management
///
/// A secure-by-default rest api stack implemented with hyper, tokio and postgres.
///
/// This project is focused on providing end-to-end encryption by default for 12-factor applications looking to customize functionality using environment variables as needed.
///
/// Comes with a working user management and authentication backend written in postgresql. 
///
/// For ease of use, you can browse the database using pg4admin for database management (deployed with docker compose).
///
/// ### Overview
///
/// - User authentication enabled by default and implemented with custom tls assets to encrypt all JWT tokens with storage in postgres.
/// - Users can upload and manage files stored on AWS S3 (assuming valid credentials are loaded outside this rust project).
/// - User password reset and user email change support using one-time-use tokens that are stored in postgres.
/// - The hyper server hosts tls assets that can be re-generated with the tools in this repository.
/// - The postgres database requires clients present the included tls Certificate Authority file for encrypting data in-transit.
/// - The rest api server accesses postgres with a bb8 client threadpool.
/// - encrypted JWT keys included and documentation to build new ones as you need.
///
/// ### TLS Encryption Status
///
/// Compoment        | Status
/// ---------------- | ------
/// Rest API Server  | Listening for encrypted client connections on tcp port **3000**
/// JWT              | Encrypting and decrypting tokens with [ECDSA using SHA-256](https://docs.rs/jsonwebtoken/latest/jsonwebtoken/enum.Algorithm.html#variant.ES256)
/// Postgres         | Listening for encrypted client connections on tcp port **5432** (tls Certificate Authority required)
/// pgAdmin          | Listening for encrypted HTTP client connections on tcp port **5433**
/// AWS S3           | Encrypted at rest with [AES256](https://en.wikipedia.org/wiki/Advanced_Encryption_Standard)
///
/// # Interation Test Guide
///
/// ## Build and run the example server
///
/// ```bash
/// time cargo build --example server && export RUST_BACKTRACE=1 && export RUST_LOG=info && time ./target/debug/examples/server
/// ```
///
/// ## Set up bash curl tests
///
/// ```bash
/// export API_TLS_DIR="./certs/tls/api"
/// export TLS_CERT_ARGS="--cacert ${API_TLS_DIR}/api-ca.pem --cert ${API_TLS_DIR}/api.crt --key ${API_TLS_DIR}/api.key"
/// ```
///
/// ## User APIs
///
/// ### Login
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/login" -XPOST -d '{"email":"user@email.com","password":"12345"}' | jq -r '.token'
/// ```
///
/// ### Create user
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -XPOST -d '{"email":"user@email.com","password":"12345"}' | jq
/// ```
///
/// ### Login and save the token as an env variable
///
/// ```bash
/// export TOKEN=$(curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/login" -XPOST -d '{"email":"user@email.com","password":"12345"}' | jq -r '.token')
/// ```
///
/// ### Get user
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/1" -XGET -H "Bearer: ${TOKEN}" | jq
/// ```
///
/// ### Update user
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -H "Bearer: ${TOKEN}" -XPUT -d '{"user_id":1,"email":"somenewemail@gmail.com","password":"321123","state":0}'
/// ```
///
/// ### Change user password
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -H "Bearer: ${TOKEN}" -XPUT -d '{"user_id":1,"password":"12345a"}' | jq
/// ```
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -H "Bearer: ${TOKEN}" -XPUT -d '{"user_id":1,"password":"12345"}' | jq
/// ```
///
/// ### Create a one-time-use-password (otp) allowing a user to reset their users.password from the users.email
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/password/reset" -H "Bearer: ${TOKEN}" -XPOST -d '{"user_id":1,"email":"user@email.com"}' | jq
/// ```
///
/// ### Consume user one-time-use-password token to reset the users.password (otp)
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/password/change" -H "Bearer: ${TOKEN}" -XPOST -d '{"user_id":1,"email":"user@email.com"}' | jq
/// ```
///
/// ### Change user email
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -H "Bearer: ${TOKEN}" -XPUT -d '{"user_id":1,"email":"unique@gmail.com"}' | jq
/// ```
///
/// ### Verify user email
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/verify?u=1&t=2" | jq
/// ```
///
/// ### Search user (token must be for the POST-ed user id)
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/search" -XPOST -H "Bearer: ${TOKEN}" -d '{"email":"user","user_id":1}' | jq
/// ```
///
/// ### Delete user
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user" -XDELETE -d '{"email":"user@email.com","user_id":1}' -H "Content-type: application/json" -H "Bearer: ${TOKEN}" | jq
/// ```
///
/// ## JWT (json web tokens)
///
/// ### Configurable JWT Environment Variables
///
/// #### Header key for the token:
///
/// ```bash
/// export TOKEN_HEADER="Bearer"
/// ```
///
/// #### Token Org (embedded in the jwt)
///
/// ```bash
/// export TOKEN_ORG="Org Name";
/// ```
///
/// #### Token Lifetime Duration
///
/// ```bash
/// # 30 days
/// export TOKEN_EXPIRATION_SECONDS_INTO_FUTURE=2592000;
/// # 7 days
/// export TOKEN_EXPIRATION_SECONDS_INTO_FUTURE=604800;
/// # 1 day
/// export TOKEN_EXPIRATION_SECONDS_INTO_FUTURE=86400;
/// ```
///
/// #### Token Encryption Keys
///
/// ```bash
/// export TOKEN_ALGO_KEY_DIR="./jwt"
/// export TOKEN_ALGO_PRIVATE_KEY_ORG="${TOKEN_ALGO_KEY_DIR}/private-key.pem"
/// export TOKEN_ALGO_PRIVATE_KEY="${TOKEN_ALGO_KEY_DIR}/private-key-pkcs8.pem"
/// export TOKEN_ALGO_PUBLIC_KEY="${TOKEN_ALGO_KEY_DIR}/public-key.pem"
/// ```
///
/// ##### Generate your own jwt keys with these commands
///
/// These commands were tested on ubuntu 21.10 with bash:
///
/// ```bash
/// openssl ecparam -name prime256v1 -genkey -out "${TOKEN_ALGO_PRIVATE_KEY_ORG}"
/// openssl pkcs8 -topk8 -nocrypt -in private-key.pem -out "${TOKEN_ALGO_PRIVATE_KEY}"
/// openssl ec -in "${TOKEN_ALGO_PRIVATE_KEY_ORG}" -pubout -out "${TOKEN_ALGO_PUBLIC_KEY}"
/// ```
///
/// ## S3
///
/// ### Setting up AWS credentials
///
/// https://docs.aws.amazon.com/cli/latest/userguide/cli-configure-envvars.html
///
/// ```bash
/// export AWS_ACCESS_KEY_ID=ACCESS_KEY
/// export AWS_SECRET_ACCESS_KEY=SECRET_KEY
/// ```
///
/// ### S3 Upload a user data file (no file type restrictions + s3 archival)
///
/// ```bash
/// export UPLOAD_FILE="./README.md"
/// export DATA_TYPE="file"
/// export S3_DATA_BUCKET="BUCKET_NAME"
/// export S3_DATA_PREFIX="user/data/file"
/// ```
///
/// ```bash
/// time curl -s ${TLS_CERT_ARGS} -XPOST --data-binary "@${UPLOAD_FILE}" "https://0.0.0.0:3000/user/data" -H "Bearer: ${TOKEN}" -H 'user_id: 1' -H 'comments: this is a test comment' -H 'encoding: na' -H 'Content-type: text/txt' -H 'filename: README.md' -H "data_type: ${DATA_TYPE}" | jq
/// ```
///
/// ### Search user data (token must be for the POST-ed user id)
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/data/search" -XPOST -H "Bearer: ${TOKEN}" -d '{"user_id":1}' | jq
/// ```
///
/// ### Update a single user data record (token must be for the PUT user id)
///
/// ```bash
/// curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/user/data" -XPUT -H "Bearer: ${TOKEN}" -d '{"user_id":1,"data_id":1,"comments":"updated comment using curl"}' | jq
/// ```
///
/// ### Login and save the token as an env variable
///
/// ```bash
/// export TOKEN=$(curl -s ${TLS_CERT_ARGS} "https://0.0.0.0:3000/login" -XPOST -d '{"email":"user@email.com","password":"12345"}' | jq -r '.token')
/// ```
///
/// ## Postgres DB
///
/// ### View DB Tables
///
/// #### Connect to postgres using tls
///
/// ```bash
/// psql --set=sslmode=require -h 0.0.0.0 -p 5432 -U postgres -d mydb
/// ```
///
/// #### Get public tables in the mydb
///
/// ```bash
/// SELECT table_name FROM information_schema.tables WHERE table_schema='public';
/// ```
///

extern crate pretty_env_logger;
#[macro_use]
extern crate log;
extern crate serde;
extern crate serde_json;
extern crate chrono;
extern crate uuid;

pub mod core;
pub mod handle_request;
pub mod is3;
pub mod jwt;
pub mod pools;
pub mod requests;
pub mod tls;
pub mod utils;
