use std::sync::Arc;

use crate::errors::{ApiError, CrateResult, Error};
use reqwest::{StatusCode, header::HeaderMap};
use serde::de::DeserializeOwned;

pub async fn handle_response<'a, ResponseDataType, T>(response: reqwest::Response, error_handler: &Option<Arc<T>>) -> CrateResult<(ResponseDataType, HeaderMap)>
where
    ResponseDataType: DeserializeOwned,
    T: Fn(StatusCode, Option<String>) -> ApiError + ?Sized
{
    let headers= response.headers().clone();
    if response.status().is_success() {
        let body_text = response.text().await?;
        let maybe_data = serde_json::from_str(&body_text);
        match maybe_data {
            Ok(response_data) => { return Ok((response_data, headers)) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }
    else {
        if let Some(handler) = error_handler {
            let status = response.status();
            let maybe_content = 
                match response.text().await {
                    Ok(body) if body == "" => None,
                    Ok(body) => Some(body),
                    Err(_) => None
                };
            return Err(handler(status, maybe_content).into());
        }
        return Err(ApiError(format!("Api error occured with status: {}\n{}", response.status(), response.text().await?)).into());
    }
}

pub async fn handle_response_alt<T>(response: reqwest::Response, error_handler: &Option<Arc<T>>) -> CrateResult<HeaderMap>
where
    T: Fn(StatusCode, Option<String>) -> ApiError + ?Sized
{
    let headers = response.headers().clone();
    if response.status().is_success() {
        return Ok(headers)
    }
    else {
        if let Some(handler) = error_handler {
            let status = response.status();
            let maybe_content = 
                match response.text().await {
                    Ok(body) if body == "" => None,
                    Ok(body) => Some(body),
                    Err(_) => None
                };
            return Err(handler(status, maybe_content).into());
        }
        return Err(ApiError(format!("Api error occured with status: {}\n{}", response.status(), response.text().await?)).into());
    }
}