use std::{future::Future, pin::Pin, sync::Arc};

// use crate::utils::handle_response;
use crate::{errors::{ApiError, CrateResult, Error}, utils::{handle_response, handle_response_alt}};
use reqwest::{IntoUrl, Response, StatusCode, Url, header::HeaderMap, multipart::{self, Form}};
use serde::{Serialize, de::DeserializeOwned};

pub struct RestApiClientBuilder
{
    http_client: reqwest::Client,
    api_base_url: Result<Url, reqwest::Error>,
    before_request_handler: Option< Pin<Arc< dyn Fn(String) -> Pin<Box<dyn Future<Output = ()>> >> > >,
    after_request_handler: Option< Pin<Arc< dyn Fn(&HeaderMap) -> Pin<Box<dyn Future<Output = ()>> >> > >,
    handle_api_error: Option< Arc<dyn Fn(StatusCode, Option<String>) -> ApiError> >
}
impl RestApiClientBuilder {
    pub fn new<T: IntoUrl>(http_client: reqwest::Client, api_base_url: T) -> Self {
        Self{
            http_client: http_client,
            api_base_url: api_base_url.into_url(),
            before_request_handler: None,
            after_request_handler: None,
            handle_api_error: None,
        }
    }
    pub fn on_api_error<Func>(mut self, handler: Func) -> Self
    where
        Func: Fn(StatusCode, Option<String>) -> ApiError + 'static
    {
        self.handle_api_error = Some(Arc::new( handler));
        self
    }
    pub fn before_request<Fut, Func>(mut self, handler: Func) -> Self 
    where
        Fut: Future<Output = ()> + 'static,
        Func: Fn(String) -> Fut + 'static
    {
        self.before_request_handler = Some(Arc::pin(move |headers| Box::pin(handler(headers))));
        self
    }
    pub fn after_request<Fut, Func>(mut self, handler: Func) -> Self 
    where
        Fut: Future<Output = ()> + 'static,
        Func: Fn(&HeaderMap) -> Fut + 'static
    {
        self.after_request_handler = Some(Arc::pin(move |headers| Box::pin(handler(headers))));
        self
    }
    pub fn build(self) -> CrateResult<RestApiClient> {
        Ok(
            RestApiClient{
                http_client: self.http_client,
                api_base_url: self.api_base_url?.to_string().strip_suffix("/").unwrap().to_string(),
                before_request_handler: self.before_request_handler,
                after_request_handler: self.after_request_handler,
                handle_api_error: self.handle_api_error
            }
        )
    }
}

#[derive(Clone)]
pub struct RestApiClient
{
    http_client: reqwest::Client,
    api_base_url: String,
    before_request_handler: Option< Pin<Arc< dyn Fn(String) -> Pin<Box<dyn Future<Output = ()>> >> > >,
    after_request_handler: Option< Pin<Arc< dyn Fn(&HeaderMap) -> Pin<Box<dyn Future<Output = ()>> >> > >,
    handle_api_error: Option< Arc<dyn Fn(StatusCode, Option<String>) -> ApiError> >
}
impl RestApiClient
{
    pub async fn get<ResponseDataType>(&self, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where 
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.clone() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async {self.http_client.get(url.clone()).send().await}, &self.after_request_handler).await?;
        let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
        return data
    }
    pub async fn post<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.clone() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.post(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
        return data
    }
    pub async fn patch<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.patch(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
        return data
    }
    pub async fn put<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.put(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
        return data
    }
    pub async fn delete<ResponseDataType>(&self, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.delete(url.clone()).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
        return data
    }
    // pub async fn post_multipart<ResponseDataType>(&self, form: multipart::Form, path: &str) -> CrateResult<ResponseDataType>
    // where
    //     ResponseDataType: DeserializeOwned
    // {
    //     let url = self.api_base_url.to_string() + path;
    //     let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.post(url.clone()).multipart(form).send().await }, &self.after_request_handler).await.unwrap();
    //     let data: CrateResult<(ResponseDataType, HeaderMap)> = handle_response(response, &self.handle_api_error).await;
    //     return data
    // }
    
    pub async fn post_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.post(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<HeaderMap> = handle_response_alt(response, &self.handle_api_error).await;
        return data
    }
    pub async fn patch_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.patch(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<HeaderMap> = handle_response_alt(response, &self.handle_api_error).await;
        return data
    }
    pub async fn put_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.put(url.clone()).json(&request_data).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<HeaderMap> = handle_response_alt(response, &self.handle_api_error).await;
        return data
    }
    pub async fn delete_alt(&self, path: &str) -> CrateResult<HeaderMap> {
        let url = self.api_base_url.to_string() + path;
        let response = run_request(url.clone(), &self.before_request_handler, || async { self.http_client.delete(url.clone()).send().await }, &self.after_request_handler).await?;
        let data: CrateResult<HeaderMap> = handle_response_alt(response, &self.handle_api_error).await;
        return data
    }

    pub async fn post_multipart_alt(&self, form: multipart::Form, path: &str) -> CrateResult<HeaderMap>
    where
    {
        let url = self.api_base_url.to_string() + path;
        // let response = self.http_client.post(url).multipart(form).send().await?;
        if let Some(handler) = self.before_request_handler.clone() {
            handler(url.clone()).await;
        }
        let response = self.http_client.post(url.clone()).multipart(form).send().await?;
        if let Some(handler) = self.after_request_handler.clone() {
            handler(response.headers()).await;
        }
        let data: CrateResult<HeaderMap> = handle_response_alt(response, &self.handle_api_error).await;
        return data
    }
}

async fn run_request<BFtr, BFn, RFtr, RFn, AFtr, AFn>(url: String, before_handler: &Option<Pin<Arc<BFn>> >, request: RFn, after_handler: &Option<Pin<Arc<AFn>>>) -> CrateResult<Response>
where
    BFtr: Future<Output = ()>,
    BFn: Fn(String) -> BFtr + ?Sized,
    RFtr: Future<Output = Result<Response, reqwest::Error>>,
    RFn: Fn() -> RFtr,
    AFtr: Future<Output = ()>,
    AFn: Fn(&HeaderMap) -> AFtr + ?Sized
{
    if let Some(handler) = before_handler {
        handler(url).await;
    }
    let response = request().await?;
    if let Some(handler) = after_handler {
        handler(response.headers()).await;
    }
    return Ok(response)
}