use crate::errors::{CrateResult, Error};
use reqwest::header::HeaderMap;
use serde::de::DeserializeOwned;

pub async fn handle_response<'a, ResponseDataType, ApiErrorType>(response: reqwest::Response) -> CrateResult<(ResponseDataType, HeaderMap)>
where
    ResponseDataType: DeserializeOwned,
    ApiErrorType: DeserializeOwned + ToString
{
    let headers= response.headers().clone();
    if response.status().is_success() {
        let body_text = response.text().await?;
        let maybe_data = serde_json::from_str(&body_text);
        match maybe_data {
            Ok(response_data) => { return Ok((response_data, headers)) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }
    else {
        let body_text = response.text().await?;
        let maybe_api_error = serde_json::from_str::<ApiErrorType>(&body_text);
        match maybe_api_error {
            Ok(api_error) => { return Err(Error::ApiError(api_error.to_string())) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }
}

pub async fn handle_response_alt<ApiErrorType>(response: reqwest::Response) -> CrateResult<HeaderMap>
where
    ApiErrorType: DeserializeOwned + ToString
{
    let headers = response.headers().clone();
    if response.status().is_success() {
        return Ok(headers)
    }
    else {
        let body_text = response.text().await?;
        let maybe_api_error = serde_json::from_str::<ApiErrorType>(&body_text);
        match maybe_api_error {
            Ok(api_error) => { return Err(Error::ApiError(api_error.to_string())) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }
}