// use crate::utils::handle_response;
use crate::{errors::{CrateResult, Error}, utils::{handle_response, handle_response_alt}};
use reqwest::{IntoUrl, Response, StatusCode, Url, header::HeaderMap, multipart};
use serde::{Serialize, de::DeserializeOwned};

pub struct RestApiClientBuilder<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    http_client: reqwest::Client,
    api_base_url: Result<Url, reqwest::Error>,

    placeholder: Option<ApiErrorType>
}
impl<ApiErrorType> RestApiClientBuilder<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    pub fn new<T: IntoUrl>(http_client: reqwest::Client, api_base_url: T) -> Self {
        Self{
            http_client: http_client,
            api_base_url: api_base_url.into_url(),

            placeholder: None
        }
    }
    pub fn build(self) -> CrateResult<RestApiClient<ApiErrorType>> {
        Ok(
            RestApiClient{
                http_client: self.http_client,
                api_base_url: self.api_base_url?,
                
                placeholder: None
            }
        )
    }
}

#[derive(Clone)]
pub struct RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    http_client: reqwest::Client,
    api_base_url: Url,

    placeholder: Option<ApiErrorType>
}
impl<ApiErrorType> RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    pub async fn get<ResponseDataType>(&self, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where 
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.get(url).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn post<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.post(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn patch<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.patch(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn put<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.put(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn delete<ResponseDataType>(&self, path: &str) -> CrateResult<(ResponseDataType, HeaderMap)>
    where
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.delete(url).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }

    // pub async fn post_multipart<ResponseDataType>(&self, form: multipart::Form, path: &str) -> CrateResult<ResponseDataType>
    // where
    //     ResponseDataType: DeserializeOwned
    // {
    //     let url = self.api_base_url.to_string() + path;
    //     let response = self.http_client.post(url).multipart(form).send().await?;
    //     handle_response::<ResponseDataType, ApiErrorType>(response).await
    // }
    
    pub async fn post_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.post(url).json(&request_data).send().await?;
        handle_response_alt::<ApiErrorType>(response).await
    }
    pub async fn patch_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.patch(url).json(&request_data).send().await?;
        handle_response_alt::<ApiErrorType>(response).await
    }
    pub async fn put_alt<RequestDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<HeaderMap>
    where
        RequestDataType: Serialize,
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.put(url).json(&request_data).send().await?;
        handle_response_alt::<ApiErrorType>(response).await
    }
    pub async fn delete_alt(&self, path: &str) -> CrateResult<HeaderMap>
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.delete(url).send().await?;
        handle_response_alt::<ApiErrorType>(response).await
    }


    pub async fn post_multipart_alt(&self, form: multipart::Form, path: &str) -> CrateResult<HeaderMap>
    where
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.post(url).multipart(form).send().await?;
        handle_response_alt::<ApiErrorType>(response).await
    }
}