// use crate::utils::handle_response;
use crate::{errors::{CrateResult, Error}, utils::handle_response};
use reqwest::{IntoUrl, Response, StatusCode, Url};
use serde::{Serialize, de::DeserializeOwned};

pub struct RestApiClientBuilder<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    http_client: reqwest::Client,
    api_base_url: Result<Url, reqwest::Error>,
    refresh_token_path: Option<String>,

    placeholder: Option<ApiErrorType>
}
impl<ApiErrorType> RestApiClientBuilder<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    pub fn new<T: IntoUrl>(http_client: reqwest::Client, api_base_url: T) -> Self {
        Self{
            http_client: http_client,
            api_base_url: api_base_url.into_url(),
            refresh_token_path: None,

            placeholder: None
        }
    }
    pub fn build(self) -> CrateResult<RestApiClient<ApiErrorType>> {
        Ok(
            RestApiClient{
                http_client: self.http_client,
                api_base_url: self.api_base_url?,
                
                placeholder: None
            }
        )
    }
}

#[derive(Clone)]
pub struct RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    http_client: reqwest::Client,
    api_base_url: Url,

    placeholder: Option<ApiErrorType>
}
impl<ApiErrorType> RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString + Clone
{
    pub async fn get<ResponseDataType>(&self, path: &str) -> CrateResult<ResponseDataType>
    where 
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.get(url).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn post<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<ResponseDataType>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.post(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn put<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<ResponseDataType>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.put(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn delete<ResponseDataType>(&self, path: &str) -> CrateResult<ResponseDataType>
    where
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.to_string() + path;
        let response = self.http_client.delete(url).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
}