use reqwest::{IntoUrl, Request, Url};
use serde::{Deserialize, Serialize, de::DeserializeOwned};

use crate::errors::{CrateResult, Error};

// use reqwest::Url;

pub struct RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString
{
    http_client: reqwest::Client,
    api_base_url: String,

    placeholder: Option<ApiErrorType>
}
impl<ApiErrorType> RestApiClient<ApiErrorType>
where
    ApiErrorType: DeserializeOwned + ToString
{
    pub fn new(http_client: reqwest::Client, api_base_url: Url) -> Self {
        Self{
            http_client: http_client,
            api_base_url: api_base_url.as_str().to_string(),

            placeholder: None
        }
    }
    pub async fn get<ResponseDataType: DeserializeOwned>(&self, path: &str) -> CrateResult<ResponseDataType> {
        let url = self.api_base_url.clone() + path;
        let response = self.http_client.get(url).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    pub async fn post<RequestDataType, ResponseDataType>(&self, request_data: RequestDataType, path: &str) -> CrateResult<ResponseDataType>
    where
        RequestDataType: Serialize,
        ResponseDataType: DeserializeOwned
    {
        let url = self.api_base_url.clone() + path;
        let response = self.http_client.post(url).json(&request_data).send().await?;
        handle_response::<ResponseDataType, ApiErrorType>(response).await
    }
    // pub async fn delete<RequestDataType, ResponseDataType>(self)
}

async fn handle_response<ResponseDataType, ApiErrorType>(response: reqwest::Response) -> CrateResult<ResponseDataType>
where
    ResponseDataType: DeserializeOwned,
    ApiErrorType: DeserializeOwned + ToString
{
    if response.status().is_success() {
        let body_text = response.text().await?;
        let maybe_data = serde_json::from_str(&body_text);
        match maybe_data {
            Ok(response_data) => { return Ok(response_data) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }else {
        let body_text = response.text().await?;
        let maybe_api_error = serde_json::from_str::<ApiErrorType>(&body_text);
        match maybe_api_error {
            Ok(api_error) => { return Err(Error::ApiError(api_error.to_string())) }
            Err(_) => { return Err(Error::UnexpectedData(body_text)) }
        }
    }
}