const CONTENT_TYPE_APPLICATION_JSON: &'static str = "application/json";

pub mod Resemble {
    use lazy_static::lazy_static;
    use std::sync::RwLock;

    lazy_static! {
        static ref API_TOKEN_RW: RwLock<String> = RwLock::new(String::new());
    }

    lazy_static! {
        static ref BASE_URL_RW: RwLock<String> =
            RwLock::new("https://app.resemble.ai/api/".to_string());
    }

    pub fn set_api_key(api_key: String) {
        let mut api_token = API_TOKEN_RW.write().unwrap();
        *api_token = api_key;
    }

    pub fn set_base_url(url: String) {
        let mut base_url = BASE_URL_RW.write().unwrap();
        *base_url = url;
    }

    fn token() -> String {
        let api_token = API_TOKEN_RW.read();
        let token = (*api_token.unwrap()).clone();

        format!("Token token={}", token)
    }

    fn endpoint(version: &str, endpoint: &str) -> String {
        let api_endpoint = if endpoint.starts_with('/') {
            endpoint.to_string()
        } else {
            format!("/{}", endpoint)
        };

        let base_url = BASE_URL_RW.read();

        format!("{}{}{}", *base_url.unwrap(), version, api_endpoint)
    }

    pub mod v2 {
        use serde::{de, Deserialize};
        type APIResponse<T> = Result<T, ErrorResponse>;

        #[derive(Clone, Deserialize, Debug)]
        pub struct ErrorResponse {
            pub success: bool,
            pub message: Option<String>,
        }
        
        #[derive(Clone, Deserialize, Debug)]
        pub struct MessageResponse {
            pub success: bool,
            pub message: Option<String>,
        }

        #[derive(Clone, Deserialize, Debug)]
        pub struct PaginationResponse<T> {
            pub success: bool,
            pub message: Option<String>,
            pub page: isize,
            pub num_pages: isize,
            pub page_size: isize,
            pub items: Vec<T>,
        }

        #[derive(Clone, Deserialize, Debug)]
        pub struct ReadResponse<T> {
            pub success: bool,
            pub message: Option<String>,
            pub item: Option<T>,
        }

        #[derive(Clone, Deserialize, Debug)]
        pub struct WriteResponse<T> {
            pub success: bool,
            pub message: Option<String>,
            /* The item is returned when the write operation succeeds */
            pub item: Option<T>,
        }

        #[derive(Clone, Deserialize, Debug)]
        pub struct UpdateResponse<T> {
            pub success: bool,
            pub message: Option<String>,
            /* The item is returned when the update operation succeeds */
            pub item: Option<T>,
        }

        #[derive(Clone, Deserialize, Debug)]
        pub struct DeleteResponse {
            pub success: bool,
            pub message: Option<String>,
        }

        fn return_json_error<T>(err: Option<reqwest::Error>) -> Result<T, ErrorResponse> {
            Err(ErrorResponse {
                message: Some(format!("Client error: {:#?}", err)),
                success: false,
            })
        }

        pub fn return_json_response<T: de::DeserializeOwned>(
            response: Result<reqwest::blocking::Response, reqwest::Error>,
        ) -> Result<T, ErrorResponse> {
            if response.is_err() {
                return return_json_error::<T>(response.err());
            }

            let response = response.unwrap();

            if response.status().as_u16() == 401 {
                let json = response.json::<ErrorResponse>();

                if json.is_err() {
                    return return_json_error::<T>(json.err());
                }

                Err(json.unwrap())
            } else {
                let json = response.json::<T>();

                if json.is_err() {
                    return return_json_error::<T>(json.err());
                }

                Ok(json.unwrap())
            }
        }

        pub mod project {
            use chrono::{DateTime, Utc};
            use reqwest::header::{AUTHORIZATION, CONTENT_TYPE};
            use serde::{Deserialize, Serialize};

            use crate::Resemble::{
                endpoint, token,
                v2::{
                    return_json_response, APIResponse, DeleteResponse, PaginationResponse,
                    ReadResponse, UpdateResponse, WriteResponse,
                },
            };

            #[derive(Clone, Deserialize, Debug)]
            pub struct Project {
                pub uuid: String,
                pub name: String,
                pub description: String,
                pub is_public: bool,
                pub is_collaborative: bool,
                pub is_archived: bool,
                pub created_at: DateTime<Utc>,
                pub updated_at: DateTime<Utc>,
            }

            #[derive(Serialize, Debug)]
            pub struct ProjectInput {
                pub name: String,
                pub description: String,
                pub is_public: bool,
                pub is_collaborative: bool,
                pub is_archived: bool,
            }

            pub fn all(
                page: usize,
                page_size: Option<usize>,
            ) -> APIResponse<PaginationResponse<Project>> {
                let mut params = vec![("page", page)];
                if page_size.is_some() {
                    params.push(("page_size", page_size.unwrap()));
                }

                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", "projects"))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .query(&params)
                    .send();

                return_json_response(response)
            }

            pub fn create(project: ProjectInput) -> APIResponse<WriteResponse<Project>> {
                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", "projects"))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&project)
                    .send();

                return_json_response(response)
            }

            pub fn update(
                uuid: String,
                project: ProjectInput,
            ) -> APIResponse<UpdateResponse<Project>> {
                let response = reqwest::blocking::Client::new()
                    .put(endpoint("v2", &format!("projects/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&project)
                    .send();

                return_json_response(response)
            }

            pub fn get(uuid: String) -> APIResponse<ReadResponse<Project>> {
                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", &format!("projects/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }

            pub fn delete(uuid: String) -> APIResponse<DeleteResponse> {
                let response = reqwest::blocking::Client::new()
                    .delete(endpoint("v2", &format!("projects/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }
        }

        pub mod voice {
            use chrono::{DateTime, Utc};
            use reqwest::header::{AUTHORIZATION, CONTENT_TYPE};
            use serde::{Deserialize, Serialize};

            use crate::Resemble::{
                endpoint, token,
                v2::{
                    return_json_response, APIResponse, DeleteResponse, PaginationResponse,
                    ReadResponse, UpdateResponse, WriteResponse, MessageResponse
                },
            };

            #[derive(Clone, Deserialize, Debug)]
            pub struct Voice {
                pub uuid: String,
                pub name: String,
                pub status: String,
                pub dataset_url: Option<String>,
                pub callback_uri: Option<String>,
                pub created_at: DateTime<Utc>,
                pub updated_at: DateTime<Utc>,
            }

            #[derive(Serialize, Debug)]
            pub struct VoiceInput {
                pub name: String,
                pub dataset_url: Option<String>,
                pub callback_uri: Option<String>
            }

            #[derive(Serialize, Debug)]
            pub struct UpdateVoiceInput {
                pub name: String,
                pub callback_uri: Option<String>
            }


            pub fn all(
                page: usize,
                page_size: Option<usize>,
            ) -> APIResponse<PaginationResponse<Voice>> {
                let mut params = vec![("page", page)];
                if page_size.is_some() {
                    params.push(("page_size", page_size.unwrap()));
                }

                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", "voices"))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .query(&params)
                    .send();

                return_json_response(response)
            }

            pub fn create(voice: VoiceInput) -> APIResponse<WriteResponse<Voice>> {
                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", "voices"))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&voice)
                    .send();

                return_json_response(response)
            }

            pub fn update(uuid: String, voice: UpdateVoiceInput) -> APIResponse<UpdateResponse<Voice>> {
                let response = reqwest::blocking::Client::new()
                    .put(endpoint("v2", &format!("voices/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&voice)
                    .send();

                return_json_response(response)
            }
            
            pub fn build(uuid: String) -> APIResponse<MessageResponse> {
                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", &format!("voices/{}/build", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();
                    
                return_json_response(response)
            }

            pub fn get(uuid: String) -> APIResponse<ReadResponse<Voice>> {
                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", &format!("voices/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }

            pub fn delete(uuid: String) -> APIResponse<DeleteResponse> {
                let response = reqwest::blocking::Client::new()
                    .delete(endpoint("v2", &format!("voices/{}", uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }
        }

        pub mod recording {
            use chrono::{DateTime, Utc};
            use reqwest::blocking::multipart::Form;
            use reqwest::header::{AUTHORIZATION, CONTENT_TYPE};
            use serde::{Deserialize, Serialize};

            use crate::Resemble::{
                endpoint, token,
                v2::{
                    return_json_error, return_json_response, APIResponse, DeleteResponse,
                    ErrorResponse, PaginationResponse, ReadResponse, UpdateResponse, WriteResponse,
                },
            };

            #[derive(Clone, Deserialize, Debug)]
            pub struct Recording {
                pub uuid: String,
                pub name: String,
                pub text: String,
                pub emotion: String,
                pub is_active: bool,
                pub audio_src: String,
                pub created_at: DateTime<Utc>,
                pub updated_at: DateTime<Utc>,
            }

            #[derive(Serialize, Debug)]
            pub struct RecordingInput {
                pub name: String,
                pub text: String,
                pub emotion: String,
                pub is_active: bool,
            }

            pub fn all(
                voice_uuid: String,
                page: usize,
                page_size: Option<usize>,
            ) -> APIResponse<PaginationResponse<Recording>> {
                let mut params = vec![("page", page)];
                if page_size.is_some() {
                    params.push(("page_size", page_size.unwrap()));
                }

                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", &format!("voices/{}/recordings", voice_uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .query(&params)
                    .send();

                return_json_response(response)
            }

            pub fn create(
                voice_uuid: String,
                recording: RecordingInput,
                audio_file: std::path::PathBuf,
            ) -> APIResponse<WriteResponse<Recording>> {
                let form = Form::new()
                    .text("name", recording.name)
                    .text("text", recording.text)
                    .text("emotion", recording.emotion)
                    .text(
                        "is_active",
                        if recording.is_active { "true" } else { "false" },
                    )
                    .file("file", audio_file);

                if form.is_err() {
                    return Err(ErrorResponse {
                        message: Some(format!("Client error: {:#?}", form.err())),
                        success: false,
                    });
                }

                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", &format!("voices/{}/recordings", voice_uuid)))
                    .multipart(form.unwrap())
                    .header(AUTHORIZATION, token())
                    .send();

                return_json_response(response)
            }

            pub fn update(
                voice_uuid: String,
                uuid: String,
                recording: RecordingInput,
            ) -> APIResponse<UpdateResponse<Recording>> {
                let response = reqwest::blocking::Client::new()
                    .put(endpoint(
                        "v2",
                        &format!("voices/{}/recordings/{}", voice_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&recording)
                    .send();

                return_json_response(response)
            }

            pub fn get(voice_uuid: String, uuid: String) -> APIResponse<ReadResponse<Recording>> {
                let response = reqwest::blocking::Client::new()
                    .get(endpoint(
                        "v2",
                        &format!("voices/{}/recordings/{}", voice_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }

            pub fn delete(voice_uuid: String, uuid: String) -> APIResponse<DeleteResponse> {
                let response = reqwest::blocking::Client::new()
                    .delete(endpoint(
                        "v2",
                        &format!("voices/{}/recordings/{}", voice_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }
        }

        pub mod clip {
            use chrono::{DateTime, Utc};
            use reqwest::header::{AUTHORIZATION, CONTENT_TYPE};
            use serde::{Deserialize, Serialize};

            use crate::Resemble::{
                endpoint, token,
                v2::{
                    return_json_error, return_json_response, APIResponse, DeleteResponse,
                    ErrorResponse, PaginationResponse, ReadResponse, UpdateResponse, WriteResponse,
                },
            };

            #[derive(Clone, Deserialize, Debug)]
            pub struct Timestamps {
                pub phonemes: String,
                pub end_times: Vec<f64>,
                pub phoneme_chars: Vec<String>,
            }

            #[derive(Clone, Deserialize, Debug)]
            pub struct Clip {
                pub uuid: String,
                pub title: Option<String>,
                pub body: String,
                pub voice_uuid: String,
                pub is_public: bool,
                pub is_archived: bool,
                pub timestamps: Option<Timestamps>,
                pub audio_src: Option<String>,
                pub raw_audio: Option<String>,
                pub created_at: DateTime<Utc>,
                pub updated_at: DateTime<Utc>,
            }

            #[derive(Serialize, Debug)]
            pub struct SyncClipInput {
                pub title: Option<String>,
                pub body: String,
                pub voice_uuid: String,
                pub is_public: bool,
                pub is_archived: bool,
                pub sample_rate: Option<usize>, // 16000 | 22050 | 44100
                pub output_format: Option<String>, // 'wav' | 'mp3'
                pub precision: Option<String>,  // 'PCM_16' | 'PCM_32'
                pub include_timestamps: Option<bool>,

                pub raw: Option<bool>,
            }

            #[derive(Serialize, Debug)]
            pub struct AsyncClipInput {
                pub title: Option<String>,
                pub body: String,
                pub voice_uuid: String,
                pub is_public: bool,
                pub is_archived: bool,
                pub sample_rate: Option<usize>, // 16000 | 22050 | 44100
                pub output_format: Option<String>, // 'wav' | 'mp3'
                pub precision: Option<String>,  // 'PCM_16' | 'PCM_32'
                pub include_timestamps: Option<bool>,

                pub callback_uri: String,
            }

            pub fn all(
                project_uuid: String,
                page: usize,
                page_size: Option<usize>,
            ) -> APIResponse<PaginationResponse<Clip>> {
                let mut params = vec![("page", page)];
                if page_size.is_some() {
                    params.push(("page_size", page_size.unwrap()));
                }

                let response = reqwest::blocking::Client::new()
                    .get(endpoint("v2", &format!("projects/{}/clips", project_uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .query(&params)
                    .send();

                return_json_response(response)
            }

            pub fn create_sync(
                project_uuid: String,
                clip: SyncClipInput,
            ) -> APIResponse<WriteResponse<Clip>> {
                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", &format!("projects/{}/clips", project_uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&clip)
                    .send();

                return_json_response(response)
            }

            pub fn create_async(
                project_uuid: String,
                clip: AsyncClipInput,
            ) -> APIResponse<WriteResponse<Clip>> {
                let response = reqwest::blocking::Client::new()
                    .post(endpoint("v2", &format!("projects/{}/clips", project_uuid)))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&clip)
                    .send();

                return_json_response(response)
            }

            pub fn update_async(
                project_uuid: String,
                uuid: String,
                clip: AsyncClipInput,
            ) -> APIResponse<UpdateResponse<Clip>> {
                let response = reqwest::blocking::Client::new()
                    .put(endpoint(
                        "v2",
                        &format!("projects/{}/clips/{}", project_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .json(&clip)
                    .send();

                return_json_response(response)
            }

            pub fn get(project_uuid: String, uuid: String) -> APIResponse<ReadResponse<Clip>> {
                let response = reqwest::blocking::Client::new()
                    .get(endpoint(
                        "v2",
                        &format!("projects/{}/clips/{}", project_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response(response)
            }

            pub fn delete(project_uuid: String, uuid: String) -> APIResponse<DeleteResponse> {
                let response = reqwest::blocking::Client::new()
                    .delete(endpoint(
                        "v2",
                        &format!("projects/{}/clips/{}", project_uuid, uuid),
                    ))
                    .header(AUTHORIZATION, token())
                    .header(CONTENT_TYPE, crate::CONTENT_TYPE_APPLICATION_JSON)
                    .send();

                return_json_response::<DeleteResponse>(response)
            }
        }
    }
}

#[cfg(test)]
mod tests {
    use crate::Resemble::{
        self,
        v2::clip::{AsyncClipInput, SyncClipInput},
        v2::project::ProjectInput,
        v2::recording::RecordingInput,
        v2::voice::VoiceInput,
    };
    use std::{env::var, panic};

    fn get_test_voice_uuid() -> String {
        std::env::var("TEST_VOICE_UUID")
            .expect("Invalid voice_uuid; please set the TEST_VOICE_UUID environment variable")
    }

    fn get_test_callback_uri() -> String {
        std::env::var("TEST_CALLBACK_URI")
            .expect("Invalid callback_uri; please set the TEST_CALLBACK_URI environment variable")
    }

    #[test]
    fn test_v2_projects() {
        run_test(|| {
            // all()
            let projects = Resemble::v2::project::all(1, None).unwrap();
            assert_eq!(projects.success, true);

            // create()
            let project = Resemble::v2::project::create(ProjectInput {
                name: "Test Project".to_string(),
                description: "Test Description".to_string(),
                is_archived: false,
                is_collaborative: false,
                is_public: false,
            })
            .unwrap();
            assert_eq!(project.success, true);

            // update()
            let updated_project = Resemble::v2::project::update(
                project.item.unwrap().uuid,
                ProjectInput {
                    name: "Updated Test Project".to_string(),
                    description: "Updated Test Description".to_string(),
                    is_archived: false,
                    is_collaborative: false,
                    is_public: false,
                },
            )
            .unwrap();
            assert_eq!(updated_project.success, true);

            // get()
            let fetched_project =
                Resemble::v2::project::get(updated_project.item.unwrap().uuid).unwrap();
            assert_eq!(fetched_project.success, true);

            // delete()
            let delete_op =
                Resemble::v2::project::delete(fetched_project.item.unwrap().uuid).unwrap();
            assert_eq!(delete_op.success, true);
        })
    }

    #[test]
    fn test_v2_voices() {
        run_test(|| {
            // all()
            let voices = Resemble::v2::voice::all(1, None).unwrap();
            assert_eq!(voices.success, true);

            // create()
            let voice = Resemble::v2::voice::create(VoiceInput {
                name: "Test voice".to_string(),
            })
            .unwrap();
            assert_eq!(voice.success, true);

            // update()
            let updated_voice = Resemble::v2::voice::update(
                voice.item.unwrap().uuid,
                VoiceInput {
                    name: "Updated Test voice".to_string(),
                },
            )
            .unwrap();
            assert_eq!(updated_voice.success, true);

            // get()
            let fetched_voice = Resemble::v2::voice::get(updated_voice.item.unwrap().uuid).unwrap();
            assert_eq!(fetched_voice.success, true);

            // delete()
            let delete_op = Resemble::v2::voice::delete(fetched_voice.item.unwrap().uuid).unwrap();
            assert_eq!(delete_op.success, true);
        })
    }

    #[test]
    fn test_v2_clips() {
        run_test(|| {
            let project = Resemble::v2::project::create(ProjectInput {
                description: "Test Description".to_string(),
                name: "Test Project".to_string(),
                is_archived: false,
                is_collaborative: false,
                is_public: false,
            })
            .unwrap();
            let project_uuid = project.item.unwrap().uuid;

            // all()
            let clips = Resemble::v2::clip::all(project_uuid.clone(), 1, None).unwrap();
            assert_eq!(clips.success, true);

            // create_sync()
            let clip_create_sync = Resemble::v2::clip::create_sync(
                project_uuid.clone(),
                SyncClipInput {
                    body: "this is a test".to_string(),
                    include_timestamps: Some(true),
                    is_archived: false,
                    is_public: false,
                    output_format: None,
                    precision: None,
                    raw: None,
                    sample_rate: None,
                    title: None,
                    voice_uuid: get_test_voice_uuid(),
                },
            )
            .unwrap();
            assert_eq!(clip_create_sync.success, true);

            let clip_uuid = clip_create_sync.clone().item.unwrap().uuid;

            // create_async()
            let clip_create_async = Resemble::v2::clip::create_async(
                project_uuid.clone(),
                AsyncClipInput {
                    body: "this is a test".to_string(),
                    include_timestamps: Some(true),
                    is_archived: false,
                    is_public: false,
                    output_format: None,
                    precision: None,
                    sample_rate: None,
                    title: None,
                    voice_uuid: get_test_voice_uuid(),
                    callback_uri: get_test_callback_uri(),
                },
            )
            .unwrap();
            assert_eq!(clip_create_async.success, true);

            // update_async()
            let clip_update_async = Resemble::v2::clip::update_async(
                project_uuid.clone(),
                clip_uuid.clone(),
                AsyncClipInput {
                    body: "this is another test".to_string(),
                    include_timestamps: Some(true),
                    is_archived: false,
                    is_public: false,
                    output_format: None,
                    precision: None,
                    sample_rate: None,
                    title: None,
                    voice_uuid: get_test_voice_uuid(),
                    callback_uri: get_test_callback_uri(),
                },
            )
            .unwrap();
            assert_eq!(clip_update_async.success, true);

            // get()
            let fetched_clip =
                Resemble::v2::clip::get(project_uuid.clone(), clip_uuid.clone()).unwrap();
            assert_eq!(fetched_clip.success, true);

            // delete()
            let delete_op =
                Resemble::v2::clip::delete(project_uuid.clone(), clip_uuid.clone()).unwrap();
            assert_eq!(delete_op.success, true);
        })
    }

    #[test]
    fn test_v2_recordings() {
        run_test(|| {
            let voice = Resemble::v2::voice::create(VoiceInput {
                name: "Test Voice".to_string(),
            })
            .unwrap();
            let voice_uuid = voice.item.unwrap().uuid;

            // all()
            let recordings = Resemble::v2::recording::all(voice_uuid.clone(), 1, None).unwrap();
            assert_eq!(recordings.success, true);

            let path = std::path::PathBuf::from("spec_sample_audio.wav");
            assert!(path.exists());

            // create()
            let created_recording = Resemble::v2::recording::create(
                voice_uuid.clone(),
                RecordingInput {
                    emotion: "neutral".to_string(),
                    is_active: true,
                    name: "recording".to_string(),
                    text: "this is a test".to_string(),
                },
                path,
            )
            .unwrap();
            assert_eq!(created_recording.success, true);

            let recording_uuid = created_recording.clone().item.unwrap().uuid;

            // update()
            let updated_recording = Resemble::v2::recording::update(
                voice_uuid.clone(),
                recording_uuid.clone(),
                RecordingInput {
                    emotion: "neutral".to_string(),
                    is_active: false,
                    name: "updated recording".to_string(),
                    text: "this is an updated test".to_string(),
                },
            )
            .unwrap();
            assert_eq!(updated_recording.success, true);

            // get()
            let fetched_recording =
                Resemble::v2::recording::get(voice_uuid.clone(), recording_uuid.clone()).unwrap();
            assert_eq!(fetched_recording.success, true);

            // delete()
            let delete_op =
                Resemble::v2::recording::delete(voice_uuid.clone(), recording_uuid.clone())
                    .unwrap();
            assert_eq!(delete_op.success, true);
        })
    }

    fn run_test<T>(test: T) -> ()
    where
        T: FnOnce() -> () + panic::UnwindSafe,
    {
        let api_key = var("TEST_API_KEY").unwrap();
        let base_url = var("TEST_BASE_URL").unwrap_or_default();
        Resemble::set_api_key(api_key);

        if !base_url.is_empty() {
            Resemble::set_base_url(base_url);
        }

        let result = panic::catch_unwind(|| test());

        assert!(result.is_ok())
    }
}
