use reqwest::blocking::{Client};
use reqwest::header::{HeaderMap, HeaderValue};
use select::document::Document;
use crate::cache::{Store};
use crate::headers::{Headers, MyHeader};
use crate::parser::{Parser};
use crate::value::Value;


pub struct Requests<'a, T> {
    pub text: String, // 保存响应的数据,
    pub url: &'a str,
    store: &'a T,
}


impl<'a, T> Requests<'a, T> where T: Store {

    fn contruct_headers(_headers: Option<MyHeader>) -> HeaderMap {
        let mut headers = HeaderMap::new();
        match _headers {
            Some(data) => {
                headers.insert("user-agent", HeaderValue::from_str(&data.user_agent).unwrap());
                headers.insert("host", HeaderValue::from_str(&data.host.unwrap_or_default()).unwrap());
                headers
            },
            None => {
                headers.insert("user-agent", HeaderValue::from_static("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.159 Safari/537.36"));
                headers
            }
        }
    }

    pub fn new(store: &'a T) -> Requests<T> {
        Requests {
            text: "".to_string(),
            url: "",
            store,
        }
    }

    pub fn connect(&self, url: &'a str, headers: Headers) -> Self {
        let text = self.get(url, headers);

        Requests {
            text,
            url,
            store: self.store
        }
    }

    pub fn parser<F>(&self, parser_call: F, key: &str) -> &T
        where F: Fn(Parser) -> Value {
            let document = Document::from(self.text.as_str());
            let mut parser = Parser::new(&document);
            parser.defined_css_processes();
            let result = parser_call(parser);
            self.store.pipes(key, result);
            self.store
    }

     pub fn free_parse<F>(&self, parse_call: F)
        where F: Fn(Parser) {
            let document = Document::from(self.text.as_str());
            let mut parser = Parser::new(&document);
            parser.defined_css_processes();
            parse_call(parser);
        }           

    // pub fn into_parse<F, S>(&self, parse_call: F)
    //     where F: Fn(Parser) -> Box<S> {
    //         let document = Document::from(self.text.as_str());
    //         let parser = Parser::new(&document);
    //         parse_call(parser);
    //     }


    pub fn get(&self, url: &str, headers: Headers) -> String {
        let builder = Client::builder();
        let client = match headers {
            Headers::None => builder.build().unwrap(),
            Headers::Default => builder.default_headers(Requests::<'a, T>::contruct_headers(None)).build().unwrap(),
            Headers::JSON(data) => {
                let my_header = serde_json::from_str::<MyHeader>(data).unwrap();
                builder.default_headers(Requests::<'a, T>::contruct_headers(Some(my_header))).build().unwrap()
            } 
        };
        let res = client.get(url).send();

        match res {
            Ok(res) =>  {
                let data = res.text().unwrap_or_else(|err| {
                    panic!("error url: {}, error: {:?}", url, err);
                });
                data
              
            },
            Err(err) => panic!("No data! error: {:?}", err),
        }
        
    }
    
}



