use std::{collections::HashMap, ops::Deref, rc::Rc, thread, time::Instant};

use requests2::{
    dbfile::DBfile,
    dbfile_derive::{dbnote, DBfile},
    select::predicate::Name, Cache, Headers, Requests,
};

#[derive(DBfile)]
#[dbnote(table_name = "b339", driver = "postgres", primary_key = "href")]
struct Item<'a> {
    book_name: &'a str,
    href: String,
    img: String,
    isbn: String,
    price: f32,
}

fn create_table() {
    let by_data = Item {
        book_name: "test",
        href: "".to_string(),
        img: "".to_string(),
        isbn: "".to_string(),
        price: 6.5,
    };
    by_data.create_table();
}

fn main() {
    create_table();
    run();
}

fn run() {
    let mut urls = Vec::new();
    let now = Instant::now();
    let max_page = 30;

    for p in 1..(max_page + 1) {
        let url = format!("https://book.kongfz.com/Cxianzhuang/cat_8001w{}/", &p);
        urls.push(url);
    }

    while urls.len() > 0 {
        let mut batch_urls = Vec::new();

        for _ in 0..10 {
            if urls.len() > 0 {
                batch_urls.push(urls.remove(0));
            }
        }

        let mut handlers = Vec::new();

        for (p, url) in batch_urls.into_iter().enumerate() {
            let handler = thread::spawn(move || {
                crawl_kongfz_by_page(p as i32, url.to_string());
            });
            handlers.push(handler);
        }

        for handler in handlers {
            handler.join().unwrap();
        }
    }

    println!("cast {:?} time", now.elapsed().as_secs());
}

pub fn crawl_kongfz_by_page(page: i32, url: String) {
    let data = Cache::new();
    let client = Requests::new(&data);
    let rq = client.connect(&url, Headers::Default);

    rq.free_parse(|mut p| {
        p.free_select(".item.clearfix", |nodes| {
            let _book_item = nodes
                .iter()
                .enumerate()
                .filter(|(_idx, n)| n.attr("itemname").is_some())
                .map(|(idx, x)| {
                    let book_name = Rc::new(x.attr("itemname").unwrap().to_string());
                    let isbn = Rc::new(x.attr("isbn").unwrap().to_string());
                    let price: f32 = Rc::new(x.attr("price").unwrap()).parse().unwrap();
                    let href = Rc::new(
                        x.find(Name("a"))
                            .find_map(|n| n.attr("href"))
                            .unwrap()
                            .to_string(),
                    );
                    let img = Rc::new(
                        x.find(Name("img"))
                            .find_map(|n| n.attr("src"))
                            .unwrap()
                            .to_string(),
                    );

                    let mut _header = false;
                    if idx == 0 && page == 1 {
                        _header = true
                    };
                    Item {
                        book_name: Rc::clone(&book_name).deref(),
                        href: Rc::clone(&href).deref().to_string(),
                        img: Rc::clone(&img).deref().to_string(),
                        isbn: Rc::clone(&isbn).deref().to_string(),
                        price,
                    }
                    .to_db();

                    // DBStore::to_csv(vec![
                    //     Item {
                    //         book_name: Rc::clone(&book_name).deref(),
                    //         price,
                    //         href: Rc::clone(&href).deref(),
                    //         img: Rc::clone(&img).deref(),
                    //         isbn: Rc::clone(&isbn).deref(),
                    //     }
                    // ], "test_links2.csv", "a", header);

                    let hash = HashMap::from([
                        ("book_name".to_string(), Rc::clone(&book_name)),
                        ("href".to_string(), Rc::clone(&href)),
                        ("isbn".to_string(), Rc::clone(&isbn)),
                        ("price".to_string(), Rc::new(price.to_string())),
                        ("img".to_string(), Rc::clone(&img)),
                    ]);
                    // }
                    hash
                })
                .collect::<Vec<HashMap<String, Rc<String>>>>();

            // print!("{:?}", book_item)
        });
    })
}
