use std::{collections::HashMap, cell::RefCell, sync::{Mutex, Arc}, borrow::BorrowMut};
use crate::value::Value;

/// # Cache 实现了trait store 它可以将数据存储到内部变量caches中
/// 新建db并写入存放到requests中，
/// '''
/// let db = Cache::new()
/// let mut rq = Requests::new(&data).connect("https://www.qq.com", Headers::Default)
/// '''
pub trait Store {
    fn pipes(&self, key: &str, result: Value);
    fn print(&self) ;
    fn print_json(&self);
    fn count(&self, key: &str) -> usize;
    fn get(&self, key:&str) -> Value;
}

#[derive(Debug)]
pub struct Cache {
    caches: Arc<Mutex<RefCell::<HashMap<String,Value>>>>
}

impl Cache {
    pub fn new() -> Self {
        Cache {caches: Arc::new(Mutex::new(RefCell::new(HashMap::new())))}
    }
}


impl Store for Cache  {
    fn pipes(&self, key: &str, result: Value) {
        let caches = Arc::clone(&self.caches);
        let caches = caches.lock().unwrap();
        (*caches).borrow_mut().insert(key.to_string(), result);
    }

    fn print_json(&self) {
        for (k, v) in self.caches.lock().unwrap().borrow().iter() {
            let data = serde_json::to_string(v).expect("not Serialize to string");
            println!("Key -- {:?} Value -- {:?}", k, data);
        }
    }

    fn print(&self) {
        for (k, v) in self.caches.lock().unwrap().borrow().iter() {
            println!("Key -- {:?} Value -- {:?}", k, v);
        }
    }

    fn count(&self, key:&str)  -> usize {
        let mut counter = 0;
        for (k, v) in self.caches.lock().unwrap().borrow().iter() {
            if k == key {
                match v {
                    Value::LIST(c) => counter = c.len(),
                    _ => counter = 1
                };
            }
        }
        counter
    }

    fn get(&self, key:&str) -> Value {
        let mut vv = Value::NULL;
        for (k, v) in self.caches.lock().unwrap().borrow().iter() {
            if k == key {
                match v {
                    Value::STR(s) => vv = Value::STR(s.to_string()),
                    Value::LIST(s) => vv = Value::LIST(
                        s.iter().map(|x| x.to_owned()).collect::<Vec<String>>()
                    ),
                    Value::INT(i) => vv = Value::INT(*i),
                    _ => (),
                }
            }
            
        }
        vv
    }

}