#![allow(dead_code)]

use anyhow::{Context, Error, Result, anyhow, bail};
use fakemap::FakeMap as Map;
use serde::{Deserialize, Serialize};

#[cfg(test)]
mod tests;

#[derive(Clone, Debug, Deserialize, Serialize)]
pub struct Package {
    pub public: Map<String, Label>,
    pub private: Map<String, Label>,
}

#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]
#[serde(try_from = "&str")]
pub struct Label {
    segments: Vec<Segment>,
    root: Root,
}

impl<'a> TryFrom<&'a str> for Label {
    type Error = Error;

    fn try_from(mut s: &'a str) -> Result<Self> {
        let orig = s;

        let root = if s.starts_with("@") {
            if let Some(pos) = s.find("//") {
                let name = s[1..pos].to_owned();
                s = &s[pos..];

                Root::Other(name)
            } else {
                bail!("could not find `//` in label {}", orig);
            }
        } else if s.starts_with("//") {
            s = &s[2..];

            Root::Local
        } else {
            Root::Relative
        };

        let mut segments = vec![];
        loop {
            dbg!(s);
            let segment = Segment::try_from(s)?;
            s = &s[segment.needed_len()..];
            segments.push(segment);
            if s.is_empty() {
                break;
            }
        }

        Ok(Label {
            root,
            segments,
        })
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]
pub enum Segment {
    Empty,
    Dir(String),
    File(String),
    Rule(String),
}

impl Segment {
    fn dir(s: &str) -> Self {
        Segment::Dir(s.to_owned())
    }

    fn file(s: &str) -> Self {
        Segment::File(s.to_owned())
    }

    fn rule(s: &str) -> Self {
        Segment::Rule(s.to_owned())
    }

    fn needed_len(&self) -> usize {
        match self {
            Segment::Empty => 0,
            Segment::Dir(s) | Segment::Rule(s) => s.len() + 1,
            Segment::File(s) => s.len(),
        }
    }

    fn take_until_sep(s: &str) -> Result<String> {
        let mut end = 0;
        
        for (byte_pos, c) in s.char_indices() {
            if c.is_control() {
                bail!("control characters not allowed in labels");
            } else if c == '\\' {
                bail!("backslashes not allowed in labels; use forward slashes (/) instead");
            } else if c == '/' || c == ':' {
                end = byte_pos;
                break;
            } else {
                end = byte_pos + 1;
            }
        };
    
        let s = &s[..end];
    
        if s.chars().next().map(|c| c.is_whitespace()).unwrap_or(false) {
            bail!("backslashes not allowed in labels; use forward slashes (/) instead");
        }
    
        Ok(s.to_owned())
    }
}

impl TryFrom<&'_ str> for Segment {
    type Error = Error;

    fn try_from(s: &str) -> Result<Self> {
        let segment = if s.is_empty() || s.starts_with('/') {
            Segment::Empty
        } else if s.starts_with(":") {
            let rule = Self::take_until_sep(&s[1..]).with_context(|| anyhow!("could not parse label segment {}", s))?;

            if rule.is_empty() {
                bail!("empty rule not allowed (got {})", s);
            }

            Segment::Rule(rule)
        } else {
            let path = Self::take_until_sep(s).with_context(|| anyhow!("could not parse label segment {}", s))?;

            if s.bytes().nth(path.len()).map(|c| c == b'/').unwrap_or(false) {
                Segment::Dir(path)
            } else {
                Segment::File(path)
            }
        };

        Ok(segment)
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]
pub enum Root {
    Relative,
    Local,
    Other(String),
}
