use gtk::{gio, glib};
use gtk::prelude::{ActionExt, ActionMapExt, ToVariant};

use std::marker::PhantomData;

pub trait ActionGroupName {
    fn group_name() -> &'static str;
}

pub trait ActionName {
    type Group: ActionGroupName;
    type Value: glib::variant::ToVariant + glib::variant::FromVariant + Default;

    fn name() -> &'static str;

    fn action_name() -> &'static str {
        &format!("{}.{}", Self::Group::group_name(), Self::name())
    }
}

pub trait StatelessAction {
    type Group: ActionGroupName;

    fn name() -> &'static str;

    fn action_name() -> &'static str {
        &format!("{}.{}", Self::Group::group_name(), Self::name())
    }
}

pub struct RelmAction<Name: ActionName> {
    name: PhantomData<Name>,
    action: gio::SimpleAction,
}

impl <Name: ActionName> RelmAction<Name>
where Name::Value: glib::variant::ToVariant + glib::variant::FromVariant + Default {
    pub fn new_stateful<Callback: FnMut(Name::Value)>(callback: Callback) -> Self {
        let value = Name::Value::default();
        let variant = value.to_variant();
        let ty = variant.type_();

        let action = gio::SimpleAction::new(Name::action_name(), Some(ty));
        Self {
            name: PhantomData,
            action
        }
    }
}

#[derive(Debug)]
pub struct ActionGroup<GroupName: ActionGroupName> {
    group_name: std::marker::PhantomData<GroupName>,
    group: gtk::gio::SimpleActionGroup,
}

impl<GroupName: ActionGroupName> ActionGroup<GroupName> {
    pub fn add_action<A: Action<Group = GroupName> + 'static>(&self, action: A) {
        let value = A::Value::default();
        let variant = value.to_variant();
        let ty = variant.type_();

        let simple_action = gtk::gio::SimpleAction::new(A::name(), Some(ty));

        simple_action.connect_activate(move |_, opt_variant| {
            let value = if let Some(variant) = opt_variant {
                variant.get::<A::Value>()
            } else {
                None
            };
            action.callback(value);
        });

        self.group.add_action(&simple_action);
    }

    pub fn add_stateless_action<A: StatelessAction<Group = GroupName> + 'static>(&self, action: A) {
        let simple_action = gtk::gio::SimpleAction::new(A::name(), None);

        simple_action.connect_activate(move |_, _| {
            action.callback();
        });

        self.group.add_action(&simple_action);
    }

    pub fn into_action_group(self) -> gtk::gio::SimpleActionGroup {
        self.group
    }

    pub fn new() -> Self {
        Self {
            group_name: std::marker::PhantomData,
            group: gtk::gio::SimpleActionGroup::new(),
        }
    }
}
