use mockito::{mock, Matcher};
use std::collections::HashMap;
use uuid::Uuid;

use crate::*;

fn create_config() -> log::ElasticConfig {
    log::ElasticConfig {
        url: mockito::server_url(),
        username: "test_user".to_string(),
        password: "test_password".to_string(),
        environment: log::LogEnvironment::Development,
        application_name: "relastic-tests".to_string(),
    }
}

#[test]
fn test_logging() {
    let reply = r#"{
        "took": 1,
        "errors": false,
        "items": [
            {
                "index": {
                    "_index": "application-production-2021.11",
                    "_type": "_doc",
                    "_id": "abc",
                    "_version": 1,
                    "result": "created",
                    "_shards": {
                    "total": 2,
                    "successful": 1,
                    "failed": 0
                    },
                    "_seq_no": 2,
                    "_primary_term": 1,
                    "status": 201
                }
            }
        ]
    }"#;

    let my_expected_test_value = Uuid::new_v4();

    let api_mock = mock("POST", "/_bulk")
        .with_status(200)
        .with_header("content-type", "application/json")
        .with_body(reply)
        .match_body(Matcher::Regex(my_expected_test_value.to_string()))
        .create();

    let config = create_config();

    log::setup_elastic_log(config, 100);
    log::information(
        "This is a test {value}",
        HashMap::from([("value", my_expected_test_value.to_string())]),
    );
    log::debug(
        "This is a test {value}",
        HashMap::from([("value", my_expected_test_value.to_string())]),
    );
    log::fatal(
        "This is a test {value}",
        HashMap::from([("value", my_expected_test_value.to_string())]),
    );
    log::warning(
        "This is a test {value}",
        HashMap::from([("value", my_expected_test_value.to_string())]),
    );
    log::flush();

    assert_eq!(api_mock.expect(4).matched(), true);
}

#[test]
fn it_returns_error_on_inner_error_replies() {
    let reply = r#"{
        "took": 3,
        "errors": true,
        "items": [
          {
            "index": {
              "_index": "application-production-2021.11",
              "_type": "_doc",
              "_id": "321",
              "status": 400,
              "error": {
                "type": "mapper_parsing_exception",
                "reason": "failed to parse",
                "caused_by": {
                  "type": "json_parse_exception",
                  "reason": "Unexpected character ('H' (code 72)): was expecting comma to separate Object entries\n at [Source: org.elasticsearch.common.bytes.AbstractBytesReference$MarkSupportingStreamInputWrapper@7bc49eab; line: 1, column: 137]"
                }
              }
            }
          },
          {
            "index": {
              "_index": "application-production-2021.11",
              "_type": "_doc",
              "_id": "1234",
              "status": 400,
              "error": {
                "type": "mapper_parsing_exception",
                "reason": "failed to parse",
                "caused_by": {
                  "type": "json_parse_exception",
                  "reason": "Unexpected character ('H' (code 72)): was expecting comma to separate Object entries\n at [Source: org.elasticsearch.common.bytes.AbstractBytesReference$MarkSupportingStreamInputWrapper@d19c3fd; line: 1, column: 137]"
                }
              }
            }
          },
          {
            "index": {
              "_index": "application-production-2021.11",
              "_type": "_doc",
              "_id": "123",
              "_version": 1,
              "result": "created",
              "_shards": {
                "total": 2,
                "successful": 1,
                "failed": 0
              },
              "_seq_no": 2,
              "_primary_term": 1,
              "status": 201
            }
          }
        ]
    }"#;

    let api_mock = mock("POST", "/_bulk")
        .with_status(200)
        .with_header("content-type", "application/json")
        .with_body(reply)
        .create();

    let config = create_config();

    // We are not testing the input. That's why we send an empty string here.
    let result = log::test_post_to_elastic(config, "".to_string());

    assert_eq!(api_mock.expect(1).matched(), true);
    assert_eq!(result.is_err(), true);
    let err = result.err().expect("Expected error here");

    match err {
        log::Error::SomeLogsWereNotAccepted { errors } => {
            assert_ne!(errors, "".to_string())
        }
        _ => {
            assert!(
                false,
                "Did not get the expected error here. Expected 'SomeLogsWereNotAccepted', got {}",
                err
            )
        }
    }
}

#[test]
fn it_returns_error_on_api_error_reply() {
    let reply = r#"{
      "error": {
        "root_cause": [
            {
                "type": "illegal_argument_exception",
                "reason": "The bulk request must be terminated by a newline [\\n]"
            }
        ],
        "type": "illegal_argument_exception",
        "reason": "The bulk request must be terminated by a newline [\\n]"
      },
      "status": 400
    }"#;

    let api_mock = mock("POST", "/_bulk")
        .with_status(400)
        .with_header("content-type", "application/json")
        .with_body(reply)
        .create();

    let config = create_config();

    // We are not testing the input. That's why we send an empty string here.
    let result = log::test_post_to_elastic(config, "".to_string());
    assert_eq!(api_mock.expect(1).matched(), true);
    assert_eq!(result.is_err(), true);

    let err = result.err().expect("Expected error here");

    match err {
        log::Error::ApiRejectedLogPayload { errors } => assert_ne!(errors, "".to_string()),
        _ => {
            assert!(
                false,
                "Did not get the expected error here. Expected 'ApiRejectedPayload', got: {}",
                err
            )
        }
    }
}

#[test]
fn it_does_not_publish_debug_in_production() {
    let reply = r#"{
        "took": 1,
        "errors": false,
        "items": [
            {
                "index": {
                    "_index": "application-production-2021.11",
                    "_type": "_doc",
                    "_id": "abc",
                    "_version": 1,
                    "result": "created",
                    "_shards": {
                    "total": 2,
                    "successful": 1,
                    "failed": 0
                    },
                    "_seq_no": 2,
                    "_primary_term": 1,
                    "status": 201
                }
            }
        ]
    }"#;

    let my_expected_test_value = Uuid::new_v4();

    let api_mock = mock("POST", "/_bulk")
        .with_status(200)
        .with_header("content-type", "application/json")
        .with_body(reply)
        .match_body(Matcher::Regex(my_expected_test_value.to_string()))
        .create();

    let config = log::ElasticConfig {
        url: mockito::server_url(),
        username: "test_user".to_string(),
        password: "test_password".to_string(),
        environment: log::LogEnvironment::Production,
        application_name: "relastic-tests".to_string(),
    };

    log::setup_elastic_log(config, 100);
    log::debug(
        "This is a test {value}",
        HashMap::from([("value", my_expected_test_value.to_string())]),
    );
    log::flush();

    assert_eq!(api_mock.expect(0).matched(), true);
}
