use chrono::Datelike;
use chrono::SecondsFormat;
use chrono::Utc;
use lazy_static::lazy_static;
use serde::Deserialize;
use serde::Serialize;
use serde_json::json;
use snafu::Snafu;
use std::collections::HashMap;
use std::string::ToString;
use std::sync::atomic::AtomicBool;
use std::sync::mpsc;
use std::sync::RwLock;
use std::thread;
use std::time::Duration;
use strum_macros::Display;
use strum_macros::EnumString;

lazy_static! {
    static ref LOG: RwLock<Option<std::sync::mpsc::SyncSender<LogMessage>>> = RwLock::new(None);
}
lazy_static! {
    static ref SHOULD_LOOP: AtomicBool = AtomicBool::new(false);
}
lazy_static! {
    static ref IS_FLUSHED: AtomicBool = AtomicBool::new(false);
}
static WAIT: Duration = Duration::from_secs(1);

#[derive(Debug, Snafu)]
pub enum Error {
    #[snafu(display("Request failed: {}", inner))]
    RequestFailed { inner: reqwest::Error },
    #[snafu(display("Failed to deserialize log API reply: {}", inner))]
    DeserializationFailed { inner: reqwest::Error },
    #[snafu(display(
        "Some, or all chunked logs, were not accepted by the log API: {}",
        errors
    ))]
    SomeLogsWereNotAccepted { errors: String },
    #[snafu(display("The log API rejected the whole chunked log request: {}", errors))]
    ApiRejectedLogPayload { errors: String },
}

#[derive(Display, Debug, PartialEq, Eq)]
pub enum LogLevel {
    Debug,
    Information,
    Warning,
    Error,
    Fatal,
}

#[derive(Serialize, Display, PartialEq, EnumString, Clone)]
pub enum LogEnvironment {
    Production,
    Development,
}

pub struct LogMessage {
    level: LogLevel,
    message_template: String,
    message: String,
    fields: HashMap<String, String>,
}

#[derive(Serialize)]
struct LogInnerIndex {
    #[serde(rename(serialize = "_index"))]
    index: String,
    #[serde(rename(serialize = "_type"))]
    log_type: String,
}

#[derive(Serialize)]
struct LogIndex {
    index: LogInnerIndex,
}

#[derive(Deserialize, Serialize, Debug)]
struct ElasticErrorCause {
    r#type: String,
    reason: String,
}

#[derive(Deserialize, Serialize, Debug)]
struct ElasticError {
    caused_by: ElasticErrorCause,
}

#[derive(Serialize, Deserialize)]
struct ElasticSingleLogStatus {
    #[serde(rename(deserialize = "_index"))]
    index: String,
    status: u16,
    #[serde(rename(deserialize = "_id"))]
    id: String,
    error: Option<ElasticError>,
}

#[derive(serde::Deserialize, Serialize)]
struct ElasticSingleReplyIndex {
    index: ElasticSingleLogStatus,
}

#[derive(serde::Deserialize, Serialize)]
struct ElasticReply {
    errors: bool,
    items: Vec<ElasticSingleReplyIndex>,
}

#[derive(serde::Deserialize, Serialize)]
struct ElasticErrorReplyError {
    reason: String,
    r#type: String,
}

#[derive(serde::Deserialize, Serialize)]
struct ElasticErrorReply {
    error: ElasticErrorReplyError,
    status: u16,
}

fn log(level: LogLevel, template: &str, fields: HashMap<&str, String>) {
    let mut message = template.to_string();
    for (key, value) in fields.iter() {
        message = message.replacen(
            format!("{{{}}}", key).as_str(),
            &(format!("{{{}}}", value).as_str()),
            1,
        );
    }

    let mut fields_copy = HashMap::<String, String>::with_capacity(fields.len());
    for (key, value) in fields.into_iter() {
        fields_copy.insert(key.to_string(), value);
    }

    let log_msg = LogMessage {
        level,
        message_template: template.to_string(),
        message,
        fields: fields_copy,
    };

    {
        let logger = LOG.read().unwrap();
        if let Some(logger_) = &*logger {
            logger_
                .send(log_msg)
                .expect("Relastic queue has stopped working");
            return;
        }
    }
    println!("[WARNING]: Relastic not initialized. Logging to console");
    println!("[{}]: ", log_msg.level);
}

/// Logs message with 'information' as severity level.
/// Feel free to use this to log any information you may want to inspect.
/// /// # Examples
///
/// ```
///
/// relastic::log::information(
///     "This is my test message: {msg}",
///     std::collections::HashMap::from([("msg", "Hello World!".to_string())])
/// );
///
/// ```
pub fn information(template: &str, fields: HashMap<&str, String>) {
    log(LogLevel::Information, template, fields)
}

/// Logs message with 'error' as severity level.
/// This should be used if a critial error that needs too be looked at occurrs.
/// The application must be able to recover, and unrelated data-loss should not happen when using this.
///
/// ```
///
/// relastic::log::error(
///     "This is my test error: {error}",
///     std::collections::HashMap::from([("error", "Hello World!".to_string())])
/// );
///
/// ```
pub fn error(template: &str, fields: HashMap<&str, String>) {
    log(LogLevel::Error, template, fields)
}

/// Logs message with 'debug' as severity level.
/// This should be used for debug-purposes only. Messages will be ignored if enviroment is set to production.
///
/// ```
///
/// relastic::log::debug(
///     "This is my test message: {msg}",
///     std::collections::HashMap::from([("msg", "Hello World!".to_string())])
/// );
///
/// ```
pub fn debug(template: &str, fields: HashMap<&str, String>) {
    log(LogLevel::Debug, template, fields)
}

/// Logs message with 'fatal' as severity level.
/// This should be used if an unrecoverable error occurrs, that leaves the application with no choice but to shut down.
///
/// ```
///
/// relastic::log::fatal(
///     "This is my test error: {error}",
///     std::collections::HashMap::from([("error", "Hello World!".to_string())])
/// );
///
/// ```
pub fn fatal(template: &str, fields: HashMap<&str, String>) {
    log(LogLevel::Fatal, template, fields)
}

/// Logs message with 'warning' as severity level.
/// This should be used if an error or issue occurrs that may may have to be looked at,
/// but is unimportant for application logic's continuous success.
///
/// ```
///
/// relastic::log::warning(
///     "This is my test error: {error}",
///     std::collections::HashMap::from([("error", "Hello World!".to_string())])
/// );
///
/// ```
pub fn warning(template: &str, fields: HashMap<&str, String>) {
    log(LogLevel::Warning, template, fields)
}

/// Closes the sender, and publishes remaining logs before closing the publish loop.
/// You should call this before your application completes/returns so that no logs are lost.
pub fn flush() {
    println!("Waiting for log grace-period to expire...");
    thread::sleep(WAIT);
    println!("Closing log transciever");
    {
        let mut sender = LOG.write().unwrap();
        *sender = None;
    }
    println!("Flushing logs");
    SHOULD_LOOP.store(false, std::sync::atomic::Ordering::SeqCst);
    while !IS_FLUSHED.load(std::sync::atomic::Ordering::SeqCst) {
        println!("Waiting for logs to flush...");
        thread::sleep(WAIT);
    }
    println!("Flushed logs");
}

#[derive(Serialize)]
struct LogPayload {
    #[serde(rename(serialize = "@timestamp"))]
    timestamp: String,
    level: String,
    #[serde(rename(serialize = "messageTemplate"))]
    message_template: String,
    message: String,
    fields: HashMap<String, String>,
}

pub struct ElasticConfig {
    pub username: String,
    pub password: String,
    pub url: String,
    pub environment: LogEnvironment,
    pub application_name: String,
}

impl Clone for ElasticConfig {
    fn clone(&self) -> Self {
        ElasticConfig {
            username: self.username.to_owned(),
            password: self.password.to_owned(),
            url: self.url.to_owned(),
            environment: self.environment.to_owned(),
            application_name: self.application_name.to_owned(),
        }
    }
}

fn concat_fields(
    default_fields: &HashMap<String, String>,
    extra_fields: HashMap<String, String>,
) -> HashMap<String, String> {
    let mut all_fields = HashMap::new();
    for (key, value) in default_fields.iter() {
        all_fields.insert(key.to_owned(), value.to_owned());
    }
    for (key, value) in extra_fields.into_iter() {
        all_fields.insert(key, value);
    }
    all_fields
}

fn log_to_console(log_payload: &LogPayload) {
    println!("[{}]: {}", log_payload.level, log_payload.message);
}

#[cfg(test)]
pub fn test_post_to_elastic(elastic_credentials: ElasticConfig, json: String) -> Result<(), Error> {
    let client = reqwest::blocking::Client::new();
    post_to_elastic(client, elastic_credentials, json)
}

fn post_to_elastic(
    client: reqwest::blocking::Client,
    elastic_credentials: ElasticConfig,
    json: String,
) -> Result<(), Error> {
    let result = client
        .post(format!("{}/_bulk", elastic_credentials.url))
        .basic_auth(
            elastic_credentials.username,
            Some(elastic_credentials.password),
        )
        .header("Content-Type", "application/json")
        .header("Accept", "application/json")
        .body(json.to_owned())
        .send()
        .map_err(|x| Error::RequestFailed { inner: x })?;
    if result.status().is_success() {
        let ok_result = result
            .json::<ElasticReply>()
            .map_err(|x| Error::DeserializationFailed { inner: x })?;
        let mut errors = Vec::<String>::new();
        for index in ok_result.items.into_iter() {
            if let Some(some_error) = index.index.error {
                errors.push(format!(
                    "[{}]: {}",
                    some_error.caused_by.r#type, some_error.caused_by.reason
                ));
            }
        }
        if errors.len() > 0 {
            return Err(Error::SomeLogsWereNotAccepted {
                errors: errors.join("\n"),
            });
        }
        return Ok(());
    }

    let err_result = result.json::<ElasticErrorReply>();
    if let Ok(some_err_result) = err_result {
        return Err(Error::ApiRejectedLogPayload {
            errors: json!(some_err_result).to_string(),
        });
    } else {
        return Err(Error::ApiRejectedLogPayload {
            errors: format!("[Unhandled Error]: Payload {}", json),
        });
    }
}

fn serialize_to_chunks(log_chunks: Vec<(LogIndex, LogPayload)>) -> String {
    let mut chunks = Vec::<String>::new();
    for log_chunk in log_chunks.into_iter() {
        chunks.push(json!(log_chunk.0).to_string());
        chunks.push(json!(log_chunk.1).to_string());
    }
    // Elastic requires the payload to end with a newline
    chunks.push("\n".to_string());
    let json = chunks.join("\n");
    json
}

fn log_to_elastic(
    client: reqwest::blocking::Client,
    elastic_credentials: ElasticConfig,
    log_chunks: Vec<(LogIndex, LogPayload)>,
) {
    let json = serialize_to_chunks(log_chunks);
    println!("{}", json.to_owned());
    let result = post_to_elastic(client, elastic_credentials, json);
    if let Err(result_err) = result {
        eprintln!("error: {}", result_err)
    }
}

/// Sets up the log service. Call this before calling the log::<severity> functions.
/// To prevent loss of logs on application shutdown, make sure call log::flush() before the fact.
///
/// # Examples
///
/// ```
///
///
/// let my_config = relastic::log::ElasticConfig {
///        url: "https://localhost".to_string(),
///        username: "test_user".to_string(),
///        password: "test_password".to_string(),
///        environment: relastic::log::LogEnvironment::Production,
///        application_name: "relastic-tests".to_string(),
/// };
///
/// relastic::log::setup_elastic_log(my_config, 100);
///
/// relastic::log::information("This is a test: {text}", std::collections::HashMap::from([("text", "Hello World".to_string())]));
///
/// relastic::log::flush();
///
/// ```
pub fn setup_elastic_log(config: ElasticConfig, buffer_size: usize) -> () {
    let client = reqwest::blocking::Client::new();
    let (tx, rx) = mpsc::sync_channel::<LogMessage>(buffer_size);
    {
        let mut logger = LOG.write().unwrap();
        *logger = Some(tx);
    }
    let app_name_lowercase = config.application_name.to_lowercase();
    let env_str_lowercase = (&config.environment).to_owned().to_string().to_lowercase();
    let default_fields = HashMap::from([
        ("ApplicationName".to_string(), app_name_lowercase.to_owned()),
        (
            "HostingEnvironment".to_string(),
            env_str_lowercase.to_owned(),
        ),
    ]);

    SHOULD_LOOP.store(true, std::sync::atomic::Ordering::SeqCst);

    thread::spawn(move || loop {
        let mut payload_chunks = Vec::<(LogIndex, LogPayload)>::new();
        let next_logs = rx.recv_timeout(WAIT);
        if next_logs.is_err() {
            // No logs to flush
            if !SHOULD_LOOP.load(std::sync::atomic::Ordering::SeqCst) {
                IS_FLUSHED.store(true, std::sync::atomic::Ordering::SeqCst);
                return;
            }
            continue;
        }
        for recevied in next_logs.into_iter() {
            let dt = Utc::now();
            let str_index: String = format!(
                "{}-{}-{}.{}",
                app_name_lowercase,
                env_str_lowercase,
                dt.date().year(),
                dt.date().month()
            );
            let index = LogIndex {
                index: LogInnerIndex {
                    index: str_index,
                    log_type: "_doc".to_string(),
                },
            };
            let log_payload = LogPayload {
                timestamp: dt.to_rfc3339_opts(SecondsFormat::Nanos, true),
                level: recevied.level.to_string(),
                message_template: recevied.message_template,
                message: recevied.message,
                fields: concat_fields(&default_fields, recevied.fields),
            };
            log_to_console(&log_payload);

            // Skip logging debug to elastic if env is production
            if recevied.level == LogLevel::Debug && config.environment == LogEnvironment::Production
            {
                continue;
            }
            payload_chunks.push((index, log_payload));
        }
        if payload_chunks.len() > 0 {
            log_to_elastic(client.to_owned(), config.clone(), payload_chunks);
        }
        if !SHOULD_LOOP.load(std::sync::atomic::Ordering::SeqCst) {
            IS_FLUSHED.store(true, std::sync::atomic::Ordering::SeqCst);
            return;
        }
    });
}
