//! complex number library

use std::fmt;
use std::ops::Add;
use std::ops::Sub;
use std::ops::Neg;

/// complex number definition.
#[derive(Clone, Copy, Debug, PartialEq)]
pub struct Complex(pub i64, pub i64);

// According to https://doc.rust-lang.org/std/fmt/trait.Display.html
// Display is similar to Debug, but Display is for user-facing output, and so cannot be derived.
impl fmt::Display for Complex {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "({}, {}i)", self.0, self.1)
    }
}

impl Add for Complex {
    type Output = Self;

    fn add(self, other: Self) -> Self {
        Self(
            self.0 + other.0,
            self.1 + other.1,
        )
    }
}

impl Sub for Complex {
    type Output = Self;

    fn sub(self, other: Self) -> Self {
        Self(
            self.0 - other.0,
            self.1 - other.1,
        )
    }
}

impl Neg for Complex {
    type Output = Self;

    fn neg(self) -> Self {
        ZERO - self
    }
}

/// the complex zero: (0, 0)
pub const ZERO: Complex = Complex(0, 0);

/// the conjugate of a+bi is a-bi
pub fn conjugate(c: Complex) -> Complex {
    Complex(c.0, -c.1)
}

impl From<(i32, i32)> for Complex {
    fn from(item: (i32, i32)) -> Self {
        Complex(item.0.into(), item.1.into())
    }
}

impl From<(i64, i64)> for Complex {
    fn from(item: (i64, i64)) -> Self {
        Complex(item.0, item.1)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn it_works() {
        let result = 2 + 2;
        assert_eq!(result, 4);
    }

    #[test]
    fn four_quadrant() {
        let _ = vec![
            Complex(1, 1),
            Complex(-1, 1),
            Complex(-1, -1),
            Complex(1, -1),
        ];
    }

    #[test]
    fn add_works() {
        assert_eq!(Complex(1, 2) + Complex(3, 4), Complex(4, 6));
    }

    #[test]
    fn sub_works() {
        assert_eq!(Complex(1, 2) - Complex(3, 4), Complex(-2, -2));
    }

    #[test]
    fn neg_works() {
        assert_eq!(-Complex(-1, 1), Complex(1, -1));
    }

    #[test]
    fn zero_works() {
        assert_eq!(-ZERO, ZERO);
        assert_eq!(ZERO+ZERO, ZERO);
        assert_eq!(ZERO-ZERO, ZERO);
    }
}
