use crate::codepointset::CodePointSet;
use crate::position::PositionType;
use std::ops;

/// A group index is u16.
/// CaptureGroupID 0 corresponds to the first capture group.
pub type CaptureGroupID = u16;

/// The maximum number of capture groups supported.
pub const MAX_CAPTURE_GROUPS: usize = 65535;

/// The maximum number of loops supported.
pub const MAX_LOOPS: usize = 65535;

#[derive(Debug, Copy, Clone, PartialEq, Eq)]
pub enum CharacterClassType {
    Digits,
    Spaces,
    Words,
}

/// The stuff in a bracket.
#[derive(Debug, Clone, Default)]
pub struct BracketContents {
    pub invert: bool,
    pub cps: CodePointSet,
}

/// An instruction pointer.
pub type IP = usize;

/// Representation of a loop.
#[derive(Debug, Copy, Clone)]
pub struct LoopData<Position: PositionType> {
    pub iters: usize,
    pub entry: Position,
}

impl<Position: PositionType> LoopData<Position> {
    pub fn new(entry: Position) -> LoopData<Position> {
        LoopData { iters: 0, entry }
    }
}

/// Representation of a capture group.
#[derive(Debug, Copy, Clone)]
pub struct GroupData<Position: PositionType> {
    pub start: Option<Position>,
    pub end: Option<Position>,
}

impl<Position: PositionType> GroupData<Position> {
    pub fn new() -> GroupData<Position> {
        GroupData {
            start: None,
            end: None,
        }
    }

    pub fn start_matched(&self) -> bool {
        self.start.is_some()
    }

    pub fn end_matched(&self) -> bool {
        self.end.is_some()
    }

    pub fn as_range(&self) -> Option<ops::Range<Position>> {
        // Note: we may have only start_matched (if forwards) or end_matched (if
        // backwards) set.
        match (self.start, self.end) {
            (Some(start), Some(end)) => Some(ops::Range { start, end }),
            _ => None,
        }
    }

    /// Reset the group to "not entered."
    pub fn reset(&mut self) {
        self.start = None;
        self.end = None;
    }
}
