//! A small library for creating regex-based lexers
//! # Example
//!
//! ```rust
//!
//! use reglex::{RuleList, rule_list, lex};
//!
//! #[derive(Debug, PartialEq)]
//! enum Token {
//!     Keyword,
//!     Number(u64),
//!     Left,
//!     Right,
//! }
//!
//! fn lexer(input: &String) -> Result<Vec<Token>, usize> {
//!     let regexes: RuleList<Token> = rule_list! [
//!         "kw" => |_| Some(Token::Keyword),
//!         r"\d+" => |s: &str| Some(Token::Number(s.parse().unwrap())),
//!         r"\{" => |_| Some(Token::Left),
//!         r"\}" => |_| Some(Token::Right),
//!         r"\s" => |_| None
//!     ];
//!
//!     lex(&regexes, input)
//! }
//!
//! fn main() {
//!     assert_eq!(
//!         lexer(&"kw  { 12 53 }".to_string()),
//!         Ok(vec![
//!             Token::Keyword,
//!             Token::Left,
//!             Token::Number(12),
//!             Token::Number(53),
//!             Token::Right
//!         ])
//!     );
//!
//!     assert_eq!(lexer(&"kw ERROR! { 12 53 }".to_string()), Err(3));
//! }
//! ```
#![deny(clippy::all, clippy::pedantic, missing_docs)]

pub use regex;

/// A list of lexing rules. Can be generated by the `rule_list` macro.
pub type RuleList<Token> = Vec<(regex::Regex, Box<dyn Fn(&str) -> Option<Token>>)>;

/// Generates a `RuleList`.
///
/// The macro takes in the list of regexes and parsing functions, using `=>`.
///
/// # Panics
///
/// Panics on an invalid regex.
///
#[macro_export]
macro_rules! rule_list {
    ($($pat:expr => $fn:expr), *) => {
        {
            vec![
                $(($crate::regex::Regex::new($pat).unwrap(), Box::new($fn)), )*
            ]
        }
    };
}

/// The lexing function.
///
/// Takes in the list of lexing rules (which can be generated by `regex_list`) and the input
/// `String`.
///
/// Returns a `Result`:
/// - `Ok`: The vector of tokens
/// - `Err`: The position of the error
pub fn lex<Token>(rules: &RuleList<Token>, s: &String) -> Result<Vec<Token>, usize> {
    let mut pos = 0; // The position the lexer is at
    let mut output: Vec<Token> = Vec::new(); // The list of tokens

    while pos < s.len() { // Main loop
        let mut no_token_found = true; // Is there an error? (Mathes no rules)

        for (pat, f) in rules { // Loop through rules
            if let Some(m) = pat.find_at(&s, pos) { // The rule has a match
                if m.start() == pos { // The match is at the current position
                    no_token_found = false;

                    match f(m.as_str()) { 
                        Some(t) => { // Add token
                            output.push(t);
                            pos = m.end();
                            break;
                        }
                        None => { // No token
                            pos = m.end();
                            break;
                        }
                    }
                }
            }
        }

        if no_token_found {
            return Err(pos);
        }
    }

    Ok(output)
}
