#![allow(clippy::pedantic)]
#![allow(clippy::cargo)]
//! # Prime Regex
//! `regex_prime` is a crate that hosts a singular function to test whether a number is a prime.

// regular regex crate does not support backtracking
use fancy_regex::Regex;

/// Tests if a positive integer is a prime or not
///
/// # Notes
/// Based on <https://www.noulakaz.net/2007/03/18/a-regular-expression-to-check-for-prime-numbers/>
///
/// Extremely inefficient
///
/// # Examples
///
/// ```
/// use prime::is_prime;
///
/// let number = 11;
/// assert_eq!(is_prime(number), true);
/// ```
///
/// # Panics
///
/// Panics if the regex cannot be created
/// Panics if number is extremely large and or many times bigger than the recursion limit
pub fn is_prime(number: usize) -> Result<bool, fancy_regex::Error> {
    let re = Regex::new(r"^1?$|^(11+?)\1+$").unwrap();
    let possible_prime_match = "1".repeat(number);

    match re.is_match(&possible_prime_match) {
        Ok(is_prime_bool) => Ok(!is_prime_bool),
        Err(err) => Err(err),
    }
}

#[cfg(test)]
mod tests {
    use crate::is_prime;

    #[test]
    fn blog_examples() {
        assert!(!is_prime(10).unwrap(), "10 is not a prime");
        assert!(is_prime(11).unwrap());
        assert!(!is_prime(12).unwrap(), "12 is not a prime");
        assert!(is_prime(13).unwrap());
        assert!(!is_prime(99).unwrap(), "99 is not a prime");
        assert!(!is_prime(100).unwrap(), "100 is not a prime");
        assert!(is_prime(101).unwrap());
    }

    #[test]
    fn psuedoprimes() {
        let psuedoprimes = vec![
            9, 15, 21, 25, 27, 33, 35, 39, 45, 49, 51, 55, 57, 63, 65, 69,
        ];

        assert_eq!(
            psuedoprimes
                .iter()
                .map(|x| is_prime(*x).unwrap())
                .collect::<Vec<bool>>(),
            vec![false; psuedoprimes.len()]
        )
    }

    #[test]
    fn large_prime() {
        // shows how slow this matching process is
        let large_prime: usize = 3010349;
        let large_non_prime: usize = 3010350;
        assert!(is_prime(large_prime).is_err(), "Stack Overflow");
        assert!(is_prime(large_non_prime).is_err(), "Stack Overflow")
    }

    #[test]
    fn prime_loop() {
        for n in 0..10000 {
            match is_prime(n) {
                Ok(bool) => match bool {
                    true => println!("{} is a prime", n),
                    false => println!("{} is not a prime", n),
                },
                Err(_) => {
                    println!("Limit is: {}", n);
                    break;
                }
            }
        }
    }
}
