#![warn(missing_docs)]

//! A small crate to enable equality checks through pointer comparison

use std::fmt;
use std::hash::{Hash, Hasher};
use std::mem::size_of_val;
use std::ops::Deref;

/// Newtype for references (or smart pointers) where [equality](Eq) is defined
/// by the address (or address range) pointed to
///
/// # Caveats / Notes
///
/// * Empty address ranges are always considered equal, disregarding different base addresses (and Rust's rules on wide-pointer comparison).
/// * Working with trait objects may cause surprising behavior due to Rust's
/// pointer comparison rules.
///
/// # Example
/// ```
/// use refid::RefId;
/// use std::collections::HashSet;
/// fn count_distinct<T>(slice: &[&T]) -> usize {
///     let mut counted: HashSet<RefId<&T>> = HashSet::new();
///     let mut count: usize = 0;
///     for item in slice {
///         if counted.insert(RefId(item)) {
///             count += 1;
///         }
///     }
///     count
/// }
/// let v1 = "X".to_string();
/// let v2 = "X".to_string();
/// let v3 = "X".to_string();
/// assert_eq!(count_distinct(&vec![&v1, &v2, &v1, &v3]), 3);
/// ```
#[derive(Clone, Copy)]
pub struct RefId<T: ?Sized + Deref>(
    /// inner value (reference or smart pointer)
    pub T,
);

/// Comparison by address (but equal if range pointed to has size of zero)
impl<T: ?Sized + Deref> PartialEq for RefId<T> {
    fn eq(&self, other: &Self) -> bool {
        (size_of_val(&*self.0) == 0 && size_of_val(&*other.0) == 0)
            || (&*self.0 as *const T::Target == &*other.0 as *const T::Target)
    }
}

/// Comparison by address (but equal if range pointed to has size of zero)
impl<T: ?Sized + Deref> Eq for RefId<T> {}

impl<T: ?Sized + Deref> Hash for RefId<T> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        if size_of_val(&*self.0) != 0 {
            (&*self.0 as *const T::Target).hash(state)
        }
    }
}

/// Debug representation appended with "`@`" and address
impl<T: ?Sized + Deref> fmt::Debug for RefId<T>
where
    T::Target: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{:?}@{:?}", &*self.0, &*self.0 as *const T::Target)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn test_two_refs_to_same_value() {
        let v = 1;
        let r1 = &v;
        let r2 = &v;
        assert_eq!(RefId(r1), RefId(r2));
    }
    #[test]
    fn test_two_boxes() {
        #[derive(Debug)]
        struct S {
            _value: i32,
        }
        let v1 = Box::new(S { _value: 1 });
        let v2 = Box::new(S { _value: 2 });
        let r1: &S = &v1;
        let r2: &S = &v2;
        assert_eq!(RefId(r1), RefId(r1));
        assert_ne!(RefId(r1), RefId(r2));
    }
    #[test]
    fn test_unit_type() {
        let v1 = ();
        let v2 = ();
        let r1: &() = &v1;
        let r2: &() = &v2;
        assert_eq!(RefId(r1), RefId(r2));
    }
    #[test]
    fn test_empty_slice() {
        let v = vec![5, 6, 7];
        let r1: &[i32] = &v[0..0];
        let r2: &[i32] = &v[1..1];
        assert_eq!(RefId(r1), RefId(r2));
    }
    #[test]
    fn test_nonempty_slices() {
        let v1 = vec![5, 6, 7];
        let v2 = vec![5, 6, 7];
        let r1: &[i32] = &v1[0..1];
        let r2: &[i32] = &v1[0..1];
        let r3: &[i32] = &v1[0..2];
        let r4: &[i32] = &v2[0..1];
        assert_eq!(RefId(r1), RefId(r1));
        assert_eq!(RefId(r1), RefId(r2));
        assert_ne!(RefId(r1), RefId(r3));
        assert_ne!(RefId(r1), RefId(r4));
    }
    #[test]
    fn test_rc() {
        use std::rc::Rc;
        let v1 = vec![9];
        let v2 = vec![9];
        let r1 = Rc::new(v1);
        let r2 = Rc::new(v2);
        let r3 = r1.clone();
        assert_ne!(RefId(r1.clone()), RefId(r2.clone()));
        assert_eq!(RefId(r1.clone()), RefId(r3.clone()));
    }
    #[test]
    fn test_hash_set() {
        use std::collections::HashSet;
        fn count_distinct<T>(slice: &[&T]) -> usize {
            let mut counted: HashSet<RefId<&T>> = HashSet::new();
            let mut count: usize = 0;
            for item in slice {
                if counted.insert(RefId(item)) {
                    count += 1;
                }
            }
            count
        }
        let v1 = "X".to_string();
        let v2 = "X".to_string();
        let v3 = "X".to_string();
        assert_eq!(count_distinct(&vec![&v1, &v2, &v1, &v3]), 3);
    }
}
