use clap::{App, Arg, ArgSettings, crate_version};
use refi::LogLevel;

fn main() {
    let matches = App::new("refi")
        .version(crate_version!())
        .author("Miika Launiainen <miika.launiainen@protonmail.com>")
        .about("refi is a command line tool to rename files in numberic order.")
        .arg(
            Arg::new("folder")
                .help("Path to the folder of files")
                .required(true),
        )        
        .arg(
            Arg::new("verbose")
                .short('v')
                .help("Adds verbose logging")
                .required(false)
                .takes_value(false)
                .conflicts_with("quiet"),
        )
        .arg(
            Arg::new("quiet")
                .short('q')
                .help("Removes non-error logging")
                .required(false)
                .takes_value(false)
                .conflicts_with("verbose"),
        )
        .arg(
            Arg::new("logfile")
                .short('l')
                .long("logfile")
                .help("Save logfile [rename.log]"),
        )
        .arg(Arg::new("yes").short('y').help("Automatic yes to prompts"))
        .arg(
            Arg::new("nro")
                .short('n')
                .long("nro")
                .takes_value(true)
                .help("Define custom number to start counting from"),
        )
        .arg(
            Arg::new("zeroes")
                .short('z')
                .long("zeroes")
                .takes_value(true)
                .help("Define custom number of zeroes before number"),
        )
        .arg(
            Arg::new("prefix")
                .short('p')
                .long("prefix")
                .takes_value(true)
                .help("Define prefix"),
        )
        .arg(
            Arg::new("mode")
                .short('m')
                .long("mode")
                .takes_value(true)
                .possible_values(&["n", "f"])
                .help("Choose what renaming mode to use")
                .long_help("Choose what renaming mode to use.\nn: normal (default)\nf: fixing missing numbers\n"),
        )
        .arg(
            Arg::new("ordering")
                .short('o')
                .long("order")
                .takes_value(true)
                .possible_values(&["a", "z", "n", "o"])
                .help("Choose ordering for the files")
                .long_help("Choose ordering for the files.\na: A-Z (default)\nz: Z-A\nn: Newest to oldest\no: Oldest to newest\n"),
        )
        .arg(
            Arg::new("ignore")
                .long("ignore")
                .takes_value(true)
                .setting(ArgSettings::MultipleValues)
                .help("Give a list of missing numbers to ignore")
        )
        .get_matches();

    let log_level = if matches.is_present("verbose") {
        LogLevel::Verbose
    } else if matches.is_present("quiet") {
        LogLevel::Quiet
    } else {
        LogLevel::Standard
    };

    refi::rename(matches.into(), log_level);
}
