// Copyright 2021 Koichi Kitahara
//
// Licensed under either of Apache License, Version 2.0:
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
// or MIT license:
//
//   Permission is hereby granted, free of charge, to any person obtaining a
//   copy of this software and associated documentation files (the "Software"),
//   to deal in the Software without restriction, including without limitation
//   the rights to use, copy, modify, merge, publish, distribute, sublicense,
//   and/or sell copies of the Software, and to permit persons to whom the
//   Software is furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in
//   all copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
//   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
//   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
//   DEALINGS IN THE SOFTWARE.
//
// at your option.

use core::ops::{Neg, Not};

macro_rules! doc {
    ($( $x:expr, )* @$item:item) => {
        $( #[doc = $x] )*
        $item
    };
}

macro_rules! def {
    ($Op:ident, $RefOp:ident, $op:ident, $ref_op:ident, $assert:expr) => {
        doc!(
            "An escape hatch for implimenting",
            concat!("`", stringify!($Op), "`"),
            "for references to newtypes.",
            "",
            "As of Rust 1.52.1, the following code does not compile:",
            "```compile_fail",
            concat!("use core::ops::", stringify!($Op), ";"),
            "",
            "#[derive(PartialEq)]",
            "struct A<T>(T);",
            "",
            concat!("impl<'a, T> ", stringify!($Op), " for &'a A<T>"),
            "where",
            concat!("    T: ", stringify!($Op), ","),
            concat!("    &'a T: ", stringify!($Op), "<Output = T::Output>,"),
            "{",
            concat!("    type Output = A<T::Output>;"),
            "",
            concat!("    fn ", stringify!($op), "(self) -> Self::Output {"),
            concat!("        A(self.0.", stringify!($op), "())"),
            "    }",
            "}",
            "",
            "fn f<T>(a: T)",
            "where",
            concat!("    for<'a> &'a T: ", stringify!($Op), ","),
            "{",
            concat!("    let ", stringify!($op), "_a = (&a).", stringify!($op), "();"),
            "",
            concat!("    // to do something with `a` and `", stringify!($op), "_a`"),
            "    todo!();",
            "}",
            "",
            "fn g<T>(a: T)",
            "where",
            concat!("    for<'a> &'a T: ", stringify!($Op), ","),
            "{",
            "    f(a);",
            "}",
            "",
            concat!("assert!(", stringify!($assert), ");"),
            "```",
            "but the following code does:",
            "```",
            concat!("use core::ops::", stringify!($Op), ";"),
            concat!("use ref_ops::", stringify!($RefOp),";"),
            "",
            "#[derive(PartialEq)]",
            "struct A<T>(T);",
            "",
            concat!("impl<T> ", stringify!($Op), " for &A<T>"),
            "where",
            concat!("    T: ", stringify!($RefOp), ","),
            "{",
            "    type Output = A<T::Output>;",
            "",
            concat!("    fn ", stringify!($op), "(self) -> Self::Output {"),
            concat!("        A(self.0.", stringify!($ref_op), "())"),
            "    }",
            "}",
            "",
            "fn f<T>(a: T)",
            "where",
            concat!("    for<'a> &'a T: ", stringify!($Op), ","),
            "{",
            concat!("    let ", stringify!($op), "_a = (&a).", stringify!($op), "();"),
            "",
            concat!("    // to do something with `a` and `", stringify!($op), "_a`"),
            "    todo!();",
            "}",
            "",
            "fn g<T>(a: T)",
            "where",
            concat!("    for<'a> &'a T: ", stringify!($Op), ","),
            "{",
            "    f(a);",
            "}",
            "",
            concat!("assert!(", stringify!($assert), ");"),
            "```",
            @pub trait $RefOp: $Op {
                fn $ref_op(&self) -> Self::Output;
            }
        );

        impl<T> $RefOp for T
        where
            Self: $Op,
            for<'a> &'a Self: $Op<Output = Self::Output>,
        {
            fn $ref_op(&self) -> Self::Output {
                self.$op()
            }
        }
    };
}

def!(Neg, RefNeg, neg, ref_neg, -&A(1.0) == A(-1.0));
def!(Not, RefNot, not, ref_not, !&A(true) == A(false));
