// Copyright 2021 Koichi Kitahara
//
// Licensed under either of Apache License, Version 2.0:
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
// or MIT license:
//
//   Permission is hereby granted, free of charge, to any person obtaining a
//   copy of this software and associated documentation files (the "Software"),
//   to deal in the Software without restriction, including without limitation
//   the rights to use, copy, modify, merge, publish, distribute, sublicense,
//   and/or sell copies of the Software, and to permit persons to whom the
//   Software is furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in
//   all copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
//   THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
//   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
//   DEALINGS IN THE SOFTWARE.
//
// at your option.

use core::ops::{Add, BitAnd, BitOr, BitXor, Div, Mul, Rem, Shl, Shr, Sub};

macro_rules! doc {
    ($( $x:expr, )* @$item:item) => {
        $( #[doc = $x] )*
        $item
    };
}

macro_rules! def {
    ($Op:ident, $RefOp:ident, $op:ident, $ref_op:ident, $assert:expr) => {
        doc!(
            "An escape hatch for implimenting",
            concat!("`", stringify!($Op), "`"),
            "for references to newtypes.",
            "",
            "As of Rust 1.52.1, the following code does not compile:",
            "```compile_fail",
            concat!("use core::ops::", stringify!($Op), ";"),
            "",
            "#[derive(PartialEq)]",
            "struct A<T>(T);",
            "",
            concat!("impl<'a, 'b, T, U> ", stringify!($Op), "<&'b A<U>> for &'a A<T>"),
            "where",
            concat!("    T: ", stringify!($Op), "<&'b U>,"),
            concat!("    &'a T: ", stringify!($Op), "<&'b U, Output = T::Output>,"),
            "{",
            concat!("    type Output = A<T::Output>;"),
            "",
            concat!("    fn ", stringify!($op), "(self, other: &'b A<U>) -> Self::Output {"),
            concat!("        A(self.0.", stringify!($op), "(&other.0))"),
            "    }",
            "}",
            "",
            "pub fn f<T, U>(a: T, b: U)",
            "where",
            concat!("    for<'a, 'b> &'a T: ", stringify!($Op), "<&'b U>,"),
            "{",
            concat!("    let a_b = (&a).", stringify!($op), "(&b);"),
            "",
            concat!("    // to do something with `a`, `b`, and `a_b`"),
            "    todo!();",
            "}",
            "",
            "pub fn g<T, U>(a: T, b: U)",
            "where",
            concat!("    for<'a, 'b> &'a T: ", stringify!($Op), "<&'b U>,"),
            "{",
            "    f(a, b);",
            "}",
            "",
            concat!("assert!(", stringify!($assert), ");"),
            "```",
            "but the following code does:",
            "```",
            concat!("use core::ops::", stringify!($Op), ";"),
            concat!("use ref_ops::", stringify!($RefOp),";"),
            "",
            "#[derive(PartialEq)]",
            "struct A<T>(T);",
            "",
            concat!("impl<'a, T, U> ", stringify!($Op), "<&'a A<U>> for &A<T>"),
            "where",
            concat!("    T: ", stringify!($RefOp), "<&'a U>,"),
            "{",
            "    type Output = A<T::Output>;",
            "",
            concat!("    fn ", stringify!($op), "(self, other: &'a A<U>) -> Self::Output {"),
            concat!("        A(self.0.", stringify!($ref_op), "(&other.0))"),
            "    }",
            "}",
            "",
            "pub fn f<T, U>(a: T, b: U)",
            "where",
            concat!("    for<'a, 'b> &'a T: ", stringify!($Op), "<&'b U>,"),
            "{",
            concat!("    let a_b = (&a).", stringify!($op), "(&b);"),
            "",
            concat!("    // to do something with `a`, `b`, and `a_b`"),
            "    todo!();",
            "}",
            "",
            "pub fn g<T, U>(a: T, b: U)",
            "where",
            concat!("    for<'a, 'b> &'a T: ", stringify!($Op), "<&'b U>,"),
            "{",
            "    f(a, b);",
            "}",
            "",
            concat!("assert!(", stringify!($assert), ");"),
            "```",
            @pub trait $RefOp<Rhs>: $Op<Rhs> {
                fn $ref_op(&self, other: Rhs) -> Self::Output;
            }
        );

        impl<T, Rhs> $RefOp<Rhs> for T
        where
            Self: $Op<Rhs>,
            for<'a> &'a Self: $Op<Rhs, Output = Self::Output>,
        {
            fn $ref_op(&self, other: Rhs) -> Self::Output {
                self.$op(other)
            }
        }
    };
}

def!(Add, RefAdd, add, ref_add, &A(1.0) + &A(2.0) == A(3.0));
def!(BitAnd, RefBitAnd, bitand, ref_bitand, &A(6) & &A(5) == A(4));
def!(BitOr, RefBitOr, bitor, ref_bitor, &A(3) | &A(5) == A(7));
def!(BitXor, RefBitXor, bitxor, ref_bitxor, &A(3) ^ &A(5) == A(6));
def!(Div, RefDiv, div, ref_div, &A(6.0) / &A(2.0) == A(3.0));
def!(Mul, RefMul, mul, ref_mul, &A(2.0) * &A(3.0) == A(6.0));
def!(Rem, RefRem, rem, ref_rem, &A(6.0) % &A(4.0) == A(2.0));
def!(Shl, RefShl, shl, ref_shl, &A(3) << &A(2) == A(12));
def!(Shr, RefShr, shr, ref_shr, &A(12) >> &A(2) == A(3));
def!(Sub, RefSub, sub, ref_sub, &A(3.0) - &A(1.0) == A(2.0));
