use crate::Status;
use anyhow::{anyhow, Context, Result};
use chrono::{DateTime, Utc};
use colored::*;
use serde::{Deserialize, Serialize};
use std::fmt::{Debug, Display, Formatter};
use std::path::{Path, PathBuf};
use std::time::{Duration, Instant};
use wait_timeout::ChildExt;

#[derive(Clone, Default, Serialize, Deserialize)]
pub struct Command {
    name: String,
    args: Vec<String>,
    exe_filename: PathBuf,
    working_dir: Option<PathBuf>,
    timeout: Option<Duration>,
    timed_out: bool,
    stdout: Vec<u8>,
    stderr: Vec<u8>,
    exit_code: Option<i32>,
    duration: Option<Duration>,
    start: Option<DateTime<Utc>>,
}

impl Command {
    pub fn new(name: &str) -> Self {
        let mut d = Command::default();
        d.name = name.to_string();
        d
    }

    pub fn arg(&mut self, arg: &str) -> &mut Command {
        self.args.push(arg.to_string());
        self
    }

    pub fn current_dir<P: AsRef<Path>>(&mut self, dir: P) -> &mut Command {
        self.working_dir = Some(dir.as_ref().to_path_buf());
        self
    }

    pub fn timeout(&mut self, timeout: Duration) -> &mut Command {
        self.timeout = Some(timeout);
        self
    }

    fn output(&mut self) -> Result<std::process::Output> {
        match self.timeout {
            Some(timeout) => {
                let mut cmd = std::process::Command::new(&self.exe_filename);
                for arg in &self.args {
                    cmd.arg(&arg);
                }
                match &self.working_dir {
                    Some(dir) => {
                        cmd.current_dir(dir);
                    }
                    None => {}
                };
                let mut child = cmd
                    .spawn()
                    .with_context(|| format!("spawning {} {}", &self.name, &self.args.join(" ")))?;
                match child.wait_timeout(timeout)? {
                    Some(_) => {}
                    None => {
                        child.kill()?;
                        child.wait()?;
                        self.timed_out = true;
                    }
                };
                Ok(child.wait_with_output()?)
            }
            None => {
                let mut cmd = std::process::Command::new(&self.exe_filename);
                for arg in &self.args {
                    cmd.arg(&arg);
                }
                match &self.working_dir {
                    Some(dir) => {
                        cmd.current_dir(dir);
                    }
                    None => {}
                };
                Ok(cmd
                    .output()
                    .with_context(|| format!("spawning {} {}", &self.name, &self.args.join(" ")))?)
            }
        }
    }

    pub fn exec(&mut self) -> Result<Command> {
        let start_instant = Instant::now();
        let start = Utc::now();
        let exe_filename = Command::which(&self.name)?;
        if exe_filename.exists() {
            self.exe_filename = exe_filename.to_path_buf();
            let output = self.output()?;
            if self.timed_out {}
            Ok(Command {
                exe_filename: exe_filename,
                working_dir: match &self.working_dir {
                    Some(dir) => Some(dir.to_path_buf()),
                    None => None,
                },
                name: self.name.to_string(),
                args: self.args.clone(),
                stdout: output.stdout.clone(),
                stderr: output.stderr.clone(),
                exit_code: output.status.code(),
                duration: Some(start_instant.elapsed()),
                timeout: self.timeout,
                timed_out: self.timed_out,
                start: Some(start),
            })
        } else {
            Err(anyhow!(
                "unable to determine executable filename for {}",
                &self.name
            ))
        }
    }

    pub fn name(&self) -> &str {
        &self.name
    }
    pub fn exit_code(&self) -> Option<i32> {
        self.exit_code
    }
    pub fn start(&self) -> &Option<DateTime<Utc>> {
        &self.start
    }
    pub fn duration(&self) -> Option<Duration> {
        self.duration
    }
    pub fn stdout(&self) -> &Vec<u8> {
        &self.stdout
    }
    pub fn stderr(&self) -> &Vec<u8> {
        &self.stderr
    }

    pub fn text(&self) -> Result<String> {
        Ok(format!(
            "{}\n{}",
            std::str::from_utf8(&self.stdout)?,
            std::str::from_utf8(&self.stderr)?
        )
        .trim()
        .to_string())
    }

    pub fn status(&self) -> Status {
        match self.exit_code() {
            Some(exit_code) => {
                if exit_code == 0 {
                    Status::Ok
                } else {
                    if self.timed_out {
                        Status::TimedOut
                    } else {
                        Status::Error
                    }
                }
            }
            None => Status::Unknown,
        }
    }

    fn which(name: &str) -> Result<PathBuf> {
        let extensions = vec![".exe", ".bat", ".cmd", ""];
        for ext in extensions.iter() {
            let exe_name = format!("{}{}", name, ext);
            match Command::which_exact(&exe_name) {
                Some(path) => {
                    return Ok(path);
                }
                None => {}
            };
        }
        Err(anyhow!("unrecognized command '{}'", name))
    }

    fn which_exact(name: &str) -> Option<PathBuf> {
        std::env::var_os("PATH").and_then(|paths| {
            std::env::split_paths(&paths)
                .filter_map(|dir| {
                    let full_path = dir.join(&name);
                    if full_path.is_file() {
                        Some(full_path)
                    } else {
                        None
                    }
                })
                .next()
        })
    }
}

impl Display for Command {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        write!(
            f,
            "{} {} {}",
            &self.status(),
            &self.name(),
            &self.args.join(" ")
        )
    }
}

impl Debug for Command {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        writeln!(f, "{}", self)?;
        write!(f, "\n{:>15}: {}", "name", &self.name().green())?;
        if self.args.len() > 0 {
            write!(f, "\n{:>15}: {}", "args", &self.args.join(" ").green())?;
        }
        write!(f, "\n{:>15}: {:#?}", "exe filename", &self.exe_filename)?;
        match &self.working_dir {
            Some(dir) => {
                write!(f, "\n{:>15}: {}", "dir", dir.display())?;
            }
            None => {}
        };
        match &self.timeout {
            Some(timeout) => {
                write!(f, "\n{:>15}: {:#?}", "timeout", timeout)?;
                write!(f, "\n{:>15}: {}", "timed out", self.timed_out)?;
            }
            None => {}
        }
        match &self.exit_code {
            Some(exit_code) => {
                write!(f, "\n{:>15}: {}", "exit code", exit_code)?;
            }
            None => {}
        }
        match &self.start() {
            Some(dt) => {
                write!(f, "\n{:>15}: {}", "start", dt)?;
            }
            None => {}
        }
        match &self.duration() {
            Some(duration) => {
                write!(f, "\n{:>15}: {:#?}", "duration", duration)?;
            }
            None => {}
        }
        let text = match self.text() {
            Ok(t) => t,
            Err(_) => format!("[{} bytes]", self.stdout.len() + self.stderr.len()),
        };
        write!(
            f,
            "\n{:>15}: {}",
            "output",
            text.split("\n")
                .collect::<Vec<&str>>()
                .join("\n                 ")
                .trim()
        )
    }
}
