use crate::Result;
use colored::Colorize;
use fern::Dispatch;
use log::{Level, LevelFilter};
use std::collections::HashMap;

pub struct Log {
    level: Level,
    filters: HashMap<String, LevelFilter>,
    format_fn: fn(&std::fmt::Arguments, &log::Record) -> String,
}

impl Log {
    pub fn new() -> Self {
        Log {
            level: Level::Info,
            filters: HashMap::new(),
            format_fn: Log::level_format,
        }
    }

    pub fn level(mut self, level: Level) -> Log {
        self.level = level;
        self
    }

    pub fn level_for(mut self, target: &str, level: Level) -> Log {
        let level_filter = match level {
            Level::Trace => LevelFilter::Trace,
            Level::Debug => LevelFilter::Debug,
            Level::Info => LevelFilter::Info,
            Level::Warn => LevelFilter::Warn,
            Level::Error => LevelFilter::Error,
        };
        self.filters.insert(target.to_string(), level_filter);
        self
    }

    pub fn format(mut self, f: fn(&std::fmt::Arguments, &log::Record) -> String) -> Log {
        self.format_fn = f;
        self
    }

    pub fn simple_format(message_args: &std::fmt::Arguments, _record: &log::Record) -> String {
        format!("{}", message_args)
    }

    pub fn level_format(message_args: &std::fmt::Arguments, record: &log::Record) -> String {
        format!(
            "{:>5} {}",
            record.level().to_string().white().bold(),
            message_args
        )
    }

    pub fn target_format(message_args: &std::fmt::Arguments, record: &log::Record) -> String {
        format!(
            "{:>5} [{}] {}",
            record.level().to_string().white().bold(),
            record.target(),
            message_args
        )
    }

    pub fn init(self) -> Result<()> {
        let level_filter = match &self.level {
            Level::Trace => LevelFilter::Trace,
            Level::Debug => LevelFilter::Debug,
            Level::Info => LevelFilter::Info,
            Level::Warn => LevelFilter::Warn,
            _ => LevelFilter::Error,
        };

        let f = self.format_fn;
        let mut dispatch = Dispatch::new()
            .format(move |out, message, record| {
                let s = f(message, record);
                out.finish(format_args!("{}", &s))
            })
            .level(level_filter)
            .chain(std::io::stdout());

        for (key, value) in self.filters.into_iter() {
            dispatch = dispatch.level_for(key, value);
        }

        dispatch.apply()?;
        Ok(())
    }
}
