use crate::errors::*;
use serde::{Deserialize, Serialize};
use std::fmt::{Display, Formatter};
use std::path::{Path, PathBuf};

#[derive(Serialize, Deserialize, Clone, Debug, Hash, PartialEq, Eq, PartialOrd, Ord)]

/// Metadata about the environment
pub struct Env {
    /// The username
    pub username: String,
    /// The hostname,
    pub hostname: String,
    /// The distro
    pub distro: String,
    /// The real name for the user
    pub realname: String,
    /// The device name
    pub devicename: String,
}

impl Env {
    /// uses the PATH environment variable to search
    /// for a filename matching the specified name.
    /// if a matching filename is not found, it
    /// will check for the existence of name.exe
    /// and name.bat
    ///
    /// Example
    ///
    /// ```
    /// use reef::Env;
    /// let git_path = Env::which("git").unwrap();
    /// assert!(git_path.exists());
    /// ```
    pub fn which(name: &str) -> Result<PathBuf> {
        super::paths::Paths::which(name)
    }

    /// extracts the text following the shebang #!
    ///
    /// Example
    ///
    /// given a file
    /// with the contents:
    ///
    /// #!C:/Ruby26-x64/bin/ruby.exe
    ///
    /// the shebang method will return C:/Ruby26-x64/bin/ruby.exe
    ///
    /// ```
    /// use reef::Env;
    /// use std::env;
    /// let path = std::env::temp_dir().join("test.rb");
    /// std::fs::write(&path,b"#!C:/Ruby26-x64/bin/ruby.exe")?;
    /// let target = Env::shebang(&path).unwrap();
    /// assert_eq!("C:/Ruby26-x64/bin/ruby.exe",target);
    /// # Ok::<(), std::io::Error>(())
    /// ```
    pub fn shebang(path: &Path) -> Result<String> {
        super::paths::Paths::shebang(path)
    }
}

impl Default for Env {
    fn default() -> Self {
        Env {
            username: whoami::username(),
            hostname: whoami::hostname(),
            distro: whoami::distro(),
            realname: whoami::realname(),
            devicename: whoami::devicename(),
        }
    }
}

impl Display for Env {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        write!(f, "username {}", self.username)
    }
}

#[cfg(test)]
#[test]
fn which() {
    let env = Env::default();
    if env.distro == "Windows 10" {
        match super::Env::which("rake") {
            Ok(rake_path) => {
                assert!(rake_path.exists());
                //let (name, args) = parse_command("rake default");
                //assert!(name.contains("ruby"), "{} did not contains 'ruby'", name);
                //assert_eq!(2, args.len(), "args: {:?}", args)
            }
            Err(_) => assert!(false, "which(\"rake\") failed"),
        }
    }
}
