use serde::{Deserialize, Serialize};
use std::convert::TryFrom;

// TODO: Consider removing this struct altogether
#[derive(Serialize, Deserialize, Default, Clone, Debug, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub struct Duration {
    secs: i32,
    nanos: i32,
}

impl Duration {
    pub fn new(secs: i32, nanos: i32) -> Self {
        Duration {
            secs: secs,
            nanos: nanos,
        }
    }
    pub fn secs(&self) -> u32 {
        u32::try_from(self.secs).unwrap()
    }

    pub fn nanos(&self) -> u32 {
        u32::try_from(self.nanos).unwrap()
    }

    pub fn from_std(
        duration: &std::time::Duration,
    ) -> std::result::Result<Duration, Box<dyn std::error::Error>> {
        let total_nanos = duration.as_nanos();
        let secs = total_nanos / 1_000_000_000;
        let nanos = total_nanos - (secs * 1_000_000_000);
        Ok(Duration {
            secs: i32::try_from(secs)?,
            nanos: i32::try_from(nanos)?,
        })
    }

    pub fn to_std(&self) -> std::result::Result<std::time::Duration, Box<dyn std::error::Error>> {
        Ok(std::time::Duration::new(
            u64::try_from(self.secs)?,
            u32::try_from(self.nanos)?,
        ))
    }
}

pub fn format(duration: &std::time::Duration) -> String {
    match duration.as_micros() > 999 {
        // greater than 999 microseconds
        false => format!("{} \u{00b5}s", duration.as_micros()),
        true => match duration.as_millis() > 999 {
            // greater that 999 milliseconds
            false => format!("{} ms", duration.as_millis()),
            true => match duration.as_secs() > 90 {
                // greater than 90 seconds
                false => format!("{} s", duration.as_secs()),
                true => match duration.as_secs() > 299 {
                    // greater than 5 minutes
                    false => {
                        format!("{} m", duration.as_secs() / 60)
                        /*
                        let whole_minutes = duration.as_secs() / 60;
                        let remaining_secs = duration.as_secs() - (whole_minutes * 60);
                        let fremaining = remaining_secs as f32;
                        let fractional_minutes = fremaining / 6.0;
                        format!("{}.{}m", duration.as_secs() / 60, fractional_minutes)*/
                    }
                    true => match duration.as_secs() > 5940 {
                        // greater than 99 minutes
                        false => format!("{} m", duration.as_secs() / 60),
                        true => match duration.as_secs() > 86400 {
                            // greater than 24 hours
                            false => format!("{} h", duration.as_secs() / 3600),
                            true => match duration.as_secs() > 5184000 {
                                // greater than 60 days
                                false => format!("{} d", duration.as_secs() / 86400),
                                true => format!("{} mo", duration.as_secs() / (86400 * 30)),
                            },
                        },
                    },
                },
            },
        },
    }
}

#[cfg(test)]
use crate::duration;

#[test]
fn usage() {}

#[test]
fn format_test() {
    assert_eq!(
        "15 µs",
        duration::format(&std::time::Duration::from_micros(15))
    );

    let d = Duration::from_std(&std::time::Duration::new(70_000, 4321)).unwrap();
    assert_eq!(70_000, d.secs());
    assert_eq!(4321, d.nanos());
    let d2 = d.to_std().unwrap();
    assert_eq!(70_000, d2.as_secs());
}
