use crate::{Command, Result};
use serde::{Deserialize, Serialize};
use std::fmt::{Display, Formatter};
use std::path::Path;

use std::vec::Vec;

#[derive(Serialize, Deserialize, Clone, Debug, Hash, Default, PartialEq, Eq, PartialOrd, Ord)]
pub struct Commands {
    pub history: Vec<Command>,
}

impl Commands {
    pub fn new() -> Self {
        Commands {
            history: Vec::new(),
        }
    }

    pub fn exec(&mut self, command_strs: &[&str]) -> Result<()> {
        for (_i, c) in command_strs.iter().enumerate() {
            let command = Command::new(c).directory(std::env::temp_dir()).exec()?;
            self.history.push(command.clone());
        }
        Ok(())
    }

    pub fn exec_in<P: AsRef<Path>>(&mut self, command_str: &str, path: P) -> Result<Command> {
        let command = Command::new(command_str).directory(&path).exec()?;
        self.history.push(command.clone());
        Ok(command)
    }

    pub fn save<P: AsRef<Path>>(&self, path: P) -> Result<()> {
        let json = serde_json::to_string_pretty(&self)?;
        std::fs::write(path.as_ref(), &json)?;
        Ok(())
    }

    pub fn open<P: AsRef<Path>>(path: P) -> Result<Commands> {
        if path.as_ref().exists() {
            let json = std::fs::read_to_string(path.as_ref())?;
            match serde_json::from_str::<Commands>(&json) {
                Ok(data) => {
                    return Ok(data);
                }
                Err(_) => {
                    std::fs::remove_file(path)?;
                    return Ok(Commands::new());
                }
            }
        } else {
            Ok(Commands::new())
        }
    }

    pub fn find(&self, pattern: &str) -> Vec<Command> {
        let mut results = Vec::new();
        for c in &self.history {
            if c.matches(pattern) {
                results.push(c.clone());
            }
        }
        results
    }

    pub fn remove(&mut self, pattern: &str) {
        for i in 0..self.history.len() - 1 {
            if self.history[i].matches(pattern) {
                self.history.remove(i);
            }
        }
    }
}

impl Display for Commands {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        for (_i, c) in self.history.iter().enumerate() {
            write!(f, "\n{}", c)?;
        }
        write!(f, "")
    }
}

#[cfg(test)]
#[test]
fn usage() {
    let mut commands = Commands::new();
    commands.exec(&["git version", "cargo version"]).unwrap();
    //println!("{}",commands);
}
