use crate::{Env, Error, ErrorKind, Paths, Result, Status};
use chrono::prelude::*;
use colored::*;
use log::{info, trace, warn};
use serde::{Deserialize, Serialize};
use std::cmp::Ordering;
use std::fmt::{Display, Formatter};
use std::path::{Path, PathBuf};
use std::time::Instant;

#[derive(Serialize, Deserialize, Clone, Debug, Hash, Default, PartialEq, Eq)]
/// Metadata about a std::process::Command
pub struct Command {
    /// The working directory
    pub working_dir: PathBuf,
    /// The command exe filename
    pub exe_filename: PathBuf,
    /// The command name
    pub name: String,
    /// The command arguments
    pub args: Vec<String>,
    /// The standard output text
    pub stdout: String,
    /// The standard error text
    pub stderr: String,
    /// Indication of success or failure
    pub status: Status,
    /// The exit code of the process
    pub exit_code: i32,
    /// The duration of the command execution
    pub duration: std::time::Duration,
    /// The timeout duration for the command execution
    pub timeout: std::time::Duration,
    /// The start time of the command execution
    pub start: String,
    /// Environment metadata
    pub env: Env,
    /// String Tags
    pub tags: Vec<String>,
    pub spinner: bool,
}

impl Command {
    pub fn new(command: &str) -> Command {
        let (name, args) = crate::text::parse_command(command);
        let mut c = Command::default();
        c.timeout = std::time::Duration::new(300, 0);
        c.name = name;
        c.args = args;
        c.working_dir = match std::env::current_dir() {
            Ok(dir) => dir.to_path_buf(),
            Err(_) => std::env::temp_dir().to_path_buf(),
        };
        c.spinner = false;
        c
    }

    pub fn arg(&self, arg: &str) -> Command {
        let mut c = self.clone();
        c.args.push(arg.to_string());
        c
    }

    pub fn timeout_nanos(mut self, nanos: u32) -> Command {
        self.timeout = std::time::Duration::new(0, nanos);
        self
    }
    pub fn timeout_secs(mut self, secs: u64) -> Command {
        self.timeout = std::time::Duration::new(secs, 0);
        self
    }

    pub fn timeout_mins(mut self, minutes: u64) -> Command {
        self.timeout = std::time::Duration::new(minutes * 60, 0);
        self
    }

    pub fn directory<P: AsRef<Path>>(mut self, path: P) -> Command {
        self.working_dir = path.as_ref().to_path_buf();
        self
    }

    pub fn spinner(mut self, show: bool) -> Command {
        self.spinner = show;
        self
    }

    pub fn tag(mut self, tag: &str) -> Command {
        self.tags.push(tag.to_string());
        self
    }

    pub fn open<P: AsRef<Path>>(path: P) -> Result<Command> {
        let json = std::fs::read_to_string(path.as_ref())?;
        Ok(serde_json::from_str::<Command>(&json)?)
    }

    pub fn save<P: AsRef<Path>>(&self, path: P) -> Result<()> {
        let json = serde_json::to_string_pretty(&self)?;
        std::fs::write(path.as_ref(), &json)?;
        Ok(())
    }

    pub fn start_utc(&self) -> DateTime<Utc> {
        match &self.start.parse::<DateTime<Utc>>() {
            Ok(dt) => *dt,
            Err(_) => Utc::now(),
        }
    }

    pub fn start_local(&self) -> DateTime<Local> {
        self.start_utc().with_timezone(&Local)
    }

    pub fn end_utc(&self) -> DateTime<Utc> {
        let chrono_duration = chrono::Duration::from_std(self.duration).unwrap();
        let end_time = self
            .start_utc()
            .checked_add_signed(chrono_duration)
            .unwrap();
        end_time
    }

    pub fn end_local(&self) -> DateTime<Local> {
        self.end_utc().with_timezone(&Local)
    }

    pub fn age(&self) -> std::time::Duration {
        let chrono_duration = Utc::now()
            .naive_utc()
            .signed_duration_since(self.start_utc().naive_utc());
        match chrono_duration.to_std() {
            Ok(duration) => duration,
            Err(_) => std::time::Duration::new(0, 0),
        }
    }

    pub fn get_tag_value(&self, name: &str) -> Option<String> {
        for tag in &self.tags {
            let prop_name = format!("{}=", name);
            if tag.contains(&prop_name) {
                return Some(tag.replace(&prop_name, ""));
            }
        }
        None
    }
    pub fn duration_string(&self) -> String {
        crate::duration::format(&self.duration)
    }

    pub fn summary(&self) -> String {
        format!(
            "{} {} {} {}",
            &self.status.symbol(),
            &self.duration_string(),
            &self.name,
            &self.args.join(" ")
        )
    }

    pub fn matches(&self, pattern: &str) -> bool {
        if format!("{}", self.working_dir.display()).contains(pattern) {
            return true;
        }
        if self.stdout.contains(pattern) {
            return true;
        }

        if self.stderr.contains(pattern) {
            return true;
        }
        for tag in &self.tags {
            if tag.contains(pattern) {
                return true;
            }
        }
        false
    }

    pub fn details(&self) -> String {
        format!(
            "{}\ndirectory: {}\nstart time: {}\noutput:\n{}\n{}",
            self.summary(),
            self.working_dir.display(),
            self.start_local().to_string(),
            self.stdout,
            self.stderr
        )
    }

    pub fn expect_exit_code(&self, exit_code: i32) -> Result<Command> {
        if self.exit_code == exit_code {
            Ok(self.clone())
        } else {
            Err(Error::from_kind(ErrorKind::IncorrectExitCode(
                exit_code,
                format!("{}", &self),
            )))
        }
    }

    fn get_std(&self) -> Result<std::process::Command> {
        let mut dir = PathBuf::new();
        dir.push(&self.working_dir);
        if !dir.exists() {
            return Err(Error::from_kind(ErrorKind::PathDoesNotExist(dir)));
        }

        let exe_filename = Env::which(&self.name)?;
        let mut process_command = std::process::Command::new(&exe_filename);
        process_command.current_dir(&dir);
        let ext = Paths::extension(&exe_filename);
        if ext == "bat" || ext == "cmd" {
            process_command =
                std::process::Command::new(&format!("{}", Paths::which("cmd").unwrap().display()));
            process_command.current_dir(&dir);

            let mut args: Vec<String> = Vec::new();
            args.push("/C".to_string());
            args.push(format!("{}", exe_filename.display()));
            args.append(&mut self.args.clone());
            process_command.args(crate::text::get_vec_osstring(&args.clone()));
        } else {
            process_command.current_dir(&dir);
            process_command.args(crate::text::get_vec_osstring(&self.args.clone()));
        }
        Ok(process_command)
    }
    // TODO: add timeout support: https://docs.rs/wait-timeout/0.2.0/wait_timeout/
    pub fn exec(&self) -> Result<Command> {
        if !&self.working_dir.exists() {
            return Err(Error::from_kind(ErrorKind::PathDoesNotExist(
                self.working_dir.to_path_buf(),
            )));
        }

        let mut process_command = self.get_std()?;

        let mut command = self.clone();
        command.exe_filename = Paths::which(&command.name)?;
        let now = Instant::now();
        command.env = Env::default();

        let mut dir = PathBuf::new();
        dir.push(&self.working_dir);

        /*
        let command_path = Env::which(&command.name)?;
        command.exe_filename = Paths::which(&command.name)?; // format!("{}", Paths::which(&command.name).unwrap().display());
        let mut process_command = std::process::Command::new(&command.exe_filename);
        command.start = Utc::now().to_string();
        process_command.current_dir(&command.working_dir);
        debug!("directory={}", &command.working_dir.display());
        debug!("command={} {}", &command.name, &command.args.join(" "));
        let ext = Paths::extension(&command_path);
        if ext == "bat" || ext == "cmd" {
            command.exe_filename = Paths::which("cmd")?;
            //command.path = format!("{}", Paths::which("cmd").unwrap().display());
            process_command =
                std::process::Command::new(&format!("{}", Paths::which("cmd").unwrap().display()));
            process_command.current_dir(&command.working_dir);

            let mut args: Vec<String> = Vec::new();
            args.push("/C".to_string());
            args.push(format!("{}", command_path.display()));
            args.append(&mut command.args.clone());
            process_command.args(crate::text::get_vec_osstring(&args.clone()));
        } else {
            process_command.current_dir(&command.working_dir);
            process_command.args(crate::text::get_vec_osstring(&command.args.clone()));
        }*/

        trace!("timeout={:?}", self.timeout);

        /*
        let mut child = process_command.spawn()?;
        let mut _spinner: Option<Spinner> = None;
        if self.spinner {
            _spinner = Some(Spinner::new(&format!(
                "executing {} {}...",
                command.name,
                command.args.join(" ")
            )));
        }*/
        // match child.wait_timeout(self.timeout)? {
        //     Some(status) => {
        //         trace!("status={}", status);
        //         command.status = match status.success() {
        //             true => Status::Ok,
        //             false => Status::Error,
        //         };
        command.start = Utc::now().to_string();
        let output = process_command.output()?;
        command.exit_code = match output.status.code() {
            Some(code) => code,
            None => 0,
        };
        trace!("exit_code={}", command.exit_code);
        command.stdout = match std::str::from_utf8(&output.stdout) {
            Ok(text) => text.trim().to_string(),
            Err(_) => "* error parsing stdout *".to_string(),
        };
        trace!("stdout={}", command.stdout);

        command.stderr = match std::str::from_utf8(&output.stderr) {
            Ok(text) => text.trim().to_string(),
            Err(_) => "* error parsing stderr *".to_string(),
        };
        trace!("stderr={}", command.stderr);

        command.duration = now.elapsed().clone(); // Duration::from_std(&now.elapsed()).unwrap();
        command.status = match output.status.success() {
            true => {
                info!("{}", &command.summary());
                Status::Ok
            }
            false => {
                warn!("{}", &command.summary());
                Status::Error
            }
        };
        //    }
        //    None => {
        //        warn!(
        //            "timeout of {} exceeded",
        //            &crate::duration::format(&command.timeout)
        //        );
        //        child.kill()?;
        //        child.wait()?;
        //        command.stderr = "timed out".to_string();
        //        command.exit_code = 99;
        //    }
        //}

        Ok(command)
    }
}

impl Display for Command {
    fn fmt(&self, f: &mut Formatter) -> std::fmt::Result {
        let mut output_lines = Vec::new();
        if self.stdout.len() > 0 {
            output_lines.append(&mut self.stdout.split('\n').collect());
        }
        if self.stderr.len() > 0 {
            output_lines.append(&mut self.stderr.split('\n').collect());
        }
        write!(
            f,
            "{}{}{}{} {}\n",
            &format!("{}", &self.working_dir.display()).white().bold(),
            ">".white().bold(),
            "".clear(),
            &self.name.green(),
            &self.args.join(" "),
        )?;
        for line in output_lines {
            writeln!(f, " {} {}", "".cyan(), line.clear())?;
        }
        Ok(())
    }
}
impl std::error::Error for Command {}

impl PartialOrd for Command {
    fn partial_cmp(&self, other: &Command) -> Option<Ordering> {
        Some(self.cmp(other))
    }
}

impl Ord for Command {
    fn cmp(&self, other: &Command) -> Ordering {
        self.start_utc().cmp(&other.start_utc())
    }
}

#[cfg(test)]
#[test]
fn usage() {
    let git_version = Command::new("git --version").exec().unwrap();
    assert_eq!(git_version.status, Status::Ok);
    assert!(git_version.stdout.contains("git version"));

    let json = serde_json::to_string_pretty(&git_version).unwrap();
    let gv2 = serde_json::from_str::<Command>(&json).unwrap();
    assert!(gv2.working_dir.exists(), "working_dir exists");
    assert!(gv2.exe_filename.exists(), "exe_filename exists");
    assert_eq!(gv2.status, Status::Ok, "status is Ok");
    assert!(
        gv2.stdout.contains("git version"),
        "stdout contains 'git version'"
    );
}
