
pub use std::sync::Arc;

pub struct Arrangement {
    name:     &'static str,
    schedule: Vec<SequencerEvent>,
}

/**
  here, our sequencer simply contains a sequence
  of pending events
  */
#[derive(Default)]
pub struct Sequencer {
    pending_events: Vec<Box<SequencerEvent>>,
}

impl Sequencer {
    pub fn schedule_event(&mut self, ev: SequencerEvent) {
        self.pending_events.push(Box::new(ev))
    }
    pub fn run(&mut self) {

        let mut bar = 0;

        for ev in self.pending_events.iter_mut() {
            match &**ev {
                SequencerEvent::Routine( routine ) => {
                    (routine)()
                },
                SequencerEvent::PlayEvent{ 
                    routine, start, bars
                } => {
                    while bar < *bars {
                        if bar == 0 {
                            if let Some(start) = start {
                                (start)();
                            }
                        }
                        println!("bar: {:?}", bar);
                        (routine)();
                        bar += 1;
                    }
                    bar = 0;
                }
            }
        }

        self.pending_events.clear();
    }
}

#[derive(Clone)]
pub enum SequencerEvent {

    /**
      nothing special -- just run this function
      when called
      */
    Routine(Arc<dyn Fn() -> ()>),

    /**
      run this function for *bars* number of
      bars, optionally calling *start* before
      running the function for the first bar
      */
    PlayEvent {
        routine: fn() -> (),
        start:   Option<fn() -> ()>,
        bars:    i32,
    }
}

impl Sequencer {

    fn schedule_arrangement(&mut self, x: Arrangement) {

        let name = x.name.clone();

        println!("scheduling: {:?}",name);

        self.schedule_event(SequencerEvent::Routine(
                Arc::new(move || {
                    println!("activate arrangement {:?}", name);
                })));

        for ev in x.schedule.iter() {
            self.schedule_event(ev.clone());
        }
    }
}

/**
  this is the first example arrangement,
  three scene changes for N bars each
  */
fn arr0() -> Arrangement {
    Arrangement {
        name: "first take",
        schedule: vec!{
            SequencerEvent::PlayEvent {
                start:   Some(|| {
                    println!("--------------------------[Scene Change: intro]");
                }),
                routine: intro,
                bars: 4,
            },
            SequencerEvent::PlayEvent {
                start:   Some(|| {
                    println!("--------------------------[Scene Change: second_build]");
                }),
                routine: second_build,
                bars: 2,
            },
            SequencerEvent::PlayEvent {
                start:   Some(|| {
                    println!("--------------------------[Scene Change: wash_outro]");
                }),
                routine: wash_outro,
                bars: 4,
            },
        }
    }
}

/**
  this function generates and example
  arrangement, named "take two"
  */
fn arr1() -> Arrangement {
    Arrangement {
        name: "take two",
        schedule: vec!{
            SequencerEvent::PlayEvent {
                start:   Some(|| {
                    println!("--------------------------[Scene Change: tight_outro]");
                }),
                routine: tight_outro,
                bars: 8,
            },
        }
    }
}

///placeholders
#[derive(Default)] pub struct RedzoneBass               {} 
#[derive(Default)] pub struct RedzoneClap               {} 
#[derive(Default)] pub struct KickDrumSample            {} 
#[derive(Default)] pub struct SurgeSynthesizer          {} 
#[derive(Default)] pub struct RedzoneMasterChannel      {} 
#[derive(Default)] pub struct RedzoneSynthGlue          {} 
#[derive(Default)] pub struct RedzoneDrumsGlue          {} 
#[derive(Default)] pub struct RedzoneLPF                {} 
#[derive(Default)] pub struct RedzoneHPF                {} 
#[derive(Default)] pub struct RedzoneWashReverb         {} 
#[derive(Default)] pub struct RedzoneRoutingState       {} 
#[derive(Default)] pub struct EnergyConservationState   {} 
#[derive(Default)] pub struct Tuner                     {} 
#[derive(Default)] pub struct TimeStretcher             {} 
#[derive(Default)] pub struct RedzoneKick  { sample: KickDrumSample, }
#[derive(Default)] pub struct RedzoneSynth { engine: SurgeSynthesizer, }

/**
  |not all of these are filed in. it is all just
  |a placeholder.  what it means it that we have,
  |for example, some trait "BassIntro" which can
  |be implemented by any song or object in our
  |world.
  |
  |for instance, the kick drum could implement
  |this to  configure itself in a manner which
  |corresponds to a "BassIntro".  Then, when we
  |call redzone.play_at(RedzoneMarker::BassIntro),
  |the kick drum configures itself appropriately
  |and executes.
  |
  |furthermore, BassIntro could be implemented for
  |the song Redzone itself, or indeed for anything
  |else which might want to know what to do if it
  |gets asked to do "BassIntro"
  */
impl BassIntro      for Redzone        { fn bass_intro(&mut self)       { } }
impl BassOutro      for Redzone        { fn bass_outro(&mut self)       { } }
impl FirstBassMain  for Redzone        { fn first_bass_main(&mut self)  { } }
impl FirstBuild     for Redzone        { fn first_build(&mut self)      { } }
impl SecondBassMain for Redzone        { fn second_bass_main(&mut self) { } }
impl SecondBuild    for Redzone        { fn second_build(&mut self)     { self.play() } }
impl TightIntro     for Redzone        { fn tight_intro(&mut self)      { } }
impl TightOutro     for Redzone        { fn tight_outro(&mut self)      { println!("redzone tight outro"); }}
impl WashIntro      for Redzone        { fn wash_intro(&mut self)       { println!("redzone wash intro"); self.kick.wash_intro(); } }
impl WashOutro      for Redzone        { fn wash_outro(&mut self)       { println!("redzone wash outro"); } }

impl Play           for RedzoneSynth   { fn play(&mut self)             { } }
impl Play           for KickDrumSample { fn play(&mut self)             { } }
impl Play           for RedzoneBass    { fn play(&mut self)             { println!("playing redzone bass"); } }
impl Play           for RedzoneClap    { fn play(&mut self)             { println!("playing redzone clap"); } }
impl Play           for RedzoneKick    { fn play(&mut self)             { println!("playing redzone kick"); self.sample.play(); } }
impl WashIntro      for RedzoneBass    { fn wash_intro(&mut self)       { println!("redzone bass [wash intro configuration]"); } }
impl WashIntro      for RedzoneClap    { fn wash_intro(&mut self)       { println!("redzone clap [wash intro configuration]"); } }
impl WashIntro      for RedzoneKick    { fn wash_intro(&mut self)       { println!("redzone kick [wash intro configuration]"); } }

///---------------------
#[derive(Default)]
pub struct Redzone {
    pub kick:  RedzoneKick,
    pub clap:  RedzoneClap,
    pub bass:  RedzoneBass,
    pub synth: RedzoneSynth,

    master_channel:              RedzoneMasterChannel,
    synth_glue:                  RedzoneSynthGlue,
    drums_glue:                  RedzoneDrumsGlue,
    global_lpf:                  RedzoneLPF,
    global_hpf:                  RedzoneHPF,
    reverb_wash:                 RedzoneWashReverb,
    routing_state:               RedzoneRoutingState,

    momentum_conservation_state: MomentumConservationState,
    energy_conservation_state:   EnergyConservationState,
    tuner:                       Tuner,
    time_stretcher:              TimeStretcher,
}

#[derive(Default)]
pub struct MomentumConservationState {
    previous: RingBuffer<f32,128>,
}

///------------------
pub struct RingBuffer<T: Default + Copy,const SIZE: usize> {
    values: [T; SIZE],
}

impl<T: Default + Copy, const SIZE: usize> Default for RingBuffer<T,SIZE> {
    fn default() -> Self {
        Self {
            values: [T::default(); SIZE],
        }
    }
}

///------------------
pub enum RedzoneMarker {
    WashIntro,
    TightIntro,
    BassIntro,
    WashOutro,
    TightOutro,
    BassOutro,
    FirstBassMain,
    SecondBassMain,
    FirstBuild,
    SecondBuild,
}

impl Play for Redzone {
    fn play(&mut self) {
        self.kick.play();
        self.clap.play();
        self.bass.play();
        self.synth.play();
    }
}

impl MomentumConservation for Redzone {
    fn ensure_conserve_momentum(&mut self) {

    }
}

impl PlayAt for Redzone {

    type Marker = RedzoneMarker;

    fn play_at(&mut self, marker: Self::Marker) {

        self.ensure_conserve_momentum();

        match marker {
            RedzoneMarker::BassIntro      => self.bass_intro(),
            RedzoneMarker::BassOutro      => self.bass_outro(),
            RedzoneMarker::FirstBassMain  => self.first_bass_main(),
            RedzoneMarker::FirstBuild     => self.first_build(),
            RedzoneMarker::SecondBassMain => self.second_bass_main(),
            RedzoneMarker::SecondBuild    => self.second_build(),
            RedzoneMarker::TightIntro     => self.tight_intro(),
            RedzoneMarker::TightOutro     => self.tight_outro(),
            RedzoneMarker::WashIntro      => self.wash_intro(),
            RedzoneMarker::WashOutro      => self.wash_outro(),
        }
    }
}

///--------------------------------------
pub fn intro() {
    let mut rz = Redzone::default();
    rz.play_at(RedzoneMarker::WashIntro);
}

pub fn second_build() {
    let mut rz = Redzone::default();
    rz.play_at(RedzoneMarker::SecondBuild);
}

pub fn wash_outro() {
    let mut rz = Redzone::default();
    rz.play_at(RedzoneMarker::WashOutro);
}

pub fn tight_outro() {
    let mut rz = Redzone::default();
    rz.play_at(RedzoneMarker::TightOutro);
}

///--------------------------------------
pub trait Play {
    fn play(&mut self);
}

pub trait PlayAt {
    type Marker;
    fn play_at(&mut self, marker: Self::Marker);
}

pub trait WashIntro      { fn wash_intro(&mut self);       } 
pub trait WashOutro      { fn wash_outro(&mut self);       } 
pub trait TightIntro     { fn tight_intro(&mut self);      } 
pub trait TightOutro     { fn tight_outro(&mut self);      } 
pub trait BassIntro      { fn bass_intro(&mut self);       } 
pub trait BassOutro      { fn bass_outro(&mut self);       } 
pub trait FirstBassMain  { fn first_bass_main(&mut self);  } 
pub trait SecondBassMain { fn second_bass_main(&mut self); } 
pub trait FirstBuild     { fn first_build(&mut self);      } 
pub trait SecondBuild    { fn second_build(&mut self);     } 

pub trait MomentumConservation {
    fn ensure_conserve_momentum(&mut self);
}

pub fn play_main() {
    let mut sq = Sequencer::default();
    sq.schedule_arrangement(arr0());
    sq.schedule_arrangement(arr1());
    sq.run();
}
