#![allow(clippy::module_name_repetitions, clippy::must_use_candidate)]

#[cfg(feature = "serde_support")]
use serde::{Deserialize, Serialize};
#[cfg(feature = "serde_support")]
use serde_bytes::{ByteBuf, Bytes};
use std::{
    any::type_name,
    fmt::{Debug, Display},
    hash::Hash,
    ops::{Deref, DerefMut},
};
#[cfg(feature = "zeroize")]
use zeroize::Zeroize;

/// Redacting wrapper type for a const-generic array
///
/// Provides redaction of contents in `Debug` and `Display`, as well as serde serialization and
/// deserialization, if the `serde` feature is enabled.
///
/// This wrapper is distinct from [`RedactedArray`](crate::RedactedArray) to allow for the use of
/// `serde_bytes`'s wrappers for efficient serialization.
#[derive(Hash, PartialEq, Eq, PartialOrd, Ord)]
pub struct RedactedBytes<const N: usize> {
    items: [u8; N],
}

// Redacting implementations
impl<const N: usize> Debug for RedactedBytes<N> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<u8>())
    }
}

impl<const N: usize> Display for RedactedBytes<N> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<u8>())
    }
}

// Deref and std::convert implementations
impl<const N: usize> Deref for RedactedBytes<N> {
    type Target = [u8];

    fn deref(&self) -> &Self::Target {
        &self.items
    }
}

impl<const N: usize> DerefMut for RedactedBytes<N> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.items
    }
}

impl<const N: usize> AsRef<[u8]> for RedactedBytes<N> {
    fn as_ref(&self) -> &[u8] {
        &self.items
    }
}

impl<const N: usize> RedactedBytes<N> {
    /// Create a new `RedactedBytes` from the provided array
    pub fn new(items: [u8; N]) -> Self {
        Self { items }
    }
    /// Convert back to the inner array
    pub fn into_inner(self) -> [u8; N] {
        self.items
    }
}

impl<const N: usize> From<[u8; N]> for RedactedBytes<N> {
    fn from(items: [u8; N]) -> Self {
        Self { items }
    }
}

// Pass through methods
impl<const N: usize> Default for RedactedBytes<N> {
    fn default() -> Self {
        Self { items: [0_u8; N] }
    }
}

impl<const N: usize> Clone for RedactedBytes<N> {
    fn clone(&self) -> Self {
        Self { items: self.items }
    }
}

impl<const N: usize> Copy for RedactedBytes<N> {}

// Optional serde tie ins
#[cfg(feature = "serde_support")]
impl<const N: usize> Serialize for RedactedBytes<N> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let items: &Bytes = Bytes::new(self.as_ref());
        items.serialize(serializer)
    }
}

#[cfg(feature = "serde_support")]
impl<'de, const N: usize> Deserialize<'de> for RedactedBytes<N> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let items: ByteBuf = Deserialize::deserialize(deserializer)?;
        let len = items.len();
        let items: [u8; N] = items
            .into_vec()
            .try_into()
            .map_err(|_| serde::de::Error::invalid_length(len, &"Wrong length"))?;
        Ok(RedactedBytes { items })
    }
}

#[cfg(feature = "zeroize")]
impl<const N: usize> Zeroize for RedactedBytes<N> {
    fn zeroize(&mut self) {
        self.items.zeroize();
    }
}
