#[cfg(feature = "serde_support")]
use serde::{Deserialize, Serialize};
use std::{
    any::type_name,
    fmt::{Debug, Display},
    ops::{Deref, DerefMut},
};
#[cfg(feature = "zeroize")]
use zeroize::Zeroize;

/// Basic redacting wrapper type
pub struct Redacted<T> {
    item: T,
}

// Redacting implementations
impl<T> Debug for Redacted<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<T>())
    }
}

impl<T> Display for Redacted<T> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<T>())
    }
}

// Deref and std::convert implementations
impl<T> Deref for Redacted<T> {
    type Target = T;

    fn deref(&self) -> &Self::Target {
        &self.item
    }
}

impl<T> DerefMut for Redacted<T> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.item
    }
}

impl<T> AsRef<T> for Redacted<T> {
    fn as_ref(&self) -> &T {
        &self.item
    }
}

impl<T> AsMut<T> for Redacted<T> {
    fn as_mut(&mut self) -> &mut T {
        &mut self.item
    }
}

impl<T> From<T> for Redacted<T> {
    fn from(item: T) -> Self {
        Self { item }
    }
}

impl<T> Redacted<T> {
    /// Creates a new `Redacted` from the given value
    pub fn new(item: T) -> Self {
        Self { item }
    }
    /// Converts this `Redacted` into its inner type
    pub fn into_inner(self) -> T {
        self.item
    }
}

// Passthrough implementations
impl<T: Default> Default for Redacted<T> {
    fn default() -> Self {
        Self {
            item: Default::default(),
        }
    }
}

impl<T: Clone> Clone for Redacted<T> {
    fn clone(&self) -> Self {
        Self {
            item: self.item.clone(),
        }
    }
}

impl<T: Copy> Copy for Redacted<T> {}

impl<T: AsRef<[u8]>> AsRef<[u8]> for Redacted<T> {
    fn as_ref(&self) -> &[u8] {
        self.item.as_ref()
    }
}

impl<T: AsMut<[u8]>> AsMut<[u8]> for Redacted<T> {
    fn as_mut(&mut self) -> &mut [u8] {
        self.item.as_mut()
    }
}

#[cfg(feature = "serde_support")]
impl<T: Serialize> Serialize for Redacted<T> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.item.serialize(serializer)
    }
}

#[cfg(feature = "serde_support")]
impl<'de, T: Deserialize<'de>> Deserialize<'de> for Redacted<T> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let item: T = Deserialize::deserialize(deserializer)?;
        Ok(Redacted { item })
    }
}

#[cfg(feature = "zeroize")]
impl<T: Zeroize> Zeroize for Redacted<T> {
    fn zeroize(&mut self) {
        self.item.zeroize();
    }
}
