#![allow(clippy::module_name_repetitions)]

#[cfg(feature = "serde_support")]
use serde::{Deserialize, Serialize};
use std::{
    any::type_name,
    fmt::{Debug, Display},
    ops::{Deref, DerefMut},
};
#[cfg(feature = "zeroize")]
use zeroize::Zeroize;

/// Redacting wrapper type for a const-generic array
///
/// Provides redaction of contents in `Debug` and `Display`, as well as serde serialization and
/// deserialization, if the `serde` feature is enabled.
pub struct RedactedArray<T, const N: usize> {
    items: [T; N],
}

// Redacting implementations
impl<T, const N: usize> Debug for RedactedArray<T, N> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<T>())
    }
}

impl<T, const N: usize> Display for RedactedArray<T, N> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}: [REDACTED]", type_name::<T>())
    }
}

// Deref and std::convert implementations
impl<T, const N: usize> Deref for RedactedArray<T, N> {
    type Target = [T];

    fn deref(&self) -> &Self::Target {
        &self.items
    }
}

impl<T, const N: usize> DerefMut for RedactedArray<T, N> {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.items
    }
}

impl<T, const N: usize> AsRef<[T]> for RedactedArray<T, N> {
    fn as_ref(&self) -> &[T] {
        &self.items
    }
}

impl<T, const N: usize> RedactedArray<T, N> {
    /// Create a new `RedactedArray` from the provided array
    pub fn new(items: [T; N]) -> Self {
        Self { items }
    }
    /// Convert back to the inner array
    pub fn into_inner(self) -> [T; N] {
        self.items
    }
}

impl<T, const N: usize> From<[T; N]> for RedactedArray<T, N> {
    fn from(items: [T; N]) -> Self {
        Self { items }
    }
}

// Pass through methods
impl<T: Default, const N: usize> Default for RedactedArray<T, N>
where
    [T; N]: Default,
{
    fn default() -> Self {
        Self {
            items: Default::default(),
        }
    }
}

impl<T: Clone, const N: usize> Clone for RedactedArray<T, N> {
    fn clone(&self) -> Self {
        Self {
            items: self.items.clone(),
        }
    }
}

impl<T: Copy, const N: usize> Copy for RedactedArray<T, N> {}

// Optional serde tie ins
#[cfg(feature = "serde_support")]
impl<T: Serialize, const N: usize> Serialize for RedactedArray<T, N> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let items: &[T] = self.as_ref();
        items.serialize(serializer)
    }
}

#[cfg(feature = "serde_support")]
impl<'de, T: Deserialize<'de>, const N: usize> Deserialize<'de> for RedactedArray<T, N> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let items: Vec<T> = Deserialize::deserialize(deserializer)?;
        let len = items.len();
        let items: [T; N] = items
            .try_into()
            .map_err(|_| serde::de::Error::invalid_length(len, &"Wrong length"))?;
        Ok(RedactedArray { items })
    }
}

#[cfg(feature = "zeroize")]
impl<T: Zeroize, const N: usize> Zeroize for RedactedArray<T, N> {
    fn zeroize(&mut self) {
        self.items.zeroize();
    }
}
